#include "stdafx.h"
#include "Bitmap.h"
#include "FileIO.h"

SDIBitmap::~SDIBitmap()
{
	if (m_pHeader != NULL)
		Delete();
}

SDIBitmap::SDIBitmap(const SDIBitmap &bmp)
{
	m_pHeader = NULL;
	if (bmp.m_pHeader != NULL)
	{
		Create(bmp.GetWidth(), bmp.GetHeight(), bmp.GetBitCount());
		memcpy(m_pbyBits, bmp.m_pbyBits, GetBitmapSize());
	}
}

SDIBitmap &SDIBitmap::operator =(const SDIBitmap &bmp)
{
	if (this == &bmp)
		return *this;

	if (m_pHeader != NULL && bmp.m_pHeader != NULL &&
		m_pHeader->biWidth == bmp.m_pHeader->biWidth &&
		m_pHeader->biHeight == bmp.m_pHeader->biHeight &&
		m_pHeader->biBitCount == bmp.m_pHeader->biBitCount)
		memcpy(m_pbyBits, bmp.m_pbyBits, GetBitmapSize());
	else
	{
		if (m_pHeader != NULL)
			Delete();
		if (bmp.m_pHeader != NULL)
		{
			Create(bmp.GetWidth(), bmp.GetHeight(), bmp.GetBitCount());
			memcpy(m_pbyBits, bmp.m_pbyBits, GetBitmapSize());
		}
	}
	return *this;
}

void SDIBitmap::Create(int nWidth, int nHeight, int nBitCount)
{
	ASSERT(m_pHeader == NULL);
	ASSERT(nBitCount == 16 || nBitCount == 24 || nBitCount == 32);

	m_nPitch = ALIGN(4 * 8, nWidth * nBitCount) / 8;
	m_pHeader = (BITMAPINFOHEADER *) new BYTE[sizeof(*m_pHeader) + m_nPitch * nHeight];
	m_pbyBits = (BYTE *) m_pHeader + sizeof(*m_pHeader);

    m_pHeader->biSize = sizeof(*m_pHeader);
    m_pHeader->biWidth = nWidth;
    m_pHeader->biHeight = nHeight;
    m_pHeader->biPlanes = 1;
    m_pHeader->biBitCount = nBitCount;
    m_pHeader->biCompression = 0;
    m_pHeader->biSizeImage = m_nPitch * nHeight;
    m_pHeader->biXPelsPerMeter = 0;
    m_pHeader->biYPelsPerMeter = 0;
    m_pHeader->biClrUsed = 0;
    m_pHeader->biClrImportant = 0;
}

void SDIBitmap::Delete()
{
	ASSERT(m_pHeader != NULL);
	delete [] m_pHeader;
	m_pHeader = NULL;
}

void SDIBitmap::Fill(COLORREF cr)
{
	ASSERT(m_pHeader != NULL);
	int x, y;
	BYTE byRed, byGreen, byBlue;
	BYTE *pbyBits, *pbyLine = m_pbyBits;
	switch (GetBitCount())
	{
	case 32:
		byRed = GetRValue(cr);
		byGreen = GetGValue(cr);
		byBlue = GetBValue(cr);
		for (y = 0;y < GetHeight();y++)
		{
			pbyBits = pbyLine;
			for (x = 0;x < GetWidth();x++)
			{
				*pbyBits++ = byBlue;
				*pbyBits++ = byGreen;
				*pbyBits++ = byRed;
				*pbyBits++ = 0;
			}
			pbyLine += m_nPitch;
		}
		break;
	case 24:
		byRed = GetRValue(cr);
		byGreen = GetGValue(cr);
		byBlue = GetBValue(cr);
		for (y = 0;y < GetHeight();y++)
		{
			pbyBits = pbyLine;
			for (x = 0;x < GetWidth();x++)
			{
				*pbyBits++ = byBlue;
				*pbyBits++ = byGreen;
				*pbyBits++ = byRed;
			}
			pbyLine += m_nPitch;
		}
		break;
	default:
		ASSERT(FALSE);
	}
}

void SDIBitmap::Blt(int xDest, int yDest, const SDIBitmap &bmpSrc, int xSrc, int ySrc, int nWidth, int nHeight)
{
	ASSERT(m_pHeader != NULL);
	ASSERT(bmpSrc.m_pHeader != NULL);
	ASSERT(GetBitCount() == bmpSrc.GetBitCount());
	ASSERT(inrange(xSrc, 0, bmpSrc.GetWidth()));
	ASSERT(inrange(ySrc, 0, bmpSrc.GetHeight()));
	ASSERT(inrange(xSrc + nWidth, 0, bmpSrc.GetWidth() + 1));
	ASSERT(inrange(ySrc + nHeight, 0, bmpSrc.GetHeight() + 1));

	if (xDest < 0)
	{
		nWidth += xDest;
		xSrc -= xDest;
		xDest = 0;
	}
	if (xDest + nWidth > GetWidth())
		nWidth = GetWidth() - xDest;
	if (nWidth <= 0)
		return;

	if (yDest < 0)
	{
		nHeight += yDest;
		ySrc -= yDest;
		yDest = 0;
	}
	if (yDest + nHeight > GetHeight())
		nHeight = GetHeight() - yDest;
	if (nHeight <= 0)
		return;

	const BYTE *pbySrc = bmpSrc.GetBits(xSrc, ySrc);
	BYTE *pbyDest = GetBits(xDest, yDest);
	for (int n = 0;n < nHeight;n++)
	{
		memcpy(pbyDest, pbySrc, nWidth * GetBitCount() / 8);
		pbySrc -= bmpSrc.m_nPitch;
		pbyDest -= m_nPitch;
	}
}

void SDIBitmap::Convert(const SDIBitmap &bmpSrc)
{
	ASSERT(m_pHeader != NULL);
	ASSERT(bmpSrc.m_pHeader != NULL);
	ASSERT(GetWidth() == bmpSrc.GetWidth());
	ASSERT(GetHeight() == bmpSrc.GetHeight());

	if (GetBitCount() == bmpSrc.GetBitCount())
	{
		memcpy(m_pbyBits, bmpSrc.m_pbyBits, GetBitmapSize());
		return;
	}

	int x, y;
	const BYTE *pbySrc, *pbySrcLine = bmpSrc.m_pbyBits;
	BYTE *pbyDest, *pbyDestLine = m_pbyBits;
	switch (GetBitCount())
	{
	case 32:
		switch (bmpSrc.GetBitCount())
		{
		case 24:
			for (y = 0;y < GetHeight();y++)
			{
				pbySrc = pbySrcLine;
				pbyDest = pbyDestLine;
				for (x = 0;x < GetWidth();x++)
				{
					*pbyDest++ = *pbySrc++;
					*pbyDest++ = *pbySrc++;
					*pbyDest++ = *pbySrc++;
					*pbyDest++ = 0;
				}
				pbySrcLine += bmpSrc.m_nPitch;
				pbyDestLine += m_nPitch;
				pbySrc = pbySrcLine;
				pbyDest = pbyDestLine;
			}
			break;
		case 16:
			for (y = 0;y < GetHeight();y++)
			{
				pbySrc = pbySrcLine;
				pbyDest = pbyDestLine;
				for (x = 0;x < GetWidth();x++)
				{
					*pbyDest++ = (*(WORD *) pbySrc & 0x1f) << 3;
					*pbyDest++ = (*(WORD *) pbySrc >> 5 & 0x1f) << 3;
					*pbyDest++ = (*(WORD *) pbySrc >> 10 & 0x1f) << 3;
					*pbyDest++ = 0;
					pbySrc += 2;
				}
				pbySrcLine += bmpSrc.m_nPitch;
				pbyDestLine += m_nPitch;
				pbySrc = pbySrcLine;
				pbyDest = pbyDestLine;
			}
			break;
		default:
			ASSERT(FALSE);
		}
		break;
	default:
		//  32Ʈ .
		ASSERT(FALSE);
	}
}

void SDIBitmap::LoadBitmapFile(LPCTSTR lpszFileName)
{
	ASSERT(m_pHeader == NULL);
	ASSERT(IsValidString(lpszFileName));

	SFile file(lpszFileName, FM_READ);

	BITMAPFILEHEADER m_bfh;
	file.Read(&m_bfh, sizeof(m_bfh));
	if (m_bfh.bfType != MAKEWORD('B', 'M'))
		file.ThrowBadFormatException();

	BITMAPINFOHEADER m_bih;
	file.Read(&m_bih, sizeof(m_bih));
	if (m_bih.biSize != sizeof(m_bih))
		file.ThrowBadFormatException();

	Create(m_bih.biWidth, m_bih.biHeight, m_bih.biBitCount);
	file.Read(m_pbyBits, GetBitmapSize());
}

void SDIBSection::Create(int nWidth, int nHeight, int nBitCount)
{
	ASSERT(m_pHeader == NULL);
	ASSERT(nBitCount == 16 || nBitCount == 24 || nBitCount == 32);

	m_pHeader = (BITMAPINFOHEADER *) new BYTE[sizeof(*m_pHeader)];
    m_pHeader->biSize = sizeof(*m_pHeader);
    m_pHeader->biWidth = nWidth;
    m_pHeader->biHeight = nHeight;
    m_pHeader->biPlanes = 1;
    m_pHeader->biBitCount = nBitCount;
    m_pHeader->biCompression = 0;
    m_pHeader->biSizeImage = m_nPitch * nHeight;
    m_pHeader->biXPelsPerMeter = 0;
    m_pHeader->biYPelsPerMeter = 0;
    m_pHeader->biClrUsed = 0;
    m_pHeader->biClrImportant = 0;
	m_nPitch = ALIGN(4 * 8, nWidth * nBitCount) / 8;

	m_hBitmap = CreateDIBSection(NULL, (BITMAPINFO *) m_pHeader, DIB_RGB_COLORS, (LPVOID *) &m_pbyBits, NULL, 0);
}

void SDIBSection::Delete()
{
	ASSERT(m_pHeader != NULL);
	DeleteObject(m_hBitmap);
	delete [] m_pHeader;
	m_pHeader = NULL;
	m_hBitmap = NULL;
}
