#ifndef __BITMAP_H__
#define __BITMAP_H__

#include "Commdef.h"

class SDIBitmap
{
public:
	SDIBitmap();
	SDIBitmap(const SDIBitmap &bmp);
	virtual ~SDIBitmap();

	SDIBitmap &operator =(const SDIBitmap &bmp);

	virtual void Create(int nWidth, int nHeight, int nBitCount);
	virtual void Delete();

public:
	int GetWidth() const;
	int GetHeight() const;
	int GetBitCount() const;
	int GetPitch() const;
	int GetBitmapSize() const;

	const BITMAPINFOHEADER *GetHeader() const;
	BYTE *GetBits();
	const BYTE *GetBits() const;
	BYTE *GetBits(int x, int y);
	const BYTE *GetBits(int x, int y) const;

public:
	void DrawBitmap(HDC hDC, int xDest, int yDest) const;
	void DrawBitmap(HDC hDC, int xDest, int yDest, int xSrc, int ySrc, int nWidth, int nHeight) const;
	void StretchBitmap(HDC hDC, int xDest, int yDest, int nDestWidth, int nDestHeight) const;
	void StretchBitmap(HDC hDC, int xDest, int yDest, int nDestWidth, int nDestHeight, int xSrc, int ySrc, int nSrcWidth, int nSrcHeight) const;

	void Fill(COLORREF cr);
	void Blt(int xDest, int yDest, const SDIBitmap &bmpSrc);
	void Blt(int xDest, int yDest, const SDIBitmap &bmpSrc, int xSrc, int ySrc, int nWidth, int nHeight);
	void Convert(const SDIBitmap &bmpSrc);

	void LoadBitmapFile(LPCTSTR lpszFileName);

protected:
	BITMAPINFOHEADER *m_pHeader;
	BYTE *m_pbyBits;
	int m_nPitch;
};

class SDIBSection : public SDIBitmap
{
public:
	SDIBSection();
	SDIBSection(const SDIBitmap &bmp);
	SDIBSection(const SDIBSection &bmp);

	SDIBSection &operator =(const SDIBitmap &bmp);
	SDIBSection &operator =(const SDIBSection &bmp);

	virtual void Create(int nWidth, int nHeight, int nBitCount);
	virtual void Delete();

public:
	operator HBITMAP();

private:
	HBITMAP m_hBitmap;
};

inline SDIBitmap::SDIBitmap()
{
	m_pHeader = NULL;
}

inline int SDIBitmap::GetWidth() const
{
	ASSERT(m_pHeader != NULL);
	return m_pHeader->biWidth;
}

inline int SDIBitmap::GetHeight() const
{
	ASSERT(m_pHeader != NULL);
	return m_pHeader->biHeight;
}

inline int SDIBitmap::GetBitCount() const
{
	ASSERT(m_pHeader != NULL);
	return m_pHeader->biBitCount;
}

inline int SDIBitmap::GetPitch() const
{
	ASSERT(m_pHeader != NULL);
	return m_nPitch;
}

inline int SDIBitmap::GetBitmapSize() const
{
	ASSERT(m_pHeader != NULL);
	return m_nPitch * m_pHeader->biHeight;
}

inline const BITMAPINFOHEADER *SDIBitmap::GetHeader() const
{
	return m_pHeader;
}

inline BYTE *SDIBitmap::GetBits()
{
	ASSERT(m_pHeader != NULL);
	return m_pbyBits;
}

inline const BYTE *SDIBitmap::GetBits() const
{
	ASSERT(m_pHeader != NULL);
	return m_pbyBits;
}

inline BYTE *SDIBitmap::GetBits(int x, int y)
{
	ASSERT(m_pHeader != NULL);
	ASSERT(inrange(x, 0, m_pHeader->biWidth));
	ASSERT(inrange(y, 0, m_pHeader->biHeight));
	return m_pbyBits + (m_nPitch * (m_pHeader->biHeight - y - 1)) + (x * m_pHeader->biBitCount / 8);
}

inline const BYTE *SDIBitmap::GetBits(int x, int y) const
{
	ASSERT(m_pHeader != NULL);
	return m_pbyBits + (m_nPitch * (m_pHeader->biHeight - y - 1)) + (x * m_pHeader->biBitCount / 8);
}

inline void SDIBitmap::DrawBitmap(HDC hDC, int xDest, int yDest) const
{
	ASSERT(m_pHeader != NULL);
	SetDIBitsToDevice(hDC, xDest, yDest, m_pHeader->biWidth, m_pHeader->biHeight, 0, 0,
		0, m_pHeader->biHeight, m_pbyBits, (BITMAPINFO *) m_pHeader, DIB_RGB_COLORS);
}

inline void SDIBitmap::DrawBitmap(HDC hDC, int xDest, int yDest, int xSrc, int ySrc, int nWidth, int nHeight) const
{
	ASSERT(m_pHeader != NULL);
	ASSERT(inrange(xSrc, 0, GetWidth()));
	ASSERT(inrange(ySrc, 0, GetHeight()));
	ASSERT(inrange(xSrc + nWidth, 0, GetWidth() + 1));
	ASSERT(inrange(ySrc + nHeight, 0, GetHeight() + 1));
	SetDIBitsToDevice(hDC, xDest, yDest, nWidth, nHeight, xSrc, 0,
		0, nHeight, m_pbyBits + m_nPitch * (m_pHeader->biHeight - ySrc - nHeight),
		(BITMAPINFO *) m_pHeader, DIB_RGB_COLORS);
}

inline void SDIBitmap::StretchBitmap(HDC hDC, int xDest, int yDest, int nDestWidth, int nDestHeight) const
{
	ASSERT(m_pHeader != NULL);
	StretchDIBits(hDC, xDest, yDest, nDestWidth, nDestHeight,
		0, 0, GetWidth(), GetHeight(), m_pbyBits,
		(BITMAPINFO *) m_pHeader, DIB_RGB_COLORS, SRCCOPY);
}

inline void SDIBitmap::StretchBitmap(HDC hDC, int xDest, int yDest, int nDestWidth, int nDestHeight, int xSrc, int ySrc, int nSrcWidth, int nSrcHeight) const
{
	ASSERT(m_pHeader != NULL);
	ASSERT(inrange(xSrc, 0, GetWidth()));
	ASSERT(inrange(ySrc, 0, GetHeight()));
	ASSERT(inrange(xSrc + nSrcWidth, 0, GetWidth() + 1));
	ASSERT(inrange(ySrc + nSrcHeight, 0, GetHeight() + 1));
	StretchDIBits(hDC, xDest, yDest, nDestWidth, nDestHeight,
		xSrc, ySrc, nSrcWidth, nSrcHeight, m_pbyBits,
		(BITMAPINFO *) m_pHeader, DIB_RGB_COLORS, SRCCOPY);
}

inline void SDIBitmap::Blt(int xDest, int yDest, const SDIBitmap &bmpSrc)
{
	ASSERT(m_pHeader != NULL);
	ASSERT(bmpSrc.m_pHeader != NULL);
	Blt(xDest, yDest, bmpSrc, 0, 0, bmpSrc.GetWidth(), bmpSrc.GetHeight());
}

inline SDIBSection::SDIBSection()
{
}

inline SDIBSection::SDIBSection(const SDIBitmap &bmp)
	: SDIBitmap(bmp)
{
}

inline SDIBSection::SDIBSection(const SDIBSection &bmp)
	: SDIBitmap(bmp)
{
}

inline SDIBSection &SDIBSection::operator =(const SDIBitmap &bmp)
{
	SDIBitmap::operator =(bmp);
	return *this;
}

inline SDIBSection &SDIBSection::operator =(const SDIBSection &bmp)
{
	SDIBitmap::operator =(bmp);
	return *this;
}

inline SDIBSection::operator HBITMAP()
{
	ASSERT(m_pHeader != NULL);
	return m_hBitmap;
}

#endif // __BITMAP_H__
