

#include "stdafx.h"
#include "Stream.h"
#include "FileIO.h"
#include "FindFile.h"

BOOL STextFileStream::ReadString(LPSTR lpsz, int nMax, BOOL *pbEndLine /*= NULL*/)
{
	ASSERT(nMax > 0);
	ASSERT(IsValidAddress(lpsz, nMax));

	UINT nRead = Read(lpsz, sizeof(*lpsz) * (nMax - 1));
	if (nRead == 0)
		return FALSE;
	for (int i = 0;i < (int) nRead;i++)
		if (lpsz[i] == NULL)
			lpsz[i] = ' ';
		else if (lpsz[i] == '\n')
		{
			Seek(-(int) (nRead - i - 1), FS_CURRENT);
			if (i > 0 && lpsz[i - 1] == '\r')
				i--;
			lpsz[i] = NULL;
			if (pbEndLine != NULL)
				*pbEndLine = TRUE;
			return TRUE;
		}

	char ch;
	if (Read(&ch, sizeof(ch)) == sizeof(ch))
		if (ch == '\n')
		{
			if (i > 0 && lpsz[i - 1] == '\r')
				i--;
			lpsz[i] = NULL;
			if (pbEndLine != NULL)
				*pbEndLine = TRUE;
			return TRUE;
		}
		else if (ch == '\r')
			if ((nRead = Read(&ch, sizeof(ch))) == sizeof(ch) && ch == '\n')
			{
				lpsz[i] = NULL;
				if (pbEndLine != NULL)
					*pbEndLine = TRUE;
				return TRUE;
			}
			else
				Seek(-(int) (sizeof(ch) + nRead), FS_CURRENT);
		else
			Seek(-(int) sizeof(ch), FS_CURRENT);
	lpsz[i] = NULL;
	if (pbEndLine != NULL)
		*pbEndLine = FALSE;
	return TRUE;
}

BOOL STextFileStream::ReadString(LPWSTR lpsz, int nMax, BOOL *pbEndLine /*= NULL*/)
{
	ASSERT(nMax > 0);
	ASSERT(IsValidAddress(lpsz, nMax));

	UINT nRead = Read(lpsz, sizeof(*lpsz) * (nMax - 1));
	if (nRead == 0)
		return FALSE;
	for (int i = 0;i < (int) nRead;i++)
		if (lpsz[i] == NULL)
			lpsz[i] = ' ';
		else if (lpsz[i] == '\n')
		{
			Seek(-(int) (nRead - i - 1), FS_CURRENT);
			if (i > 0 && lpsz[i - 1] == '\r')
				i--;
			lpsz[i] = NULL;
			if (pbEndLine != NULL)
				*pbEndLine = TRUE;
			return TRUE;
		}

	WCHAR ch;
	if ((nRead = Read(&ch, sizeof(ch))) == sizeof(ch))
		if (ch == '\n')
		{
			if (i > 0 && lpsz[i - 1] == '\r')
				i--;
			lpsz[i] = NULL;
			if (pbEndLine != NULL)
				*pbEndLine = TRUE;
			return TRUE;
		}
		else if (ch == '\r')
			if ((nRead = Read(&ch, sizeof(ch))) == sizeof(ch) && ch == '\n')
			{
				lpsz[i] = NULL;
				if (pbEndLine != NULL)
					*pbEndLine = TRUE;
				return TRUE;
			}
			else
				Seek(-(int) (sizeof(ch) + nRead), FS_CURRENT);
		else
			Seek(-(int) sizeof(ch), FS_CURRENT);
	else
		Seek(-(int) nRead, FS_CURRENT);
	lpsz[i] = NULL;
	if (pbEndLine != NULL)
		*pbEndLine = FALSE;
	return TRUE;
}

BOOL STextFileStream::ReadChar(char *pch)
{
	ASSERT(IsValidAddress(pch));
	return Read(pch, sizeof(*pch)) == sizeof(*pch);
}

BOOL STextFileStream::ReadChar(WCHAR *pch)
{
	ASSERT(IsValidAddress(pch));
	return Read(pch, sizeof(*pch)) == sizeof(*pch);
}

void STextFileStream::WriteString(LPCSTR lpsz)
{
	ASSERT(IsValidString(lpsz));
	Write(lpsz, sizeof(*lpsz) * strlen(lpsz));
}

void STextFileStream::WriteString(LPCWSTR lpsz)
{
	ASSERT(IsValidString(lpsz));
	Write(lpsz, sizeof(*lpsz) * wcslen(lpsz));
}

void STextFileStream::WriteChar(char ch)
{
	Write(&ch, sizeof(ch));
}

void STextFileStream::WriteChar(WCHAR ch)
{
	Write(&ch, sizeof(ch));
}

SFile::~SFile()
{
	if (m_hFile != INVALID_HANDLE_VALUE)
		Close();
}

void SFile::Open(LPCTSTR lpszFileName, UINT nMode)
{
	ASSERT(m_hFile == INVALID_HANDLE_VALUE);
	ASSERT(IsValidString(lpszFileName));

	DWORD dwDesiredAccess = 0, dwShareMode = 0, dwCreationDisposition;
	if (nMode & FM_READ)
		dwDesiredAccess |= GENERIC_READ;
	if (nMode & FM_WRITE)
		dwDesiredAccess |= GENERIC_WRITE;
	if (nMode & FM_SHARE_READ)
		dwShareMode |= FILE_SHARE_READ;
	if (nMode & FM_SHARE_WRITE)
		dwShareMode |= FILE_SHARE_READ;
	if (nMode & FM_CREATE)
		dwCreationDisposition = CREATE_ALWAYS;
	else
		dwCreationDisposition = OPEN_EXISTING;

	m_hFile = CreateFile(lpszFileName, dwDesiredAccess, dwShareMode, NULL, dwCreationDisposition, FILE_ATTRIBUTE_NORMAL, NULL);
	if (m_hFile == INVALID_HANDLE_VALUE)
	{
		BOOL bExisting = IsFileExisting(lpszFileName);
		if (nMode & FM_CREATE)
			ThrowFileException(bExisting ? FE_CREATEEXIST : FE_CREATE, lpszFileName);
		if (bExisting)
			ThrowFileException(nMode & FM_WRITE ? FE_OPENWRITE : FE_OPENREAD, lpszFileName);
		ThrowFileException(FE_FILENOTFOUND, lpszFileName);
	}
	SetFileName(lpszFileName);
}

void SFile::Close()
{
	ASSERT(m_hFile != INVALID_HANDLE_VALUE);
	CloseHandle(m_hFile);
	m_hFile = INVALID_HANDLE_VALUE;
	SetFileName(_T(""));
}

void SFile::Flush()
{
	ASSERT(m_hFile != INVALID_HANDLE_VALUE);
	if (!FlushFileBuffers(m_hFile))
		ThrowException(FE_WRITE);
}

UINT SFile::Read(LPVOID lpBuf, UINT nSize, BOOL bThrowEOF /*= TRUE*/)
{
	ASSERT(m_hFile != INVALID_HANDLE_VALUE);
	ASSERT(IsValidAddress(lpBuf, nSize));

	DWORD dwRead;
	if (!ReadFile(m_hFile, lpBuf, nSize, &dwRead, NULL))
		ThrowException(FE_READ);
	if (bThrowEOF && nSize != dwRead)
		ThrowException(FE_ENDOFFILE);
	return dwRead;
}

void SFile::Write(LPCVOID lpBuf, UINT nSize)
{
	ASSERT(m_hFile != INVALID_HANDLE_VALUE);
	ASSERT(IsValidReadOnlyAddress(lpBuf, nSize));

	DWORD dwWritten;
	if (!WriteFile(m_hFile, lpBuf, nSize, &dwWritten, NULL))
		ThrowException(FE_WRITE);
	if (nSize != dwWritten)
		ThrowException(FE_DISKFULL);
}

DWORD SFile::Seek(long lOffset, UINT nFrom)
{
	ASSERT(m_hFile != INVALID_HANDLE_VALUE);
	switch (nFrom)
	{
	case FS_BEGIN:
		nFrom = FILE_BEGIN;
		break;
	case FS_CURRENT:
		nFrom = FILE_CURRENT;
		break;
	case FS_END:
		nFrom = FILE_END;
		break;
	default:
		ASSERT(FALSE);
	}
	return SetFilePointer(m_hFile, lOffset, NULL, nFrom);
}

DWORD SFile::GetPosition() const
{
	ASSERT(m_hFile != INVALID_HANDLE_VALUE);
	return SetFilePointer(m_hFile, 0, NULL, FILE_CURRENT);
}

DWORD SFile::GetLength() const
{
	ASSERT(m_hFile != INVALID_HANDLE_VALUE);
	DWORD dwCurrent = SetFilePointer(m_hFile, 0, NULL, FILE_CURRENT);
	DWORD dwLength = SetFilePointer(m_hFile, 0, NULL, FILE_END);
	VERIFY(dwCurrent == SetFilePointer(m_hFile, dwCurrent, NULL, FILE_BEGIN));
	return dwLength;
}

void SFile::SetLength(DWORD dwLength)
{
	ASSERT(m_hFile != INVALID_HANDLE_VALUE);
	SetFilePointer(m_hFile, dwLength, NULL, FILE_BEGIN);
	if (!SetEndOfFile(m_hFile))
	{
		if (GetLastError() == ERROR_HANDLE_DISK_FULL)
			ThrowException(FE_DISKFULL);
		ThrowException(FE_WRITE);
	}
}

void SFile::ThrowException(UINT nCause) const
{
	ASSERT(m_hFile != INVALID_HANDLE_VALUE);
	ThrowFileException(nCause, GetFileName());
}

void SFile::ThrowBadFormatException() const
{
	ASSERT(m_hFile != INVALID_HANDLE_VALUE);
	ThrowFileException(FE_BADFORMAT, GetFileName());
}

UINT STextFile::Read(LPVOID lpBuf, UINT nSize)
{
	return m_file.Read(lpBuf, nSize, FALSE);
}

void STextFile::Write(LPCVOID lpBuf, UINT nSize)
{
	m_file.Write(lpBuf, nSize);
}

void STextFile::Open(LPCTSTR lpszFileName, UINT nMode)
{
	m_file.Open(lpszFileName, nMode);
}

void STextFile::Close()
{
	m_file.Close();
}

void STextFile::Flush()
{
	m_file.Flush();
}

BOOL STextFile::ReadString(LPSTR lpsz, int nMax, BOOL *pbEndLine /*= NULL*/)
{
	return STextFileStream::ReadString(lpsz, nMax, pbEndLine);
}

BOOL STextFile::ReadString(LPWSTR lpsz, int nMax, BOOL *pbEndLine /*= NULL*/)
{
	ASSERT(nMax > 0);
	ASSERT(IsValidAddress(lpsz, nMax));

	for (int i = 0;i < nMax - 1;i++)
	{
		if (!ReadChar(lpsz + i))
		{
			if (i == 0)
				return FALSE;
			lpsz[i] = NULL;
			if (pbEndLine != NULL)
				*pbEndLine = FALSE;
			return TRUE;
		}
		if (lpsz[i] == NULL)
			lpsz[i] = ' ';
		else if (lpsz[i] == '\n')
		{
			if (i > 0 && lpsz[i - 1] == '\r')
				i--;
			lpsz[i] = NULL;
			if (pbEndLine != NULL)
				*pbEndLine = TRUE;
			return TRUE;
		}
	}

	char ch;
	if (ReadChar(&ch))
		if (ch == '\n')
		{
			if (i > 0 && lpsz[i - 1] == '\r')
				i--;
			lpsz[i] = NULL;
			if (pbEndLine != NULL)
				*pbEndLine = TRUE;
			return TRUE;
		}
		else if (ch == '\r' && ReadChar(&ch))
			if (ch == '\n')
			{
				lpsz[i] = NULL;
				if (pbEndLine != NULL)
					*pbEndLine = TRUE;
				return TRUE;
			}
			else
				Seek(-(int) (sizeof(ch) * 2), FS_CURRENT);
		else
			Seek(-(int) sizeof(ch), FS_CURRENT);
	lpsz[i] = NULL;
	if (pbEndLine != NULL)
		*pbEndLine = FALSE;
	return TRUE;
}

BOOL STextFile::ReadChar(char *pch)
{
	return STextFileStream::ReadChar(pch);
}

BOOL STextFile::ReadChar(WCHAR *pch)
{
	ASSERT(IsValidAddress(pch));
	char s[2];
	if (!ReadChar(s))
		return FALSE;
	int nRead = 1;
	if (IsDBCSLeadByte(s[0]) && !ReadChar(s + nRead++))
		return FALSE;
	MultiByteToWideChar(CP_ACP, 0, s, nRead, pch, 1);
	return TRUE;
}

void STextFile::WriteString(LPCSTR lpsz)
{
	STextFileStream::WriteString(lpsz);
}

void STextFile::WriteString(LPCWSTR lpsz)
{
	int nLength = WideCharToMultiByte(CP_ACP, 0, lpsz, -1, NULL, 0, NULL, NULL);
	SArrHeap<char> psz = new char[nLength];
	WideCharToMultiByte(CP_ACP, 0, lpsz, -1, psz, nLength, NULL, NULL);
	WriteString(psz);
}

void STextFile::WriteChar(char ch)
{
	STextFileStream::WriteChar(ch);
}

void STextFile::WriteChar(WCHAR ch)
{
	char s[2];
	int nLength = WideCharToMultiByte(CP_ACP, 0, &ch, 1, s, 2, NULL, NULL);
	Write(s, sizeof(*s) * nLength);
}

DWORD STextFile::Seek(long lOffset, UINT nFrom)
{
	return m_file.Seek(lOffset, nFrom);
}

DWORD STextFile::GetPosition() const
{
	return m_file.GetPosition();
}

DWORD STextFile::GetLength() const
{
	return m_file.GetLength();
}

void STextFile::SetLength(DWORD dwLength)
{
	m_file.SetLength(dwLength);
}

void STextFile::ThrowException(UINT nCause) const
{
	m_file.ThrowException(nCause);
}

SFileException::SFileException(UINT nCause, LPCTSTR lpszFileName)
{
	ASSERT(IsValidString(lpszFileName));
	m_nCause = nCause;
	lstrcpyn(m_szFileName, lpszFileName, countof(m_szFileName));
}

BOOL SFileException::GetErrorMessage(LPTSTR lpszError, UINT nMaxError, PUINT pnHelpContext /*= NULL*/)
{
	ASSERT(IsValidAddress(lpszError, nMaxError));
	TCHAR szBuffer[MAX_PATH + 256];
	switch (m_nCause)
	{
	case FE_CREATE:
		lstrcpy(szBuffer, m_szFileName);
		lstrcat(szBuffer, _T("    ϴ.\nũ   Ǿ ִ,  ̸ Ȯ ȮϽʽÿ."));
		break;
	case FE_CREATEEXIST:
		lstrcpy(szBuffer, m_szFileName);
		lstrcat(szBuffer, _T("    ϴ.\nũ   Ǿ ִ,  ̹ , б   ȮϽʽÿ."));
		break;
	case FE_OPENREAD:
		lstrcpy(szBuffer, m_szFileName);
		lstrcat(szBuffer, _T("    ϴ.\n̹  ȮϽʽÿ."));
		break;
	case FE_OPENWRITE:
		lstrcpy(szBuffer, m_szFileName);
		lstrcat(szBuffer, _T("    ϴ.\n̹ , б   ȮϽʽÿ."));
		break;
	case FE_FILENOTFOUND:
		lstrcpy(szBuffer, m_szFileName);
		lstrcat(szBuffer, _T("  ã  ϴ."));
		break;
	case FE_READ:
		lstrcpy(szBuffer, m_szFileName);
		lstrcat(szBuffer, _T(" Ͽ д   ϴ."));
		break;
	case FE_WRITE:
		lstrcpy(szBuffer, m_szFileName);
		lstrcat(szBuffer, _T(" Ͽ    ϴ."));
		break;
	case FE_DISKFULL:
		lstrcpy(szBuffer, _T("ũ   մϴ."));
		break;
	case FE_ENDOFFILE:
	case FE_SEEK:
	case FE_BADFORMAT:
		lstrcpy(szBuffer, m_szFileName);
		lstrcat(szBuffer, _T("   ٸų ջ Դϴ."));
		break;
	default:
		ASSERT(FALSE);
		return FALSE;
	}
	lstrcpyn(lpszError, szBuffer, nMaxError);
	return TRUE;
}

void ThrowFileException(UINT nCause, LPCTSTR lpszFileName)
{
	ASSERT(IsValidString(lpszFileName));
	THROW(new SFileException(nCause, lpszFileName));
}
