

#ifndef __LINKEDLIST_H__
#define __LINKEDLIST_H__

#include "Commdef.h"

typedef struct { } *PNODE;

template <class T, class A = const T &, class F = const T &>
class SList
{
public:
	SList()
	{
		m_pHead = m_pTail = NULL;
		m_nCount = 0;
	}

	SList(const SList<T, A, F> &list)
	{
		SList<T, A, F> listTemp;
		for (SNode *pNode = list.m_pHead;pNode;pNode = pNode->pNext)
			listTemp.AddTail(pNode->t);
		m_pHead = listTemp.m_pHead;
		m_pTail = listTemp.m_pTail;
		m_nCount = listTemp.m_nCount;
		listTemp.m_pHead = listTemp.m_pTail = NULL;
		listTemp.m_nCount = 0;
	}

	virtual ~SList()
	{
		for (SNode *pNode = m_pHead;pNode;)
		{
			SNode *pNext = pNode->pNext;
			delete pNode;
			pNode = pNext;
		}
	}

// Attributes
public:
	int GetCount() const
	{
		return m_nCount;
	}

	BOOL IsEmpty() const
	{
		return m_pHead == NULL;
	}

	T &PeekHead()
	{
		ASSERT(m_pHead != NULL);
		return m_pHead->t;
	}

	T &PeekTail()
	{
		ASSERT(m_pTail != NULL);
		return m_pTail->t;
	}

	const T &PeekHead() const
	{
		ASSERT(m_pHead != NULL);
		return m_pHead->t;
	}

	const T &PeekTail() const
	{
		ASSERT(m_pTail != NULL);
		return m_pTail->t;
	}

	PNODE GetHead() const
	{
		return (PNODE) m_pHead;
	}

	PNODE GetTail() const
	{
		return (PNODE) m_pTail;
	}

	PNODE GetPrev(PNODE pNode) const
	{
		ASSERT(IsValidNode(pNode));
		return (PNODE) ((SNode *) pNode)->pPrev;
	}

	PNODE GetNext(PNODE pNode) const
	{
		ASSERT(IsValidNode(pNode));
		return (PNODE) ((SNode *) pNode)->pNext;
	}

	T &operator [](PNODE pNode)
	{
		ASSERT(IsValidNode(pNode));
		return ((SNode *) pNode)->t;
	}

	const T &operator [](PNODE pNode) const
	{
		ASSERT(IsValidNode(pNode));
		return ((SNode *) pNode)->t;
	}

// Operations
public:
	PNODE AddHead(A a)
	{
		SNode *pAdd = new SNode(a);
		LinkHead(pAdd);
		return (PNODE) pAdd;
	}

	PNODE AddTail(A a)
	{
		SNode *pAdd = new SNode(a);
		LinkTail(pAdd);
		return (PNODE) pAdd;
	}

	PNODE InsertBefore(PNODE pNode, A a)
	{
		SNode *pInsert = new SNode(a);
		LinkBefore((SNode *) pNode, pInsert);
		return (PNODE) pInsert;
	}

	PNODE InsertAfter(PNODE pNode, A a)
	{
		SNode *pInsert = new SNode(a);
		LinkAfter((SNode *) pNode, pInsert);
		return (PNODE) pInsert;
	}

	T &AddHead()
	{
		SNode *pAdd = new SNode;
		LinkHead(pAdd);
		return pAdd->t;
	}

	T &AddTail()
	{
		SNode *pAdd = new SNode;
		LinkTail(pAdd);
		return pAdd->t;
	}

	T &InsertBefore(PNODE pNode)
	{
		SNode *pInsert = new SNode;
		LinkBefore((SNode *) pNode, pInsert);
		return pInsert->t;
	}

	T &InsertAfter(PNODE pNode)
	{
		SNode *pInsert = new SNode;
		LinkAfter((SNode *) pNode, pInsert);
		return pInsert->t;
	}

	void AddHead(const SList<T, A, F> &list)
	{
		SList<T, A, F> listTemp = list;
		listTemp.MoveToHeadAll(*this);
	}

	void AddTail(const SList<T, A, F> &list)
	{
		SList<T, A, F> listTemp = list;
		listTemp.MoveToTailAll(*this);
	}

	void InsertBefore(PNODE pNode, const SList<T, A, F> &list)
	{
		ASSERT(IsValidNode(pNode));
		SList<T, A, F> listTemp = list;
		listTemp.MoveToBeforeAll(*this, pNode);
	}

	void InsertAfter(PNODE pNode, const SList<T, A, F> &list)
	{
		ASSERT(IsValidNode(pNode));
		SList<T, A, F> listTemp = list;
		listTemp.MoveToAfterAll(*this, pNode);
	}

	void RemoveHead()
	{
		ASSERT(m_pHead->pPrev == NULL);
		SNode *pNode = m_pHead->pNext;
		if (pNode == NULL)
		{
			ASSERT(m_nCount == 1);
			m_pHead = m_pTail = NULL;
		}
		else
		{
			ASSERT(m_nCount > 1);
			pNode->pPrev = NULL;
			delete m_pHead;
			m_pHead = pNode;
		}
		m_nCount--;
	}

	void RemoveTail()
	{
		ASSERT(IsValidNode((PNODE) m_pTail));
		ASSERT(m_pTail->pNext == NULL);
		SNode *pNode = m_pTail->pPrev;
		if (pNode == NULL)
		{
			ASSERT(m_nCount == 1);
			m_pHead = m_pTail = NULL;
		}
		else
		{
			ASSERT(m_nCount > 1);
			pNode->pNext = NULL;
			delete m_pTail;
			m_pTail = pNode;
		}
		m_nCount--;
	}

	void Remove(PNODE pNode)
	{
		Unlink((SNode *) pNode);
		delete (SNode *) pNode;
	}

	void RemoveAll()
	{
		for (SNode *pNode = m_pHead;pNode;)
		{
			SNode *pNext = pNode->pNext;
			delete pNode;
			pNode = pNext;
		}
		m_pHead = m_pTail = NULL;
		m_nCount = 0;
	}

	void MoveToHead(PNODE pCurNode)
	{
		ASSERT(IsValidNode(pCurNode));
		Unlink((SNode *) pCurNode);
		LinkHead((SNode *) pCurNode);
	}

	void MoveToTail(PNODE pCurNode)
	{
		ASSERT(IsValidNode(pCurNode));
		Unlink((SNode *) pCurNode);
		LinkTail((SNode *) pCurNode);
	}

	void MoveToBefore(PNODE pCurNode, PNODE pNode)
	{
		ASSERT(IsValidNode(pCurNode));
		ASSERT(IsValidNode(pNode));
		Unlink((SNode *) pCurNode);
		LinkBefore((SNode *) pNode, (SNode *) pCurNode);
	}

	void MoveToAfter(PNODE pCurNode, PNODE pNode)
	{
		ASSERT(IsValidNode(pCurNode));
		ASSERT(IsValidNode(pNode));
		Unlink((SNode *) pCurNode);
		LinkAfter((SNode *) pNode, (SNode *) pCurNode);
	}

	void MoveToHead(PNODE pCurNode, SList<T, A, F> &list)
	{
		ASSERT(IsValidNode(pCurNode));
		Unlink((SNode *) pCurNode);
		list.LinkHead((SNode *) pCurNode);
	}

	void MoveToTail(PNODE pCurNode, SList<T, A, F> &list)
	{
		ASSERT(IsValidNode(pCurNode));
		Unlink((SNode *) pCurNode);
		list.LinkTail((SNode *) pCurNode);
	}

	void MoveToBefore(PNODE pCurNode, SList<T, A, F> &list, PNODE pNode)
	{
		ASSERT(IsValidNode(pCurNode));
		ASSERT(list.IsValidNode(pNode));
		Unlink((SNode *) pCurNode);
		list.LinkBefore((SNode *) pNode, (SNode *) pCurNode);
	}

	void MoveToAfter(PNODE pCurNode, SList<T, A, F> &list, PNODE pNode)
	{
		ASSERT(IsValidNode(pCurNode));
		ASSERT(list.IsValidNode(pNode));
		Unlink((SNode *) pCurNode);
		list.LinkAfter((SNode *) pNode, (SNode *) pCurNode);
	}

	void MoveToHeadAll(SList<T, A, F> &list)
	{
		if (m_nCount <= 0)
			return;
		if (list.m_nCount <= 0)
		{
			list.m_pHead = m_pHead;
			list.m_pTail = m_pTail;
			list.m_nCount = m_nCount;
		}
		else
		{
			m_pTail->pNext = list.m_pHead;
			list.m_pHead->pPrev = m_pTail;
			list.m_pHead = m_pHead;
			list.m_nCount += m_nCount;
		}
		m_pHead = m_pTail = NULL;
		m_nCount = 0;
	}

	void MoveToTailAll(SList<T, A, F> &list)
	{
		if (m_nCount <= 0)
			return;
		if (list.m_nCount <= 0)
		{
			list.m_pHead = m_pHead;
			list.m_pTail = m_pTail;
			list.m_nCount = m_nCount;
		}
		else
		{
			m_pHead->pPrev = list.m_pTail;
			list.m_pTail->pNext = m_pHead;
			list.m_pTail = m_pTail;
			list.m_nCount += m_nCount;
		}
		m_pHead = m_pTail = NULL;
		m_nCount = 0;
	}

	void MoveToBeforeAll(SList<T, A, F> &list, PNODE pNode)
	{
		ASSERT(list.IsValidNode(pNode));
		if (m_nCount <= 0)
			return;
		m_pHead->pPrev = ((SNode *) pNode)->pPrev;
		m_pTail->pNext = (SNode *) pNode;
		if (((SNode *) pNode)->pPrev)
			((SNode *) pNode)->pPrev->pNext = m_pHead;
		else
			list.m_pHead = m_pHead;
		((SNode *) pNode)->pPrev = m_pTail;
		list.m_nCount += m_nCount;

		m_pHead = m_pTail = NULL;
		m_nCount = 0;
	}

	void MoveToAfterAll(SList<T, A, F> &list, PNODE pNode)
	{
		ASSERT(list.IsValidNode(pNode));
		if (m_nCount <= 0)
			return;
		m_pHead->pPrev = (SNode *) pNode;
		m_pTail->pNext = ((SNode *) pNode)->pNext;
		if (((SNode *) pNode)->pNext)
			((SNode *) pNode)->pNext->pPrev = m_pTail;
		else
			list.m_pTail = m_pTail;
		((SNode *) pNode)->pNext = m_pHead;
		list.m_nCount += m_nCount;

		m_pHead = m_pTail = NULL;
		m_nCount = 0;
	}

	PNODE Find(F f, PNODE pNode = NULL) const
	{
		if(pNode == NULL)
			pNode = (PNODE) m_pHead;
		else
		{
			ASSERT(IsValidNode(pNode));
			pNode = (PNODE) ((SNode *) pNode)->pNext;
		}
		for(;pNode;pNode = (PNODE) ((SNode *) pNode)->pNext)
			if (((SNode *) pNode)->t == f)
				return pNode;
		return NULL;
	}

	PNODE FindIndex(int nIndex) const
	{
		ASSERT(nIndex >= 0 && nIndex < m_nCount);
		SNode *pNode;
		if (nIndex <= m_nCount / 2)
			for (pNode = m_pHead;nIndex > 0;nIndex--, pNode = pNode->pNext)
				NULL;
		else
		{
			nIndex = m_nCount - nIndex - 1;
			for (pNode = m_pTail;nIndex > 0;nIndex--, pNode = pNode->pPrev)
				NULL;
		}
		return (PNODE) pNode;
	}

	void Reverse()
	{
		SNode *pNode = m_pHead;
		m_pHead = m_pTail;
		m_pTail = pNode;
		for (;pNode;pNode = pNode->pPrev)
		{
			SNode *pPrev = pNode->pPrev;
			pNode->pPrev = pNode->pNext;
			pNode->pNext = pPrev;
		}
	}

	void Swap(PNODE pNode1, PNODE pNode2)
	{
		ASSERT(IsValidNode(pNode1));
		ASSERT(IsValidNode(pNode2));

		if (((SNode *) pNode1)->pPrev == (SNode *) pNode2)
		{
			(((SNode *) pNode1)->pNext ? ((SNode *) pNode1)->pNext->pPrev : m_pTail) = ((SNode *) pNode2);
			(((SNode *) pNode2)->pPrev ? ((SNode *) pNode2)->pPrev->pNext : m_pHead) = ((SNode *) pNode1);
			SNode *pTemp = ((SNode *) pNode1)->pNext;
			((SNode *) pNode1)->pPrev = ((SNode *) pNode2)->pPrev;
			((SNode *) pNode1)->pNext = (SNode *) pNode2;
			((SNode *) pNode2)->pPrev = (SNode *) pNode1;
			((SNode *) pNode2)->pNext = pTemp;
		}
		else if (((SNode *) pNode1)->pNext == (SNode *) pNode2)
		{
			(((SNode *) pNode1)->pPrev ? ((SNode *) pNode1)->pPrev->pNext : m_pHead) = ((SNode *) pNode2);
			(((SNode *) pNode2)->pNext ? ((SNode *) pNode2)->pNext->pPrev : m_pTail) = ((SNode *) pNode1);
			SNode *pTemp = ((SNode *) pNode1)->pPrev;
			((SNode *) pNode1)->pPrev = (SNode *) pNode2;
			((SNode *) pNode1)->pNext = ((SNode *) pNode2)->pNext;
			((SNode *) pNode2)->pPrev = pTemp;
			((SNode *) pNode2)->pNext = (SNode *) pNode1;
		}
		else
		{
			(((SNode *) pNode1)->pPrev ? ((SNode *) pNode1)->pPrev->pNext : m_pHead) = ((SNode *) pNode2);
			(((SNode *) pNode1)->pNext ? ((SNode *) pNode1)->pNext->pPrev : m_pTail) = ((SNode *) pNode2);
			(((SNode *) pNode2)->pPrev ? ((SNode *) pNode2)->pPrev->pNext : m_pHead) = ((SNode *) pNode1);
			(((SNode *) pNode2)->pNext ? ((SNode *) pNode2)->pNext->pPrev : m_pTail) = ((SNode *) pNode1);
			swap(((SNode *) pNode1)->pPrev, ((SNode *) pNode2)->pPrev);
			swap(((SNode *) pNode1)->pNext, ((SNode *) pNode2)->pNext);
		}
	}

	void Swap(PNODE pNode1, SList<T, A, F> &list, PNODE pNode2)
	{
		ASSERT(IsValidNode(pNode1));
		ASSERT(list.IsValidNode(pNode2));

		if (((SNode *) pNode1)->pPrev == (SNode *) pNode2)
		{
			ASSERT(this == &list);
			(((SNode *) pNode1)->pNext ? ((SNode *) pNode1)->pNext->pPrev : m_pTail) = ((SNode *) pNode2);
			(((SNode *) pNode2)->pPrev ? ((SNode *) pNode2)->pPrev->pNext : m_pHead) = ((SNode *) pNode1);
			SNode *pTemp = ((SNode *) pNode1)->pNext;
			((SNode *) pNode1)->pPrev = ((SNode *) pNode2)->pPrev;
			((SNode *) pNode1)->pNext = (SNode *) pNode2;
			((SNode *) pNode2)->pPrev = (SNode *) pNode1;
			((SNode *) pNode2)->pNext = pTemp;
		}
		else if (((SNode *) pNode1)->pNext == (SNode *) pNode2)
		{
			ASSERT(this == &list);
			(((SNode *) pNode1)->pPrev ? ((SNode *) pNode1)->pPrev->pNext : m_pHead) = ((SNode *) pNode2);
			(((SNode *) pNode2)->pNext ? ((SNode *) pNode2)->pNext->pPrev : m_pTail) = ((SNode *) pNode1);
			SNode *pTemp = ((SNode *) pNode1)->pPrev;
			((SNode *) pNode1)->pPrev = (SNode *) pNode2;
			((SNode *) pNode1)->pNext = ((SNode *) pNode2)->pNext;
			((SNode *) pNode2)->pPrev = pTemp;
			((SNode *) pNode2)->pNext = (SNode *) pNode1;
		}
		else
		{
			(((SNode *) pNode1)->pPrev ? ((SNode *) pNode1)->pPrev->pNext : m_pHead) = ((SNode *) pNode2);
			(((SNode *) pNode1)->pNext ? ((SNode *) pNode1)->pNext->pPrev : m_pTail) = ((SNode *) pNode2);
			(((SNode *) pNode2)->pPrev ? ((SNode *) pNode2)->pPrev->pNext : list.m_pHead) = ((SNode *) pNode1);
			(((SNode *) pNode2)->pNext ? ((SNode *) pNode2)->pNext->pPrev : list.m_pTail) = ((SNode *) pNode1);
			swap(((SNode *) pNode1)->pPrev, ((SNode *) pNode2)->pPrev);
			swap(((SNode *) pNode1)->pNext, ((SNode *) pNode2)->pNext);
		}
	}

	void Swap(SList<T, A, F> &list)
	{
		swap(m_pHead, list.m_pHead);
		swap(m_pTail, list.m_pTail);
		swap(m_nCount, list.m_nCount);
	}

	void InsertSort(BOOL bAscending = TRUE)
	{
		if (m_nCount < 2)
			return;
		if (bAscending)
			for (SNode *pNode1 = m_pHead->pNext;pNode1;)
			{
				T *pt = &pNode1->t;
				for (SNode *pNode2 = pNode1->pPrev;pNode2 && pNode2->t > *pt;pNode2 = pNode2->pPrev)
					NULL;
				SNode *pNext = pNode1->pNext;
				Unlink(pNode1);
				if (pNode2 == NULL)
					LinkHead(pNode1);
				else
					LinkAfter(pNode2, pNode1);
				pNode1 = pNext;
			}
		else
			for (SNode *pNode1 = m_pHead->pNext;pNode1;)
			{
				T *pt = &pNode1->t;
				for (SNode *pNode2 = pNode1->pPrev;pNode2 && pNode2->t < *pt;pNode2 = pNode2->pPrev)
					NULL;
				SNode *pNext = pNode1->pNext;
				Unlink(pNode1);
				if (pNode2 == NULL)
					LinkHead(pNode1);
				else
					LinkAfter(pNode2, pNode1);
				pNode1 = pNext;
			}
	}

	void MergeSort(BOOL bAscending = TRUE)
	{
		SList<T, A, F> list;
		if (bAscending)
			for (int nSize = 1;nSize < m_nCount;nSize *= 2)
			{
				SNode *pRightNode = m_pHead->pNext;
				for (int nCount = 1;nCount < nSize;nCount++)
					pRightNode = pRightNode->pNext;
				for (SNode *pLeftNode = m_pHead;pRightNode;)
				{
					SNode *pEnd = pRightNode->pNext;
					for (nCount = 1;nCount < nSize && pEnd;nCount++)
						pEnd = pEnd->pNext;
					for (SNode *pLeft = pLeftNode, *pRight = pRightNode;;)
						if (pLeft == pRight)
						{
							if (pRight == pEnd)
								break;
							SNode *pNext = pRight->pNext;
							Unlink(pRight);
							list.LinkTail(pRight);
							pLeft = pRight = pNext;
						}
						else
							if (pRight == pEnd || pLeft->t <= pRight->t)
							{
								SNode *pNext = pLeft->pNext;
								Unlink(pLeft);
								list.LinkTail(pLeft);
								pLeft = pNext;
							}
							else
							{
								SNode *pNext = pRight->pNext;
								Unlink(pRight);
								list.LinkTail(pRight);
								pRight = pNext;
							}
					pLeftNode = pRightNode = pEnd;
					for (nCount = 0;nCount < nSize && pRightNode;nCount++)
						pRightNode = pRightNode->pNext;
				}
				if (m_nCount <= 0)
				{
					m_pHead = list.m_pHead;
					m_pTail = list.m_pTail;
					m_nCount = list.m_nCount;
				}
				else
				{
					list.m_pTail->pNext = m_pHead;
					m_pHead->pPrev = list.m_pTail;
					m_pHead = list.m_pHead;
					m_nCount += list.m_nCount;
				}
				list.m_pHead = list.m_pTail = NULL;
				list.m_nCount = 0;
			}
		else
			for (int nSize = 1;nSize < m_nCount;nSize *= 2)
			{
				SNode *pRightNode = m_pHead->pNext;
				for (int nCount = 1;nCount < nSize;nCount++)
					pRightNode = pRightNode->pNext;
				for (SNode *pLeftNode = m_pHead;pRightNode;)
				{
					SNode *pEnd = pRightNode->pNext;
					for (nCount = 1;nCount < nSize && pEnd;nCount++)
						pEnd = pEnd->pNext;
					for (SNode *pLeft = pLeftNode, *pRight = pRightNode;;)
						if (pLeft == pRight)
						{
							if (pRight == pEnd)
								break;
							SNode *pNext = pRight->pNext;
							Unlink(pRight);
							list.LinkTail(pRight);
							pLeft = pRight = pNext;
						}
						else
							if (pRight == pEnd || pLeft->t >= pRight->t)
							{
								SNode *pNext = pLeft->pNext;
								Unlink(pLeft);
								list.LinkTail(pLeft);
								pLeft = pNext;
							}
							else
							{
								SNode *pNext = pRight->pNext;
								Unlink(pRight);
								list.LinkTail(pRight);
								pRight = pNext;
							}
					pLeftNode = pRightNode = pEnd;
					for (nCount = 0;nCount < nSize && pRightNode;nCount++)
						pRightNode = pRightNode->pNext;
				}
				if (m_nCount <= 0)
				{
					m_pHead = list.m_pHead;
					m_pTail = list.m_pTail;
					m_nCount = list.m_nCount;
				}
				else
				{
					list.m_pTail->pNext = m_pHead;
					m_pHead->pPrev = list.m_pTail;
					m_pHead = list.m_pHead;
					m_nCount += list.m_nCount;
				}
				list.m_pHead = list.m_pTail = NULL;
				list.m_nCount = 0;
			}
	}

	void Sort(BOOL bAscending = TRUE)
	{
		MergeSort(bAscending);
	}

	bool operator ==(const SList<T, A, F> &list) const
	{
		if (m_nCount != list.m_nCount)
			return false;
		for (SNode *pNode1 = m_pHead, *pNode2 = list.m_pHead;pNode1;)
		{
			if (pNode1->t != pNode2->t)
				return false;
			pNode1 = pNode1->pNext;
			pNode2 = pNode2->pNext;
		}
		return true;
	}

	bool operator !=(const SList<T, A, F> &list) const
	{
		if (m_nCount != list.m_nCount)
			return true;
		for (SNode *pNode1 = m_pHead, *pNode2 = list.m_pHead;pNode1;)
		{
			if (pNode1->t != pNode2->t)
				return true;
			pNode1 = pNode1->pNext;
			pNode2 = pNode2->pNext;
		}
		return false;
	}

	SList<T, A, F> &operator =(const SList<T, A, F> &list)
	{
		if (this == &list)
			return *this;

		SList<T, A, F> listTemp;
		for (SNode *pNode = list.m_pHead;pNode;pNode = pNode->pNext)
			listTemp.AddTail(pNode->t);
		Swap(listTemp);
		return *this;
	}

	SList<T, A, F> &operator +=(const SList<T, A, F> &list)
	{
		AddTail(list);
		return *this;
	}

#ifdef _STREAM

	void ReadFromStream(SInputStream &in)
	{
		SList<T, A, F> listTemp;
		int nCount;
		for (in >> nCount;nCount > 0;nCount--)
			in >> listTemp.AddTail();
		Swap(listTemp);
	}

	void WriteToStream(SOutputStream &out) const
	{
		out << m_nCount;
		for (PNODE pNode = GetHead();pNode;pNode = GetNext(pNode))
			out << (*this)[pNode];
	}

#endif // _STREAM

#ifdef _MFC_VER
	friend CArchive &operator >>(CArchive &ar, SList<T, A, F> &list);
	friend CArchive &operator <<(CArchive &ar, const SList<T, A, F> &list);
#endif // _MFC_VER

#ifdef _DEBUG

	BOOL IsValidNode(PNODE pNode) const
	{
		if (!IsValidAddress((SNode *) pNode))
			return FALSE;
		if (((SNode *) pNode)->pPrev != NULL && ((SNode *) pNode)->pPrev == ((SNode *) pNode)->pNext)
			return FALSE;
		for (SNode *pTemp = m_pHead;pTemp;pTemp = pTemp->pNext)
			if (pTemp == (SNode *) pNode)
				return TRUE;
		return FALSE;
	}

#endif // _DEBUG

// Implementation
private:
	struct SNode
	{
		SNode() { }
		SNode(A a) : t(a) { }
		T t;
		SNode *pPrev, *pNext;
	};
	SNode *m_pHead, *m_pTail;
	int m_nCount;

protected:
	void LinkHead(SNode *pNode)
	{
		ASSERT(IsValidAddress(pNode));
		if (m_pHead == NULL)
		{
			ASSERT(m_pTail == NULL);
			ASSERT(m_nCount == 0);
			pNode->pNext = NULL;
			m_pTail = pNode;
		}
		else
		{
			ASSERT(m_nCount > 0);
			ASSERT(m_pHead->pPrev == NULL);
			pNode->pNext = m_pHead;
			m_pHead->pPrev = pNode;
		}
		pNode->pPrev = NULL;
		m_pHead = pNode;
		m_nCount++;
	}

	void LinkTail(SNode *pNode)
	{
		ASSERT(IsValidAddress(pNode));
		if (m_pTail == NULL)
		{
			ASSERT(m_pHead == NULL);
			ASSERT(m_nCount == 0);
			pNode->pPrev = NULL;
			m_pHead = pNode;
		}
		else
		{
			ASSERT(m_nCount > 0);
			ASSERT(IsValidNode((PNODE) m_pTail));
			ASSERT(m_pTail->pNext == NULL);
			pNode->pPrev = m_pTail;
			m_pTail->pNext = pNode;
		}
		pNode->pNext = NULL;
		m_pTail = pNode;
		m_nCount++;
	}

	void LinkBefore(SNode *pCurNode, SNode *pNewNode)
	{
		ASSERT(IsValidNode((PNODE) pCurNode));
		ASSERT(IsValidAddress(pNewNode));
		pNewNode->pPrev = pCurNode->pPrev;
		pNewNode->pNext = pCurNode;
		if (pCurNode->pPrev)
			pCurNode->pPrev->pNext = pNewNode;
		else
			m_pHead = pNewNode;
		pCurNode->pPrev = pNewNode;
		m_nCount++;
	}

	void LinkAfter(SNode *pCurNode, SNode *pNewNode)
	{
		ASSERT(IsValidNode((PNODE) pCurNode));
		ASSERT(IsValidAddress(pNewNode));
		pNewNode->pPrev = pCurNode;
		pNewNode->pNext = pCurNode->pNext;
		if (pCurNode->pNext)
			pCurNode->pNext->pPrev = pNewNode;
		else
			m_pTail = pNewNode;
		pCurNode->pNext = pNewNode;
		m_nCount++;
	}

	void Unlink(SNode *pNode)
	{
		ASSERT(IsValidNode((PNODE) pNode));
		if (m_nCount == 1)
		{
			ASSERT(pNode == m_pHead && pNode == m_pTail);
			m_pHead = m_pTail = NULL;
		}
		else
		{
			if (pNode == m_pHead)
				m_pHead = pNode->pNext;
			else
				pNode->pPrev->pNext = pNode->pNext;
			if (pNode == m_pTail)
				m_pTail = pNode->pPrev;
			else
				pNode->pNext->pPrev = pNode->pPrev;
		}
		m_nCount--;
	}
};

template <class T, class A = const T &, class F = const T &>
class SRefList
{
// Attributes
public:
	int GetCount() const
	{
		return m_list.GetCount();
	}

	BOOL IsEmpty() const
	{
		return m_list.IsEmpty();
	}

	T &PeekHead()
	{
		return m_list.PeekHead();
	}

	T &PeekTail()
	{
		return m_list.PeekTail();
	}

	const T &PeekHead() const
	{
		return m_list.PeekHead();
	}

	const T &PeekTail() const
	{
		return m_list.PeekTail();
	}

	PNODE GetHead() const
	{
		return m_list.GetHead();
	}

	PNODE GetTail() const
	{
		return m_list.GetTail();
	}

	PNODE GetPrev(PNODE pNode) const
	{
		return m_list.GetPrev(pNode);
	}

	PNODE GetNext(PNODE pNode) const
	{
		return m_list.GetNext(pNode);
	}

	T &operator [](PNODE pNode)
	{
		return m_list[pNode];
	}

	const T &operator [](PNODE pNode) const
	{
		return m_list[pNode];
	}

// Operations
public:
	PNODE AddHead(A a)
	{
		return m_list.AddHead(new T(a));
	}

	PNODE AddTail(A a)
	{
		return m_list.AddTail(new T(a));
	}

	PNODE InsertBefore(PNODE pNode, A a)
	{
		return m_list.InsertBefore(pNode, new T(a));
	}

	PNODE InsertAfter(PNODE pNode, A a)
	{
		return m_list.InsertAfter(pNode, new T(a));
	}

	T &AddHead()
	{
		T *pt = new T;
		m_list.AddHead(pt);
		return *pt;
	}

	T &AddTail()
	{
		T *pt = new T;
		m_list.AddTail(pt);
		return *pt;
	}

	T &InsertBefore(PNODE pNode)
	{
		T *pt = new T;
		m_list.InsertBefore(pNode, pt);
		return *pt;
	}

	T &InsertAfter(PNODE pNode)
	{
		T *pt = new T;
		m_list.InsertAfter(pNode, pt);
		return *pt;
	}

	void AddHead(const SRefList<T, A, F> &list)
	{
		m_list.AddHead(list.m_list);
	}

	void AddTail(const SRefList<T, A, F> &list)
	{
		m_list.AddTail(list.m_list);
	}

	void InsertBefore(PNODE pNode, const SRefList<T, A, F> &list)
	{
		m_list.InsertBefore(list.m_list);
	}

	void InsertAfter(PNODE pNode, const SRefList<T, A, F> &list)
	{
		m_list.InsertAfter(list.m_list);
	}

	PNODE AddRefHead(T *pt)
	{
		ASSERT(IsValidAddress(pt));
		return m_list.AddHead(pt);
	}

	PNODE AddRefTail(T *pt)
	{
		ASSERT(IsValidAddress(pt));
		return m_list.AddTail(pt);
	}

	PNODE InsertRefBefore(PNODE pNode, T *pt)
	{
		ASSERT(IsValidAddress(pt));
		return m_list.InsertBefore(pNode, pt);
	}

	PNODE InsertRefAfter(PNODE pNode, T *pt)
	{
		ASSERT(IsValidAddress(pt));
		return m_list.InsertAfter(pNode, pt);
	}

	void RemoveHead()
	{
		m_list.RemoveHead();
	}

	void RemoveTail()
	{
		m_list.RemoveTail();
	}

	void Remove(PNODE pNode)
	{
		m_list.Remove(pNode);
	}

	void RemoveAll()
	{
		m_list.RemoveAll();
	}

	void ModifyRef(PNODE pNode, T *pt)
	{
		ASSERT(IsValidNode(pNode));
		ASSERT(IsValidAddress(pt));
		m_list[pNode].ModifyRef(pt);
	}

	void MoveToHead(PNODE pCurNode)
	{
		m_list.MoveToHead(pCurNode);
	}

	void MoveToTail(PNODE pCurNode)
	{
		m_list.MoveToTail(pCurNode);
	}

	void MoveToBefore(PNODE pCurNode, PNODE pNode)
	{
		m_list.MoveToBefore(pCurNode, pNode);
	}

	void MoveToAfter(PNODE pCurNode, PNODE pNode)
	{
		m_list.MoveToAfter(pCurNode, pNode);
	}

	void MoveToHead(PNODE pCurNode, SRefList<T, A, F> &list)
	{
		m_list.MoveToHead(pCurNode, list.m_list);
	}

	void MoveToTail(PNODE pCurNode, SRefList<T, A, F> &list)
	{
		m_list.MoveToTail(pCurNode, list.m_list);
	}

	void MoveToBefore(PNODE pCurNode, SRefList<T, A, F> &list, PNODE pNode)
	{
		m_list.MoveToBefore(pCurNode, list.m_list, pNode);
	}

	void MoveToAfter(PNODE pCurNode, SRefList<T, A, F> &list, PNODE pNode)
	{
		m_list.MoveToAfter(pCurNode, list.m_list, pNode);
	}

	void MoveToHeadAll(SRefList<T, A, F> &list)
	{
		m_list.MoveToHeadAll(list.m_list);
	}

	void MoveToTailAll(SRefList<T, A, F> &list)
	{
		m_list.MoveToTailAll(list.m_list);
	}

	void MoveToBeforeAll(SRefList<T, A, F> &list, PNODE pNode)
	{
		m_list.MoveToBeforeAll(list.m_list, pNode);
	}

	void MoveToAfterAll(SRefList<T, A, F> &list, PNODE pNode)
	{
		m_list.MoveToAfterAll(list.m_list, pNode);
	}

	PNODE Find(F f, PNODE pNode = NULL) const
	{
		return m_list.Find(f, pNode);
	}

	PNODE FindIndex(int nIndex) const
	{
		return m_list.FindIndex(nIndex);
	}

	void Reverse()
	{
		m_list.Reverse();
	}

	void Swap(PNODE pNode1, PNODE pNode2)
	{
		m_list.Swap(pNode1, pNode2);
	}

	void Swap(PNODE pNode1, SRefList<T, A, F> &list, PNODE pNode2)
	{
		m_list.Swap(pNode1, list.m_list, pNode2);
	}

	void Swap(SRefList<T, A, F> &list)
	{
		m_list.Swap(list);
	}

	void SwapRef(PNODE pNode1, PNODE pNode2)
	{
		ASSERT(IsValidNode(pNode1));
		ASSERT(IsValidNode(pNode2));
		m_list[pNode1].Swap(m_list[pNode2]);
	}

	void SwapRef(PNODE pNode1, SRefList<T, A, F> &list, PNODE pNode2)
	{
		ASSERT(IsValidNode(pNode1));
		ASSERT(list.IsValidNode(pNode2));
		m_list[pNode1].Swap(list.m_list[pNode2]);
	}

	void InsertSort(BOOL bAscending = TRUE)
	{
		m_list.InsertSort(bAscending);
	}

	void MergeSort(BOOL bAscending = TRUE)
	{
		m_list.MergeSort(bAscending);
	}

	void Sort(BOOL bAscending = TRUE)
	{
		m_list.Sort(bAscending);
	}

	bool operator ==(const SRefList<T, A, F> &list)
	{
		return m_list == list.m_list;
	}

	bool operator !=(const SRefList<T, A, F> &list)
	{
		return m_list != list.m_list;
	}

	SRefList<T, A, F> &operator =(const SRefList<T, A, F> &list)
	{
		m_list = list.m_list;
		return *this;
	}

	SRefList<T, A, F> &operator +=(const SRefList<T, A, F> &list)
	{
		m_list += list.m_list;
		return *this;
	}

#ifdef _DEBUG

	BOOL IsValidNode(PNODE pNode) const
	{
		return m_list.IsValidNode(pNode);
	}

#endif // _DEBUG

// Implementation
private:
	class SNode
	{
	public:
		SNode(T *pt)
			{ ASSERT(IsValidAddress(pt));
				m_pt = pt; }
		~SNode()
			{ delete m_pt; }

	public:
		void ModifyRef(T *pt)
			{ ASSERT(IsValidAddress(pt));
				delete m_pt; m_pt = pt; }
		void Swap(SNode &node)
			{ swap(m_pt, node.m_pt); }

		bool operator >(const SNode &node)
			{ return *m_pt > *node.m_pt; }
		bool operator <(const SNode &node)
			{ return *m_pt < *node.m_pt; }
		bool operator >=(const SNode &node)
			{ return *m_pt >= *node.m_pt; }
		bool operator <=(const SNode &node)
			{ return *m_pt <= *node.m_pt; }
		bool operator ==(const SNode &node)
			{ return *m_pt == *node.m_pt; }
		bool operator !=(const SNode &node)
			{ return *m_pt != *node.m_pt; }

		bool operator ==(F f)
			{ return *m_pt == f; }

		operator T &()
			{ return *m_pt; }
		operator T *()
			{ m_pt->Clone(); }

	private:
		T *m_pt;
	};

	SList<SNode, T *, F> m_list;
};

#endif // __LINKEDLIST_H__
