#include "stdafx.h"
#include "Map.h"

/////////////////////////////////////////////////////////////////////////////
// SMap

SMap::~SMap()
{
	if (m_ppBackMap != NULL)
		Delete();
}

void SMap::Create(int nWidth, int nHeight, STerrainManager *pManager, const STerrainPattern *pPattern)
{
	ASSERT(m_ppBackMap == NULL && m_ppFrontMap == NULL);
	ASSERT(nWidth > 0 && nHeight > 0);
	ASSERT(IsValidAddress(pManager));

	m_nWidth = nWidth;
	m_nHeight = nHeight;
	m_ppBackMap = new const STerrain *[m_nWidth * m_nHeight];
	m_ppFrontMap = new const STerrain *[m_nWidth * m_nHeight];

	m_pTerrainManager = pManager;
	for (int y = 0;y < m_nHeight;y++)
		for (int x = 0;x < m_nWidth;x++)
			m_ppBackMap[x + y * m_nWidth] = pPattern->GetRandomTerrain();
	ClearFrontMap();
}

void SMap::Delete()
{
	ASSERT(m_ppBackMap != NULL && m_ppFrontMap != NULL);
	delete m_ppBackMap;
	delete m_ppFrontMap;

	m_nWidth = m_nHeight = 0;
	m_ppBackMap = m_ppFrontMap = NULL;
}

/*void SMap::GetTerrainRange(int x, int y, SStoreMap *pMap) const
{
	ASSERT(IsValidAddress(pMap));
	pMap->SetX(x);
	pMap->SetY(y);
	for (int yMap = 0;yMap < pMap->GetHeight();yMap++)
		for (int xMap = 0;xMap < pMap->GetWidth();xMap++)
			if (IsValidPoint(x + xMap, y + yMap))
				pMap->SetTerrain(xMap, yMap, GetTerrain(x + xMap, y + yMap));
			else
				pMap->SetTerrain(xMap, yMap, NULL);
}

void SMap::SetTerrainRange(const SStoreMap &map)
{
	for (int yMap = 0;yMap < map.GetHeight();yMap++)
		for (int xMap = 0;xMap < map.GetWidth();xMap++)
		{
			const STerrain *pTerrain = map.GetTerrain(xMap, yMap);
			if (pTerrain != NULL)
			{
				ASSERT(IsValidPoint(map.GetX() + xMap, map.GetY() + yMap));
				SetTerrain(map.GetX() + xMap, map.GetY() + yMap, pTerrain);
			}
		}
}*/

void SMap::ClearFrontMap()
{
	memset(m_ppFrontMap, 0, sizeof(LPVOID) * m_nWidth * m_nHeight);
}

const STerrain *SMap::GetTerrain(int x, int y) const
{
	ASSERT(m_ppBackMap != NULL && m_ppFrontMap != NULL);
	ASSERT(IsValidPoint(x, y));

	const STerrain *pTerrain = m_ppFrontMap[x + y * m_nWidth];
	if (pTerrain == NULL)
		pTerrain = m_ppBackMap[x + y * m_nWidth];
	return pTerrain;
}

void SMap::SetTerrain(int x, int y, const STerrain *pTerrain)
{
	ASSERT(m_ppBackMap != NULL && m_ppFrontMap != NULL);
	ASSERT(IsValidPoint(x, y));
	ASSERT(IsValidReadOnlyAddress(pTerrain));

	int nClass = GetTerrain(x, y)->GetType()->GetClass();
	if (nClass != STerrainType::CLIFF_SOLID && nClass != STerrainType::CLIFF_PATTERN)
	{
		const STerrainSet *pSet = GetTerrain(x, y)->GetSet();
		int nWidth = pSet->GetWidth();
		int nHeight = pSet->GetHeight();
		if (nWidth > 1 || nHeight > 1)
		{
			int nIndex = GetTerrain(x, y)->GetIndex();
			int xSet = x - nIndex % nWidth;
			int ySet = y - nIndex / nWidth;
			const STerrainPattern *pPattern = pSet->GetPattern()->GetType()->GetPattern(0);
			SetTerrainRange(xSet, ySet, nWidth, nHeight, pPattern);
		}
	}
	m_ppFrontMap[x + y * m_nWidth] = pTerrain;
}

void SMap::SetTerrainRange(int x, int y, int nWidth, int nHeight, const STerrainPattern *pPattern)
{
	for (int ySet = 0;ySet < nHeight;ySet++)
		for (int xSet = 0;xSet < nWidth;xSet++)
		{
			int i = (x + xSet) + (y + ySet) * m_nWidth;
			m_ppFrontMap[i] = pPattern->GetRandomTerrain();
		}
}

void SMap::SetPattern(int x, int y, const STerrainPattern *pPattern)
{
	ASSERT(m_ppBackMap != NULL && m_ppFrontMap != NULL);
	ASSERT(IsValidPoint(x, y));
	ASSERT(IsValidReadOnlyAddress(pPattern));

	const STerrain *pTerrain = GetTerrain(x, y);
	if (pTerrain->GetID().GetType() == pPattern->GetType()->GetTypeID())
	{
		SPatternID idPattern1 = pTerrain->GetID().GetPattern();
		SPatternID idPattern2 = pPattern->GetPatternID();
		if (idPattern1 == 1)
			idPattern1 = 0;
		if (idPattern2 == 1)
			idPattern2 = 0;
		if (idPattern1 == idPattern2)
			return;
	}
	SetTerrain(x, y, pPattern->GetRandomTerrain());
}

void SMap::MergeMap()
{
	for (int y = 0;y < m_nHeight;y++)
		for (int x = 0;x < m_nWidth;x++)
		{
			const STerrain *pTerrain = m_ppFrontMap[x + y * m_nWidth];
			if (pTerrain != NULL)
				m_ppBackMap[x + y * m_nWidth] = pTerrain;
		}
	ClearFrontMap();
}

void SMap::Load(LPCTSTR lpszFileName)
{
}

void SMap::Save(LPCTSTR lpszFileName)
{
	SaveForEngine(lpszFileName);
}

void SMap::SaveForEngine(LPCTSTR lpszFileName)
{
	SFile file(lpszFileName, FM_CREATE | FM_WRITE);

	LPCSTR lpszID = " Map";
	file.WriteString(strlen(lpszID), lpszID);

	SArray<SUsedType, const SUsedType &, int> arrUsedType;
	for (int i = 0;i < GetWidth() * GetHeight();i++)
	{
		const STerrainType *pType = m_ppBackMap[i]->GetType();
		if (arrUsedType.Find(pType->GetTypeID()) < 0)
			arrUsedType.Add(SUsedType(pType->GetTypeID(), pType->GetName()));
	}
	arrUsedType.Sort();
	file << arrUsedType;

	file << GetWidth() << GetHeight();
	for (i = 0;i < GetWidth() * GetHeight();i++)
	{
		SEngineID id = m_ppBackMap[i]->GetEngineID();
		int nType = id.GetType();
		int nIndex = id.GetIndex();

		nType = arrUsedType.Find(nType);
		ASSERT(nType >= 0);
		file << nType << nIndex;
	}
}

/////////////////////////////////////////////////////////////////////////////
// SMapDrawer

BOOL SMapDrawer::IsEqualType(int x, int y, int nType)
{
	ASSERT(IsValidPoint(x, y));
	const STerrain *pTerrain = GetTerrain(x, y);
	return pTerrain->GetType()->GetTypeID() == nType;
}

BOOL SMapDrawer::AnalPatternHorz(int x, int y, const STerrainPattern **ppPattern)
{
	STerrainSideInfo info;
	info.Fill(-1);

	STerrainSideInfo infoTemp;
	if (IsValidPoint(x - 1, y))
	{
		GetTerrain(x - 1, y)->GetSideInfo(&infoTemp);
		info.SetSideInfo(STerrainSideInfo::LEFT_TOP, infoTemp.GetSideInfo(STerrainSideInfo::RIGHT_TOP));
		info.SetSideInfo(STerrainSideInfo::LEFT_BOTTOM, infoTemp.GetSideInfo(STerrainSideInfo::RIGHT_BOTTOM));
	}
	else if (IsValidPoint(x, y))
	{
		GetTerrain(x, y)->GetSideInfo(&infoTemp);
		info.SetSideInfo(STerrainSideInfo::LEFT_TOP, infoTemp.GetSideInfo(STerrainSideInfo::LEFT_TOP));
		info.SetSideInfo(STerrainSideInfo::LEFT_BOTTOM, infoTemp.GetSideInfo(STerrainSideInfo::LEFT_BOTTOM));
	}
	if (IsValidPoint(x + 1, y))
	{
		GetTerrain(x + 1, y)->GetSideInfo(&infoTemp);
		info.SetSideInfo(STerrainSideInfo::RIGHT_TOP, infoTemp.GetSideInfo(STerrainSideInfo::LEFT_TOP));
		info.SetSideInfo(STerrainSideInfo::RIGHT_BOTTOM, infoTemp.GetSideInfo(STerrainSideInfo::LEFT_BOTTOM));
	}
	else if (IsValidPoint(x, y))
	{
		GetTerrain(x, y)->GetSideInfo(&infoTemp);
		info.SetSideInfo(STerrainSideInfo::RIGHT_TOP, infoTemp.GetSideInfo(STerrainSideInfo::RIGHT_TOP));
		info.SetSideInfo(STerrainSideInfo::RIGHT_BOTTOM, infoTemp.GetSideInfo(STerrainSideInfo::RIGHT_BOTTOM));
	}
	for (int i = 0;i < 8;i++)
	{
		int nType = info.GetSideInfo(i);
		if (nType >= 0)
		{
			const STerrainType *pType = GetTerrainManager()->GetType(nType);
			int nClass = pType->GetClass();
			if (nClass == STerrainType::CLIFF_SOLID || nClass == STerrainType::CLIFF_PATTERN)
			{
				*ppPattern = NULL;
				return TRUE;
			}
		}
	}
	*ppPattern = GetTerrainManager()->FindPattern(info);
	return *ppPattern != NULL;
}

BOOL SMapDrawer::AnalPatternVert(int x, int y, const STerrainPattern **ppPattern)
{
	STerrainSideInfo info;
	info.Fill(-1);

	STerrainSideInfo infoTemp;
	if (IsValidPoint(x, y - 1))
	{
		GetTerrain(x, y - 1)->GetSideInfo(&infoTemp);
		info.SetSideInfo(STerrainSideInfo::TOP_LEFT, infoTemp.GetSideInfo(STerrainSideInfo::BOTTOM_LEFT));
		info.SetSideInfo(STerrainSideInfo::TOP_RIGHT, infoTemp.GetSideInfo(STerrainSideInfo::BOTTOM_RIGHT));
	}
	else if (IsValidPoint(x, y))
	{
		GetTerrain(x, y)->GetSideInfo(&infoTemp);
		info.SetSideInfo(STerrainSideInfo::TOP_LEFT, infoTemp.GetSideInfo(STerrainSideInfo::TOP_LEFT));
		info.SetSideInfo(STerrainSideInfo::TOP_RIGHT, infoTemp.GetSideInfo(STerrainSideInfo::TOP_RIGHT));
	}
	if (IsValidPoint(x, y + 1))
	{
		GetTerrain(x, y + 1)->GetSideInfo(&infoTemp);
		info.SetSideInfo(STerrainSideInfo::BOTTOM_LEFT, infoTemp.GetSideInfo(STerrainSideInfo::TOP_LEFT));
		info.SetSideInfo(STerrainSideInfo::BOTTOM_RIGHT, infoTemp.GetSideInfo(STerrainSideInfo::TOP_RIGHT));
	}
	else if (IsValidPoint(x, y))
	{
		GetTerrain(x, y)->GetSideInfo(&infoTemp);
		info.SetSideInfo(STerrainSideInfo::BOTTOM_LEFT, infoTemp.GetSideInfo(STerrainSideInfo::BOTTOM_LEFT));
		info.SetSideInfo(STerrainSideInfo::BOTTOM_RIGHT, infoTemp.GetSideInfo(STerrainSideInfo::BOTTOM_RIGHT));
	}
	for (int i = 0;i < 8;i++)
	{
		int nType = info.GetSideInfo(i);
		if (nType >= 0)
		{
			const STerrainType *pType = GetTerrainManager()->GetType(nType);
			int nClass = pType->GetClass();
			if (nClass == STerrainType::CLIFF_SOLID || nClass == STerrainType::CLIFF_PATTERN)
			{
				*ppPattern = NULL;
				return TRUE;
			}
		}
	}
	*ppPattern = GetTerrainManager()->FindPattern(info);
	return *ppPattern != NULL;
}

BOOL SMapDrawer::AnalPatternAll(int x, int y, const STerrainPattern **ppPattern)
{
	STerrainSideInfo info;
	info.Fill(-1);

	STerrainSideInfo infoTemp;
	if (IsValidPoint(x - 1, y))
	{
		GetTerrain(x - 1, y)->GetSideInfo(&infoTemp);
		info.SetSideInfo(STerrainSideInfo::LEFT_TOP, infoTemp.GetSideInfo(STerrainSideInfo::RIGHT_TOP));
		info.SetSideInfo(STerrainSideInfo::LEFT_BOTTOM, infoTemp.GetSideInfo(STerrainSideInfo::RIGHT_BOTTOM));
	}
	else if (IsValidPoint(x, y))
	{
		GetTerrain(x, y)->GetSideInfo(&infoTemp);
		info.SetSideInfo(STerrainSideInfo::LEFT_TOP, infoTemp.GetSideInfo(STerrainSideInfo::LEFT_TOP));
		info.SetSideInfo(STerrainSideInfo::LEFT_BOTTOM, infoTemp.GetSideInfo(STerrainSideInfo::LEFT_BOTTOM));
	}
	if (IsValidPoint(x + 1, y))
	{
		GetTerrain(x + 1, y)->GetSideInfo(&infoTemp);
		info.SetSideInfo(STerrainSideInfo::RIGHT_TOP, infoTemp.GetSideInfo(STerrainSideInfo::LEFT_TOP));
		info.SetSideInfo(STerrainSideInfo::RIGHT_BOTTOM, infoTemp.GetSideInfo(STerrainSideInfo::LEFT_BOTTOM));
	}
	else if (IsValidPoint(x, y))
	{
		GetTerrain(x, y)->GetSideInfo(&infoTemp);
		info.SetSideInfo(STerrainSideInfo::RIGHT_TOP, infoTemp.GetSideInfo(STerrainSideInfo::RIGHT_TOP));
		info.SetSideInfo(STerrainSideInfo::RIGHT_BOTTOM, infoTemp.GetSideInfo(STerrainSideInfo::RIGHT_BOTTOM));
	}

	if (IsValidPoint(x, y - 1))
	{
		GetTerrain(x, y - 1)->GetSideInfo(&infoTemp);
		info.SetSideInfo(STerrainSideInfo::TOP_LEFT, infoTemp.GetSideInfo(STerrainSideInfo::BOTTOM_LEFT));
		info.SetSideInfo(STerrainSideInfo::TOP_RIGHT, infoTemp.GetSideInfo(STerrainSideInfo::BOTTOM_RIGHT));
	}
	else if (IsValidPoint(x, y))
	{
		GetTerrain(x, y)->GetSideInfo(&infoTemp);
		info.SetSideInfo(STerrainSideInfo::TOP_LEFT, infoTemp.GetSideInfo(STerrainSideInfo::TOP_LEFT));
		info.SetSideInfo(STerrainSideInfo::TOP_RIGHT, infoTemp.GetSideInfo(STerrainSideInfo::TOP_RIGHT));
	}
	if (IsValidPoint(x, y + 1))
	{
		GetTerrain(x, y + 1)->GetSideInfo(&infoTemp);
		info.SetSideInfo(STerrainSideInfo::BOTTOM_LEFT, infoTemp.GetSideInfo(STerrainSideInfo::TOP_LEFT));
		info.SetSideInfo(STerrainSideInfo::BOTTOM_RIGHT, infoTemp.GetSideInfo(STerrainSideInfo::TOP_RIGHT));
	}
	else if (IsValidPoint(x, y))
	{
		GetTerrain(x, y)->GetSideInfo(&infoTemp);
		info.SetSideInfo(STerrainSideInfo::BOTTOM_LEFT, infoTemp.GetSideInfo(STerrainSideInfo::BOTTOM_LEFT));
		info.SetSideInfo(STerrainSideInfo::BOTTOM_RIGHT, infoTemp.GetSideInfo(STerrainSideInfo::BOTTOM_RIGHT));
	}
	for (int i = 0;i < 8;i++)
	{
		int nType = info.GetSideInfo(i);
		if (nType >= 0)
		{
			const STerrainType *pType = GetTerrainManager()->GetType(nType);
			int nClass = pType->GetClass();
			if (nClass == STerrainType::CLIFF_SOLID || nClass == STerrainType::CLIFF_PATTERN)
			{
				*ppPattern = NULL;
				return TRUE;
			}
		}
	}
	*ppPattern = GetTerrainManager()->FindPattern(info);
	return *ppPattern != NULL;
}

BOOL SMapDrawer::AnalCliff(int nCliff, int x, int y, BOOL bUp)
{
	if (!IsValidPoint(x, y))
		return FALSE;
	const STerrain *pTerrain = GetTerrain(x, y);
	if (pTerrain->GetType()->GetTypeID() != nCliff)
		return FALSE;

	y += (bUp ? -1 : 1);
	if (!IsValidPoint(x, y))
		return FALSE;
	pTerrain = GetTerrain(x, y);
	if (pTerrain->GetType()->GetTypeID() == nCliff)
		return FALSE;
	return TRUE;
}

const STerrainPattern *SMapDrawer::AnalCliffTop(int nCliff, int x, int y, BOOL bDrawLeftRight)
{
	int nLeft;
	if (IsValidPoint(x - 1, y))
	{
		if (AnalCliff(nCliff, x - 1, y, TRUE))
		{
			nLeft = 0;
			if (bDrawLeftRight)
				DrawCliffTop(nCliff, x - 1, y - 1, 1, FALSE);
		}
		else if (AnalCliff(nCliff, x - 1, y + 1, TRUE))
		{
			nLeft = 1;
			if (bDrawLeftRight)
				DrawCliffTop(nCliff, x - 1, y, 1, FALSE);
		}
		else if (AnalCliff(nCliff, x - 1, y + 2, TRUE))
		{
			nLeft = 2;
			if (bDrawLeftRight)
				DrawCliffTop(nCliff, x - 1, y + 1, 1, FALSE);
		}
		else
		{
			if (IsEqualType(x - 1, y, nCliff) ||
				IsEqualType(x - 1, y + 1, nCliff) ||
				IsEqualType(x - 1, y + 2, nCliff))
				return NULL;
			nLeft = 1;
		}
	}
	else
		nLeft = 1;

	int nRight;
	if (IsValidPoint(x + 1, y))
	{
		if (AnalCliff(nCliff, x + 1, y, TRUE))
		{
			nRight = 0;
			if (bDrawLeftRight)
				DrawCliffTop(nCliff, x + 1, y - 1, 1, FALSE);
		}
		else if (AnalCliff(nCliff, x + 1, y + 1, TRUE))
		{
			nRight = 1;
			if (bDrawLeftRight)
				DrawCliffTop(nCliff, x + 1, y, 1, FALSE);
		}
		else if (AnalCliff(nCliff, x + 1, y + 2, TRUE))
		{
			nRight = 2;
			if (bDrawLeftRight)
				DrawCliffTop(nCliff, x + 1, y + 1, 1, FALSE);
		}
		else
		{
			if (IsEqualType(x + 1, y, nCliff) ||
				IsEqualType(x + 1, y + 1, nCliff) ||
				IsEqualType(x + 1, y + 2, nCliff))
				return NULL;
			nRight = 1;
		}
	}
	else
		nRight = 1;

	SPatternID idPattern;
	if (nLeft == 0 && nRight == 0)
	{
		idPattern.SetPattern(1, 8);
		y--;
	}
	else if (nLeft == 0 && nRight >= 1)
	{
		idPattern.SetPattern(1, 4);
		y--;
	}
	else if (nLeft >= 1 && nRight == 0)
	{
		idPattern.SetPattern(1, 1);
		y--;
	}
	else if (nLeft >= 1 && nRight >= 0)
		idPattern.SetPattern(1, 7);
	else
		return NULL;

	const STerrainType *pType = GetTerrain(x, y)->GetType();
	if (pType->GetClass() != STerrainType::CLIFF_PATTERN)
	{
		int nBack = pType->GetTypeID();
		if (nCliff == nBack)
			return NULL;
		pType = GetTerrainManager()->FindType(nCliff, nBack);
		if (pType == NULL)
			return NULL;
	}
	return pType->GetPattern(idPattern);
}

const STerrainPattern *SMapDrawer::AnalCliffBottom(int nCliff, int x, int y, BOOL bDrawLeftRight)
{
	int nLeft;
	if (IsValidPoint(x - 1, y))
	{
		if (AnalCliff(nCliff, x - 1, y, FALSE))
		{
			nLeft = 0;
			if (bDrawLeftRight)
				DrawCliffBottom(nCliff, x - 1, y - 1, 1, FALSE);
		}
		else if (AnalCliff(nCliff, x - 1, y - 1, FALSE))
		{
			nLeft = 1;
			if (bDrawLeftRight)
				DrawCliffBottom(nCliff, x - 1, y, 1, FALSE);
		}
		else if (AnalCliff(nCliff, x - 1, y - 2, FALSE))
		{
			nLeft = 2;
			if (bDrawLeftRight)
				DrawCliffBottom(nCliff, x - 1, y - 1, 1, FALSE);
		}
		else
		{
			if (IsEqualType(x - 1, y, nCliff) ||
				IsEqualType(x - 1, y - 1, nCliff) ||
				IsEqualType(x - 1, y - 2, nCliff))
				return NULL;
			nLeft = 1;
		}
	}
	else
		nLeft = 1;

	int nRight;
	if (IsValidPoint(x + 1, y))
	{
		if (AnalCliff(nCliff, x + 1, y, FALSE))
		{
			nRight = 0;
			if (bDrawLeftRight)
				DrawCliffBottom(nCliff, x + 1, y + 1, 1, FALSE);
		}
		else if (AnalCliff(nCliff, x + 1, y - 1, FALSE))
		{
			nRight = 1;
			if (bDrawLeftRight)
				DrawCliffBottom(nCliff, x + 1, y, 1, FALSE);
		}
		else if (AnalCliff(nCliff, x + 1, y - 2, FALSE))
		{
			nRight = 2;
			if (bDrawLeftRight)
				DrawCliffBottom(nCliff, x + 1, y - 1, 1, FALSE);
		}
		else
		{
			if (IsEqualType(x + 1, y, nCliff) ||
				IsEqualType(x + 1, y - 1, nCliff) ||
				IsEqualType(x + 1, y - 2, nCliff))
				return NULL;
			nRight = 1;
		}
	}
	else
		nRight = 1;

	SPatternID idPattern;
	if (nLeft == 0 && nRight == 0)
	{
		idPattern.SetPattern(2, 8);
		y++;
	}
	else if (nLeft == 0 && nRight >= 1)
	{
		idPattern.SetPattern(2, 1);
		y++;
	}
	else if (nLeft >= 1 && nRight == 0)
	{
		idPattern.SetPattern(2, 4);
		y++;
	}
	else if (nLeft >= 1 && nRight >= 0)
		idPattern.SetPattern(2, 7);
	else
		return NULL;

	const STerrainType *pType = GetTerrain(x, y)->GetType();
	if (pType->GetClass() != STerrainType::CLIFF_PATTERN)
	{
		int nBack = pType->GetTypeID();
		if (nCliff == nBack)
			return NULL;
		pType = GetTerrainManager()->FindType(nCliff, nBack);
		if (pType == NULL)
			return NULL;
	}
	return pType->GetPattern(idPattern);
}

BOOL SMapDrawer::DrawSideHorz(int x, int y, int nWidth)
{
	for (int n = 0;n < nWidth;n++)
		if (IsValidPoint(x + n, y))
		{
			const STerrainPattern *pPattern;
			if (!AnalPatternVert(x + n, y, &pPattern))
				return FALSE;
			if (pPattern != NULL)
				SetPattern(x + n, y, pPattern);
		}
	return TRUE;
}

BOOL SMapDrawer::DrawSideVert(int x, int y, int nHeight)
{
	for (int n = 0;n < nHeight;n++)
		if (IsValidPoint(x, y + n))
		{
			const STerrainPattern *pPattern;
			if (!AnalPatternHorz(x, y + n, &pPattern))
				return FALSE;
			if (pPattern != NULL)
				SetPattern(x, y + n, pPattern);
		}
	return TRUE;
}

BOOL SMapDrawer::DrawCorner(int x, int y)
{
	if (IsValidPoint(x, y))
	{
		const STerrainPattern *pPattern;
		if (!AnalPatternAll(x, y, &pPattern))
			return FALSE;
		if (pPattern != NULL)
			SetPattern(x, y, pPattern);
	}
	return TRUE;
}

BOOL SMapDrawer::DrawCliffTop(int nCliff, int x, int y, int nWidth, BOOL bDrawLeftRight)
{
	for (int n = 0;n < nWidth;n++)
		if (IsValidPoint(x + n, y))
		{
			const STerrainPattern *pPattern = AnalCliffTop(nCliff, x + n, y, bDrawLeftRight);
			if (pPattern == NULL)
				return FALSE;

			const STerrainSet *pSet;
			const STerrain *pTerrain1, *pTerrain2;
			switch (pPattern->GetPatternID().GetEnd())
			{
			case 1:
			case 4:
				if (rand() % 4 == 0)
				{
					pSet = pPattern->GetRandomSet(1, 2);
					pTerrain1 = pSet->GetTerrain(0);
					pTerrain2 = pSet->GetTerrain(1);
				}
				else
				{
					int nEnd = pPattern->GetPatternID().GetEnd();
					pSet = pPattern->GetType()->GetPattern(SPatternID(1, nEnd + 1))->GetRandomSet(1, 1);
					pTerrain1 = pSet->GetTerrain(0);
					pSet = pPattern->GetType()->GetPattern(SPatternID(1, nEnd + 2))->GetRandomSet(1, 1);
					pTerrain2 = pSet->GetTerrain(0);
				}
				break;
			case 7:
				pSet = pPattern->GetRandomSet(1, 1);
				pTerrain1 = pSet->GetTerrain(0);
				pTerrain2 = NULL;
				break;
			case 8:
				pSet = pPattern->GetRandomSet(1, 2);
				pTerrain1 = pSet->GetTerrain(0);
				pTerrain2 = pSet->GetTerrain(1);
				break;
			default:
				ASSERT(FALSE);
			}

			if (pTerrain2 == NULL)
				SetTerrain(x + n, y, pTerrain1);
			else
			{
				if (IsValidPoint(x + n, y - 1))
					SetTerrain(x + n, y - 1, pTerrain1);
				SetTerrain(x + n, y, pTerrain2);
			}
		}
	return TRUE;
}

BOOL SMapDrawer::DrawCliffBottom(int nCliff, int x, int y, int nWidth, BOOL bDrawLeftRight)
{
	for (int n = 0;n < nWidth;n++)
		if (IsValidPoint(x + n, y))
		{
			const STerrainPattern *pPattern = AnalCliffBottom(nCliff, x + n, y, bDrawLeftRight);
			if (pPattern == NULL)
				return FALSE;

			const STerrainSet *pSet;
			const STerrain *pTerrain1, *pTerrain2;
			switch (pPattern->GetPatternID().GetEnd())
			{
			case 1:
			case 4:
				if (rand() % 4 == 0)
				{
					pSet = pPattern->GetRandomSet(1, 2);
					pTerrain1 = pSet->GetTerrain(0);
					pTerrain2 = pSet->GetTerrain(1);
				}
				else
				{
					int nEnd = pPattern->GetPatternID().GetEnd();
					pSet = pPattern->GetType()->GetPattern(SPatternID(2, nEnd + 1))->GetRandomSet(1, 1);
					pTerrain1 = pSet->GetTerrain(0);
					pSet = pPattern->GetType()->GetPattern(SPatternID(2, nEnd + 2))->GetRandomSet(1, 1);
					pTerrain2 = pSet->GetTerrain(0);
				}
				break;
			case 7:
				pSet = pPattern->GetRandomSet(1, 1);
				pTerrain1 = pSet->GetTerrain(0);
				pTerrain2 = NULL;
				break;
			case 8:
				pSet = pPattern->GetRandomSet(1, 2);
				pTerrain1 = pSet->GetTerrain(0);
				pTerrain2 = pSet->GetTerrain(1);
				break;
			default:
				ASSERT(FALSE);
			}

			if (pTerrain2 == NULL)
				SetTerrain(x + n, y, pTerrain1);
			else
			{
				SetTerrain(x + n, y, pTerrain1);
				if (IsValidPoint(x + n, y + 1))
					SetTerrain(x + n, y + 1, pTerrain2);
			}
		}
	return TRUE;
}

BOOL SMapDrawer::DrawNormalTerrain(int x, int y, const STerrainSet *pSet)
{
	ASSERT(IsValidPoint(x, y));
	ASSERT(IsValidReadOnlyAddress(pSet));
	ASSERT(pSet->GetPattern()->GetType()->GetClass() == STerrainType::NORMAL_SOLID);

	ClearFrontMap();

	for (int ySet = 0;ySet < pSet->GetHeight();ySet++)
		for (int xSet = 0;xSet < pSet->GetWidth();xSet++)
			if (IsValidPoint(x + xSet, y + ySet))
				SetTerrain(x + xSet, y + ySet, pSet->GetTerrain(xSet, ySet));

	int nLeft = x - 1;
	int nRight = x + pSet->GetWidth();
	int nTop = y - 1;
	int nBottom = y + pSet->GetHeight();
	if (!DrawSideVert(nLeft, y, pSet->GetHeight()) ||
		!DrawSideVert(nRight, y, pSet->GetHeight()) ||
		!DrawSideHorz(x, nTop, pSet->GetWidth()) ||
		!DrawSideHorz(x, nBottom, pSet->GetWidth()) ||
		!DrawCorner(nLeft, nTop) ||
		!DrawCorner(nRight, nTop) ||
		!DrawCorner(nLeft, nBottom) ||
		!DrawCorner(nRight, nBottom))
	{
		ClearFrontMap();
		return FALSE;
	}
	return TRUE;
}

BOOL SMapDrawer::DrawCliffTerrain(int x, int y, const STerrainSet *pSet)
{
	ASSERT(IsValidPoint(x, y));
	ASSERT(IsValidReadOnlyAddress(pSet));
	ASSERT(pSet->GetPattern()->GetType()->GetClass() == STerrainType::CLIFF_SOLID);

	ClearFrontMap();

	for (int ySet = 0;ySet < pSet->GetHeight();ySet++)
		for (int xSet = 0;xSet < pSet->GetWidth();xSet++)
			if (IsValidPoint(x + xSet, y + ySet))
				SetTerrain(x + xSet, y + ySet, pSet->GetTerrain(xSet, ySet));

	int nCliff = pSet->GetPattern()->GetType()->GetTypeID();
	int nTop = y - 1;
	int nBottom = y + pSet->GetHeight();
	if (!DrawCliffTop(nCliff, x, nTop, pSet->GetWidth(), TRUE) ||
		!DrawCliffBottom(nCliff, x, nBottom, pSet->GetWidth(), TRUE))
	{
		ClearFrontMap();
		return FALSE;
	}
	return TRUE;
}

BOOL SMapDrawer::DrawTerrain(int x, int y, const STerrainSet *pSet)
{
	ASSERT(IsValidReadOnlyAddress(pSet));
	if (pSet->GetPattern()->GetType()->GetClass() == STerrainType::NORMAL_SOLID)
		return DrawNormalTerrain(x, y, pSet);
	else
		return DrawCliffTerrain(x, y, pSet);
}

SStoreMap::SStoreMap(int nWidth, int nHeight)
{
	ASSERT(nWidth > 0 && nHeight > 0);
	m_nWidth = nWidth;
	m_nHeight = nHeight;
	m_ppMap = new const STerrain *[m_nWidth * m_nHeight];
	memset(m_ppMap, NULL, sizeof(const STerrain *) * m_nWidth * m_nHeight);
}

SStoreMap::~SStoreMap()
{
	delete [] m_ppMap;
}
