#include "stdafx.h"
#include "Terrain.h"

static void ThrowTerrainException(LPCTSTR lpszMessage);

/////////////////////////////////////////////////////////////////////////////
// STerrainSideInfo

bool STerrainSideInfo::operator ==(const STerrainSideInfo &info) const
{
	for (int i = 0;i < countof(m_anType);i++)
		if (m_anType[i] != -1 && m_anType[i] != info.m_anType[i])
			return false;
	return true;
}

/////////////////////////////////////////////////////////////////////////////
// SPatternID

void SPatternID::GetSideInfo(STerrainSideInfo *pInfo, int nInside, int nOutside) const
{
	pInfo->Fill(nOutside);
	if (inrange(GetPattern(), 0, 2))
		return;

	for (int i = GetStart() - 1;i != GetEnd() - 1;)
	{
		pInfo->SetSideInfo(i, nInside);
		if (++i >= 8)
			i = 0;
	}
}

#ifdef _DEBUG

BOOL SPatternID::IsValidPattern(int nPattern)
{
	return IsValidPattern(nPattern / 10, nPattern % 10);
}

BOOL SPatternID::IsValidPattern(int nStart, int nEnd)
{
	if (nStart == 0 && inrange(nEnd, 0, 2))
		return TRUE;
	if (nStart < 1 || nStart > 8)
		return FALSE;
	if (nEnd < 1 || nEnd > 8)
		return FALSE;
	return TRUE;
}

#endif // _DEBUG

/////////////////////////////////////////////////////////////////////////////
// STerrainID

void STerrainID::GetID(int *pnType, SPatternID *pidPattern, int *pnSet, int *pnIndex) const
{
	ASSERT(IsValidAddress(pnType));
	ASSERT(IsValidAddress(pidPattern));
	ASSERT(IsValidAddress(pnSet));
	ASSERT(IsValidAddress(pnIndex));
	*pnType = GetType();
	*pidPattern = GetPattern();
	*pnSet = GetSet();
	*pnIndex = GetIndex();
}

void STerrainID::SetID(int nType, SPatternID idPattern, int nSet, int nIndex)
{
	ASSERT(inrange(nType, 0, 256));
	ASSERT(inrange(nSet, 0, 256));
	ASSERT(inrange(nIndex, 0, 256));
	m_dwID = MAKELONG(MAKEWORD(nIndex, nSet), MAKEWORD(idPattern.GetPattern(), nType));
}

/////////////////////////////////////////////////////////////////////////////
// STerrainSetKind

void STerrainSetKind::AddKind(int nWidth, int nHeight)
{
	int n = nWidth << 8 | nHeight;
	if (m_arrKind.Find(n) < 0)
		m_arrKind.Add(n);
}

/////////////////////////////////////////////////////////////////////////////
// STerrain

const STerrainType *STerrain::GetType() const
{
	return m_pSet->GetPattern()->GetType();
}

void STerrain::GetSideInfo(STerrainSideInfo *pInfo) const
{
	const STerrainType *pType = m_pSet->GetPattern()->GetType();
	SPatternID id = m_id.GetPattern();
	m_id.GetPattern().GetSideInfo(pInfo, pType->GetInside(), pType->GetOutside());
}

void STerrain::Load(const STerrainSet *pSet, SPatternID idPattern, const SDIBitmap &bmp, int x, int y, int *pnIndex)
{
	Blt(0, 0, bmp, x * STerrain::GetWidth(), y * STerrain::GetHeight(),
		STerrain::GetWidth(), STerrain::GetHeight());

	m_pSet = pSet;

	int nType = GetType()->GetTypeID();
	int nSet = m_pSet->GetSetID();
	m_id.SetID(nType, idPattern, nSet, x + bmp.GetWidth() / STerrain::GetWidth() * y);

	m_idEngine.SetID(nType, (*pnIndex)++);
}

void STerrain::WriteToStream(SOutputStream &out) const
{
	for (int y = 0;y < GetHeight();y++)
		for (int x = 0;x < GetWidth();x++)
		{
			const BYTE *pbyBits = GetBits(x, y);
			BYTE byRed = pbyBits[2] >> 3;
			BYTE byGreen = pbyBits[1] >> 2;
			BYTE byBlue = pbyBits[0] >> 3;

			WORD w = (byRed << 11) | (byGreen << 5) | byBlue;
			out << w;
		}
}

/////////////////////////////////////////////////////////////////////////////
// STerrainSet

void STerrainSet::Load(const STerrainPattern *pPattern, STerrainSetKind *pSetKind, LPCTSTR lpszFileName, int nSet, int *pnIndex)
{
	ASSERT(m_arrTerrain.GetSize() == 0);
	ASSERT(IsValidString(lpszFileName));

	m_pPattern = pPattern;
	m_nSet = nSet;

	SDIBitmap bmp;
	LoadBitmap(lpszFileName, &bmp);

	m_nWidth = bmp.GetWidth() / STerrain::GetWidth();
	m_nHeight = bmp.GetHeight() / STerrain::GetHeight();
	for (int y = 0;y < m_nHeight;y++)
		for (int x = 0;x < m_nWidth;x++)
		{
			STerrain *pTerrain = &m_arrTerrain.Add();
			pTerrain->Load(this, m_pPattern->GetPatternID(), bmp, x, y, pnIndex);
		}

	if (pSetKind != NULL)
		pSetKind->AddKind(m_nWidth, m_nHeight);
}

void STerrainSet::WriteToStream(SOutputStream &out) const
{
	for (int i = 0;i < m_arrTerrain.GetSize();i++)
		out << m_arrTerrain[i];
}

void STerrainSet::LoadBitmap(LPCTSTR lpszFileName, SDIBitmap *pBitmap)
{
	pBitmap->LoadBitmapFile(lpszFileName);
	if (pBitmap->GetBitCount() != 24)
	{
		CString strMessage = lpszFileName;
		strMessage += _T("  24Ʈ ÷ Ʈ  ƴմϴ.");
		ThrowTerrainException(strMessage);
	}
	if (pBitmap->GetWidth() % STerrain::GetWidth() != 0 ||
		pBitmap->GetHeight() % STerrain::GetHeight() != 0)
	{
		CString strMessage = lpszFileName;
		strMessage += _T("   ̰  ʽϴ.");
		ThrowTerrainException(strMessage);
	}
}

/////////////////////////////////////////////////////////////////////////////
// SCliffSet

void SCliffSet::Load(const STerrainPattern *pPattern, STerrainSetKind *pSetKind, LPCTSTR lpszFileName, int nSet, int *pnIndex)
{
	ASSERT(m_arrTerrain.GetSize() == 0);
	ASSERT(IsValidString(lpszFileName));

	m_pPattern = pPattern;
	m_nSet = nSet;

	SDIBitmap bmp;
	LoadBitmap(lpszFileName, &bmp);

	m_nWidth = bmp.GetWidth() / STerrain::GetWidth();
	m_nHeight = bmp.GetHeight() / STerrain::GetHeight();

	if (m_pPattern->GetPatternID() != 0)
	{
		BOOL bValidBitmap;
		switch (m_pPattern->GetPatternID().GetEnd())
		{
		case 1:
		case 4:
		case 8:
			bValidBitmap = m_nWidth == 1 && m_nHeight == 2;
			break;
		case 2:
		case 3:
		case 5:
		case 6:
		case 7:
			bValidBitmap = m_nWidth == 1 && m_nHeight == 1;
			break;
		default:
			ASSERT(FALSE);
		}
		if (!bValidBitmap)
		{
			CString strMessage = lpszFileName;
			strMessage += _T("   ̰  ʽϴ.");
			ThrowTerrainException(strMessage);
		}
	}

	for (int y = 0;y < m_nHeight;y++)
		for (int x = 0;x < m_nWidth;x++)
		{
			SPatternID idPattern = m_pPattern->GetPatternID();
			if (idPattern != 0)
			{
				switch (m_pPattern->GetPatternID().GetEnd())
				{
				case 1:
					idPattern = y == 0 ? 46 : 28;
					break;
				case 2:
					idPattern = 46;
					break;
				case 3:
					idPattern = 28;
					break;
				case 4:
					idPattern = y == 0 ? 68 : 42;
					break;
				case 5:
					idPattern = 42;
					break;
				case 6:
					idPattern = 68;
					break;
				case 7:
					idPattern = 48;
					break;
				case 8:
					idPattern = y == 0 ? 48 : 0;
					break;
				default:
					ASSERT(FALSE);
				}
				if (m_pPattern->GetPatternID().GetStart() == 2)
					idPattern.Reverse();
			}

			STerrain *pTerrain = &m_arrTerrain.Add();
			pTerrain->Load(this, idPattern, bmp, x, y, pnIndex);
		}

	if (pSetKind != NULL)
		pSetKind->AddKind(m_nWidth, m_nHeight);
}

/////////////////////////////////////////////////////////////////////////////
// STerrainPattern

const STerrainSet *STerrainPattern::GetRandomSet(int nWidth, int nHeight) const
{
	int nCount = rand() % m_arrSet.GetSize();
	for (int i = 0;;)
	{
		if (m_arrSet[i].GetWidth() == nWidth && m_arrSet[i].GetHeight() == nHeight)
			if (nCount-- == 0)
				return &m_arrSet[i];
		if (++i >= m_arrSet.GetSize())
			i = 0;
	}
}

const STerrain *STerrainPattern::GetRandomTerrain() const
{
	int i = rand() % m_arrSet.GetSize();
	return m_arrSet[i].GetTerrain(0);
}

//kkk
void STerrainPattern::Load(const STerrainType *pType, STerrainSetKind *pSetKind, LPCTSTR lpszPath, SPatternID idPattern, int *pnIndex)
{
	ASSERT(m_arrSet.GetSize() == 0);
	ASSERT(IsValidString(lpszPath));

	m_pType = pType;
	m_idPattern = idPattern;
	for (int nSet = 0;;nSet++)
	{
		CString str;
		if (idPattern.GetPattern() == 0)
		{
			str.Format(_T("%s\\B%02d.bmp"), lpszPath, nSet);
			if (!IsFileExisting(str))
				break;
		}
		else if (idPattern.GetPattern() == 1)
		{
			str.Format(_T("%s\\S%02d *.bmp"), lpszPath, nSet);
			SFindFile find;
			//kkk
				ThrowTerrainException("kkk");
			if (!find.FindFirst(str))
				break;
			str.Format(_T("%s\\%s"), lpszPath, find.GetFileName());
		}
		else if (idPattern.GetPattern() == 2)
		{
			str.Format(_T("%s\\%02d.bmp"), lpszPath, nSet);
			if (!IsFileExisting(str))
				break;
		}
		else
		{
			str.Format(_T("%s\\%d%02d.bmp"), lpszPath, idPattern.GetPattern(), nSet);
			if (!IsFileExisting(str))
				break;
		}

		STerrainSet *pSet = &m_arrSet.Add();
		pSet->Load(this, pSetKind, str, nSet, pnIndex);
	}//for_end

	if (m_arrSet.GetSize() == 0)
	{
		CString strMessage = lpszPath;
		strMessage += _T("  Ÿ ϴ.");
		ThrowTerrainException(strMessage);
	}
}

void STerrainPattern::WriteToStream(SOutputStream &out) const
{
	for (int i = 0;i < m_arrSet.GetSize();i++)
		out << m_arrSet[i];
}

const STerrainPattern *STerrainType::GetPattern(SPatternID idPattern) const
{
	int i = m_arrPattern.Find(idPattern);
	ASSERT(i >= 0);
	return &m_arrPattern[i];
}

/////////////////////////////////////////////////////////////////////////////
// SCliffPattern

void SCliffPattern::Load(const STerrainType *pType, STerrainSetKind *pSetKind, LPCTSTR lpszPath, SPatternID idPattern, int *pnIndex)
{
	ASSERT(m_arrSet.GetSize() == 0);
	ASSERT(IsValidString(lpszPath));

	m_pType = pType;
	m_idPattern = idPattern;
	for (int nSet = 0;;nSet++)
	{
		CString str;
		if (idPattern.GetPattern() == 0)
		{
			str.Format(_T("%s\\%02d.bmp"), lpszPath, nSet);
			if (!IsFileExisting(str))
				break;
		}
		else
		{
			LPCTSTR lpszHead1, lpszHead2;
			switch (idPattern.GetStart())
			{
			case 1:
				lpszHead1 = _T("1T");
				break;
			case 2:
				lpszHead1 = _T("1B");
				break;
			case 3:
				lpszHead1 = _T("2T");
				break;
			}
			switch (idPattern.GetEnd())
			{
			case 1:
				lpszHead2 = _T("L1");
				break;
			case 2:
				lpszHead2 = _T("L2");
				break;
			case 3:
				lpszHead2 = _T("L3");
				break;
			case 4:
				lpszHead2 = _T("R1");
				break;
			case 5:
				lpszHead2 = _T("R2");
				break;
			case 6:
				lpszHead2 = _T("R3");
				break;
			case 7:
				lpszHead2 = _T("M1");
				break;
			case 8:
				lpszHead2 = _T("M2");
				break;
			}
			str.Format(_T("%s\\%s%s%02d.bmp"), lpszPath, lpszHead1, lpszHead2, nSet);
			if (!IsFileExisting(str))
				break;
		}

		SCliffSet *pSet = new SCliffSet;
		pSet->Load(this, pSetKind, str, nSet, pnIndex);
		m_arrSet.AddRef(pSet);
	}

	if (m_arrSet.GetSize() == 0)
	{
		CString strMessage = lpszPath;
		strMessage += _T("  Ÿ ϴ.");
		ThrowTerrainException(strMessage);
	}
}

/////////////////////////////////////////////////////////////////////////////
// STerrainType

void STerrainType::SaveForEngine(LPCTSTR lpszPath)
{
	SFile file((CString) lpszPath + _T("\\") + m_strName + _T(".trn"), FM_CREATE | FM_WRITE);

	LPCSTR lpszID = "  Tile";
	file.WriteString(strlen(lpszID), lpszID);

	file << m_nTerrainCount;
	for (int i = 0;i < m_arrPattern.GetSize();i++)
		file << m_arrPattern[i];
}

/////////////////////////////////////////////////////////////////////////////
// STerrainSolidType

void STerrainSolidType::Load(const STerrainManager *pManager, LPCTSTR lpszPath, int nType, LPCTSTR lpszName)
{
	ASSERT(m_strName.IsEmpty());
	ASSERT(m_arrPattern.GetSize() == 0);
	ASSERT(IsValidString(lpszPath));
	ASSERT(IsValidString(lpszName));

	if (pManager->FindType(lpszName) != NULL)
	{
		CString str = lpszName;
		str += _T("  ΰ ̻Դϴ.");
		ThrowTerrainException(str);
	}

	m_nType = nType;
	m_strName = lpszName;
	ASSERT(m_strName.Find('&') < 0);

	m_nInside = m_nOutside = nType;
	m_nTerrainCount = 0;

	STerrainPattern *pPattern = &m_arrPattern.Add();
	pPattern->Load(this, NULL, lpszPath, 0, &m_nTerrainCount);
//kkk
	CString str = lpszPath;
	str += _T("\\S00 *.bmp");
	if (IsFileExisting(str, TRUE))
	{
		pPattern = &m_arrPattern.Add();
    	pPattern->Load(this, &m_kindSet, lpszPath, 1, &m_nTerrainCount);
	}
}

/////////////////////////////////////////////////////////////////////////////
// STerrainPatternType

void STerrainPatternType::Load(const STerrainManager *pManager, LPCTSTR lpszPath, int nType, LPCTSTR lpszName)
{
	ASSERT(m_strName.IsEmpty());
	ASSERT(m_arrPattern.GetSize() == 0);
	ASSERT(IsValidString(lpszPath));
	ASSERT(IsValidString(lpszName));

	if (pManager->FindType(lpszName) != NULL)
	{
		CString str = lpszName;
		str += _T("  ΰ ̻Դϴ.");
		ThrowTerrainException(str);
	}

	m_nType = nType;
	m_strName = lpszName;

	int nSplit = m_strName.Find('&');
	CString strInside, strOutside;
	if (nSplit >= 0)
	{
		strInside = m_strName.Left(nSplit);
		strOutside = m_strName.Mid(nSplit + 1);
	}

	const STerrainType *pInsideType = pManager->FindType(strInside);
	const STerrainType *pOutsideType = pManager->FindType(strOutside);
	if (pInsideType == NULL || pOutsideType == NULL)
	{
		CString strMessage = _T("   ̸Դϴ. - ");
		strMessage += m_strName;
		ThrowTerrainException(strMessage);
	}

	m_nInside = pInsideType->GetTypeID();
	m_nOutside = pOutsideType->GetTypeID();
	m_nTerrainCount = 0;

	for (int nStart = 1;nStart <= 8;nStart++)
		for (int nEnd = 1;nEnd <= 8;nEnd++)
		{
			int nPattern = nStart * 10 + nEnd;
			CString str;
			str.Format(_T("%s\\%d*.bmp"), lpszPath, nPattern);
			if (IsFileExisting(str, TRUE))
			{
				STerrainPattern *pPattern = &m_arrPattern.Add();
				pPattern->Load(this, &m_kindSet, lpszPath, nPattern, &m_nTerrainCount);
			}
		}

	if (m_arrPattern.GetSize() == 0)
	{
		CString strMessage = lpszPath;
		strMessage += _T("  Ÿ ϴ.");
		ThrowTerrainException(strMessage);
	}
}

/////////////////////////////////////////////////////////////////////////////
// SCliffSolidType

void SCliffSolidType::Load(const STerrainManager *pManager, LPCTSTR lpszPath, int nType, LPCTSTR lpszName)
{
	ASSERT(m_strName.IsEmpty());
	ASSERT(m_arrPattern.GetSize() == 0);
	ASSERT(IsValidString(lpszPath));
	ASSERT(IsValidString(lpszName));

	if (pManager->FindType(lpszName) != NULL)
	{
		CString str = lpszName;
		str += _T("  ΰ ̻Դϴ.");
		ThrowTerrainException(str);
	}

	m_nType = nType;
	m_strName = lpszName;
	ASSERT(m_strName.Find('&') < 0);

	m_nInside = m_nOutside = nType;
	m_nTerrainCount = 0;

	SCliffPattern *pPattern = new SCliffPattern;
	pPattern->Load(this, &m_kindSet, lpszPath, 0, &m_nTerrainCount);
	m_arrPattern.AddRef(pPattern);
}

/////////////////////////////////////////////////////////////////////////////
// SCliffPatternType

void SCliffPatternType::Load(const STerrainManager *pManager, LPCTSTR lpszPath, int nType, LPCTSTR lpszName)
{
	ASSERT(m_strName.IsEmpty());
	ASSERT(m_arrPattern.GetSize() == 0);
	ASSERT(IsValidString(lpszPath));
	ASSERT(IsValidString(lpszName));

	if (pManager->FindType(lpszName) != NULL)
	{
		CString str = lpszName;
		str += _T("  ΰ ̻Դϴ.");
		ThrowTerrainException(str);
	}

	m_nType = nType;
	m_strName = lpszName;

	int nSplit = m_strName.Find('&');
	CString strInside, strOutside;
	if (nSplit >= 0)
	{
		strInside = m_strName.Left(nSplit);
		strOutside = m_strName.Mid(nSplit + 1);
	}

	const STerrainType *pInsideType = pManager->FindType(strInside);
	const STerrainType *pOutsideType = pManager->FindType(strOutside);
	if (pInsideType == NULL || pOutsideType == NULL)
	{
		CString strMessage = _T("   ̸Դϴ. - ");
		strMessage += m_strName;
		ThrowTerrainException(strMessage);
	}
	if (pInsideType->GetClass() != CLIFF_SOLID || pOutsideType->GetClass() != NORMAL_SOLID)
	{
		CString strMessage = _T("߸ ŸԴϴ. - ");
		strMessage += m_strName;
		ThrowTerrainException(strMessage);
	}

	m_nInside = pInsideType->GetTypeID();
	m_nOutside = pOutsideType->GetTypeID();
	m_nTerrainCount = 0;

	for (int nStart = 1;nStart <= 2;nStart++)
		for (int nEnd = 1;nEnd <= 8;nEnd++)
		{
			int nPattern = nStart * 10 + nEnd;
			SCliffPattern *pPattern = new SCliffPattern;
			pPattern->Load(this, NULL, lpszPath, nPattern, &m_nTerrainCount);
			m_arrPattern.AddRef(pPattern);
		}
}

/////////////////////////////////////////////////////////////////////////////
// STerrainManager

const STerrainType *STerrainManager::FindType(LPCTSTR lpszTypeName) const
{
	int i = m_arrType.Find(lpszTypeName);
	if (i < 0)
		return NULL;
	return &m_arrType[i];
}

const STerrainType *STerrainManager::FindType(int nInside, int nOutside) const
{
	for (int i = 0;i < m_arrType.GetSize();i++)
		if (m_arrType[i].GetInside() == nInside && m_arrType[i].GetOutside() == nOutside)
			return &m_arrType[i];
	return NULL;
}

const STerrain *STerrainManager::GetTerrain(STerrainID id) const
{
	return GetType(id.GetType())->GetPattern(id.GetPattern())->GetSet(id.GetSet())->GetTerrain(id.GetIndex());
}

const STerrainPattern *STerrainManager::FindPattern(const STerrainSideInfo &info) const
{
	for (int nType = 0;nType < GetTypeCount();nType++)
	{
		const STerrainType *pType = GetType(nType);
		for (int nPattern = 0;nPattern < pType->GetPatternCount();nPattern++)
		{
			const STerrainPattern *pPattern = pType->GetPattern(nPattern);
			const STerrain *pTerrain = pPattern->GetSet(0)->GetTerrain(0);

			STerrainSideInfo infoTemp;
			pTerrain->GetSideInfo(&infoTemp);
			if (info == infoTemp)
				return pPattern;
		}
	}
	return NULL;
}

void STerrainManager::Load(LPCTSTR lpszPath)
{
	ASSERT(m_arrType.GetSize() == 0);
	ASSERT(IsValidString(lpszPath));

	for (int nType = 0;;nType++)
	{
		SFindFile find;
		CString str;
		str.Format(_T("%s\\%02d *"), lpszPath, nType);
		if (!find.FindFirst(str))
			break;

		CString strPath = (CString) lpszPath + '\\' + find.GetFileName();
		CString strName = ((CString) find.GetFileName()).Mid(3);

		STerrainType *pType;
		if (strName.Find('&') < 0)
		{
			if (strName.Find(_T("Cliff")) < 0)
				pType = new STerrainSolidType;
			else
				pType = new SCliffSolidType;
		}
		else
		{
			if (strName.Find(_T("Cliff")) < 0)
				pType = new STerrainPatternType;
			else
				pType = new SCliffPatternType;
		}
		pType->Load(this, strPath, nType, strName);
		m_arrType.AddRef(pType);

		if (find.FindNext())
		{
			CString strMessage = _T(" ȣ  ֽϴ. - ");
			strMessage += find.GetFileName();
			ThrowTerrainException(strMessage);
		}
	}

	if (m_arrType.GetSize() == 0)
		ThrowTerrainException(_T("Ÿ ϴ."));
}

void STerrainManager::SaveForEngine(LPCTSTR lpszPath)
{
	for (int i = 0;i < m_arrType.GetSize();i++)
		m_arrType[i].SaveForEngine(lpszPath);
}

/////////////////////////////////////////////////////////////////////////////
// Static Functions

static void ThrowTerrainException(LPCTSTR lpszMessage)
{
	AfxMessageBox(lpszMessage);
	AfxThrowUserException();
}
