#ifndef __TERRAIN_H__
#define __TERRAIN_H__

class STerrainManager;
class STerrainType;
class STerrainPattern;
class STerrainSet;

class STerrainSideInfo
{
// Attributes
public:
	enum
	{
		TOP_LEFT, TOP_RIGHT, RIGHT_TOP, RIGHT_BOTTOM,
		BOTTOM_RIGHT, BOTTOM_LEFT, LEFT_BOTTOM, LEFT_TOP
	};

	int GetSideInfo(int nIndex) const
		{ ASSERT(inrange(nIndex, 0, countof(m_anType)));
			return m_anType[nIndex]; }
	void SetSideInfo(int nIndex, int nType)
		{ ASSERT(inrange(nIndex, 0, countof(m_anType)));
			m_anType[nIndex] = nType; }

	void Fill(int nType)
		{ for (int i = 0;i < countof(m_anType);i++) m_anType[i] = nType; }

	bool operator ==(const STerrainSideInfo &info) const;

// Implementation
private:
	int m_anType[8];
};

class SPatternID
{
public:
	SPatternID()
		{ SetPattern(0); }
	SPatternID(int nPattern)
		{ SetPattern(nPattern); }
	SPatternID(int nStart, int nEnd)
		{ SetPattern(nStart, nEnd); }

// Attributes
public:
	int GetPattern() const
		{ return m_nPattern; }
	void SetPattern(int nPattern)
		{ ASSERT(IsValidPattern(nPattern));
			m_nPattern = nPattern; }
	void SetPattern(int nStart, int nEnd)
		{ ASSERT(IsValidPattern(nStart, nEnd));
			m_nPattern = nStart * 10 + nEnd; }

	int GetStart() const
		{ return m_nPattern / 10; }
	int GetEnd() const
		{ return m_nPattern % 10; }
	void SetStart(int nStart)
		{ SetPattern(nStart, GetEnd()); }
	void SetEnd(int nEnd)
		{ SetPattern(GetStart(), nEnd); }

// Operations
public:
	void GetSideInfo(STerrainSideInfo *pInfo, int nInside, int nOutside) const;
	void Reverse()
		{ SetPattern(GetEnd(), GetStart()); }

	bool operator ==(SPatternID id) const
		{ return m_nPattern == id.m_nPattern; }
	bool operator !=(SPatternID id) const
		{ return m_nPattern != id.m_nPattern; }

// Implementation
private:
	int m_nPattern;

#ifdef _DEBUG
public:
	static BOOL IsValidPattern(int nPattern);
	static BOOL IsValidPattern(int nStart, int nEnd);
#endif // _DEBUG
};

class STerrainID
{
public:
	STerrainID()
		{ SetID(0, 0, 0, 0); }
	STerrainID(int nType, SPatternID idPattern, int nSet, int nIndex)
		{ SetID(nType, idPattern, nSet, nIndex); }

// Attributes
public:
	void GetID(int *pnType, SPatternID *pidPattern, int *pnSet, int *pnIndex) const;
	void SetID(int nType, SPatternID idPattern, int nSet, int nIndex);

	int GetType() const
		{ return (m_dwID & (0xff << 3 * 8)) >> 3 * 8; }
	SPatternID GetPattern() const
		{ return (m_dwID & (0xff << 2 * 8)) >> 2 * 8; }
	int GetSet() const
		{ return (m_dwID & (0xff << 1 * 8)) >> 1 * 8; }
	int GetIndex() const
		{ return (m_dwID & (0xff << 0 * 8)) >> 0 * 8; }
	void SetType(int nType)
		{ SetID(nType, GetPattern(), GetSet(), GetIndex()); }
	void SetPattern(SPatternID idPattern)
		{ SetID(GetType(), idPattern, GetSet(), GetIndex()); }
	void SetSet(int nSet)
		{ SetID(GetType(), GetPattern(), nSet, GetIndex()); }
	void SetIndex(int nIndex)
		{ SetID(GetType(), GetPattern(), GetSet(), nIndex); }

	bool operator ==(const STerrainID &id) const
		{ return m_dwID == id.m_dwID; }

// Implementation
private:
	DWORD m_dwID;
};

class SEngineID
{
public:
	SEngineID()
		{ }
	SEngineID(int nType, int nIndex)
		{ SetID(nType, nIndex); }

// Attributes
public:
	void GetID(int *pnType, int *pnIndex) const
		{ *pnType = m_nType; *pnIndex = m_nIndex; }
	void SetID(int nType, int nIndex)
		{ m_nType = nType; m_nIndex = nIndex; }

	int GetType()
		{ return m_nType; }
	int GetIndex()
		{ return m_nIndex; }

// Implementation
private:
	int m_nType;
	int m_nIndex;
};

class STerrainSetKind
{
// Attributes
public:
	int GetKindCount() const
		{ return m_arrKind.GetSize(); }
	void GetKind(int nIndex, int *pnWidth, int *pnHeight) const
		{ *pnWidth = m_arrKind[nIndex] >> 8;
			*pnHeight = m_arrKind[nIndex] & 0xff; }

// Operations
	void AddKind(int nWidth, int nHeight);

// Implementation
private:
	SArray<int> m_arrKind;
};

class STerrain : public SDIBitmap
{
public:
	STerrain()
		{ Create(GetWidth(), GetHeight(), 24); }

// Attributes
public:
	static int GetWidth()
		{ return 96; }
	static int GetHeight()
		{ return 63; }

	const STerrainSet *GetSet() const
		{ return m_pSet; }
	STerrainID GetID() const
		{ return m_id; }
	SEngineID GetEngineID() const
		{ return m_idEngine; }
	int GetIndex() const
		{ return m_id.GetIndex(); }

	const STerrainType *GetType() const;
	void GetSideInfo(STerrainSideInfo *pInfo) const;

// Operations
public:
	void Load(const STerrainSet *pSet, SPatternID idPattern, const SDIBitmap &bmp, int x, int y, int *pnIndex);
	void WriteToStream(SOutputStream &out) const;

// Implementation
private:
	const STerrainSet *m_pSet;
	STerrainID m_id;
	SEngineID m_idEngine;
};

class STerrainSet
{
// Attributes
public:
	const STerrainPattern *GetPattern() const
		{ return m_pPattern; }
	int GetSetID() const
		{ return m_nSet; }

	int GetWidth() const
		{ return m_nWidth; }
	int GetHeight() const
		{ return m_nHeight; }

	int GetTerrainCount() const
		{ return m_arrTerrain.GetSize(); }
	const STerrain *GetTerrain(int nIndex) const
		{ return &m_arrTerrain[nIndex]; }
	const STerrain *GetTerrain(int x, int y) const
		{ ASSERT(inrange(x, 0, m_nWidth)); ASSERT(inrange(y, 0, m_nHeight));
			return &m_arrTerrain[x + y * m_nWidth]; }

// Operations
public:
	virtual void Load(const STerrainPattern *pPattern, STerrainSetKind *pSetKind, LPCTSTR lpszFileName, int nSet, int *pnIndex);
	virtual void WriteToStream(SOutputStream &out) const;

// Implementation
protected:
	const STerrainPattern *m_pPattern;
	int m_nSet;
	int m_nWidth, m_nHeight;
	SArray<STerrain> m_arrTerrain;

	void LoadBitmap(LPCTSTR lpszFileName, SDIBitmap *pBitmap);
};

class SCliffSet : public STerrainSet
{
// Overrides
public:
	virtual void Load(const STerrainPattern *pPattern, STerrainSetKind *pSetKind, LPCTSTR lpszFileName, int nSet, int *pnIndex);
};

class STerrainPattern
{
// Attributes
public:
	const STerrainType *GetType() const
		{ return m_pType; }
	SPatternID GetPatternID() const
		{ return m_idPattern; }

	int GetSetCount() const
		{ return m_arrSet.GetSize(); }
	const STerrainSet *GetSet(int nIndex) const
		{ return &m_arrSet[nIndex]; }

	const STerrainSet *GetRandomSet(int nWidth, int nHeight) const;
	const STerrain *GetRandomTerrain() const;

// Operations
public:
	virtual void Load(const STerrainType *pType, STerrainSetKind *pSetKind, LPCTSTR lpszPath, SPatternID idPattern, int *pnIndex);
	virtual void WriteToStream(SOutputStream &out) const;

	bool operator ==(SPatternID idPattern) const
		{ return m_idPattern == idPattern; }

// Implementation
protected:
	const STerrainType *m_pType;
	SPatternID m_idPattern;
	SRefArray<STerrainSet> m_arrSet;
};

class SCliffPattern : public STerrainPattern
{
// Overrides
public:
	virtual void Load(const STerrainType *pType, STerrainSetKind *pSetKind, LPCTSTR lpszPath, SPatternID idPattern, int *pnIndex);
};

class STerrainType
{
// Attributes
public:
	int GetTypeID() const
		{ return m_nType; }
	int GetInside() const
		{ return m_nInside; }
	int GetOutside() const
		{ return m_nOutside; }
	LPCTSTR GetName() const
		{ return m_strName; }

	int GetPatternCount() const
		{ return m_arrPattern.GetSize(); }
	const STerrainPattern *GetPattern(int nIndex) const
		{ return &m_arrPattern[nIndex]; }
	const STerrainPattern *GetPattern(SPatternID idPattern) const;

	int GetTerrainCount() const
		{ return m_nTerrainCount; }
	const STerrain *GetFirstTerrain() const
		{ return GetPattern(0)->GetSet(0)->GetTerrain(0); }

	const STerrainSetKind *GetSetKind() const
		{ return &m_kindSet; }

	enum { NORMAL_SOLID, NORMAL_PATTERN, CLIFF_SOLID, CLIFF_PATTERN };
	virtual UINT GetClass() const = 0;
	virtual BOOL IsSolidClass() const = 0;

// Operations
public:
	virtual void Load(const STerrainManager *pManager, LPCTSTR lpszPath, int nType, LPCTSTR lpszName) = 0;
	void SaveForEngine(LPCTSTR lpszPath);

	bool operator ==(LPCTSTR lpszTypeName) const
		{ return lstrcmp(m_strName, lpszTypeName) == 0; }

// Implementation
protected:
	int m_nType, m_nInside, m_nOutside;
	int m_nTerrainCount;
	CString m_strName;
	SRefArray<STerrainPattern, const STerrainPattern &, SPatternID> m_arrPattern;
	STerrainSetKind m_kindSet;
};

class STerrainSolidType : public STerrainType
{
// Overrides
public:
	virtual UINT GetClass() const
		{ return NORMAL_SOLID; }
	virtual BOOL IsSolidClass() const
		{ return TRUE; }
	virtual void Load(const STerrainManager *pManager, LPCTSTR lpszPath, int nType, LPCTSTR lpszName);
};

class STerrainPatternType : public STerrainType
{
// Overrides
public:
	virtual UINT GetClass() const
		{ return NORMAL_PATTERN; }
	virtual BOOL IsSolidClass() const
		{ return FALSE; }
	virtual void Load(const STerrainManager *pManager, LPCTSTR lpszPath, int nType, LPCTSTR lpszName);
};

class SCliffSolidType : public STerrainType
{
// Overrides
public:
	virtual UINT GetClass() const
		{ return CLIFF_SOLID; }
	virtual BOOL IsSolidClass() const
		{ return TRUE; }
	virtual void Load(const STerrainManager *pManager, LPCTSTR lpszPath, int nType, LPCTSTR lpszName);
};

class SCliffPatternType : public STerrainType
{
// Overrides
public:
	virtual UINT GetClass() const
		{ return CLIFF_PATTERN; }
	virtual BOOL IsSolidClass() const
		{ return FALSE; }
	virtual void Load(const STerrainManager *pManager, LPCTSTR lpszPath, int nType, LPCTSTR lpszName);
};

class STerrainManager
{
public:
	STerrainManager()
		{ srand(time(NULL)); }

// Attributes
public:
	int GetTypeCount() const
		{ return m_arrType.GetSize(); }
	const STerrainType *GetType(int nIndex) const
		{ return &m_arrType[nIndex]; }

	const STerrainType *FindType(LPCTSTR lpszTypeName) const;
	const STerrainType *FindType(int nInside, int nOutside) const;

	const STerrain *GetTerrain(STerrainID id) const;

	const STerrainPattern *FindPattern(const STerrainSideInfo &info) const;

// Operations
public:
	void Load(LPCTSTR lpszPath);
	void SaveForEngine(LPCTSTR lpszPath);

// Implementation
private:
	SRefArray<STerrainType, const STerrainType &, LPCTSTR> m_arrType;
};

#endif // __TERRAIN_H__
