// Scene.cpp: implementation of the CScene class.
//
//////////////////////////////////////////////////////////////////////

#include "Scene.h"

#include "Camera.h"
#include "Rasterizer.h"
#include "Font.h"

#include "..\Common\Timer.h"

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

CScene::CScene()
{
	m_fFPS = 0.0f;

	m_pCamera = NULL;

	m_pRasterizer = NULL;
	m_pFont = NULL;

	// Setup Value
	m_iDriver = GL_DEFAULT;
	//m_iDriver = GL_VOODOO;
}

CScene::~CScene()
{
	Destroy();
}

void CScene::Destroy()
{
	CKernel::Destroy();

	///////////////////////////////////////////////////////////////////////////    
	Timer(TIMER_STOP);

	if (m_pCamera)
	{
		delete m_pCamera;
		m_pCamera = NULL;
	}

	if (m_pFont)
	{
		delete m_pFont;
		m_pFont = NULL;
	}

	if (m_pRasterizer)
	{
		m_pRasterizer->RestoreDisplayMode();

		delete m_pRasterizer;
		m_pRasterizer = NULL;
	}

	// Unload OpenGL Driver
	UnloadOpenGLDriver();
}

bool CScene::Create(HDC hDC, DWORD dwWidth, DWORD dwHeight, USHORT bitsPerPel)
{
	// Load OpenGL Driver
	if (!LoadOpenGLDriver(m_iDriver))
		return false;

	// Create Renderer
	m_pRasterizer = new CRasterizer;
	if (!m_pRasterizer)
		return false;

	//if (!m_pRasterizer->SetDisplayMode(dwWidth, dwHeight, bitsPerPel))
	//	return false;

	if (!m_pRasterizer->Create(hDC))
		return false;

	m_pRasterizer->Initialize();
	m_pRasterizer->SetViewport(0, 0, dwWidth, dwHeight);

	m_pFont = new CFont;
	if (!m_pFont)
		return false;

	if (!m_pFont->Create("Comic Sans MS", 16))
		return false;

	m_dwWidth  = dwWidth;
	m_dwHeight = dwHeight;

	///////////////////////////////////////////////////////////////////////////
	m_pCamera = new CCamera;
	if (!m_pCamera)
		return false;

	m_pCamera->m_position.z = 10;//1350;

    
	// Setup the app so it can support single-stepping
    Timer(TIMER_START);

	return true;
}

void CScene::Render()
{
	m_pRasterizer->BeginScene();
	m_pRasterizer->SetPerspective(m_dwWidth, m_dwHeight);

	m_pCamera->Apply();
}

void CScene::Present()
{
	m_pRasterizer->SetOrtho(m_dwWidth, m_dwHeight);

	// for textout
	glEnable(GL_BLEND);
	glBlendFunc(GL_ONE, GL_ONE_MINUS_SRC_ALPHA);

		char szText[80];
		wsprintf(szText, "%s", GetCurrentOpenGLDriverName());

		m_pFont->SetColor(RGB(0xff, 0xff, 0));
		m_pFont->TextOutXY(0, 0, szText);

		wsprintf(szText, "FPS: %03d", (int) m_fFPS);

		m_pFont->SetColor(RGB(0, 0xff, 0xff));
		m_pFont->TextOutXY(0, 16, szText);

	glDisable(GL_BLEND);

	m_pRasterizer->EndScene();
}

void CScene::Update()
{
	POINT point;
	GetCursorPos(&point);

	if (GetAsyncKeyState(VK_LBUTTON))
	{
		if (GetAsyncKeyState(VK_LSHIFT))
		{
			m_pCamera->m_rotate.z -= (float) (m_ptMouse.x - point.x);
		}
		else
		{
			m_pCamera->m_rotate.x -= (float) (m_ptMouse.y - point.y);
			m_pCamera->m_rotate.y -= (float) (m_ptMouse.x - point.x);
		}
	}
	if (GetAsyncKeyState(VK_RBUTTON))
	{
		if (GetAsyncKeyState(VK_LSHIFT))
		{
			m_pCamera->m_position.z -= (float) (m_ptMouse.y - point.y) / 50.0f;
			m_pCamera->m_position.z -= (float) (m_ptMouse.x - point.x) / 5.0f;
		}
		else
		{
			m_pCamera->m_position.x -= (float) (m_ptMouse.x - point.x) / 50.0f;
			m_pCamera->m_position.y -= (float) (m_ptMouse.y - point.y) / 50.0f;
		}
	}

	m_ptMouse.x = point.x;
	m_ptMouse.y = point.y;

	UpdateFPS();
}

void CScene::UpdateFPS()
{
	static float fLastTime = 0.0f;
    static DWORD dwFrames  = 0L;
    float fTime = Timer(TIMER_GETABSOLUTETIME);
    ++dwFrames;

    // Update the scene stats once per second
    if (fTime - fLastTime > 1.0f)
    {
        m_fFPS    = dwFrames / (fTime - fLastTime);
        fLastTime = fTime;
        dwFrames  = 0L;
	}
}
