//-----------------------------------------------------------------------------
// File: main.cpp	
//
// SDK Version: 8
//
// Desc: Roam Optimizations Demo
//
//       Note: This code uses the D3D Framework helper library.
//
//
// Copyright (c) 2001 Yordan Gyurchev
//-----------------------------------------------------------------------------
#define STRICT
#include <math.h>
#include <stdio.h>
#include "D3DFont.h"
#include "wj_D3DApp.h"
#include "..\warjo_pscript.h"
#include "..\warjo_roam.h"

#define MAX_TRIANGLE_NODES 4*8192

//-----------------------------------------------------------------------------
// Name: class CMyD3DApplication
// Desc: Application class. The base class (CD3DApplication) provides the 
//       generic functionality needed in all Direct3D samples. CMyD3DApplication 
//       adds functionality specific to this sample program.
//-----------------------------------------------------------------------------
class CMyD3DApplication : public CD3DApplication
{
    CD3DFont*   m_pFont;          // Font for drawing text
	IDirect3DVertexBuffer8* pVB;   
	
	//old mouse possition
	int oldxt;
	int oldyt;

	//time stamp
	float ftmp;

	BYTE last;
	D3DXVECTOR3  pl; //place of the camera		
	D3DXVECTOR3  at; //at looking

	//some useless stuff
	float fvvl;
	int inVert;

	//pure script
	PureScript pscr;

	//terrain object
	Terrain tr;

	//texture object
	TextureMngr txmng;

	//name of the map
	char szMap[32];

	//tesselation time
	DWORD dwTessTime;

	//angles
	float frm;
	float vang;
	float vyang;

	// fog parameters
	float fog_g,fog_b,fog_r;
	int  bfog;

	//move flags
	bool bmovef;
	bool bmoveb;

	bool first;

    // Internal member functions
    HRESULT ConfirmDevice( D3DCAPS8*, DWORD, D3DFORMAT );

protected:
    HRESULT OneTimeSceneInit();
    HRESULT InitDeviceObjects();
    HRESULT RestoreDeviceObjects();
    HRESULT InvalidateDeviceObjects();
    HRESULT DeleteDeviceObjects();
    HRESULT Render();
    HRESULT FrameMove();
    HRESULT FinalCleanup();

public:
    LRESULT MsgProc( HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam );
    CMyD3DApplication();
};




//-----------------------------------------------------------------------------
// Name: WinMain()
// Desc: Entry point to the program. Initializes everything, and goes into a
//       message-processing loop. Idle time is used to render the scene.
//-----------------------------------------------------------------------------
INT WINAPI WinMain( HINSTANCE hInst, HINSTANCE, LPSTR, INT )
{
    CMyD3DApplication d3dApp;
	try {
		if( FAILED( d3dApp.Create( hInst ) ) )
			return 0;
    return d3dApp.Run();
	}
	catch (MyException e) {
		MessageBox(NULL,e.GetMessage(),"Demo",MB_OK);
	}
	return 0;
}

//-----------------------------------------------------------------------------
// Name: CMyD3DApplication()
// Desc: Application constructor. Sets attributes for the app.
//-----------------------------------------------------------------------------
CMyD3DApplication::CMyD3DApplication()
{
    m_strWindowTitle    = _T("ROAM Optimizations Demo");
    m_bUseDepthBuffer   = TRUE;
    m_bShowCursorWhenFullscreen = TRUE;

    m_pFont           = new CD3DFont( _T("Arial"), 12, D3DFONT_BOLD );

	ftmp=0.0;	
	frm=0;
	fog_g=fog_b=fog_r=0.0;
	vyang=vang=0.0f;
	vyang=0;
	pl.x=0.0;
	pl.z=+256.0;
	pl.y=64;
	bmovef=false;
	bmoveb=false;
	first=true;
}




//-----------------------------------------------------------------------------
// Name: OneTimeSceneInit()
// Desc: Called during initial app startup, this function performs all the
//       permanent initialization.
//-----------------------------------------------------------------------------
HRESULT CMyD3DApplication::OneTimeSceneInit()
{
	//map all the values that will be read from the txt file
	g_pscript.Map(&(Terrain::side),"PatchSide");
	g_pscript.Map(&(tr.iResolution),"TextureResolution");
	g_pscript.Map(&(tr.iNumMips),"TextureNMipMaps");
	g_pscript.Map((char *)&(tr.texname),"TexturePrefix");

	g_pscript.Map(&(Patch::tvar),"TypeOfVariance");

	g_pscript.Map(&(Patch::lVariance),"variance");
	g_pscript.Map(&bfog,"fog");
	g_pscript.Map(&(Patch::sroam),"StaticRoam");

	g_pscript.Map(szMap,"TerrainMap");
	if (!g_pscript.Load("data.txt")) {
		throw MyException("Configuration file 'data.txt' missing");
	}


	TriangleTreeNode::Init(MAX_TRIANGLE_NODES); 
	Patch::pvert=new CUSTOMVERTEX[WJ_VERTEXBUFFERSIZE];

    return S_OK;
}

//-----------------------------------------------------------------------------
// Name: FrameMove()
// Desc: Called once per frame, the call is the entry point for animating
//       the scene.
//-----------------------------------------------------------------------------
HRESULT CMyD3DApplication::FrameMove()
{
	float fTimeKey=m_fTime;
	if (ftmp!=0) {
		float delta;
		delta=fTimeKey-ftmp;
		ftmp=fTimeKey;
		fTimeKey=delta;
	} else ftmp=fTimeKey;

	//is it the first frame?
	if (first) {

		//resset the tessellation
		tr.Reset();
		//reset the triangle node pool
		TriangleTreeNode::Reset();
		first=false;
		}

	//construct the new point and look at
	//this camera bihavior is quite primitive... shame on me..
	at=D3DXVECTOR3(sinf(vang) , 0, -cosf(vang));
	D3DXVECTOR3 yat=D3DXVECTOR3(0,-cosf(vyang),sinf(vyang));

	D3DXVECTOR3 vFrom( pl.x, pl.y, pl.z );
	D3DXVECTOR3 vAt( pl.x +at.x, pl.y +vyang, at.z+pl.z);
	at=vAt-vFrom;
	if (bmovef) pl+=(at*m_fTime/5.0f);
	if (bmoveb) pl-=(at*m_fTime/5.0f);

    D3DXVECTOR3 vUp( 0, 1.0f, 0);

	//the view matrix
	D3DXMATRIX matView;
	D3DXMatrixLookAtLH((D3DXMATRIX*) &matView, (D3DXVECTOR3*)&vFrom,
		(D3DXVECTOR3*)&vAt,(D3DXVECTOR3*) &vUp );
	
	//set the view matrix
    m_pd3dDevice->SetTransform( D3DTS_VIEW, &matView );

	if (!Patch::sroam) {
		// if dynamic roam then set the current transform to our frustum class
		// and update the tesselation
		Patch::frst.SetViewTransform((D3DXMATRIX)matView);
		tr.Tesselate();
	}
    return S_OK;
}

//-----------------------------------------------------------------------------
// Name: Render()
// Desc: Called once per frame, the call is the entry point for 3d
//       rendering. This function sets up render states, clears the
//       viewport, and renders the scene.
//-----------------------------------------------------------------------------
HRESULT CMyD3DApplication::Render()
{

    // Clear the viewport
    m_pd3dDevice->Clear( 0L, NULL, D3DCLEAR_TARGET|D3DCLEAR_ZBUFFER,
                         0x000000ff, 1.0f, 0L );

    // Begin the scene
    if( SUCCEEDED( m_pd3dDevice->BeginScene() ) )
    {
        // Set render mode to lit, solid triangles
        m_pd3dDevice->SetRenderState( D3DRS_FILLMODE, D3DFILL_SOLID );
        m_pd3dDevice->SetRenderState( D3DRS_LIGHTING, FALSE );
		m_pd3dDevice->SetRenderState( D3DRS_CULLMODE ,D3DCULL_CW);
		
		//vertex shader
		m_pd3dDevice->SetVertexShader( D3DFVF_CUSTOMVERTEX );

		//dynamic or static?
		if (!Patch::sroam) {

			//dynamic roam  - render
			tr.Render();

			//output stats
			char sztmp[128];
			sprintf(sztmp,"Vertices:%d  TriNodes:%d",Patch::vcnt,	TriangleTreeNode::GetTriCount());
		    m_pFont->DrawText( 2,  40, D3DCOLOR_ARGB(255,255,255,0), sztmp );

			sprintf(sztmp,"Triangles: %d",g_texture.GetTriangleCount());
			m_pFont->DrawText( 2,  60, D3DCOLOR_ARGB(255,255,255,0), sztmp );
		}
		else {
			// static - just issue draw primitive calls

			//set the vertex stream
			HRESULT hr=m_pd3dDevice->SetStreamSource(0,pVB,sizeof(CUSTOMVERTEX));

			//iterate thorugh all index buffers and render them 
			//realy realy brute force...
			for(int i=0;i<g_texture.GetCount();i++) {
				IDirect3DIndexBuffer8* pIB=g_texture.GetIndexStream(i);
				if (pIB==NULL) continue;
				m_pd3dDevice->SetIndices(pIB,0);
				m_pd3dDevice->SetTexture(0,g_texture.GetTexture(i));
				WORD wt=g_texture.GetTris(i);
				m_pd3dDevice->DrawIndexedPrimitive(D3DPT_TRIANGLELIST,
					0,inVert,0,wt);
				}
			}

		// Output fps statistics
        m_pFont->DrawText( 2,  0, D3DCOLOR_ARGB(255,255,255,0), m_strFrameStats );
        m_pFont->DrawText( 2, 20, D3DCOLOR_ARGB(255,255,255,0), m_strDeviceStats );

        // End the scene.
        m_pd3dDevice->EndScene();
    }

    return S_OK;
}

//-----------------------------------------------------------------------------
// Name: InitDeviceObjects()
// Desc: Initialize scene objects.
//-----------------------------------------------------------------------------
HRESULT CMyD3DApplication::InitDeviceObjects()
{
    // Restore the font
    m_pFont->InitDeviceObjects( m_pd3dDevice );

	//init devices in the texutre manager
	g_texture.InitDevices(m_pd3dDevice);

	//load the map and if dynamic load textures 
	tr.Init(szMap);   
	
	return S_OK;
}

//-----------------------------------------------------------------------------
// Name: RestoreDeviceObjects()
// Desc: Initialize scene objects.
//-----------------------------------------------------------------------------
HRESULT CMyD3DApplication::RestoreDeviceObjects()
{
    m_pFont->RestoreDeviceObjects();
	
	//recreate textures and index buffers
	if (!Patch::sroam)
		g_texture.Recreate();
	else {
		if (!g_texture.LoadIndices("indices.raw")) //load inidices
			throw MyException("File 'indices.raw' missing");
	}



    // Set up the textures
    m_pd3dDevice->SetTextureStageState( 0, D3DTSS_COLORARG1, D3DTA_TEXTURE );
    m_pd3dDevice->SetTextureStageState( 0, D3DTSS_COLORARG2, D3DTA_DIFFUSE );
    m_pd3dDevice->SetTextureStageState( 0, D3DTSS_COLOROP,   D3DTOP_MODULATE );
    m_pd3dDevice->SetTextureStageState( 0, D3DTSS_MINFILTER, D3DTEXF_LINEAR );
    m_pd3dDevice->SetTextureStageState( 0, D3DTSS_MAGFILTER, D3DTEXF_LINEAR );

    m_pd3dDevice->SetTextureStageState( 0, D3DTSS_ADDRESSU, D3DTADDRESS_MIRROR);
	m_pd3dDevice->SetTextureStageState( 0, D3DTSS_ADDRESSV, D3DTADDRESS_MIRROR);

    // Set miscellaneous render states
    m_pd3dDevice->SetRenderState( D3DRS_DITHERENABLE,   FALSE );
    m_pd3dDevice->SetRenderState( D3DRS_SPECULARENABLE, FALSE );
    m_pd3dDevice->SetRenderState( D3DRS_ZENABLE,        TRUE );
    m_pd3dDevice->SetRenderState( D3DRS_AMBIENT,        0x00444444 );

    // Set the world matrix
    D3DXMATRIX matIdentity;
    D3DXMatrixIdentity( &matIdentity );
    m_pd3dDevice->SetTransform( D3DTS_WORLD,  &matIdentity );

    m_pd3dDevice->SetRenderState( D3DRS_LIGHTING, FALSE);

    // Set the projection matrix
    D3DXMATRIX matProj;
    FLOAT fAspect = ((FLOAT)m_d3dsdBackBuffer.Width) / m_d3dsdBackBuffer.Height;
    D3DXMatrixPerspectiveFovLH( &matProj, 1.57f, 1.0f, 1.0f, 1000.0f );
    m_pd3dDevice->SetTransform( D3DTS_PROJECTION, &matProj );

	//set presprctive specs to our user frustum
	Patch::frst.SetPerspectiveSpecs(1.57f/2, 1.0f, 1.0f, 800.0f );

    // Setup a material
    D3DMATERIAL8 mtrl;
    ZeroMemory( &mtrl, sizeof(D3DMATERIAL8) );
    mtrl.Diffuse.r=1.0f;
	mtrl.Diffuse.g=1.0f; 
	mtrl.Diffuse.b=1.0f; 
	mtrl.Diffuse.a=1.0f;
    m_pd3dDevice->SetMaterial( &mtrl );
    m_pd3dDevice->SetRenderState( D3DRS_LIGHTING, FALSE ); //no lighting

	//dynamic - init devices
	tr.InitDevices(m_pd3dDevice); 

	//static load the vertex buffer from the raw file
	FILE *fp=fopen("vertices.raw","rb");

	if (fp==NULL) 
		throw MyException("File 'vertices.raw' missing");

	//get the length of the file
	fseek(fp,0,SEEK_END);
	long t=ftell(fp);
	fseek(fp,0,SEEK_SET);

	//read the file in a memory buffer nad close it
	BYTE *pb=(BYTE *)malloc(t);
	fread(pb,1,t,fp);
	fclose(fp);	

	//create the static vertex buffer for the static algorithm
	m_pd3dDevice->CreateVertexBuffer(t,D3DUSAGE_WRITEONLY,
									D3DFVF_CUSTOMVERTEX,D3DPOOL_MANAGED,
									&pVB);

	//it has been saved by directx7 LVERTEX FVF so we need some ajustment in reading
	inVert=t/32;
	BYTE *pvb;
	//lock, write, unlock 
	pVB->Lock(0,t,&pvb,0);
	for(int i=0;i<inVert;i++) {
		memmove(pvb+i*sizeof(CUSTOMVERTEX),pb+i*32,12); //xyz
		memmove(pvb+i*sizeof(CUSTOMVERTEX)+12,pb+i*32+16,4); //color 
		memmove(pvb+i*sizeof(CUSTOMVERTEX)+16,pb+i*32+24,8); //tu,tv 
	}
	pVB->Unlock();

	//free memory used
	free(pb);

    return S_OK;
}

//-----------------------------------------------------------------------------
// Name: InvalidateDeviceObjects()
// Desc:
//-----------------------------------------------------------------------------
HRESULT CMyD3DApplication::InvalidateDeviceObjects()
{
    m_pFont->InvalidateDeviceObjects();

	//cleanup all device dependent stuff
	tr.ClearDevices();
	pVB->Release();
	g_texture.FreeDevices();
	return S_OK;
}

//-----------------------------------------------------------------------------
// Name: DeleteDeviceObjects()
// Desc: Called when the app is exiting, or the device is being changed,
//       this function deletes any device dependent objects.
//-----------------------------------------------------------------------------
HRESULT CMyD3DApplication::DeleteDeviceObjects()
{
    m_pFont->DeleteDeviceObjects();
    return S_OK;
}

//-----------------------------------------------------------------------------
// Name: FinalCleanup()
// Desc: Called before the app exits, this function gives the app the chance
//       to cleanup after itself.
//-----------------------------------------------------------------------------
HRESULT CMyD3DApplication::FinalCleanup()
{
    SAFE_DELETE( m_pFont );
    return S_OK;
}

//-----------------------------------------------------------------------------
// Name: ConfirmDevice()
// Desc: Called during device intialization, this code checks the device
//       for some minimum set of capabilities
//-----------------------------------------------------------------------------
HRESULT CMyD3DApplication::ConfirmDevice( D3DCAPS8* pCaps, DWORD dwBehavior,
                                          D3DFORMAT Format )
{
    if( dwBehavior & D3DCREATE_PUREDEVICE )
        return E_FAIL; // GetTransform doesn't work on PUREDEVICE

    // If this is a TnL device, make sure it supports directional lights
    if( (dwBehavior & D3DCREATE_HARDWARE_VERTEXPROCESSING ) ||
        (dwBehavior & D3DCREATE_MIXED_VERTEXPROCESSING ) )
    {
        if( !(pCaps->VertexProcessingCaps & D3DVTXPCAPS_DIRECTIONALLIGHTS ) )
            return E_FAIL;
    }

    return S_OK;
}

//-----------------------------------------------------------------------------
// Name: MsgProc()
// Desc: Overrrides the main WndProc, so the sample can do custom message
//       handling (e.g. processing mouse, keyboard, or menu commands).
//-----------------------------------------------------------------------------
LRESULT CMyD3DApplication::MsgProc( HWND hWnd, UINT msg, WPARAM wParam,
                                    LPARAM lParam )
{
    switch( msg )
    {
		case WM_KEYDOWN:
			if (wParam==VK_F1) {
				InvalidateDeviceObjects();
				Patch::sroam=(Patch::sroam)?0:1;
				RestoreDeviceObjects();
			}

			// reload the data.txt file - F3
			// some realy nice GPFs can be achieved using this feature
			if (wParam==VK_F3) 	g_pscript.Load("data.txt");

			// toggle full screen
			if (wParam==VK_F4) {
				ToggleFullscreen();
			}

			//move flags
			if (wParam==VK_UP) bmovef=true;
			if (wParam==VK_DOWN) bmoveb=true;
			 break;

		case WM_KEYUP:
			//move flags cleanup
			if (wParam==VK_UP) bmovef=false;
			if (wParam==VK_DOWN) bmoveb=false;
			break;

		case WM_LBUTTONDOWN:
			oldxt=LOWORD(lParam);
			oldyt=HIWORD(lParam);
			break;

		case WM_MOUSEMOVE:
			//mouse move change the camera orientation
			if (wParam)
			{
				int t=LOWORD(lParam);
				t=t-oldxt;
				oldxt=LOWORD(lParam);
				vang-=(float)t/100.0f;
				t=HIWORD(lParam);
				t=t-oldyt;
				oldyt=HIWORD(lParam);
				vyang+=(float)t/100.0f;
			}
			break;
    }

    return CD3DApplication::MsgProc( hWnd, msg, wParam, lParam );
}





