//-----------------------------------------------------------------------------
// File: Vertices.cpp
//
// Desc: In this tutorial, we are rendering some vertices. This introduces the
//       concept of the vertex buffer, a Direct3D object used to store
//       vertices. Vertices can be defined any way we want by defining a
//       custom structure and a custom FVF (flexible vertex format). In this
//       tutorial, we are using vertices that are transformed (meaning they
//       are already in 2D window coordinates) and lit (meaning we are not
//       using Direct3D lighting, but are supplying our own colors).
//
// Copyright (c) Microsoft Corporation. All rights reserved.
//-----------------------------------------------------------------------------
#include <d3d9.h>
#include <d3dx9.h>
#include <stdio.h>

#include <mmsystem.h>


#pragma comment(lib, "D3D9.lib")
#pragma comment(lib, "D3Dx9.lib")
#pragma comment(lib, "Winmm.lib")

DWORD LastQueryFPS;
int FPSCount;
double FPS;


//-----------------------------------------------------------------------------
// Global variables
//-----------------------------------------------------------------------------
LPDIRECT3D9             g_pD3D       = NULL; // Used to create the D3DDevice
LPDIRECT3DDEVICE9       g_pd3dDevice = NULL; // Our rendering device
LPDIRECT3DVERTEXBUFFER9 g_pVB        = NULL; // Buffer to hold vertices


// A structure for our custom vertex type
struct CUSTOMVERTEX
{
    FLOAT x, y, z, rhw; // The transformed position for the vertex
    DWORD color;        // The vertex color
};

// Our custom FVF, which describes our custom vertex structure
#define D3DFVF_CUSTOMVERTEX (D3DFVF_XYZRHW|D3DFVF_DIFFUSE)




//-----------------------------------------------------------------------------
// Name: InitD3D()
// Desc: Initializes Direct3D
//-----------------------------------------------------------------------------
HRESULT InitD3D( HWND hWnd )
{
    // Create the D3D object.
    if( NULL == ( g_pD3D = Direct3DCreate9( D3D_SDK_VERSION ) ) )
        return E_FAIL;

    // Set up the structure used to create the D3DDevice
    D3DPRESENT_PARAMETERS d3dpp;
    ZeroMemory( &d3dpp, sizeof(d3dpp) );
    d3dpp.Windowed = TRUE;
    d3dpp.SwapEffect = D3DSWAPEFFECT_DISCARD;
    d3dpp.BackBufferFormat = D3DFMT_UNKNOWN;
	
	d3dpp.Flags = D3DPRESENTFLAG_LOCKABLE_BACKBUFFER; //̰ ݵ ߰ ڰ ò??^^;;

    // Create the D3DDevice
    if( FAILED( g_pD3D->CreateDevice( D3DADAPTER_DEFAULT, D3DDEVTYPE_HAL, hWnd,
                                      D3DCREATE_SOFTWARE_VERTEXPROCESSING,
                                      &d3dpp, &g_pd3dDevice ) ) )
    {
        return E_FAIL;
    }
	

    // Device state would normally be set here

    return S_OK;
}




//-----------------------------------------------------------------------------
// Name: InitVB()
// Desc: Creates a vertex buffer and fills it with our vertices. The vertex
//       buffer is basically just a chuck of memory that holds vertices. After
//       creating it, we must Lock()/Unlock() it to fill it. For indices, D3D
//       also uses index buffers. The special thing about vertex and index
//       buffers is that they can be created in device memory, allowing some
//       cards to process them in hardware, resulting in a dramatic
//       performance gain.
//-----------------------------------------------------------------------------
HRESULT InitVB()
{
    // Initialize three vertices for rendering a triangle
    CUSTOMVERTEX vertices[] =
    {
        { 150.0f,  50.0f, 0.5f, 1.0f, 0xffff0000, }, // x, y, z, rhw, color
        { 250.0f, 250.0f, 0.5f, 1.0f, 0xff00ff00, },
        {  50.0f, 250.0f, 0.5f, 1.0f, 0xff00ffff, },
    };

    // Create the vertex buffer. Here we are allocating enough memory
    // (from the default pool) to hold all our 3 custom vertices. We also
    // specify the FVF, so the vertex buffer knows what data it contains.
    if( FAILED( g_pd3dDevice->CreateVertexBuffer( 3*sizeof(CUSTOMVERTEX),
                                                  0, D3DFVF_CUSTOMVERTEX,
                                                  D3DPOOL_DEFAULT, &g_pVB, NULL ) ) )
    {
        return E_FAIL;
    }

    // Now we fill the vertex buffer. To do this, we need to Lock() the VB to
    // gain access to the vertices. This mechanism is required becuase vertex
    // buffers may be in device memory.
    VOID* pVertices;
    if( FAILED( g_pVB->Lock( 0, sizeof(vertices), (void**)&pVertices, 0 ) ) )
        return E_FAIL;
    memcpy( pVertices, vertices, sizeof(vertices) );
    g_pVB->Unlock();

    return S_OK;
}




//-----------------------------------------------------------------------------
// Name: Cleanup()
// Desc: Releases all previously initialized objects
//-----------------------------------------------------------------------------
VOID Cleanup()
{
    if( g_pVB != NULL )        
        g_pVB->Release();

    if( g_pd3dDevice != NULL ) 
        g_pd3dDevice->Release();

    if( g_pD3D != NULL )       
        g_pD3D->Release();
}


//-----------------------------------------------------------------------------
// Name: DrawHDCText()
// Desc: ѱϱ..
//        ̽  ͸  DC   API 
//       ̿Ͽ ڸ Ѵ.
//
//       ********   ********
//       1. DC  ǽ  D3DFMT_R5G6B5, D3DFMT_X1R5G5B5, D3DFMT_R8G8B8, 
//          D3DFMT_X8R8G8B8 ϸ ĸ  信 GetDC  Ұϴ
//
//       2.  ̽  D3DPRESENTFLAG_LOCKABLE_BACKBUFFER÷׸ ݵ
//          ؾѴ.
//
//       3.  ε    ּ^^
//
//-----------------------------------------------------------------------------

HRESULT DrawHDCText(LPDIRECT3DDEVICE9 pDevice, int X, int Y, LPCTSTR Text)
{	
	HRESULT hr;
	HDC hDc;	

	LPDIRECT3DSURFACE9 BackBuffer; //۸  ӽ ǽ̴.
    if(FAILED(hr=pDevice->GetBackBuffer(0, 0, D3DBACKBUFFER_TYPE_MONO,&BackBuffer))) return hr; // 

	BackBuffer->GetDC(&hDc); //DC ´.

	if(hDc != NULL) // ڸ Ѵ.
	{
		SetBkMode(hDc, TRANSPARENT);
		SetTextColor(hDc, RGB(255, 255, 255));
		TextOut(hDc, X, Y, Text, strlen(Text));
		BackBuffer->ReleaseDC(hDc);
	}

	BackBuffer->Release(); //Ѵ.
	return hr;
}


//-----------------------------------------------------------------------------
// Name: Render()
// Desc: Draws the scene
//-----------------------------------------------------------------------------
VOID Render()
{
    // Clear the backbuffer to a blue color
    g_pd3dDevice->Clear( 0, NULL, D3DCLEAR_TARGET, D3DCOLOR_XRGB(0,0,255), 1.0f, 0 );

    // Begin the scene
    if( SUCCEEDED( g_pd3dDevice->BeginScene() ) )
    {
        // Draw the triangles in the vertex buffer. This is broken into a few
        // steps. We are passing the vertices down a "stream", so first we need
        // to specify the source of that stream, which is our vertex buffer. Then
        // we need to let D3D know what vertex shader to use. Full, custom vertex
        // shaders are an advanced topic, but in most cases the vertex shader is
        // just the FVF, so that D3D knows what type of vertices we are dealing
        // with. Finally, we call DrawPrimitive() which does the actual rendering
        // of our geometry (in this case, just one triangle).
        
		g_pd3dDevice->SetStreamSource( 0, g_pVB, 0, sizeof(CUSTOMVERTEX) );
        g_pd3dDevice->SetFVF( D3DFVF_CUSTOMVERTEX );
        g_pd3dDevice->DrawPrimitive( D3DPT_TRIANGLELIST, 0, 1 );	


        // End the scene
        g_pd3dDevice->EndScene();
    }	

	// Fps..
	FPSCount++;
	DWORD Now = GetTickCount();
	if(Now - LastQueryFPS>=1000)
	{
		FPS = (double)FPSCount * 1000.0 / (double)(Now - LastQueryFPS);
		FPSCount = 0;
		LastQueryFPS = Now;
	}


	//
	char FPSText[20];
	sprintf(FPSText, "FPS : %.2f", FPS); 
	DrawHDCText(g_pd3dDevice, 0, 0, FPSText);
	DrawHDCText(g_pd3dDevice, 10, 50, "DC ̿ !@#$%^&*()");
	DrawHDCText(g_pd3dDevice, 10, 70, " M$   ,,");
	DrawHDCText(g_pd3dDevice, 10, 90, "... ӵ ׷  ±...");
	DrawHDCText(g_pd3dDevice, 10, 110, "..  ѱۿ  ġ   ذ   ϴ.");
	DrawHDCText(g_pd3dDevice, 10, 130, "׷ ڵ Ͻð  Լ  ļ ~");	
	DrawHDCText(g_pd3dDevice, 100, 180, "... XP ߵǴµ ٸü  𸣰ڱ..");
	DrawHDCText(g_pd3dDevice, 100, 200, " Ϻ ۿ 8.0 ѱۿ 2000Ͽ ȵǴ...");
	DrawHDCText(g_pd3dDevice, 100, 220, "̰͵ ȵǸ  ּ^^");
	DrawHDCText(g_pd3dDevice, 100, 240, "Hitel:wodawoda");
	DrawHDCText(g_pd3dDevice, 100, 260, "Xmaker@hanmail.net");
	DrawHDCText(g_pd3dDevice, 100, 280, ".......ε...^^v");
	DrawHDCText(g_pd3dDevice, 100, 300, "غ  ſ~~^^");
	DrawHDCText(g_pd3dDevice, 200, 340, "ٶ󸶹ٻīŸ");
	DrawHDCText(g_pd3dDevice, 200, 360, "߲");
	DrawHDCText(g_pd3dDevice, 200, 380, "١ڡۡ");
	DrawHDCText(g_pd3dDevice, 200, 400, "ABCDEFGHIJKLMNOPQRSTUVWXYZ");
	DrawHDCText(g_pd3dDevice, 200, 420, "abcdefghijklmnopqrstuvwxyz");
	

    // Present the backbuffer contents to the display
    g_pd3dDevice->Present( NULL, NULL, NULL, NULL );
}






//-----------------------------------------------------------------------------
// Name: MsgProc()
// Desc: The window's message handler
//-----------------------------------------------------------------------------
LRESULT WINAPI MsgProc( HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam )
{
    switch( msg )
    {
        case WM_DESTROY:
            Cleanup();
            PostQuitMessage( 0 );
            return 0;
    }

    return DefWindowProc( hWnd, msg, wParam, lParam );
}




//-----------------------------------------------------------------------------
// Name: WinMain()
// Desc: The application's entry point
//-----------------------------------------------------------------------------
INT WINAPI WinMain( HINSTANCE hInst, HINSTANCE, LPSTR, INT )
{
    // Register the window class
    WNDCLASSEX wc = { sizeof(WNDCLASSEX), CS_CLASSDC, MsgProc, 0L, 0L,
                      GetModuleHandle(NULL), NULL, NULL, NULL, NULL,
                      "D3D Tutorial", NULL };
    RegisterClassEx( &wc );

    // Create the application's window
    HWND hWnd = CreateWindow( "D3D Tutorial", "D3D Tutorial 02: Vertices",
                              WS_OVERLAPPEDWINDOW, 0, 0, 640, 480,
                              GetDesktopWindow(), NULL, wc.hInstance, NULL );

    // Initialize Direct3D
    if( SUCCEEDED( InitD3D( hWnd ) ) )
    {
        // Create the vertex buffer
        if( SUCCEEDED( InitVB() ) )
        {
            // Show the window
            ShowWindow( hWnd, SW_SHOWDEFAULT );
            UpdateWindow( hWnd );

            // Enter the message loop
            MSG msg;
            ZeroMemory( &msg, sizeof(msg) );
            while( msg.message!=WM_QUIT )
            {
                if( PeekMessage( &msg, NULL, 0U, 0U, PM_REMOVE ) )
                {
                    TranslateMessage( &msg );
                    DispatchMessage( &msg );
                }
                else
                    Render();
            }
        }
    }

    UnregisterClass( "D3D Tutorial", wc.hInstance );
    return 0;
}
