/*  setup.c - Digital Sound Interface Kit V1.01a setup program.

    Copyright 1993,94 Carlos Hasan
*/

#include <stdio.h>
#include "sound.h"

#define VideoSeg        0xB800
#define VideoWidth      80
#define VideoHeight     25

#define Black           0
#define Blue            1
#define Green           2
#define Cyan            3
#define Red             4
#define Magenta         5
#define Brown           6
#define LightGray       7
#define DarkGray        8
#define LightBlue       9
#define LightGreen      10
#define LightCyan       11
#define LightRed        12
#define LightMagenta    13
#define Yellow          14
#define White           15
#define Blink           128

#define kbEsc           0x001B
#define kbEnter         0x000D
#define kbUp            0x4800
#define kbDown          0x5000


typedef char TFrame[6];

typedef struct {
    char    *Title;
    int     Option;
    char    *Items[8];
} TMenu;

static TFrame SingleFrame = { 218,191,192,217,196,179 };
static TFrame DoubleFrame = { 201,187,200,188,205,186 };


void SetTextMode(void)
{
    asm {
        mov     ax,0x0003
        int     10h
    }
}

void WaitVR(void)
{
    asm {
        mov     dx,3DAh
    }
    L0:
    asm {
        in      al,dx
        test    al,0x08
        jne     L0
    }
    L1:
    asm {
        in      al,dx
        test    al,0x08
        je      L1
    }
}

void DrawRect(int XLeft,int YTop,int XRight,int YBottom, char C, char Color)
{
    asm {
        push    si
        push    di
        push    ds
        push    es
        mov     ax,VideoSeg
        mov     es,ax
        mov     ax,[YTop]
        mov     dx,VideoWidth
        mul     dx
        add     ax,[XLeft]
        add     ax,ax
        mov     di,ax
        mov     bx,[XRight]
        sub     bx,[XLeft]
        jl      L0
        mov     dx,[YBottom]
        sub     dx,[YTop]
        jl      L0
        inc     bx
        inc     dx
        mov     al,[C]
        mov     ah,[Color]
        cld
    }
    L1:
    asm {
        push    di
        mov     cx,bx
        rep     stosw
        pop     di
        add     di,2*VideoWidth
        dec     dx
        jne     L1
    }
    L0:
    asm {
        pop     es
        pop     ds
        pop     di
        pop     si
    }
}

void DrawBox(int XLeft,int YTop,int XRight,int YBottom, char *Frame, char Color)
{
    asm {
        push    si
        push    di
        push    ds
        push    es
        lds     si,[Frame]
        mov     ax,VideoSeg
        mov     es,ax
        mov     ax,[YTop]
        mov     dx,VideoWidth
        mul     dx
        add     ax,[XLeft]
        add     ax,ax
        mov     di,ax
        mov     bx,[XRight]
        sub     bx,[XLeft]
        dec     bx
        jl      L0
        mov     dx,[YBottom]
        sub     dx,[YTop]
        dec     dx
        jl      L0
        mov     ah,[Color]
        cld
        push    di
        mov     al,[si+0]
        stosw
        mov     al,[si+4]
        mov     cx,bx
        rep     stosw
        mov     al,[si+1]
        stosw
        pop     di
        add     di,2*VideoWidth
    }
    L1:
    asm {
        push    di
        mov     al,[si+5]
        stosw
        mov     al,20h
        mov     cx,bx
        rep     stosw
        mov     al,[si+5]
        stosw
        pop     di
        add     di,2*VideoWidth
        dec     dx
        jne     L1
        mov     al,[si+2]
        stosw
        mov     al,[si+4]
        mov     cx,bx
        rep     stosw
        mov     al,[si+3]
        stosw
    }
    L0:
    asm {
        pop     es
        pop     ds
        pop     di
        pop     si
    }
}

void DrawText(int X,int Y, char *Text, char Color)
{
    asm {
        push    si
        push    di
        push    ds
        push    es
        lds     si,[Text]
        mov     ax,VideoSeg
        mov     es,ax
        mov     ax,[Y]
        mov     dx,VideoWidth
        mul     dx
        add     ax,[X]
        add     ax,ax
        mov     di,ax
        mov     ah,[Color]
    }
    L1:
    asm {
        mov     al,[si]
        inc     si
        test    al,al
        je      L0
        mov     [es:di],ax
        add     di,2
        jmp     L1
    }
    L0:
    asm {
        pop     es
        pop     ds
        pop     di
        pop     si
    }
}

int StrLen(char *S)
{
    asm {
        push    di
        push    es
        les     di,[S]
        xor     ax,ax
    }
    L1:
    asm {
        mov     dl,[es:di]
        test    dl,dl
        je      L0
        inc     di
        inc     ax
        jne     L1
    }
    L0:
    asm {
        pop     es
        pop     di
    }
    return _AX;
}

int ReadKey(void)
{
    asm {
        mov     ah,00h
        int     16h
        test    al,al
        je      L0
        xor     ah,ah
    }
    L0:
    return _AX;
}

void DrawDesktop(void)
{
    DrawRect(0,0,79,24,178,LightGray);
    DrawBox(0,0,79,2,DoubleFrame,White | Blue << 4);
    DrawRect(0,24,79,24,32,White | Blue << 4);
    DrawText(22,1,"Digital Sound System Setup Program",White | Blue << 4);
    DrawText(21,24,"Copyright (C) 1993,1994 Carlos Hasan",White | Blue << 4);
}

void DrawMenu(const TMenu *Menu)
{
    int I,J,X,Y,Width,Height;

    Width = StrLen(Menu->Title);
    Height = 0;
    for (I = 0; I < 8; I++) {
        if (Menu->Items[I] == NULL) break;
        J = StrLen(Menu->Items[I]);
        if (Width < J) Width = J;
        Height++;
    }
    Width += 6;
    Height += 2;

    X = (VideoWidth-Width) >> 1;
    Y = (VideoHeight-Height) >> 1;
    DrawBox(X,Y,X+Width-1,Y+Height-1,SingleFrame,White | Blue << 4);
    DrawText(X+((Width-StrLen(Menu->Title)) >> 1),Y,Menu->Title,White | Blue << 4);

    for (I = 0; I < 8; I++) {
        if (Menu->Items[I] == NULL) break;
        J = StrLen(Menu->Items[I]);
        if (Menu->Option == I) {
            DrawRect(X+1,Y+1+I,X+Width-2,Y+1+I,32,Blue | LightGray << 4);
            DrawText(X+((Width-J) >> 1),Y+1+I,Menu->Items[I],Blue | LightGray << 4);
        }
        else {
            DrawText(X+((Width-J) >> 1),Y+1+I,Menu->Items[I],White | Blue << 4);
        }
    }
}

int DoMenu(TMenu *Menu)
{
    int Key,Option;

    Option = Menu->Option;
    do {
        WaitVR();
        DrawMenu(Menu);
        Key = ReadKey();
        switch (Key) {
        case kbUp:
            if (Menu->Option > 0) Menu->Option--;
            break;
        case kbDown :
            if ((Menu->Option < 7) && (Menu->Items[Menu->Option+1] != NULL)) Menu->Option++;
            break;
        }
    } while ((Key != kbEsc) && (Key != kbEnter));

    if (Key==kbEsc) Menu->Option = Option;
    return (Key==kbEsc);
}


static TMenu MainMenu =
    { " Main Menu ",
      0,
      { "Select Soundcard",
        "Save and Exit",
         NULL,
         NULL,
         NULL,
         NULL,
         NULL,
         NULL }
    };

static TMenu CardMenu =
    { " Select Soundcard ",
      1,
      { "None               ",
        "Sound Blaster      ",
        "Sound Blaster 2.01 ",
        "Sound Blaster Pro  ",
        "Sound Blaster 16   ",
        "Gravis Ultrasound  ",
        NULL,
        NULL }
    };

static TMenu PortMenu =
    { " Select I/O Port ",
      1,
      { "210",
        "220",
        "230",
        "240",
        "250",
        "260",
        "270",
        "280" }
    };

static TMenu IRQLineMenu =
    { " Select IRQ Line ",
      3,
      { "IRQ 2 ",
        "IRQ 3 ",
        "IRQ 5 ",
        "IRQ 7 ",
        "IRQ 10",
        "IRQ 11",
        "IRQ 12",
        "IRQ 15" }
    };

static TMenu DMAChanMenu =
    { " Select DMA Channel ",
      1,
      { "DMA 0",
        "DMA 1",
        "DMA 3",
        NULL,
        NULL,
        NULL,
        NULL,
        NULL }
    };

static TMenu MixRateMenu =
    { " Select Mixing Rate ",
      2,
      { " 8000 Hz",
        "12000 Hz",
        "16000 Hz",
        "20000 hz",
        "22000 Hz",
        "28000 Hz",
        "36000 Hz",
        "44100 Hz" }
    };


int main(void)
{
    DSMCard SoundCard;
    FILE *SoundFile;

    SetTextMode();

    while (1) {
        DrawDesktop();
        if (DoMenu(&MainMenu)) {
            MainMenu.Option = 2;
            break;
        }
        if (MainMenu.Option==1) break;
        if (DoMenu(&CardMenu)) continue;
        if (CardMenu.Option == 0) continue;
        if (DoMenu(&PortMenu)) continue;
        if (DoMenu(&IRQLineMenu)) continue;
        if (DoMenu(&DMAChanMenu)) continue;
        if ((CardMenu.Option != 5)) if (DoMenu(&MixRateMenu)) continue;
        MainMenu.Option = 1;
    }

    SetTextMode();

    if (MainMenu.Option == 1) {
        switch (CardMenu.Option) {
            case 0: SoundCard.CardID = ID_NONE;     break;
            case 1: SoundCard.CardID = ID_SB;       break;
            case 2: SoundCard.CardID = ID_SB201;    break;
            case 3: SoundCard.CardID = ID_SBPRO;    break;
            case 4: SoundCard.CardID = ID_SB16;     break;
            case 5: SoundCard.CardID = ID_GUS;      break;
        }
        switch (PortMenu.Option) {
            case 0: SoundCard.IOAddr = 0x210;       break;
            case 1: SoundCard.IOAddr = 0x220;       break;
            case 2: SoundCard.IOAddr = 0x230;       break;
            case 3: SoundCard.IOAddr = 0x240;       break;
            case 4: SoundCard.IOAddr = 0x250;       break;
            case 5: SoundCard.IOAddr = 0x260;       break;
            case 6: SoundCard.IOAddr = 0x270;       break;
            case 7: SoundCard.IOAddr = 0x280;       break;
        }
        switch (IRQLineMenu.Option) {
            case 0: SoundCard.IRQNum = 2;           break;
            case 1: SoundCard.IRQNum = 3;           break;
            case 2: SoundCard.IRQNum = 5;           break;
            case 3: SoundCard.IRQNum = 7;           break;
            case 4: SoundCard.IRQNum = 10;          break;
            case 5: SoundCard.IRQNum = 11;          break;
            case 6: SoundCard.IRQNum = 12;          break;
            case 7: SoundCard.IRQNum = 15;          break;
        }
        switch (DMAChanMenu.Option) {
            case 0: SoundCard.DRQNum = 0;           break;
            case 1: SoundCard.DRQNum = 1;           break;
            case 2: SoundCard.DRQNum = 3;           break;
        }
        switch (MixRateMenu.Option) {
            case 0: SoundCard.MixRate = 8000;       break;
            case 1: SoundCard.MixRate = 12000;      break;
            case 2: SoundCard.MixRate = 16000;      break;
            case 3: SoundCard.MixRate = 20000;      break;
            case 4: SoundCard.MixRate = 22000;      break;
            case 5: SoundCard.MixRate = 28000;      break;
            case 6: SoundCard.MixRate = 36000;      break;
            case 7: SoundCard.MixRate = 44100;      break;
        }

        if (SoundCard.CardID == ID_SB || SoundCard.CardID == ID_SBPRO) {
          if (SoundCard.MixRate >= 22050) SoundCard.MixRate = 22050;
        }

        if ((SoundFile = fopen("SOUND.CFG","wb")) != NULL) {
            fwrite(&SoundCard,sizeof(SoundCard),1,SoundFile);
            fclose(SoundFile);
            printf("Setup file saved.\n");
        }
        else {
            printf("Error saviing SOUND.CFG file.\n");
        }
    }
    else {
        printf("Setup aborted.\n");
    }
    return 0;
}
