{$X+,V-}
unit FileUnit;

interface

uses
  Objects, Dos;

type
  PFileService = ^TFileService;
  TFileService = object
    Source     : PString;
    Dest       : PString;

    constructor Init(ASource, ADest : string);
    destructor  Done; virtual;
    procedure   SetSource(ASource : string); virtual;
    procedure   SetDest(ADest : string); virtual;
    function    Copy : Boolean; virtual;
    function    Move : Boolean; virtual;
    function    Rename : Boolean; virtual;
    function    Delete : Boolean; virtual;

  private
    function    CopyFile : Boolean;
    function    MoveFile : Boolean;
    function    RenameFile : Boolean;
    function    DeleteFile : Boolean;
  end;

  PWildService = ^TWildService;
  TWildService = object(TFileService)
    SrcDir   : PString;
    SrcSpec  : PString;
    DestDir  : PString;
    DestSpec : PString;
    SubDir   : Boolean;

    constructor Init(ASource, ADest : string);
    destructor  Done; virtual;
    procedure   SetSrcName(ASource : string); virtual;
    procedure   SetDestName(ADest : string); virtual;
    procedure   SetSubDir(ASubDir : Boolean); virtual;
    function    Copy : Boolean; virtual;
    function    Move : Boolean; virtual;
    function    Rename : Boolean; virtual;
    function    Delete : Boolean; virtual;
    function    InSpec(Name : string) : Boolean; virtual;
    function    DestToName(Name : string) : string; virtual;

  private
    function    CopyMoveWild(IsCopy : Boolean) : Boolean;
    function    RenameWild : Boolean;
    function    DeleteWild(Dir : string) : Boolean;
  end;

type
  PFCBName = ^TFCBName;
  TFCBName = record
    Name : array [0..7] of Char;
    Ext  : array [0..2] of Char;
  end;

  PFileInfo = ^TFileInfo;
  TFileInfo = record
    Attr : Byte;
    Time : Longint;
    Size : Longint;
    Name : string[12];
  end;

type
  SeekType = (SeekSet, SeekCur, SeekEnd);

const
  nfDir     = $0100;
  nfName    = $0200;
  nfExt     = $0400;

  nfDirName = nfDir or nfName;
  nfNameExt = nfName or nfExt;

  function  GetName(Path : string; AOptions : Word) : string;
  function  DriveValid(Drive : Char) : Boolean;
  function  PathValid(Path : string) : Boolean;
  function  ValidFileName(FileName : string) : Boolean;
  function  FileInPath(FileName : string) : string;
  function  FileExists(Name : string) : Boolean;
  function  GetFileInfo(Name : string; var FileInfo : TFileInfo) : Boolean;
  function  IsWild(S : String) : Boolean;
  function  IsDir(Name : string) : Boolean;
  procedure Han2FCB(Han : string; var FCB);
  function  FCB2Han(var FCB) : string;
  function  EraseFile(FileName : string) : Word;
  function  GetFileSize(FileName : string) : Longint;
  procedure FSeek(var F : File; Offset : Longint; Whence : SeekType);
  function  FGetC(var F : File) : Integer;
  function  FPutC(var F : File; C : Integer) : Integer;

implementation

type
  PFileBufRec = ^TFileBufRec;
  TFileBufRec = record
    Buffer   : Pointer;
    Size     : Word;
  end;

  PFileBufCollection = ^TFileBufCollection;
  TFileBufCollection = object(TCollection)
    MaxBuffer : Word;

    constructor Init(ALimit, ADelta : Integer; AMaxBuffer : Word);
    procedure   FreeItem(Item : Pointer); virtual;
  end;

function HeapFunc(Size : Word) : Integer; far;
begin
  HeapFunc := 1;
end;

constructor TFileService.Init(ASource, ADest : string);
begin
  SetSource(ASource);
  SetDest(ADest);
end;

destructor TFileService.Done;
begin
  if Source <> nil then DisposeStr(Source);
  if Dest <> nil then DisposeStr(Dest);
end;

procedure TFileService.SetSource(ASource : string);
begin
  if Source <> nil then DisposeStr(Source);

  Source := NewStr(FExpand(ASource));
end;

procedure TFileService.SetDest(ADest : string);
begin
  if Dest <> nil then DisposeStr(Dest);

  Dest := NewStr(FExpand(ADest));
end;

function TFileService.Copy : Boolean;
begin
  Copy := CopyFile;
end;

function TFileService.Move : Boolean;
begin
  Move := MoveFile;
end;

function TFileService.Rename : Boolean;
begin
  Rename := RenameFile;
end;

function TFileService.Delete : Boolean;
begin
  Delete := DeleteFile;
end;

function TFileService.CopyFile : Boolean;
var
  F1, F2        : file;
  EndOfFile     : Boolean;
  Error         : Boolean;
  FileBuffer    : PFileBufCollection;

function Min(A, B : Longint) : Longint;
begin
  if A < B then Min := A
  else Min := B;
end;

procedure ReadFile;

function NewFileBuffer(Buffer : Pointer; Size : Word) :
  PFileBufRec;
var
  FileBuffer : PFileBufRec;

begin
  New(FileBuffer);

  FileBuffer^.Buffer := Buffer;
  FileBuffer^.Size := Size;
  NewFileBuffer := FileBuffer;
end;

var
  Buffer   : Pointer;
  Size     : Word;

begin       (* ReadFile *)
  repeat
    GetMem(Buffer, FileBuffer^.MaxBuffer);
    if Buffer <> nil then
    begin
      {$I-}
      BlockRead(F1, Buffer^, FileBuffer^.MaxBuffer, Size);
      {$I+}
      Error := (IOResult <> 0) and (Size = 0);

      FileBuffer^.Insert(NewFileBuffer(Buffer, Size));
      EndOfFile := Eof(F1);
    end;
  until EndOfFile or Error or (Buffer = nil);
end;

procedure WriteFile;
var
  I : Integer;

begin
  if Error then Exit;

  I := 0;
  while (I < FileBuffer^.Count) and (not Error) do
  begin
    if FileBuffer^.At(I) <> nil then
    begin
      {$I-}
      BlockWrite(F2, PFileBufRec(FileBuffer^.At(I))^.Buffer,
        PFileBufRec(FileBuffer^.At(I))^.Size);
      {$I+}
      Error := IOResult <> 0;
    end;

    Inc(I);
  end;
end;

begin      (* TFileService.CopyFile *)
  CopyFile := False;
  if Source^ = Dest^ then Exit;

  Assign(F1, Source^);
  {$I-}
  Reset(F1, 1);
  {$I+}
  if IOResult <> 0 then Exit;

  Assign(F2, Dest^);
  {$I-}
  Rewrite(F2, 1);
  {$I+}
  if IOResult <> 0 then Exit;

  EndOfFile := False;
  Error := False;

  FileBuffer := New(PFileBufCollection, Init(10, 5, Min(
    MaxInt * 2 + 1, MaxAvail)));

  repeat
    ReadFile;
    WriteFile;

    FileBuffer^.FreeAll;
  until EndOfFile or Error;

  Dispose(FileBuffer, Done);
  Close(F1);
  Close(F2);

  CopyFile := not Error;
end;

function TFileService.MoveFile : Boolean;
begin
  MoveFile := False;

  if Source^ = Dest^ then Exit;

  if Source^[1] = Dest^[1] then MoveFile := RenameFile
  else
    if CopyFile then
      if DeleteFile then MoveFile := True;
end;

function TFileService.RenameFile : Boolean;
var
  F       : File;

begin
  Assign(F, Source^);

  {$I-}
  System.Rename(F, Dest^);
  {$I+}

  RenameFile := IOResult = 0;
end;

function TFileService.DeleteFile : Boolean;
var
  F : File;

begin
  Assign(F, Source^);
  {$I-}
  Erase(F);
  {$I+}

  DeleteFile := IOResult = 0;
end;

constructor TWildService.Init(ASource, ADest : string);
begin
  TFileService.Init('', '');

  SetSrcName(ASource);
  SetDestName(ADest);
  SubDir := False;
end;

destructor TWildService.Done;
begin
  TFileService.Done;

  DisposeStr(SrcDir);
  DisposeStr(SrcSpec);
  DisposeStr(DestDir);
  DisposeStr(DestSpec);
end;

procedure TWildService.SetSrcName(ASource : string);
var
  D     : DirStr;
  N     : NameStr;
  E     : ExtStr;
  Spec  : string;
  I     : Integer;

begin
  DisposeStr(SrcDir);
  DisposeStr(SrcSpec);

  FSplit(FExpand(ASource), D, N, E);

  Spec := N + E;
  I := 1;
  while (Length(Spec) > 0) and (I < Length(Spec)) do
  begin
    if (Spec[I] = '*') and (Spec[I + 1] in ['?', '*']) then
      System.Delete(Spec, I + 1, 1)
    else Inc(I);
  end;

  SrcDir := NewStr(D);
  SrcSpec := NewStr(Spec);
end;

procedure TWildService.SetDestName(ADest : string);
var
  Position : Byte;
  I        : Integer;
  D        : DirStr;
  N        : NameStr;
  E        : ExtStr;

begin
  DisposeStr(DestDir);
  DisposeStr(DestSpec);

  FSplit(FExpand(ADest), D, N, E);

  Position := Pos('*', N);
  if Position > 0 then
  begin
    N[0] := #8;
    for I := Position to 8 do N[I] := '?';
  end;

  Position := Pos('*', E);
  if Position > 0 then
  begin
    E[0] := #4;
    for I := Position to 4 do E[I] := '?';
  end;

  DestDir := NewStr(D);
  DestSpec := NewStr(N + E);
end;

procedure TWildService.SetSubDir(ASubDir : Boolean);
begin
  SubDir := ASubDir;
end;

function TWildService.Copy : Boolean;
begin
  Copy := CopyMoveWild(True);
end;

function TWildService.Move : Boolean;
begin
  Move := CopyMoveWild(False);
end;

function TWildService.Rename : Boolean;
begin
  Rename := RenameWild;
end;

function TWildService.Delete : Boolean;
begin
  Delete := DeleteWild(SrcDir^);
end;

function TWildService.InSpec(Name : string) : Boolean;

function Compare(Name, Spec : string) : Boolean;
var
  I1, I2, J1, J2 : Integer;

begin
  J1 := 1;
  for I1 := 1 to Length(Name) do
  begin
    if J1 > Length(Spec) then
    begin
       Compare := False;
       Exit;
    end;

    if (Name[I1] = Spec[J1]) or (Spec[J1] in ['?', '*']) then
    begin
      if Spec[J1] <> '*' then Inc(J1)
      else
      begin
        if J1 < Length(Spec) then
        begin
          I2 := I1;
          J2 := J1 + 1;
          while (I2 <= Length(Name)) and (J2 <= Length(Spec)) and
                (not (Spec[J2] in ['?', '*'])) and
                (Name[I2] = Spec[J2]) do
          begin
            Inc(I2);
            Inc(J2);
          end;

          if (J2 > Length(Spec)) or
             ((J2 <= Length(Spec)) and (Spec[J2] in ['?', '*'])) then
          begin
            I1 := I2 - 1;
            J1 := J2;
          end;
        end;
      end;
    end
    else
    begin
      Compare := False;
      Exit;
    end;
  end;

  for I1 := J1 to Length(Spec) do
  begin
    if not (Spec[I1] in ['?', '*']) then
    begin
      Compare := False;
      Exit;
    end;
  end;

  Compare := True;
end;

var
  PosName, PosSpec : Byte;
  I                : Integer;

begin           (* TWildService.InSpec *)
  PosName := Pos('.', Name);
  if PosName in [0..1] then PosName := Length(Name) + 1;

  PosSpec := Pos('.', SrcSpec^);

  InSpec := False;
  if Compare(System.Copy(Name, 1, PosName - 1),
             System.Copy(SrcSpec^, 1, PosSpec - 1)) then
    if Compare(System.Copy(Name, PosName + 1, 3),
               System.Copy(SrcSpec^, PosSpec + 1, 3)) then InSpec := True;
end;

function TWildService.DestToName(Name : string) : string;

procedure Change(var S : string; Start, Last : Integer);
var
  I1, I2 : Integer;

begin
  I2 := 1;
  for I1 := Start to Last do
  begin
    if (I2 <= Length(S)) and (S[I2] = '?') then
      S[I2] := Name[I1];

    Inc(I2);
  end;

  while (Length(S) > 0) and (S[Length(S)] = '?') do
    Dec(S[0]);
end;

var
  D : DirStr;
  N : NameStr;
  E : ExtStr;
  P : Byte;

begin
  FSplit(DestSpec^, D, N, E);

  P := Pos('.', Name);
  if P = 0 then P := Length(Name) + 1;

  Change(N, 1, P - 1);

  if E[1] = '.' then System.Delete(E, 1, 1);
  Change(E, P + 1, Length(Name));

  DestToName := N + '.' + E;
end;


function TWildService.CopyMoveWild(IsCopy : Boolean) : Boolean;

function CopyMoveRecur(SDir, TDir : string) : Boolean;
var
  F        : File;
  SInfo    : SearchRec;
  FileInfo : TFileInfo;

begin
  CopyMoveRecur := False;

  FindFirst(SDir + '*.*', AnyFile, SInfo);
  while DosError = 0 do
  begin
    if (SInfo.Name[1] <> '.') and InSpec(SInfo.Name) then
    begin
      if (SInfo.Attr and Directory) = Directory then
      begin
        if SubDir then
        begin
          if GetFileInfo(TDir + DestToName(SInfo.Name), FileInfo) then
            if (FileInfo.Attr and Directory) <> Directory then Exit
            else
          else
          begin
            {$I-}
            MkDir(TDir + DestToName(SInfo.Name));
            {$I+}
            if IOResult <> 0 then Exit;
          end;

          if not CopyMoveRecur(SDir + SInfo.Name + '\',
                              TDir + SInfo.Name + '\') then Exit;
        end;
      end
      else
      begin
        SetSource(SDir + SInfo.Name);
        SetDest(TDir + DestToName(SInfo.Name));

        if IsCopy then
          if not TFileService.Copy then Exit
          else
        else
          if not TFileService.Move then Exit;
      end;

(*
      if ((SInfo.Attr and Directory) <> Directory) or SubDir then
      begin
        Assign(F, TDir + '\' + SInfo.Name);
        {$I-}
        SetFAttr(F, SInfo.Attr);
        {$I+}
        if IOResult <> 0 then Exit;
      end;
*)
    end;

    FindNext(SInfo);
  end;

  CopyMoveRecur := True;
end;

begin
  CopyMoveWild := CopyMoveRecur(SrcDir^, DestDir^);
end;

function TWildService.RenameWild : Boolean;
var
  SInfo : SearchRec;

begin
  RenameWild := False;
  FindFirst(SrcDir^ + '*.*', AnyFile, SInfo);

  while DosError = 0 do
  begin
    if (SInfo.Name[1] <> '.') and InSpec(SInfo.Name) then
    begin
      SetSource(SrcDir^ + SInfo.Name);
      SetDest(DestDir^ + DestToName(SInfo.Name));

      if not TFileService.Rename then Exit;
    end;

    FindNext(SInfo);
  end;

  RenameWild := True;
end;

function TWildService.DeleteWild(Dir : string) : Boolean;
var
  SInfo : SearchRec;

begin
  DeleteWild := False;

  FindFirst(Dir + '*.*', AnyFile, SInfo);
  while DosError = 0 do
  begin
    if (SInfo.Name[1] <> '.') and InSpec(SInfo.Name) then
    begin
      if ((SInfo.Attr and Directory) = Directory) then
        if SubDir then
          if not DeleteWild(Dir + SInfo.Name + '\') then Exit
          else
        else
      else
      begin
        SetSource(Dir + SInfo.Name);
        if not TFileService.Delete then Exit;
      end;
    end;

    FindNext(SInfo);
  end;

  DeleteWild := True;
end;

constructor TFileBufCollection.Init(ALimit, ADelta : Integer;
  AMaxBuffer : Word);
begin
  TCollection.Init(ALimit, ADelta);

  MaxBuffer := AMaxBuffer;
end;

procedure TFileBufCollection.FreeItem(Item : Pointer);
var
  FileBuffer : PFileBufRec absolute Item;

begin
  if FileBuffer <> nil then
  begin
    if FileBuffer^.Buffer <> nil then
      FreeMem(FileBuffer^.Buffer, MaxBuffer);

    Dispose(FileBuffer);
  end;
end;

function GetName(Path : string; AOptions : Word) : string;
var
  FileName : string;
  Dir      : DirStr;
  Name     : NameStr;
  Ext      : ExtStr;

begin
  FileName := '';
  FSplit(Path, Dir, Name, Ext);

  if (AOptions and nfDir) <> 0 then
    FileName := FileName + Dir;

  if (AOptions and nfName) <> 0 then
    FileName := FileName + Name;

  if (AOptions and nfExt) <> 0 then
    FileName := FileName + Ext;

  GetName := FileName;
end;

function DriveValid(Drive: Char): Boolean; assembler;
asm
	MOV	DL,Drive
        MOV	AH,36H
        SUB	DL,'A'-1
        INT	21H
        INC	AX
        JE	@@2
@@1:	MOV	AL,1
@@2:
end;

function PathValid(Path : string) : Boolean;
var
  ExpPath : string;
  F       : File;
  SR      : SearchRec;

begin
  ExpPath := FExpand(Path);
  if Length(ExpPath) <= 3 then PathValid := DriveValid(ExpPath[1])
  else
  begin
    if ExpPath[Length(ExpPath)] = '\' then Dec(ExpPath[0]);
    FindFirst(ExpPath, Directory, SR);
    PathValid := (DosError = 0) and (SR.Attr and Directory <> 0);
  end;
end;

function ValidFileName(FileName : string) : Boolean;
const
  IllegalChars = ';,=+<>|"[]  ';

var
  Dir   : DirStr;
  Name  : NameStr;
  Ext   : ExtStr;

{ Contains returns true if S1 contains any characters in S2 }
function Contains(S1, S2 : String) : Boolean; near; assembler;
asm
	PUSH	DS
        CLD
        LDS	SI,S1
        LES	DI,S2
        MOV	DX,DI
        XOR	AH,AH
        LODSB
        MOV	BX,AX
        OR      BX,BX
        JZ      @@2
        MOV	AL,ES:[DI]
        XCHG	AX,CX
@@1:	PUSH	CX
	MOV	DI,DX
	LODSB
        REPNE	SCASB
        POP	CX
        JE	@@3
	DEC	BX
        JNZ	@@1
@@2:	XOR	AL,AL
	JMP	@@4
@@3:	MOV	AL,1
@@4:	POP	DS
end;

begin
  ValidFileName := True;
  FSplit(FileName, Dir, Name, Ext);
  if not ((Dir = '') or PathValid(Dir)) or Contains(FileName, IllegalChars)
    then ValidFileName := False;
end;

function FileInPath(FileName : string) : string;
begin
  FileInPath := FSearch(FileName, GetEnv('PATH'));
end;

function FileExists(Name : string) : Boolean;
var
  SR : SearchRec;

begin
  FindFirst(Name, AnyFile, SR);
  FileExists := DosError = 0;
end;

function GetFileInfo(Name : string; var FileInfo : TFileInfo) : Boolean;
var
  SR : SearchRec;

begin
  FindFirst(Name, AnyFile, SR);
  Move(SR.Attr, FileInfo, SizeOf(TFileInfo));
  GetFileInfo := DosError = 0;
end;

function IsWild(S : String) : Boolean;
begin
  IsWild := (Pos('?', S) > 0) or (Pos('*', S) > 0);
end;

function IsDir(Name : string) : Boolean;
var
  SR : SearchRec;

begin
  FindFirst(Name, AnyFile, SR);
  IsDir := (DosError = 0) and ((SR.Attr and Directory) = Directory);
end;

procedure Han2FCB(Han : string; var FCB);
var
  FCBName  : TFCBName absolute FCB;
  D        : DirStr;
  N        : NameStr;
  E        : ExtStr;
  Position : Byte;
  I        : Integer;

begin
  FSplit(Han, D, N, E);
  if E[1] = '.' then Delete(E, 1, 1);

  while Length(N) < 8 do N := N + ' ';

  while Length(E) < 3 do E := E + ' ';

  Position := Pos('*', N);
  if Position > 0 then
    for I := Position to 8 do N[I] := '?';

  Position := Pos('*', E);
  if Position > 0 then
    for I := Position to 3 do E[I] := '?';

  Move(N[1], FCBName.Name, 8);
  Move(E[1], FCBName.Ext, 3);
end;

function FCB2Han(var FCB) : string;
var
  FCBName  : TFCBName absolute FCB;
  Han      : string;
  Position : Byte;

begin
  Han[0] := #12;
  Han[9] := '.';
  Position := 8;
  Move(FCBName.Name, Han[1], 8);
  Move(FCBName.Ext, Han[10], 3);

  while (Position > 0) and (Han[Position] = ' ') do
  begin
    Delete(Han, Position, 1);
    Dec(Position);
  end;

  while (Length(Han) > 0) and (Han[Length(Han)] in [' ', '.']) do
    Dec(Han[0]);

  FCB2Han := Han;
end;

function EraseFile(FileName : string) : Word;
var
  F : file;

begin
  Assign(F, FileName);
  {$I-}
  Erase(F);
  {$I+}

  EraseFile := IOResult;
end;

function GetFileSize(FileName : string) : Longint;
var
  FileInfo : TFileInfo;

begin
  if GetFileInfo(FileName, FileInfo) then GetFileSize := FileInfo.Size
  else GetFileSize := -1;
end;

procedure FSeek(var F : File; Offset : Longint; Whence : SeekType);
begin
  case Whence of
    SeekCur : Inc(Offset, FilePos(F));
    SeekEnd : Inc(Offset, FileSize(F));
  end;

  Seek(F, Offset);
end;

function FGetC(var F : File) : Integer;
var
  C : Integer;

begin
  C := 0;
  {$I-}
  BlockRead(F, C, 1);
  {$I+}

  if IOResult <> 0 then
    FGetC := -1
  else
    FGetC := C;
end;

function FPutC(var F : File; C : Integer) : Integer;
begin
  {$I-}
  BlockWrite(F, C, 1);
  {$I+}

  if IOResult <> 0 then
    FPutC := -1
  else
    FPutC := C;
end;

begin
  HeapError := @HeapFunc;
end.