{$X+}
unit WAVUNIT;

INTERFACE

 procedure playWAV(frequency : word; raw_file : string);

IMPLEMENTATION

uses
   dos;

const
   BASE_IO : word = $220;
   IRQ   = 5;
   DMA16 = 5;
   LOAD_CHUNK_SIZE = 8192;
   BUFFER_BLOCK_LENGTH   = 256;

type
   p_move_params = ^move_params_T;
   move_params_T = record
      length       : longint;
      source_handle : word;
      source_offset : longint;
      dest_handle   : word;
      dest_offset   : longint;
   end;

   IO_mode_T = (INPUT, OUTPUT);

   pBuffer = ^buffer_T;
   buffer_T = array[1..2] of array[1..BUFFER_BLOCK_LENGTH] of integer;

var
   XMSDriver: pointer;

   interrupt_count : longint;
   SBIO_done     : boolean;
   current_block : byte;

   reset_port        : word;
   read_port         : word;
   write_port        : word;
   poll_port         : word;
   poll16_port       : word;

   PIC_rotate_port    : word;
   PIC_mask_port      : word;

   DMA_mask_port      : word;
   DMA_clear_ptr_port : word;
   DMA_mode_port      : word;
   DMA_base_address_port  : word;
   DMA_count_port     : word;
   DMA_page_port      : word;

   IRQ_start_mask     : byte;
   IRQ_stop_mask      : byte;
   IRQ_interrupt_vector    : byte;
   interrupt_controller    : byte;

   DMA_start_mask     : byte;
   DMA_stop_mask      : byte;
   DMA_mode           : byte;

   OldIntVector       : pointer;
   OldExitProc        : pointer;

   handler_installed  : boolean;

   memory_area        : pointer;
   memory_area_size   : longint;
   buffer_address     : longint;
   buffer_page        : byte;
   buffer_offset      : word;
   buffer_length      : word;
   block_length       : word;

   IO_mode            : IO_mode_T;

   samples_remaining  : longint;
   sampling_rate      : word;

type
   HandlerProc = procedure;

var
   SBIO_handler : HandlerProc;

   number_of_samples : longint;
   SBPlay_buffer : pBuffer;

   SBPlay_handle : word;
   current_sample : longint;
   move_params : move_params_T;

function XMS_installed: boolean; assembler;
asm
        mov    ax, 4300h
        int    2Fh
        cmp    al, 80h
        jne    @NoXMSDriver
        mov    al, TRUE
        jmp    @Done
   @NoXMSDriver:
        mov    al, FALSE
   @Done:
end;

procedure XMS_initialize; assembler;
asm
        mov    ax, 4310h
        int    2Fh
        mov    word ptr [XMSDriver], bx
        mov    word ptr [XMSDriver+2], es
end;

function XMS_getVersion: word; assembler;
asm
        mov    ah, 00h
        call   XMSDriver
end;

function XMS_getFreeMemory: word; assembler;
asm
        mov    ah, 08h
        call   XMSDriver
        mov    ax, dx
end;

function XMS_allocateMemory(var handle: word; size: word): boolean; assembler;
asm
        mov    ah, 09h
        mov    dx, size
        call   XMSDriver
        les    di, handle
        mov    es:[di], dx
end;

function XMS_reallocateMemory(handle: word; new_size: word): boolean; assembler;
asm
        mov    ah, 0Fh
        mov    bx, new_size
        mov    dx, handle
        call   XMSDriver
end;

function XMS_freeMemory(handle: word): boolean; assembler;
asm
        mov    ah, 0Ah
        mov    dx, handle
        call   XMSDriver
end;

function XMS_moveMemory(params: p_move_params): boolean; assembler;
asm
        push   ds
        mov    ah, 0Bh
        lds    si, params
        call   XMSDriver
        pop    ds
end;

procedure writeDSP(value: byte);
begin
   repeat until (port[write_port] and $80) = 0;
   port[write_port] := value;
end;

function readDSP: byte;
begin
   repeat until (port[poll_port] and $80) <> 0;
   readDSP := port[read_port];
end;

function resetDSP: boolean;
var
   i: byte;
begin
   port[reset_port] := 1;
   port[reset_port] := 0;
   i := 100;
   while (readDSP <> $AA) and (i > 0) do dec(i);
   if i > 0 then resetDSP := TRUE
            else resetDSP := FALSE;
end;

function DMAcount: word;
var
   x: word;
begin
   x := port[DMA_count_port];
   x := x + port[DMA_count_port] * 256;

   DMAcount := x;
end;

 procedure installHandler; forward;
 procedure uninstallHandler; forward;

function initializeSB(base_IO : word; IRQ : byte; DMA16 : byte; IO : IO_mode_T; rate : word): boolean;
begin

   reset_port  := base_IO + $6;
   read_port   := base_IO + $A;
   write_port  := base_IO + $C;
   poll_port   := base_IO + $E;
   poll16_port := base_IO + $F;

   initializeSB := TRUE;
   if not (resetDSP) then begin
      initializeSB := FALSE;
      exit;
   end;

   if IRQ <= 7 then begin
      IRQ_interrupt_vector  := $08+IRQ;
      PIC_rotate_port := $20;
      PIC_mask_port   := $21;
      interrupt_controller := 1;
   end else begin
      IRQ_interrupt_vector  := $70+IRQ-8;
      PIC_rotate_port := $A0;
      PIC_mask_port   := $A1;
      interrupt_controller := 2;
   end;

   IRQ_stop_mask  := 1 shl (IRQ mod 8);
   IRQ_start_mask := not(IRQ_stop_mask);

   DMA_mask_port     := $D4;
   DMA_clear_ptr_port   := $D8;
   DMA_mode_port     := $D6;
   DMA_base_address_port := $C0 + 4*(DMA16-4);
   DMA_count_port    := $C2 + 4*(DMA16-4);

   case DMA16 of
      5 :  DMA_page_port := $8B;
      6 :  DMA_page_port := $89;
      7 :  DMA_page_port := $8A;
   end;

   DMA_stop_mask  := DMA16-4 + $04;
   DMA_start_mask := DMA16-4 + $00;

   if IO = INPUT then DMA_mode := DMA16-4 + $54
                 else DMA_mode := DMA16-4 + $58;

   IO_mode := IO;
   sampling_rate := rate;

   installHandler;
end;

procedure SBIO_shutDown;
begin
   uninstallHandler;
   resetDSP;
end;

procedure startIO(length : longint);
begin
   SBIO_done := FALSE;
   samples_remaining := length;
   current_block := 1;

   port[DMA_mask_port]      := DMA_stop_mask;
   port[DMA_clear_ptr_port] := $00;
   port[DMA_mode_port]      := DMA_mode;
   port[DMA_base_address_port] := Lo(buffer_offset);
   port[DMA_base_address_port] := Hi(buffer_offset);
   port[DMA_count_port]     := Lo(buffer_length - 1);
   port[DMA_count_port]     := Hi(buffer_length - 1);
   port[DMA_page_port]      := buffer_page;
   port[DMA_mask_port]      := DMA_start_mask;

   if IO_mode = OUTPUT then writeDSP($41)
                       else writeDSP($42);
   writeDSP(Hi(sampling_rate));
   writeDSP(Lo(sampling_rate));
   if IO_mode = OUTPUT then writeDSP($B6)
                       else writeDSP($BE);
   writeDSP($10);
   writeDSP(Lo(block_length - 1));
   writeDSP(Hi(block_length - 1));
end;

procedure setHandler(Ptr : pointer);
begin
   SBIO_handler := HandlerProc(Ptr);
end;

procedure toggleBlock;
begin
   if current_block = 1 then current_block := 2
                        else current_block := 1;
end;

procedure shutDown; forward;

procedure InterruptHandler; interrupt;
var
   temp: byte;
begin
   inc(interrupt_count);

   if @SBIO_handler <> nil then SBIO_handler;

   if samples_remaining > 0 then
      dec(samples_remaining,block_length)
   else begin
      SBIO_done := TRUE;
      writeDSP($D9);
   end;
   toggleBlock;

   temp := port[poll16_port];
   port[$20] := $20;

end;

procedure enableInterrupts;  InLine($FB);
procedure disableInterrupts; InLine($FA);

procedure installHandler;
begin
   disableInterrupts;
   port[PIC_mask_port] := port[PIC_mask_port] or IRQ_stop_mask;
   getIntVec(IRQ_interrupt_vector,OldIntVector);
   setIntVec(IRQ_interrupt_vector,@InterruptHandler);
   port[PIC_mask_port] := port[PIC_mask_port] and IRQ_start_mask;
   enableInterrupts;
   handler_installed := TRUE;
end;

procedure uninstallHandler;
begin
   disableInterrupts;
   port[PIC_mask_port] := port[PIC_mask_port] or IRQ_stop_mask;
   setIntVec(IRQ_interrupt_vector, OldIntVector);
   enableInterrupts;
   handler_installed := FALSE;
end;

function getLinearAddress(Ptr : pointer): longint;
begin
   getLinearAddress := longint(Seg(Ptr^))*16 + longint(Ofs(Ptr^));
end;

function normalizePtr(p : pointer): pointer;
var
   LinearAddress: longint;
begin
   LinearAddress := getLinearAddress(p);
   normalizePtr := Ptr(LinearAddress div 16,LinearAddress mod 16);
end;

procedure getBuffer(var buffer_ptr : pointer; length : word);
begin
   memory_area_size := 8 * length;
   getMem(memory_area,memory_area_size);
   if memory_area = nil then halt;
   if ((getLinearAddress(memory_area) div 2) mod 65536)+length*2 < 65536 then
      buffer_ptr := memory_area
   else
      buffer_ptr := normalizePtr(Ptr(Seg(memory_area^),Ofs(memory_area^)+4*length));

   buffer_address := getLinearAddress(buffer_ptr);
   block_length   := length;
   buffer_length  := length * 2;
   buffer_page    := buffer_address div 65536;
   buffer_offset  := (buffer_address div 2) mod 65536;
end;

procedure freeBuffer(var buffer_ptr : pointer);
begin
   buffer_ptr := nil;
   freeMem(memory_area,memory_area_size);
end;

procedure copyBlock(offset: longint; block: byte);
begin
   with move_params do begin
      if (current_sample + BUFFER_BLOCK_LENGTH) <= number_of_samples then
         length := BUFFER_BLOCK_LENGTH*2
      else begin
         length := (number_of_samples-current_sample) * 2;
         fillChar(SBPlay_buffer^[Block][BUFFER_BLOCK_LENGTH - (length div 2) + 1],(BUFFER_BLOCK_LENGTH-(length div 2))*2, $00);
      end;
      source_handle  := SBPlay_handle;
      source_offset  := offset;
      dest_handle    := 0;
      dest_offset    := longint(addr(SBPlay_buffer^[Block]));
   end;
   XMS_moveMemory(@move_params);
end;

procedure playHandler; far;
var
   result: word;
   i: word;
begin
   if current_sample < number_of_samples then begin
      copyBlock(current_sample*2,current_block);
      inc(current_sample,BUFFER_BLOCK_LENGTH);
   end else begin
      fillChar(SBPlay_buffer^[current_block],BUFFER_BLOCK_LENGTH*2, $00);
   end;
end;

function loadRAWData(raw_file : string) : boolean;
var
   f: file;
   chunk: array[1..LOAD_CHUNK_SIZE] of byte;
   size: longint;
   result, i: word;

   rID, rLen, wID, fID, fLen, fNext : longint;
   wFormatTag, nChannels, nSamplesPerSec, nAvgBytesPerSec : word;
   dID, dLen : longint;

begin
   loadRAWData := FALSE;
   assign(f,raw_file);
   {$I-}
   reset(f, 1);
   {$I+}
   if IOResult <> 0 then exit;

   size := FileSize(f);
   number_of_samples := size div 2;

   XMS_initialize;
   if not(XMS_allocateMemory(SBPlay_handle,(number_of_samples*2 div 1024)+1)) then begin
      exit;
   end;

   with move_params do begin
      source_handle := 0;
      source_offset := longint(@chunk);
      dest_handle   := SBPlay_handle;
      dest_offset   := 0;
   end;

   repeat
      if size > LOAD_CHUNK_SIZE then move_params.length := LOAD_CHUNK_SIZE
                                else move_params.length := size;
      BlockRead(f,chunk,move_params.length,result);
      if result < LOAD_CHUNK_SIZE then
         for i := result+1 to LOAD_CHUNK_SIZE do chunk[i] := 0;
      XMS_moveMemory(@move_params);
      inc(move_params.dest_offset,move_params.length);
      dec(size,move_params.length);
   until size <= 0;

   close(f);

   loadRAWData := TRUE;
end;

function initializeSBPlay(SBPlay_sampling_rate : word; raw_file : string) : boolean;
begin
   if loadRAWData(raw_file) then begin
      getBuffer(pointer(SBPlay_buffer),BUFFER_BLOCK_LENGTH);
      copyBlock(0,1);
      copyBlock(BUFFER_BLOCK_LENGTH,2);
      current_sample := BUFFER_BLOCK_LENGTH * 2;

      setHandler(@playHandler);
      initializeSB(BASE_IO,IRQ,DMA16,OUTPUT,SBPlay_sampling_rate);
      StartIO(number_of_samples);

      initializeSBPlay := TRUE;
   end else
      initializeSBPlay := FALSE;
end;

procedure shutDown;
begin
   SBIO_shutDown;
   setHandler(nil);

   XMS_freeMemory(SBPlay_handle);

   freeBuffer(pointer(SBPlay_buffer));
end;

procedure playWAV(frequency : word; raw_file : string);
begin

   if XMS_installed then begin

      interrupt_count := 0;
      @SBIO_handler    := nil;

      if MAXAVAIL < BUFFER_BLOCK_LENGTH*8 then exit;

      if initializeSBPlay(frequency,raw_file) then begin
         repeat until SBIO_done;
         shutDown;
      end;

   end;
end;

end.
