// AdLib Soung ToolKit For WATCOM C/C++   Modified by Kim Youngsik .WOODPARK.
//                                        From AdLib ToolKit
// Support : YM3812/YMF-262M FM Chip
//           Melody mode 9 channel/Percussion mode 11 channel
//           Melody mode 18 channel/Percussion mode 20 channel
//           Stereo sound
// Thanks to Park Jeenhong(pkchn) and Music club members (go mul 7 12)

#include <conio.h>
#include "ADLIB.H"

void InitSlotVolume(void);
void InitSlotParams(void);
long CalcPremFNum(int numDeltaDemiTon, int denDeltaDemiTon);
void SetFNum(unsigned *fNumVec, int num, int den);
void InitFNums(void);
void ChangePitch(unsigned voice, int pitchBend);
void SetASlotParam(unsigned slot, int param, unsigned val);
void SetSlotParam(unsigned slot, unsigned char *param, unsigned waveSel);
void SetCharSlotParam(unsigned slot, unsigned char *cParam, unsigned waveSel);
void SndSetPrm(int slot, int prm);
void SndSetAllPrm(int slot);
void SndSAmVibRhythm(void);
void SndSKslLevel(int slot);
void SndSNoteSel(void);
void SndSFeedFm(int slot);
void SndSAttDecay(int slot);
void SndSSusRelease(int slot);
void SndSAVEK(int slot);
void SndWaveSelect(int slot);
void SetFreq(unsigned voice, int pitch, unsigned keyOn);
void SoundChut(unsigned voice);

#define TOM_PITCH   24          /* best frequency, in range of 0 to 95 */
#define TOM_TO_SD   7           /* 7 half-tones between voice 7 & 8 */
#define SD_PITCH    (TOM_PITCH + TOM_TO_SD)

#define NR_STEP_PITCH 25        /* 25 steps within a half-tone for pitch bend */

#define GetLocPrm(slot, prm)    ((unsigned)paramSlot[slot][prm])
#define HighByte(word)          (((char *)(&word))[1])      /* 80x86-8 only .. */
#define RhythmMode()            (percussion)

unsigned AdLibPort;
unsigned AdLibType;             /* YM3812 or YMF-262M Chip */

char percBits;
char percMasks[] = { 0x10, 0x08, 0x04, 0x02, 0x01 };

char voiceNote[20];             /* pitch of last note-on of each voice */
char voiceKeyOn[20];            /* state of keyOn bit of each voice */
char voicePan[20];

char noteDIV12[96];
char noteMOD12[96];
int  halfToneOffset[20];
unsigned  fNumNotes[NR_STEP_PITCH][12];
unsigned *fNumFreqPtr[20];

char lVoiceVolume[20];          /* volume for each of 11 logical voices */
char slotRelMax[20] = { 127, 127, 127, 127, 127, 127, 127, 127, 127, 127,
                        127, 127, 127, 127, 127, 127, 127, 127, 127, 127 };

unsigned modeVoices = 11;       /* 9 or 11, depending on 'percussion'*/

unsigned char paramSlot[36][nbLocParam]; /* all the parameters of slots...  */

char amDepth;                   /* chip global parameters .. */
char vibDepth;                  /* ... */
char noteSel;                   /* ... */
char percussion;                /* percussion mode parameter */

int  pitchRange;                /* pitch variation, half-tone[+1,+12] */
int  pitchRangeStep;            /* == pitchRange * NR_STEP_PITCH */
int  modeWaveSel;               /* != 0 if used with the 'wave-select' parameters */

unsigned char slotMVoice[18][2] = {
    {  0,  3 },         /* 0 */
    {  1,  4 },         /* 1 */
    {  2,  5 },         /* 2 */
    {  6,  9 },         /* 3 */
    {  7, 10 },         /* 4 */
    {  8, 11 },         /* 5 */
    { 12, 15 },         /* 6 */
    { 13, 16 },         /* 7 */
    { 14, 17 },         /* 8 */
    { 18, 21 },         /* 9  Below YMF262M's slots */
    { 19, 22 },         /* 10 */
    { 20, 23 },         /* 11 */
    { 24, 27 },         /* 12 */
    { 25, 28 },         /* 13 */
    { 26, 29 },         /* 14 */
    { 30, 33 },         /* 15 */
    { 31, 34 },         /* 16 */
    { 32, 35 }          /* 17 */
};

unsigned char slotPVoice[20][2] = {
    {  0,  3 },         /* 0 */
    {  1,  4 },         /* 1 */
    {  2,  5 },         /* 2 */
    {  6,  9 },         /* 3 */
    {  7, 10 },         /* 4 */
    {  8, 11 },         /* 5 */
    { 12, 15 },         /* Bass Drum: slot 12 et 15 */
    { 16, 255},         /* SD: slot 16 */
    { 14, 255},         /* TOM: slot 14 */
    { 17, 255},         /* TOP-CYM: slot 17 */
    { 13, 255},         /* HH: slot 13 */
    { 18, 21 },         /* 11 */
    { 19, 22 },         /* 12 */
    { 20, 23 },         /* 13 */
    { 24, 27 },         /* 14 */
    { 25, 28 },         /* 15 */
    { 26, 29 },         /* 16 */
    { 30, 33 },         /* 17 */
    { 31, 34 },         /* 18 */
    { 32, 35 }          /* 19 */
};

char offsetSlot[36] = {
     0,  1,  2,  3,  4,  5,
     8,  9, 10, 11, 12, 13,
    16, 17, 18, 19, 20, 21,
     0,  1,  2,  3,  4,  5,
     8,  9, 10, 11, 12, 13,
    16, 17, 18, 19, 20, 21
};

char carrierSlot[36] = {
    0, 0, 0,        /* 1 2 3 */
    1, 1, 1,        /* 4 5 6 */
    0, 0, 0,        /* 7 8 9 */
    1, 1, 1,        /* 10 11 12 */
    0, 0, 0,        /* 13 14 15 */
    1, 1, 1,        /* 16 17 18 */
    0, 0, 0,        /* 19 20 21 */
    1, 1, 1,        /* 22 23 24 */
    0, 0, 0,        /* 25 26 27 */
    1, 1, 1,        /* 28 29 30 */
    0, 0, 0,        /* 31 32 33 */
    1, 1, 1         /* 34 35 36 */
};

char voiceMSlot[36] = {
     0,  1,  2,
     0,  1,  2,
     3,  4,  5,
     3,  4,  5,
     6,  7,  8,
     6,  7,  8,
     9, 10, 11,
     9, 10, 11,
    12, 13, 14,
    12, 13, 14,
    15, 16, 17,
    15, 16, 17
};

char voicePSlot[36] = {
    0, 1, 2,
    0, 1, 2,
    3, 4, 5,
    3, 4, 5,
    BD, HIHAT, TOM,
    BD, SD, CYMB,
    11, 12, 13,
    11, 12, 13,
    14, 15, 16,
    14, 15, 16,
    17, 18, 19,
    17, 18, 19
};

char voicePVoice[11] = { 0, 1, 2, 3, 4, 5, 6, 7, 8, 8, 7 };

unsigned char DefInstPiano[28] = {
    1, 1, 3, 15, 5, 0, 1, 3, 15, 0, 0, 0, 1,
    0, 1, 1, 15, 7, 0, 2, 4,  0, 0, 0, 1, 0,
    0, 0
};

unsigned char DefInstBassDrum[28] = {
    0, 0, 0, 10, 4, 0, 8, 12, 11, 0, 0, 0, 1,
    0, 0, 0, 13, 4, 0, 6, 15,  0, 0, 0, 0, 1,
    0, 0,
};

unsigned char DefInstSnareDrum[28] = {
    0, 12, 0, 15, 11, 0, 8, 5, 0, 0, 0, 0, 0,
    0, 12, 0, 15, 11, 0, 8, 5, 0, 0, 0, 0, 0,
    0, 0
};

unsigned char DefInstTomTom[28] = {
    0, 4, 0, 15, 11, 0, 7,  5, 0, 0, 0, 0, 0,
    0, 4, 0, 15, 11, 0, 7,  5, 0, 0, 0, 0, 0,
    0, 0
};

unsigned char DefInstCymbal[28] = {
    0, 1, 0, 15, 11, 0, 5,  5, 0, 0, 0, 0, 0,
    0, 1, 0, 15, 11, 0, 5,  5, 0, 0, 0, 0, 0,
    0, 0
};

unsigned char DefInstHihat[28] = {
    0, 1, 0, 15, 11, 0, 7, 5,  0, 0, 0, 0, 0,
    0, 1, 0, 15, 11, 0, 7, 5,  0, 0, 0, 0, 0,
    0, 0
};

void SndOutputInline(unsigned char addr, unsigned char data);
#pragma aux SndOutputInline = \
                "MOV     EBX,EDX" \
                "MOV     EDX,AdLibPort" \
                "OUT     DX,AL" \
                "IN      AL,DX" \
                "IN      AL,DX" \
                "IN      AL,DX" \
                "IN      AL,DX" \
                "IN      AL,DX" \
                "IN      AL,DX" \
                "INC     EDX" \
                "MOV     EAX,EBX" \
                "OUT     DX,AL" \
                "DEC     EDX" \
                "MOV     ECX,46" \
    "DELAY23uS:  IN      AL,DX" \
                "LOOP    DELAY23uS"\
                parm [EAX][EDX] \
                modify [EAX EDX EBX ECX];

void SndOutput(unsigned char addr, unsigned char data)
{
    SndOutputInline(addr, data);
}

void SndOutput2(unsigned char addr, unsigned char data)
{
    if (AdLibType != OPL3) return;
    AdLibPort += 2;
    SndOutputInline(addr, data);
    AdLibPort -= 2;
}

void vSndOutput(unsigned char addr, unsigned char voice, unsigned char data)
{
    if (percussion) {
        if (voice < 11) SndOutput(addr + voicePVoice[voice], data);
        else            SndOutput2(addr + voice - 11, data);
    } else {
        if (voice < 9) SndOutput(addr + voice, data);
        else           SndOutput2(addr + voice - 9, data);
    }
}

//
//  a aa  a · (OPL2/OPL3) *BoardInstalled
//
//   : Zero - a gq
//         1-OPL2 / 2-OPL3
//
int CheckAdlib(unsigned port)
{
    int T1, T2, i;

    AdLibPort = port;
    AdLibType = 0;
    SndOutput(4, 0x60);
    SndOutput(4, 0x80);
    T1 = inp(AdLibPort);
    SndOutput(2, 0xFF);
    SndOutput(4, 0x21);
    for (i = 0; i < 200; i++) inp(AdLibPort);
    T2 = inp(AdLibPort);
    SndOutput(4, 0x60);
    SndOutput(4, 0x80);
    if (!((T1 & 0xE0) == 0) && ((T2 & 0xE0) == 0xC0)) return(0);
    T1 = inp(AdLibPort);
    AdLibType = (inp(AdLibPort) & 0x06) ? OPL2: OPL3;

    return(AdLibType);
}

//
//  a aai aa xa
//
//   : Zero - a gq
//         aAe xe a aa ͡a i a
//
int AutoDetectAdlib(void)
{
    int i, Port[] = { 0x220, 0x240, 0x260, 0x280, 0x228, 0x248, 0x268, 0x288, 0x388 };

    for (i = 0; i < 9; i++) if (CheckAdlib(Port[i])) return(AdLibPort);

    return(0);
}

//
//  Set the volume of all slots.
//
void InitSlotVolume(void)
{
    int i;

    for (i = 0; i < 20; i++) {
        lVoiceVolume[i] = slotRelMax[i];
    }
}

//
//  Initialize the chip in melodic mode (mode == 0),
//  set all 9 voices to electric-piano timbres,
//  set the 3 global parameters to zero,
//  set the pitch bend range to 1 half-tone,
//  set the pitch bend of each voice to 0x2000 (no detune),
//  set the volume of each voice to maximum level,
//  and enable the wave-select parameter.
//
void SoundInit(void)
{
    int i;

    for (i = 1; i <= 0xF5; i++) SndOutput(i, 0);    /* clear all registers */
    SndOutput(4, 0x60);                             /* mask T1 & T2 */
    SndOutput2(5, 0);

    if (AdLibType == OPL3) {
        SndOutput2(5, 1);       /* change OPL3 mode */
        SndOutput2(4, 0);
    }

    for (i = 0; i < 20; i++) {
        voiceNote[i] = 0;
        voiceKeyOn[i] = 0;
        voicePan[i] = PAN_BOTH;
    }

    InitSlotVolume();
    InitFNums();

    SetMode(0);                 /* melodic mode */
    SetGParam(0, 0, 0);         /* init global parameters */
    SetPitchRange(1);           /* default pitch range is 1 half-tone */
    SetWaveformSel(1);          /* Set Wave select YMF262-MAᕡ aa.  */
}

void SoundClose(void)
{
    SoundInit();
    if (AdLibType == OPL3) SndOutput2(5, 0);
}

void SetVoicePan(unsigned voice, int pan)
{
    int slots;

    if (AdLibType == OPL2) return;

    if (percussion) {
        voicePan[voice] = pan;
        if (voice == 7)      voice += 3;
        else if (voice == 9) voice--;
        if (voice == 10)     voicePan[7] = pan;
        else if (voice == 8) voicePan[9] = pan;
        slots = slotPVoice[voice][0];
    } else {
        slots = slotMVoice[voice][0];
    }
    voicePan[voice] = pan;
    SndSFeedFm(slots);
}

//
//  Put the chip in melodic mode (mode == 0),
//  or in percussive mode ( mode != 0).
//
//  If the melodic mode is chosen, all voices are
//  set to electric-piano, else the first 5 are set
//  to electric-piano, and the percussion voices
//  to their default timbres.
//
void SetMode(int mode)
{
    int voice;

    modeVoices = mode ? 11 : 9;
    if (AdLibType = OPL3) modeVoices += 9;
    percBits = 0;
    percussion = mode;
    SndSAmVibRhythm();

    if (mode) {     /* set the frequency for the last 4 percussion voices: */
        for (voice = 0; voice < modeVoices; voice++) {
            SetVoiceVolume(voice, 0);
        }
        SetFreq(TOM, TOM_PITCH, 0);
        SetFreq(SD, SD_PITCH, 0);
    }
    InitSlotParams();
}

//
//  Enable (state != 0) / disable (state == 0)
//  the wave-select parameters.
//
//  If you do not want to use the wave-select parameters, call
//  this function with a value of 0 AFTER calling SoundColdInit()
//  or SoundWarmInit().
//
void SetWaveformSel(int state)
{
    int i;

    modeWaveSel = state ? 0x20 : 0; /* Enable Wave Mode */
    SndOutput(1, modeWaveSel);
    SndOutput2(1, modeWaveSel);
    SndOutput(4, 0xE0);
    for (i = 0; i < 18; i++) SndOutput(0xE0 + offsetSlot[i], 0);
    for (i = 18; i < 36; i++) SndOutput2(0xE0 + offsetSlot[i], 0);
}

//
//  Routine to change the pitch bend range. The value can be from
//  1 to 12 (in half-tones).
//
//  For example, the value 12 means that the pitch bend will
//  range from -12 (pitchBend == 0, see function 'SetVoicePitch()')
//  to +12 (pitchBend == 0x3fff) half-tones.
//
//  The change will be effective as of the next call to
//  'SetVoicePitch()'.
//
void SetPitchRange(unsigned pR)
{
    if (pR > 12) pR = 12;
    if (pR < 1)  pR = 1;
    pitchRange = pR;
    pitchRangeStep = pitchRange * NR_STEP_PITCH;
}

//
//  Set the 3 global parameters AmDepth,
//  VibDepth & NoteSel
//
void SetGParam(int amD, int vibD, int nSel)
{
    amDepth = amD;
    vibDepth = vibD;
    noteSel = nSel;

    SndSAmVibRhythm();
    SndSNoteSel();
}

//
//  A timbre (melodic or percussive) is defined as follows:
//  the 13 first parameters of operator 0 ( ksl, multi, feedBack,
//  attack, sustain, eg-typem decay, release, level, am, vib, ksr, fm)
//  followed by the 13 parameters of operator 1 (if a percussive voice, all
//  the parameters are zero), followed by the wave-select parameter for
//  the operators 0 and 1.
//
//  'paramArray' is structured as follows:
//      struct {
//          int opr0Prm[ 13];       first 13 parameters
//          int opr1Prm[ 13];       last  13 parameter
//          int opr0WaveSel;        must be 0 if percussive timbre
//          int opr1WaveSel;        must be 0 if percussive timbre
//      } TimbreDef;
//
//  The old timbre files (*.INS) do not contain the parameters
//  'opr0WaveSel' and 'opr1WaveSel'.
//  Set these two parameters to zero if you are using the old file
//  format.
//
void SetVoiceTimbre(unsigned voice, unsigned char *paramArray)
{
    unsigned wave0, wave1;
    unsigned char *prm1, *wavePtr;
    unsigned char *slots;

    if (voice >= modeVoices) return;

    wavePtr = paramArray + 2 * (nbLocParam - 1);
    wave0 = (unsigned)*wavePtr++;
    wave1 = (unsigned)*wavePtr;
    prm1 = paramArray + nbLocParam -1;

    if (percussion) slots = slotPVoice[voice];
    else            slots = slotMVoice[voice];
    SetSlotParam(slots[0], paramArray, wave0);
    if (slots[1] != 255) SetSlotParam(slots[1], prm1, wave1);
}

//
//  Set the volume of the voice 'voice' to 'volume'.
//
//  The resulting output level is (timbreVolume * volume / 127).
//  The change takes place immediately.
//
void SetVoiceVolume(unsigned voice, unsigned volume)
{
    unsigned char *slots;

    if (voice >= modeVoices) return;
    if (volume > MAX_VOLUME) volume = MAX_VOLUME;
    lVoiceVolume[voice] = volume;
    if (percussion) slots = slotPVoice[voice];
    else            slots = slotMVoice[voice];
    SndSKslLevel(slots[0]);
    if (255 != slots[1]) SndSKslLevel(slots[1]);
}

//
//  Change the pitch value of a voice.
//  The variation in pitch is a function of the previous call
//  to 'SetPitchRange()' and the value of 'pitchBend'.
//  A value of 0 means -half-tone * pitchRange,
//  0x2000 means no variation (exact pitch) and
//  0x3fff means +half-tone * pitchRange.
//  Does not affect the percussive voices, except for the bass drum.
//  The change takes place immediately.
//
void SetVoicePitch(unsigned voice, unsigned pitchBend)
{
    if (!RhythmMode() || voice <= BD || voice > HIHAT) {    /* melodic + bass-drum */
        if (pitchBend > MAX_PITCH) pitchBend = MAX_PITCH;
        ChangePitch(voice, pitchBend);
        SetFreq(voice, voiceNote[voice], voiceKeyOn[voice]);
    }
}

//
//  Routine to start a note playing.
//
//  0 <= voice <= 8 in melodic mode,
//  0 <= voice <= 10 in percussive mode;
//  0 <= pitch <= 127, 60 == MID_C ( the card can play between 12 and 107 )
//
void NoteOn(unsigned voice, int pitch)
{
    pitch -= (MID_C - CHIP_MID_C);

    if (pitch < 0) pitch = 0;
    voiceKeyOn[voice] = 1;

    if (!RhythmMode() || voice < BD || voice > HIHAT)   /* this is a melodic voice */
        SetFreq(voice, pitch, 1);
    else {                              /* this is a percussive voice */
        if (voice == BD) SetFreq(BD, pitch, 0);
        else if (voice == TOM) {        /* for the last 4 percussions, only the TOM may change in frequency, modifying the three others: */
            SetFreq(TOM, pitch, 0);
            SetFreq(SD, pitch + TOM_TO_SD, 0);  /* f7 = 3 * f8 */
        } else if (voice != SD) voiceNote[voice] = pitch;
        percBits |= percMasks[voice - BD];
        SndSAmVibRhythm();
    }
}

//
//  Routine to stop playing the note which was started in 'NoteOn()'.
//
void NoteOff(unsigned voice)
{
    voiceKeyOn[voice] = 0;

    if (!RhythmMode() || voice < BD || voice > HIHAT) {
        SetFreq(voice, voiceNote[voice], 0);    /* shut off */
    } else {
        percBits &= ~percMasks[voice - BD];
        SndSAmVibRhythm();
    }
}

void InitSlotParams(void)
{
    for (int i = 0; i < 20; i++) SetVoiceTimbre(i, DefInstPiano);

    if (RhythmMode()) {
        SetVoiceTimbre(6, DefInstBassDrum);
        SetVoiceTimbre(7, DefInstSnareDrum);
        SetVoiceTimbre(8, DefInstTomTom);
        SetVoiceTimbre(9, DefInstCymbal);
        SetVoiceTimbre(10, DefInstHihat);
    }
}

//
//  Return binary value of the frequency 260.44 ( C)
//  shifted by +/- numdeltaDemiTon/denDeltaDemiTon multiplied by 8.
//
//  If the numerator (numDeltaDemiTon) is positive, the frequency is
//  increased; if negative, it is decreased.
//
//  Fo = Fb( 1 + 0.06 num /den)
//  Fnum8 = Fo * 65536 * 72 / 3.58e6
//
//  -100 <= numDeltaDemiTon <= +100
//  1 <= denDeltaDemiTon <= 100
//
long CalcPremFNum(int numDeltaDemiTon, int denDeltaDemiTon)
{
    long    f8, fNum8, d100;

    d100 = denDeltaDemiTon * 100L;
    f8 = (d100 + 6L * numDeltaDemiTon) * (26044L * 2L); /* 260.44 * 100 * 2 */
    f8 /= d100 * 25L;
    fNum8 = f8 * 16384L * 9L ;     /*( 16384L * 9L);   */
    fNum8 /= 179L * 625L;

    return(fNum8);
}

//
//  Initialize a line in the frequency table with shifted frequency values.
//  The values are shifted a fraction (num/den) of a half-tone.
//  See following routine.
//
void SetFNum(unsigned *fNumVec, int num, int den)
{
    int i;
    long val;

    *fNumVec++ = (unsigned)(4 + (val = CalcPremFNum(num, den))) >> 3;
    for (i = 1; i < 12; i++) {  /* 1.06 is 2^(1/12)   , 1.06^12 = 2 */
        val *= 106;
        *fNumVec++ = (unsigned)(4 + (val /= 100)) >> 3;
    }
}

//
//  Initialize all lines of the frequency table. Each line represents
//  12 half-tones shifted by (n/NR_STEP_PITCH), where 'n' is the line number
//  and ranges from 1 to NR_STEP_PITCH.
//
void InitFNums(void)
{
    unsigned i, j, k, num, numStep, pas;

    numStep = 100 / NR_STEP_PITCH;
    for (num = pas = 0; pas < NR_STEP_PITCH; pas++, num += numStep)
        SetFNum(fNumNotes[pas], num, 100);
    for (i = 0; i < 20; i++) {
        fNumFreqPtr[i] = (unsigned *)fNumNotes[0];
        halfToneOffset[i] = 0;
    }

    k = 0;
    for(i = 0; i < 8; i++)
        for(j = 0; j < 12; j++, k++) {
            noteDIV12[k] = i;
            noteMOD12[k] = j;
        }
}

//
//  Routine to set 'halfToneOffset[]' & 'fNumFreqPtr[]'.
//  These two global variables are used to determine the frequency
//  variation to use when a note is played.
//
void ChangePitch(unsigned voice, int pitchBend)
{
    int t1, t2, delta;
    long l;
    static long oldL = ~0;
    static int oldHt;
    static unsigned *oldPtr;

    l = (long)(pitchBend - MID_PITCH) * pitchRangeStep;
    if (oldL == l) {                    /* optimisation ... */
        fNumFreqPtr[voice] = oldPtr;
        halfToneOffset[voice] = oldHt;
    } else {
        t1 = l / MID_PITCH;
        if (t1 < 0) {
            t2 = NR_STEP_PITCH - 1 - t1;
            oldHt = halfToneOffset[voice] = -(t2 / NR_STEP_PITCH);
            delta = (t2 - NR_STEP_PITCH + 1) % NR_STEP_PITCH;
            if (delta) delta = NR_STEP_PITCH - delta;
        } else {
            oldHt = halfToneOffset[voice] = t1 / NR_STEP_PITCH;
            delta = t1 % NR_STEP_PITCH;
        }
        oldPtr = fNumFreqPtr[voice] = (unsigned *)fNumNotes[delta];
        oldL = l;
    }
}

//
//  Used to change the parameter 'param' of the slot 'slot'
//  with the value 'val'. The chip registers are updated.
//
/*
void SetASlotParam(unsigned slot, int param, unsigned val)
{
    paramSlot[slot][param] = val;
    SndSetPrm(slot, param);
}
*/

//
//  Set the 14 parameters ( 13 in 'param', 1 in 'waveSel')
//  of slot 'slot'. Update the parameter array and the chip.
//
void SetSlotParam(unsigned slot, unsigned char *param, unsigned waveSel)
{
    int i;
    unsigned char *ptr;

    for(i = 0, ptr = &paramSlot[slot][0]; i < nbLocParam - 1; i++) *ptr++ = *param++;
    *ptr = waveSel &= 0x3;
    SndSetAllPrm(slot);
}

/*
void SetCharSlotParam(unsigned slot, unsigned char *cParam, unsigned waveSel)
{
    unsigned param[nbLocParam];

    for (int i = 0; i < nbLocParam - 1; i++) param[i] = *cParam++;
    SetSlotParam(slot, param, waveSel);
}
*/

//
//  Update the parameter 'prm' for the slot 'slot'.
//  Update the chip registers.
//
void SndSetPrm(int slot, int prm)
{
    switch(prm) {
        case prmPercussion :
        case prmAmDepth :
        case prmVibDepth :
            SndSAmVibRhythm();
            break;
        case prmNoteSel :
            SndSNoteSel();
            break;
        case prmKsl :
        case prmLevel :
            SndSKslLevel(slot);
            break;
        case prmFm :
        case prmFeedBack :
            SndSFeedFm(slot);
            break;
        case prmAttack :
        case prmDecay :
            SndSAttDecay(slot);
            break;
        case prmRelease :
        case prmSustain :
            SndSSusRelease(slot);
            break;
        case prmMulti :
        case prmVib :
        case prmStaining :
        case prmKsr :
        case prmAm :
            SndSAVEK(slot);
            break;
        case prmWaveSel :
            SndWaveSelect(slot);
            break;
    }
}

//
//  Transfer all the parameters from slot 'slot' to the chip.
//
void SndSetAllPrm(int slot)
{
    SndWaveSelect(slot);
    SndSAttDecay(slot);
    SndSSusRelease(slot);
    SndSAmVibRhythm();
    SndSNoteSel();
    SndSKslLevel(slot);
    SndSFeedFm(slot);
    SndSAVEK(slot);
}

//
//  Set AM Depth, VIB depth & Rhythm
//
void SndSAmVibRhythm(void)
{
    unsigned t1;

    t1 = amDepth ? 0x80 : 0;
    t1 |= vibDepth ? 0x40 : 0;
    t1 |= RhythmMode() ? 0x20 : 0;
    t1 |= percBits ;
    SndOutput(0xBD, t1);
    SndOutput2(0xBD, t1);
}

//
//  Write to the register which controls output level and does
//  key-on/key-offs for the percussive voice slots.
//
//  KSL ( Key Scale Level )
//
void SndSKslLevel(int slot)
{
    unsigned t1,tl,tr, tc, vc, singleSlot;

    if (percussion) vc = voicePSlot[slot];
    else            vc = voiceMSlot[slot];

    t1 = 63 - (GetLocPrm(slot, prmLevel) & 63);    /* amplitude */
    singleSlot = percussion && vc > BD && vc <= HIHAT;

    if (carrierSlot[slot] || singleSlot) {
        t1 = (t1 * lVoiceVolume[vc] + (MAX_VOLUME + 1) / 2) >> LOG2_VOLUME;
    } else if (!GetLocPrm(slot, prmFm)) {
        t1 = (t1 * slotRelMax[vc] + (MAX_VOLUME + 1) / 2) >> LOG2_VOLUME;  /* ១ */
    }

    tc = GetLocPrm(slot, prmKsl) << 6;
    t1 = 63 - t1;
    t1 |= tc;
    if (slot < 18) SndOutput(0x40 + (int)offsetSlot[slot], t1);
    else           SndOutput2(0x40 + (int)offsetSlot[slot], t1);
}

//
//  Note sel
//
void SndSNoteSel(void)
{
    SndOutput(0x08, noteSel ? 64 : 0);
    SndOutput2(0x08, noteSel ? 64 : 0);
}

//
//  FEED-BACK and FM (connection).
//  Applicable only to operator 0 in melodic mode.
//  OK
//
void SndSFeedFm(int slot)
{
    unsigned t1;
    unsigned vc;

    if (percussion) vc = voicePSlot[slot];
    else            vc = voiceMSlot[slot];

    if (carrierSlot[slot]) {
        if (!RhythmMode() || slot < 12 || slot >= 18) {
//            SndOutput(0xC0 + vc, t1);  /* (int)voiceSlot[slot] */
        }
        return;
    }
    t1 = GetLocPrm(slot, prmFeedBack) << 1;
    t1 &= 0xcf;
    t1 |= GetLocPrm(slot, prmFm) ? 0 : 1;
    t1 |= voicePan[vc];
    vSndOutput(0xC0, vc, t1);
}

//
//  ATTACK, DECAY
//  OK
//
void SndSAttDecay(int slot)
{
    unsigned t1;

    t1 = GetLocPrm(slot, prmAttack) << 4;
    t1 |= GetLocPrm(slot, prmDecay) & 0xf;
    if (slot < 18) SndOutput(0x60 + (int)offsetSlot[slot], t1);  /* offsetSlot */
    else           SndOutput2(0x60 + (int)offsetSlot[slot], t1);
}

//
//  SUSTAIN, RELEASE
//  OK
//
void SndSSusRelease(int slot)
{
    unsigned t1;

    t1 = GetLocPrm(slot, prmSustain) << 4;
    t1 |= GetLocPrm(slot, prmRelease) & 0xf;
    if (slot < 18) SndOutput(0x80 + (int)offsetSlot[slot], t1);  /* offsetSlot */
    else           SndOutput2(0x80 + (int)offsetSlot[slot], t1);
}

//
//  AM, VIB, EG-TYP(Sustaining), KSR, MULTI
//
void SndSAVEK(int slot)
{
    unsigned t1;

    t1 = GetLocPrm(slot, prmAm) ? 0x80 : 0;
    t1 += GetLocPrm(slot, prmVib) ? 0x40 : 0;
    t1 += GetLocPrm(slot, prmStaining) ? 0x20 : 0;
    t1 += GetLocPrm(slot, prmKsr) ? 0x10 : 0;
    t1 += GetLocPrm(slot, prmMulti) & 0xf;
    if (slot < 18) SndOutput(0x20 + (int)offsetSlot[slot], t1);  /* offsetSlot */
    else           SndOutput2(0x20 + (int)offsetSlot[slot], t1);
}

//
//  aw wȁ i
//
void SndWaveSelect(int slot)
{
    unsigned wave;

    if (modeWaveSel) wave = GetLocPrm(slot, prmWaveSel) & 0x03;
    else             wave = 0;
    if (slot < 18) SndOutput(0xE0 + offsetSlot[slot], wave);  /* offsetSlot */
    else           SndOutput2(0xE0 + offsetSlot[slot], wave);
}

//
//  ϡái a. a .
//
void SetFreq(unsigned voice, int pitch, unsigned keyOn)
{
    unsigned int fNbr, t1;

    voiceNote[voice] = pitch;
    pitch += halfToneOffset[voice];
    if(pitch > 95) pitch = 95;
    if(pitch < 0)  pitch = 0;
    fNbr = *(fNumFreqPtr[voice] + noteMOD12[pitch]);
    t1 = keyOn ? 0x20 : 0;
    t1 |= ((unsigned)noteDIV12[pitch] << 2) + (0x3 & HighByte(fNbr));
    vSndOutput(0xA0, voice, fNbr);
    vSndOutput(0xB0, voice, t1);
}

//
//  ai 0 a 
//
void SoundChut(unsigned voice)
{
    vSndOutput(0xA0, voice, 0);
    vSndOutput(0xB0, voice, 0);
    voiceKeyOn[voice] = 0;
}

