/*
    Tiny Hangul Library ver 1.98                Last update : 96-10-20
    Input string
    (c) Kim Youngsik (WOODPARK)
*/

#include <string.h>

#include "WPGRAPH.H"
#include "WPHANIN.H"
#include "WPHANOUT.H"
#include "WPKEYDEF.H"

int _WPCallSpInpKey = F4;
int _WPCallHjInpKey = F9;
int _WPCallKBKindKey = ALT_3;

char KBKind   = WPKBKIND2;      /* Hangul japan type */
char KBMode   = 0;              /* Hangul or English mode - default English */
char KBInsert = 1;              /* Insert mode - default on */

#ifndef __WPAUTOMATA
#define __WPAUTOMATA

typedef struct {
    char CurState;
    int  KeyCode;
    int  CombiCode;
} INPSTACK;

#endif

INPSTACK InpStack[10];
int  OutStack[5];
int  InpSP = 0;
int  OutSP = 0;
int  CurState = 0;

int  CursorState = 0;
int  CursorX, CursorY;
int  CursorMode = 0;

int HanKbTable2[] = {
    0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,  /*  !"#$%&' */
    0x28, 0x29, 0x2A, 0x2B, 0x2C, 0x2D, 0x2E, 0x2F,  /* ()*+,-./ */
    0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,  /* 01234567 */
    0x38, 0x39, 0x3A, 0x3B, 0x3C, 0x3D, 0x3E, 0x3F,  /* 89:;<=>? */
    0x40, 0x41, 0x42, 0x43, 0x44, 0x86, 0x46, 0x47,  /* @ABCDEFG */
    0x48, 0x49, 0x4A, 0x4B, 0x4C, 0x4D, 0x4E, 0xA6,  /* HIJKLMNO */
    0xAC, 0x8A, 0x83, 0x53, 0x8C, 0x55, 0x56, 0x8F,  /* PQRSTUVW */
    0x58, 0x59, 0x5A, 0x5B, 0x5C, 0x5D, 0x5E, 0x5F,  /* XYZ[\]^_ */
    0x60, 0x88, 0xBA, 0x90, 0x8D, 0x85, 0x87, 0x94,  /* `abcdefg */
    0xAD, 0xA5, 0xA7, 0xA3, 0xBD, 0xBB, 0xB4, 0xA4,  /* hijklmno */
    0xAA, 0x89, 0x82, 0x84, 0x8B, 0xAB, 0x93, 0x8E,  /* pqrstuvw */
    0x92, 0xB3, 0x91, 0x7B, 0x7C, 0x7D, 0x7E, 0x7F   /* xyz{|}~  */
};

int HanKbTable3[] = {
    0x20, 0xD8, 0x22, 0x23, 0x24, 0x25, 0x26, 0x92,  /*  !"#$%&' */
    0x28, 0x29, 0x2A, 0x2B, 0x2C, 0x2D, 0x2E, 0xAD,  /* ()*+,-./ */
    0x91, 0xDD, 0xD6, 0xD3, 0xB3, 0xBA, 0xA5, 0xAC,  /* 01234567 */
    0xBC, 0xB4, 0x3A, 0x89, 0x32, 0x3D, 0x33, 0x3F,  /* 89:;<=>? */
    0x40, 0xC8, 0x21, 0xCB, 0xCA, 0xDA, 0xC3, 0x3B,  /* @ABCDEFG */
    0x27, 0x38, 0x34, 0x35, 0x36, 0x31, 0x30, 0x39,  /* HIJKLMNO */
    0x3E, 0xDC, 0xA6, 0xC7, 0x3A, 0x37, 0xD0, 0xDB,  /* PQRSTUVW */
    0xD4, 0x3C, 0xD9, 0x5B, 0x5C, 0x5D, 0x5E, 0x5F,  /* XYZ[\]^_ */
    0x60, 0xD7, 0xB4, 0xAA, 0xBD, 0xAB, 0xA3, 0xBB,  /* `abcdefg */
    0x84, 0x88, 0x8D, 0x82, 0x8E, 0x94, 0x8B, 0x90,  /* hijklmno */
    0x93, 0xD5, 0xA4, 0xC5, 0xA7, 0x85, 0xAD, 0xC9,  /* pqrstuvw */
    0xC2, 0x87, 0xD1, 0x7B, 0x7C, 0x7D, 0x7E, 0x7F   /* xyz{|}~  */
};

extern int __KeyBuff;

void (*_WPHookKBKindToggle)(int mode)  = NUL;
void (*_WPHookHanModeToggle)(int mode) = NUL;
void (*_WPHookInsertToggle)(int mode)  = NUL;
void (*_WPHookWaitKey)(void)           = NUL;
word (*_WPHookSpInp)(void)             = NUL;
word (*_WPHookHjInp)(void)             = NUL;

int  WPHanAutomata2(int key);
int  WPHanAutomata3(int key);

int WPConvAsciiToHan(int key)
{
    int     *HanKbTable;

    if (key == ' ' && (_WPKeyFlags & (STATE_LSHIFT | STATE_RSHIFT))) {
        key = (_WPKeyFlags & STATE_LSHIFT) ? LSHIFT_SPACE : RSHIFT_SPACE;
    }

    if (!KBMode || key <= 32 || key >= 127) return(key);
    if ((_WPKeyFlags & 0x40) && ((key >= 0x41 && key <= 0x5A) || (key >= 0x61 && key <= 0x7A)))
        key = key ^ 0x0020;

    HanKbTable = (KBKind == WPKBKIND2) ? HanKbTable2 : HanKbTable3;

    return(HanKbTable[key - 32]);
}

int WPInpKey(int mode)
{
    int     key;

    if (__KeyBuff) {
        key = __KeyBuff;
        __KeyBuff = 0;
    } else {
        key = WPGetKey();
        if (mode) key = WPConvAsciiToHan(key);
    }
    return(key);
}

int  WPHanAutomata(int key)
{
    return (KBKind == WPKBKIND2) ? WPHanAutomata2(key) : WPHanAutomata3(key);
}

void WPDispCursor(int px, int py, int bytex)
{
    int     SaveOp;

    asm     xor     ax, ax
    asm     mov     es, ax
    asm     mov     bx, 046Ch       /* 0000:046Ch */
    asm     mov     di, es:[bx]
    asm     and     di, 7

    if (_DI < 5 && CursorState == 0) {
        CursorState = bytex;
        CursorX = px;
        CursorY = py;
        CursorMode = KBMode;
        SaveOp = WPGetOp();
        WPSetOp(WPOP_XOR);
        if (CursorMode) WPFillBox(CursorX, CursorY, CursorState << 3, 16, 0x0F);
        else            WPFillBox(CursorX, CursorY + 13, CursorState << 3, 3, 0x0F);
        WPSetOp(SaveOp);
    } else if (_DI >= 5 && CursorState) {
        SaveOp = WPGetOp();
        WPSetOp(WPOP_XOR);
        if (CursorMode) WPFillBox(CursorX, CursorY, CursorState << 3, 16, 0x0F);
        else            WPFillBox(CursorX, CursorY + 13, CursorState << 3, 3, 0x0F);
        WPSetOp(SaveOp);
        CursorState = 0;
    }
}

void WPEraseCursor(void)
{
    int     SaveOp;

    if (CursorState == 0) return;
    SaveOp = WPGetOp();
    WPSetOp(WPOP_XOR);
    if (CursorMode) WPFillBox(CursorX, CursorY, CursorState << 3, 16, 0x0F);
    else            WPFillBox(CursorX, CursorY + 13, CursorState << 3, 3, 0x0F);
    WPSetOp(SaveOp);
    CursorState = 0;
}

int WPGetch(int px, int py, int inmode, int *code, int bytex)
{
    int     i, key, result, ar;
    static int restart = 0;

    while(1) {
        if (restart) {
            restart = 0;
            result = WPHANST;
            *code = InpStack[InpSP - 1].CombiCode;
            break;
        }
        WPDispCursor(px, py, bytex);
        if (!WPKeyHit()) {
            if (inmode & WPNOTWAIT) {
                result = WPNOKEY;
                break;
            } else {
                if (_WPHookWaitKey != NUL) _WPHookWaitKey();
                continue;
            }
        }
        key = *code = WPInpKey(!(inmode & WPONLYENG));
        if (key == BS) {
            if (InpSP) {
                InpSP--;
                if (InpSP) {
                    i = InpSP - 1;
                    CurState = InpStack[i].CurState;
                    *code = InpStack[i].CombiCode;
                    result = WPHANBS;
                } else {
                    result = WPNOHAN;
                    CurState = 0;
                }
            } else {
                result = WPASC;
            }
        } else if (key >= 128 && key <= 255) {
            i = CurState;
            ar = WPHanAutomata(key);
            *code = InpStack[InpSP-1].CombiCode;
            if (ar) {
                CurState = 0;
                InpSP = 0;
                result = WPHANEND;
                if (OutSP) {
                    restart = 1;
                    while (OutSP) WPHanAutomata(OutStack[--OutSP]);
                }
            } else if (!i) {
                result = WPHANST;
            } else {
                result = WPHANIN;
            }
        } else if (CurState) {
            WPUnGetKey(key);
            *code = InpStack[InpSP-1].CombiCode;
            result = WPHANEND;
            CurState = InpSP = 0;
        } else {
            result = WPASC;
        }
        break;
    }
    if (result != WPNOKEY) WPEraseCursor();

    return(result);
}

void WPEraseKeyBuff(void)
{
    WPEraseCursor();
    CurState = InpSP = OutSP = 0;
    while (WPKeyHit()) WPInpKey(0);
}

int  CurLen, ClrLen, MaxLen, CurPos, StrPos, Redraw;
char *StrBuff;

void WPDeleteSpace(void)
{
    while(StrBuff[CurLen - 1]) {
        if (StrBuff[CurLen - 1] != ' ' || CurPos >= CurLen) {
            StrBuff[CurLen] = 0;
            break;
        }
        CurLen--;
    }
}

void WPWriteInpBuff(int code)
{
    int     nbyte, i;

    nbyte = (code & 0x8000) ? 2 : 1;
    Redraw = 1;

    if (CurLen + nbyte > MaxLen) {
        WPEraseKeyBuff();
        return;
    }

    if (KBInsert && CurPos < CurLen) {
        for (i = CurLen - 1; i >= CurPos; i--) StrBuff[i + nbyte] = StrBuff[i];
        CurLen += nbyte;
    }

    if ((i = CurLen - CurPos - nbyte) < 0) CurLen = CurLen + ((i >= 0) ? i : -i);
    if (nbyte == 2) {
        if (WPCheckCharKind(StrBuff, CurPos + 1) == WPCHKIND_HAN1) StrBuff[CurPos + 2] = ' ';
        if (CurPos == StrPos + ClrLen) StrPos++;
        StrBuff[CurPos++] = (code >> 8) & 0xFF;
    }

    if (CurPos == StrPos + ClrLen) StrPos++;
    StrBuff[CurPos++] = code & 0xFF;
    StrBuff[CurLen] = 0;
}

void WPDeleteInpBuff(void)
{
    int     nbyte, i;

    if (CurLen < 1 || CurPos == CurLen) return;
    nbyte = (WPCheckCharKind(StrBuff, CurPos) == WPCHKIND_HAN1) ? 2 : 1;
    CurLen -= nbyte;
    for (i = CurPos; i < CurLen; i++)  StrBuff[i] = StrBuff[i + nbyte];
    StrBuff[CurLen] = 0;
    Redraw = 1;
}

int WPInpStr(int px, int py, int color, char *str, int clrlen, int maxlen, int mode)
{
    int     key, result, i, bch, SaveWMode;
    int     inpmode = (mode) ? 0 : WPONLYENG;
    word    temp;

    SaveWMode = WPGetWMode();
    WPSetWMode(WPMODE_OVERWRITE);

    StrBuff = str;
    MaxLen = maxlen;
    ClrLen = clrlen;
    CurLen = CurPos = strlen(StrBuff);
    StrPos = CurPos - ClrLen;
    if (StrPos < 0) StrPos = 0;
    KBInsert = Redraw = 1;

    if (_WPHookKBKindToggle != NUL) _WPHookKBKindToggle(KBKind);
    if (_WPHookHanModeToggle != NUL) _WPHookHanModeToggle(KBMode);
    if (_WPHookInsertToggle != NUL) _WPHookInsertToggle(KBInsert);

    WPPutsn(px, py, ((color << 4) & 0xf0) | (color >> 4), StrPos, ClrLen, StrBuff);
    while (!WPKeyHit()) WPDispCursor(px + ((CurPos - StrPos) << 3), py, 1);
    key = WPInpKey(mode);
    if (key >= 32 && key < 256) CurLen = CurPos = StrPos = *StrBuff = 0;
    WPUnGetKey(key);

    do {
        if (Redraw) {
            WPEraseCursor();
            WPPutsn(px, py, color, StrPos, ClrLen, StrBuff);
            for (i = CurLen - StrPos; i < ClrLen; i++) WPEngPutch(px + (i << 3), py, color, ' ');
            Redraw = 0;
        }
        bch =  ((i = StrBuff[CurPos]) & 0x80) ? 2 : 1;
        WPDeleteSpace();
        result = WPGetch(px + ((CurPos - StrPos) << 3), py, inpmode, &key, bch);
        if (result == WPASC) {
            switch (key) {
                case LSHIFT_SPACE :
                case RSHIFT_SPACE :
                    KBMode = !KBMode;
                    WPEraseCursor();
                    if (_WPHookHanModeToggle != NUL) _WPHookHanModeToggle(KBMode);
                    break;
                case INSKEY :
                    KBInsert = !KBInsert;
                    if (_WPHookInsertToggle != NUL) _WPHookInsertToggle(KBInsert);
                    break;
                case LEFTARROW :
                    if (CurPos) {
                        if (WPCheckCharKind(StrBuff, CurPos - 1) == WPCHKIND_HAN2) {
                            if (CurPos == StrPos) {
                                StrPos -= 2;
                                Redraw = 1;
                            } else if (CurPos - 1 == StrPos) {
                                StrPos--;
                                Redraw = 1;
                            }
                            CurPos -= 2;
                        } else {
                            if (CurPos == StrPos) {
                                StrPos--;
                                Redraw = 1;
                            }
                            CurPos--;
                        }
                    }
                    break;
                case RIGHTARROW :
                    if (CurPos < CurLen) {
                        if (WPCheckCharKind(StrBuff, CurPos) == WPCHKIND_HAN1) {
                            if (CurPos == StrPos + ClrLen - 2) {
                                StrPos++;
                                Redraw = 1;
                            }
                            CurPos += 2;
                        } else {
                            if (CurPos == StrPos + ClrLen - 1) {
                                StrPos++;
                                Redraw = 1;
                            }
                            CurPos++;
                        }
                    }
                    if ((CurPos == StrPos + ClrLen - 1) && (WPCheckCharKind(StrBuff, CurPos) == WPCHKIND_HAN1)) {
                        StrPos++;
                        Redraw = 1;
                    }
                    break;
                case HOMEKEY :
                    StrPos = 0;
                    CurPos = 0;
                    Redraw = 1;
                    break;
                case ENDKEY :
                    CurPos = CurLen;
                    if (CurLen > ClrLen) StrPos = CurLen - ClrLen;
                    else                 StrPos = 0;
                    Redraw = 1;
                    break;
                case BS :
                    if (CurPos) {
                        if (WPCheckCharKind(StrBuff, CurPos - 1) == WPCHKIND_HAN2) {
                            if (CurPos == StrPos) {
                                StrPos -= 2;
                                Redraw = 1;
                            }
                            CurPos -= 2;
                        } else {
                            if (CurPos == StrPos) {
                                StrPos--;
                                Redraw = 1;
                            }
                            CurPos--;
                        }
                    }
                    WPDeleteInpBuff();
                    break;
                case DELKEY :
                    WPDeleteInpBuff();
                    break;
                default :
                    if (key != ENTER && key != ESC && key < 128) {
                        if (!KBInsert && (StrBuff[CurPos] & 0x80)) StrBuff[CurPos + 1] = ' ';
                        WPWriteInpBuff(key);
                    } else {
                        if (key == _WPCallSpInpKey) {
                            if (_WPHookSpInp != NUL) {
                                temp = _WPHookSpInp();
                                if (temp) WPWriteInpBuff(temp);
                                Redraw = 1;
                            }
                        } else if (key == _WPCallHjInpKey) {
                            if (_WPHookHjInp != NUL) {
                                temp = _WPHookHjInp();
                                if (temp) WPWriteInpBuff(temp);
                                Redraw = 1;
                            }
                        } else if (key == _WPCallKBKindKey) {
                            KBKind = !KBKind;
                            if (_WPHookKBKindToggle != NUL) _WPHookKBKindToggle(KBKind);
                        }
                    }
                    break;
            }
        } else if (result == WPHANST) {
            i = CurPos;
            WPWriteInpBuff(key);
            CurPos = i;
        } else if (result == WPHANIN || result == WPHANBS) {
            asm     mov     ax, key
            StrBuff[CurPos]   = _AH;
            StrBuff[CurPos + 1] = _AL;
            Redraw = 1;
        } else if (result == WPHANEND) {
            if (CurPos == StrPos + ClrLen) StrPos += 2;
            asm     mov     ax, key
            StrBuff[CurPos++] = _AH;
            StrBuff[CurPos++] = _AL;
            Redraw = 1;
        } else if (result == WPNOHAN) {
            WPDeleteInpBuff();
        }
        WPDeleteSpace();
    } while (key != ENTER && key != ESC);

    WPSetWMode(SaveWMode);

    return(key);
}

void initNoWait(int px, int py, int color, char *str, int clrlen, int maxlen)
{
    StrBuff = str;
    MaxLen = maxlen;
    ClrLen = clrlen;
    CurLen = CurPos = strlen(StrBuff);
    StrPos = CurPos - ClrLen;
    if (StrPos < 0) StrPos = 0;
    KBInsert = Redraw = 1;

    if (_WPHookKBKindToggle != NUL) _WPHookKBKindToggle(KBKind);
    if (_WPHookHanModeToggle != NUL) _WPHookHanModeToggle(KBMode);
    if (_WPHookInsertToggle != NUL) _WPHookInsertToggle(KBInsert);

    WPPutsn(px, py, color, StrPos, ClrLen, StrBuff);
//    CurLen = CurPos = StrPos = *StrBuff = 0;
    //WPDispCursor(px + ((CurPos - StrPos) << 3), py, 1);
}

int WPInpStrNoWait(int px, int py, int color )
{
    int     key, result, i, bch, SaveWMode;
    int     inpmode, mode;
    inpmode = mode = WPNOTWAIT;
    word    temp;

    if (!WPKeyHit())
    {
        WPDispCursor(px + ((CurPos - StrPos) << 3), py, 1);
        return WPNOKEY;
    }

    key = WPInpKey(mode);
//    if (key >= 32 && key < 256) CurLen = CurPos = StrPos = *StrBuff = 0;
    WPUnGetKey(key);

    do {
        if (Redraw) {
            WPEraseCursor();
            WPPutsn(px, py, color, StrPos, ClrLen, StrBuff);
            for (i = CurLen - StrPos; i < ClrLen; i++) WPEngPutch(px + (i << 3), py, color, ' ');
            Redraw = 0;
        }
        bch =  ((i = StrBuff[CurPos]) & 0x80) ? 2 : 1;
        WPDeleteSpace();
        result = WPGetch(px + ((CurPos - StrPos) << 3), py, inpmode, &key, bch);
        if (result == WPNOKEY) return WPNOKEY;
        if (result == WPASC) {
            switch (key) {
                case ALT_1 :
                case ALT_2 :
                case ALT_4 :
                case ALT_5 :
                case ALT_6 :
                case ALT_7 :
                case ALT_8 :
                case ALT_D :
                case ALT_R :
                case ALT_E :
                case ALT_Z :
                case ALT_V :
                case ALT_S :
                case ALT_X :
                case F1 :
                case F2 :
                case F3 :
                case F4 :
                case F5 :
                case F6 :
                case F7 :
                case F8 :
                case F9 :
                case F10 :
                case F11 :
                case F12 :
                case UPARROW :
                case DOWNARROW :
                    return key;
                case LSHIFT_SPACE :
                case RSHIFT_SPACE :
                    KBMode = !KBMode;
                    WPEraseCursor();
                    if (_WPHookHanModeToggle != NUL) _WPHookHanModeToggle(KBMode);
                    break;
                case INSKEY :
                    KBInsert = !KBInsert;
                    if (_WPHookInsertToggle != NUL) _WPHookInsertToggle(KBInsert);
                    break;
                case LEFTARROW :
                    if (CurPos) {
                        if (WPCheckCharKind(StrBuff, CurPos - 1) == WPCHKIND_HAN2) {
                            if (CurPos == StrPos) {
                                StrPos -= 2;
                                Redraw = 1;
                            } else if (CurPos - 1 == StrPos) {
                                StrPos--;
                                Redraw = 1;
                            }
                            CurPos -= 2;
                        } else {
                            if (CurPos == StrPos) {
                                StrPos--;
                                Redraw = 1;
                            }
                            CurPos--;
                        }
                    }
                    break;
                case RIGHTARROW :
                    if (CurPos < CurLen) {
                        if (WPCheckCharKind(StrBuff, CurPos) == WPCHKIND_HAN1) {
                            if (CurPos == StrPos + ClrLen - 2) {
                                StrPos++;
                                Redraw = 1;
                            }
                            CurPos += 2;
                        } else {
                            if (CurPos == StrPos + ClrLen - 1) {
                                StrPos++;
                                Redraw = 1;
                            }
                            CurPos++;
                        }
                    }
                    if ((CurPos == StrPos + ClrLen - 1) && (WPCheckCharKind(StrBuff, CurPos) == WPCHKIND_HAN1)) {
                        StrPos++;
                        Redraw = 1;
                    }
                    break;
                case HOMEKEY :
                    StrPos = 0;
                    CurPos = 0;
                    Redraw = 1;
                    break;
                case ENDKEY :
                    CurPos = CurLen;
                    if (CurLen > ClrLen) StrPos = CurLen - ClrLen;
                    else                 StrPos = 0;
                    Redraw = 1;
                    break;
                case BS :
                    if (CurPos) {
                        if (WPCheckCharKind(StrBuff, CurPos - 1) == WPCHKIND_HAN2) {
                            if (CurPos == StrPos) {
                                StrPos -= 2;
                                Redraw = 1;
                            }
                            CurPos -= 2;
                        } else {
                            if (CurPos == StrPos) {
                                StrPos--;
                                Redraw = 1;
                            }
                            CurPos--;
                        }
                    }
                    WPDeleteInpBuff();
                    break;
                case DELKEY :
                    WPDeleteInpBuff();
                    break;
                default :
                    if (key != ENTER && key != ESC && key < 128) {
                        if (key<32) return key;
                        if (!KBInsert && (StrBuff[CurPos] & 0x80)) StrBuff[CurPos + 1] = ' ';
                        WPWriteInpBuff(key);
                    } else {
                        if (key == _WPCallSpInpKey) {
                            if (_WPHookSpInp != NUL) {
                                temp = _WPHookSpInp();
                                if (temp) WPWriteInpBuff(temp);
                                Redraw = 1;
                            }
                        } else if (key == _WPCallHjInpKey) {
                            if (_WPHookHjInp != NUL) {
                                temp = _WPHookHjInp();
                                if (temp) WPWriteInpBuff(temp);
                                Redraw = 1;
                            }
                        } else if (key == _WPCallKBKindKey) {
                            KBKind = !KBKind;
                            if (_WPHookKBKindToggle != NUL) _WPHookKBKindToggle(KBKind);
                        }
                    }
                    break;
            }
        } else if (result == WPHANST) {
            i = CurPos;
            WPWriteInpBuff(key);
            CurPos = i;
        } else if (result == WPHANIN || result == WPHANBS) {
            asm     mov     ax, key
            StrBuff[CurPos]   = _AH;
            StrBuff[CurPos + 1] = _AL;
            Redraw = 1;
        } else if (result == WPHANEND) {
            if (CurPos == StrPos + ClrLen) StrPos += 2;
            asm     mov     ax, key
            StrBuff[CurPos++] = _AH;
            StrBuff[CurPos++] = _AL;
            Redraw = 1;
        } else if (result == WPNOHAN) {
            WPDeleteInpBuff();
        }
        WPDeleteSpace();
    } while (key != ENTER && key != ESC);

    return(key);
}


int WPInpHanStr(int px, int py, int color, char *str, int clrlen, int maxlen)
{
    return(WPInpStr(px, py, color, str, clrlen, maxlen, 1));
}

int WPInpEngStr(int px, int py, int color, char *str, int clrlen, int maxlen)
{
    return(WPInpStr(px, py, color, str, clrlen, maxlen, 0));
}

