uses
   TMFUnit;
const

   _SCAN_CODE    = 256;
   _UP_KEY       = 72 + _SCAN_CODE;
   _DOWN_KEY     = 80 + _SCAN_CODE;
   _LEFT_KEY     = 75 + _SCAN_CODE;
   _RIGHT_KEY    = 77 + _SCAN_CODE;
   _SPACE_KEY    = 32;
   _ESC_KEY      = 27;

   VESA_MODE     = $0101;
   MAX_X_LINE    = 640;
   MAX_Y_LINE    = 480;
   BUFFER_LINE   = 64;
   MAP_X_WIDE    = 4;
   MAP_Y_WIDE    = 5;
   MAX_MAP_X     = 100;
   MAX_MAP_Y     = 100;
   BUFFER_LENGTH = MAX_X_LINE * BUFFER_LINE;
   MAX_BUFFER    = 6;
   MAP_Y_CENTER_MODIFY = -30;

   MAX_PALETTE   = 256;

   MAX_TILE      = 4*4;
   MAX_PERSON    = 3;
   MAX_CHARA     = MAX_PERSON * 12;
   TILE_X_SIZE   = 60;
   TILE_Y_SIZE   = 51;
   MAP_X_CENTER  = (MAX_X_LINE - TILE_X_SIZE) div 2;
   MAP_Y_CENTER  = (MAX_Y_LINE - TILE_Y_SIZE) div 2 + MAP_Y_CENTER_MODIFY;

   MAP_X_SIZE    = (MAP_X_WIDE + MAP_Y_WIDE+ 1) * TILE_X_SIZE;

   MAIN_CHARACTER : word = 1;

   MAX_BOX_STRING   = 20;

type
   palette_data_T = array[0..pred(MAX_PALETTE)] of record
      _RED, _GREEN, _BLUE : byte;
   end;

   PBuffer  = ^TBuffer;
   TBuffer  = array[0..64000] of byte;
   TVirtual = array[0..pred(MAX_BUFFER)] of PBuffer;

   THangulMode = (Epixel, EGradation);

const
   SCAN_CODE         : char = #0;
   CURRENT_BANK      : word = 0;
   CURRENT_READ_BANK : word = 0;
   IS_SAME_RW_BANK   : boolean = TRUE;
var
   write_bank, read_bank, shift_bank : word;
   virtual_buffer : TVirtual;
   palette_data   : palette_data_T;
   tile_data      : array[0..pred(MAX_TILE)] of PBuffer;
   object_data    : array[0..pred(MAX_CHARA)] of PBuffer;

 procedure occurError(message : string); forward;
 procedure drawScroll; forward;

function  pressedKey : boolean; assembler;
asm
      mov  ah, 1
      int  16h
      jnz  @@KEY_PRESSED
      mov  al, FALSE
      jmp  @@END
@@KEY_PRESSED:
      mov  al, TRUE
@@END:
end;

function  readKey : char; assembler;
asm
      mov  al, SCAN_CODE
      mov  SCAN_CODE, 0
      test al, 0FFh
      jnz  @@JUMP1
@@JUMP0:
      mov  ah, 1
      int  16h
      jz   @@JUMP0
      mov  ah, 0
      int  16h
      cmp  al, 0
      jne  @@JUMP1
      mov  SCAN_CODE, ah
@@JUMP1:
end;

function  getKey : integer;
var
   key_code : integer;
begin
   key_code := ord(readKey);
   if key_code = 0 then key_code := _SCAN_CODE + ord(readKey);
   while pressedKey do if readKey = #0 then readKey;
   getKey := key_code;
end;

function  sign(value : integer) : integer;
begin
   if value > 0 then sign := 1
   else if value < 0 then sign := -1
   else sign := 0;
end;

procedure initGraph;
type
   TVESAInfo = record
      Mode_Attribute     : word;
      Window_A_Attribute : byte;
      Window_B_Attribute : byte;
      Window_Granularity : word;
      gabage : array[1..250] of byte;
   end;

var
   VESAinfo  : TVESAInfo;
   pVESAinfo : ^TVESAInfo;
   result    : word;
begin

   pVESAinfo := @VESAinfo;
   asm
      mov  ax, 4F02h
      mov  bx, VESA_MODE
      int  10h
      mov  ax, 4F01h
      mov  cx, VESA_MODE
      les  di, pVESAinfo
      int  10h
      mov  result, ax
   end;

   if result <> $004F then occurError('VESA DRIVER not found.');

   for shift_bank := 0 to 15 do begin
      if ($40 shr shift_bank) = pVESAinfo^.Window_Granularity then break;
   end;

   if (pVESAinfo^.Window_A_Attribute and 3) = 3 then read_bank  := 0;
   if (pVESAinfo^.Window_A_Attribute and 5) = 5 then write_bank := 0;
   if (pVESAinfo^.Window_B_Attribute and 3) = 3 then read_bank  := 1;
   if (pVESAinfo^.Window_B_Attribute and 5) = 5 then write_bank := 1;

   IS_SAME_RW_BANK := read_bank = write_bank;

end;

procedure closeGraph; assembler;
asm
      mov  ax, 03h
      int  10h
end;

procedure occurError(message : string);
begin
   closeGraph;
   writeLn(message);
   halt;
end;

procedure clearDevice(fill : byte);
var
   i : integer;
begin
   for i := 0 to pred(MAX_BUFFER) do
      fillChar(virtual_buffer[i]^,BUFFER_LENGTH,fill);
end;

procedure putPixel(x, y : word; color : byte);
var
   bank : byte;
begin
   if y >= BUFFER_LINE * MAX_BUFFER then exit;
   bank := y div BUFFER_LINE;
   y := y mod BUFFER_LINE;
   virtual_buffer[bank]^[y*MAX_X_LINE+x] := color;
end;

procedure lineX(x1, x2, y : word; color : byte);
var
   bank : byte;
begin
   if (x1 > x2) or (y >= BUFFER_LINE * MAX_BUFFER) then exit;
   bank := y div BUFFER_LINE;
   y := y mod BUFFER_LINE;
   fillChar(virtual_buffer[bank]^[y*MAX_X_LINE+x1],succ(x2-x1),color);
end;

procedure lineY(x, y1, y2 : word; color : byte);
var
   bank : byte;
begin
   if (y1 > y2) or (y1 >= BUFFER_LINE * MAX_BUFFER) then exit;
   y2 := succ(y2 - y1);
   bank := y1 div BUFFER_LINE;
   y1 := y1 mod BUFFER_LINE;
   while y2 > 0 do begin
      virtual_buffer[bank]^[y1*MAX_X_LINE+x] := color;
      dec(y2);
      inc(y1);
      if y1 >= BUFFER_LINE then begin
         inc(bank);
         if bank >= MAX_BUFFER then exit;
         y1 := 0;
      end;
   end;
end;

procedure bar(x,y,x_len,y_len : word; color : byte);
var
   bank : byte;
begin
   if y >= BUFFER_LINE * MAX_BUFFER then exit;
   bank := y div BUFFER_LINE;
   y := y mod BUFFER_LINE;
   while y_len > 0 do begin
      fillChar(virtual_buffer[bank]^[y*MAX_X_LINE+x],x_len,color);
      dec(y_len);
      inc(y);
      if y >= BUFFER_LINE then begin
         inc(bank);
         if bank >= MAX_BUFFER then exit;
         y := 0;
      end;
   end;
end;

procedure putImage(x, y : word; var bitmap_data);
var
   pb : ^TBuffer;
   target : TVirtual;
   x_len  : word;
begin
   if y >= BUFFER_LINE * MAX_BUFFER then exit;
   pb := @bitmap_data;
   target := virtual_buffer;
   asm
         push ds

         mov  ax, y
         shr  ax, 4
         and  ax, 0FFFCh
         lea  bx, target
         add  bx, ax
         mov  ax, y
         and  ax, BUFFER_LINE - 1
         mov  y, ax

         cld
         lds  si, bitmap_data
         lodsw
         mov  x_len, ax
         lodsw
         mov  cx, ax

         mov  di, ss:[bx]
         mov  es, ss:[bx+2]

         mov  ax, y
         mov  dx, ax
         shl  ax, 2
         add  ax, dx
         shl  ax, 7
         add  ax, x
         add  di, ax

@@MAIN_LOOP:
         push cx

         mov  cx, x_len
      @@LOOP:
         mov  al, ds:[si]
         inc  al
         jz   @@SKIP
         dec  al
         mov  es:[di], al
      @@SKIP:
         inc  si
         inc  di
         loop @@LOOP

         add  di, MAX_X_LINE
         sub  di, x_len

         mov  ax, y
         inc  ax
         mov  y, ax
         cmp  ax, BUFFER_LINE
         jne  @@NOT_CHANGE_BANK
         mov  y, 0

         add  bx, 4
         mov  di, ss:[bx]
         mov  es, ss:[bx+2]
         add  di, x

      @@NOT_CHANGE_BANK:
         pop  cx
         loop @@MAIN_LOOP

         pop  ds
   end;
end;

procedure flipPage;
var
   source : TVirtual;
   x_len, _write_bank, _shift_bank, page : word;
begin
   source := virtual_buffer;
   asm

         cld
         mov  ax, write_bank
         mov  _write_bank, ax
         mov  ax, shift_bank
         mov  _shift_bank, ax
         mov  page, 0
         mov  dx, CURRENT_BANK
         cmp  dx, 0
         je   @@SKIP_SET_BANK
         mov  ax, 4F05h
         xor  dx, dx
         mov  bx, _write_bank
         int  10h

      @@SKIP_SET_BANK:
         push ds

         mov  es, SegA000
         mov  di, 0
         lea  bx, source
         mov  si, ss:[bx]
         mov  ds, ss:[bx+2]
         add  bx, 4
         mov  cx, MAX_X_LINE * BUFFER_LINE
         jmp  @@MAIN_LOOP

   @@PAGE_BOUNDARY:
         sub  cx, ax
         dec  cx
         push cx
         mov  cx, ax
         inc  cx
         shr  cx, 1
         jnc  @@NO_CARRY21
         movsb
      @@NO_CARRY21:
         rep  movsw
         pop  cx
         push bx
         inc  dx
         mov  ax, 4F05h
         mov  bx, _write_bank
         push cx
         mov  cx, _shift_bank
         shl  dx, cl
         int  10h
         shr  dx, cl
         pop  cx
         pop  bx

@@MAIN_LOOP:
         mov  ax, di
         not  ax
         cmp  ax, cx
         jb   @@PAGE_BOUNDARY
         shr  cx, 1
         jnc  @@NO_CARRY11
         movsb
      @@NO_CARRY11:
         rep  movsw

         mov  ax, page
         inc  ax
         mov  page, ax
         cmp  page, MAX_BUFFER
         je   @@END

         mov  si, ss:[bx]
         mov  ds, ss:[bx+2]
         add  bx, 4
         mov  cx, MAX_X_LINE * BUFFER_LINE
         jmp  @@MAIN_LOOP

@@END:
         pop  ds

         mov  CURRENT_BANK, dx
   end;
end;

(* Hangul Unit Begin *)

type
   THangulFontData = array[0..15,0..1] of byte;
   TAsciiFontData  = array[0..15] of byte;
   THangulBuffer   = array[0..33] of byte;
var
   max_font : integer;
   ascii_font_data : array[0.. 95] of ^TAsciiFontData;
   han_font_data   : array[0..499] of ^THangulFontData;
   buffer : THangulBuffer;
   hangul_mode : THangulMode;

procedure setHangulMode(mode : THangulMode);
begin
   hangul_mode := mode;
end;

procedure printHangul(x, y : integer; s : string; color : byte; is_bold : boolean);
const
   convert_table : array[1..3,0..30] of byte = (
      ( 0, 0, 1, 2, 3, 4, 5, 6, 7, 8,
        9,10,11,12,13,14,15,16,17,18,
       19, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0),
      ( 0, 0, 0, 1, 2, 3, 4, 5, 0, 0,
        6, 7, 8, 9,10,11, 0, 0,12,13,
       14,15,16,17, 0, 0,18,19,20,21,0),
      ( 0, 0, 1, 2, 3, 4, 5, 6, 7, 8,
        9,10,11,12,13,14,15,16, 0,17,
       18,19,20,21,22,23,24,25,26,27,0)
   );
var
   jamo : array[1..3] of byte;
   font_number : array[1..3] of integer;
   i, j : integer;
   s_ptr : integer;
   temp_w : word;
   temp_b : byte;
   pb1, pb2 : ^byte;
   temp_buffer : THangulBuffer;

 procedure putUnitHangul(x,y : integer; color : byte);
 var
    i, j, k : integer;
 begin
    case hangul_mode of
       Epixel :
       for j := 0 to 255 do begin
          if (buffer[j shr 3] and ($80 shr (j and $07))) > 0 then
             putPixel(x+((j shr 3) and $01) shl 3 + (j and $07),y+j shr 4,color{ + j shr 5});
       end;
       EGradation :
       for j := 0 to 255 do begin
          if (buffer[j shr 3] and ($80 shr (j and $07))) > 0 then
             putPixel(x+((j shr 3) and $01) shl 3 + (j and $07),y+j shr 4,color + j shr 5);
       end;
    end;
 end;

begin

   s_ptr := 1;
   while s_ptr <= length(s) do begin
      if ((ord(s[s_ptr]) and $80) > 0) and (s_ptr < length(s)) then begin
         jamo[1] := (ord(s[s_ptr]) and $7C) shr 2;
         jamo[2] := ((ord(s[s_ptr]) and $03) shl 3) + (ord(s[s_ptr+1]) shr 5);
         jamo[3] := (ord(s[s_ptr+1]) and $1F);
         for i := 1 to 3 do begin
            if jamo[i] in [0..30] then
               jamo[i] := convert_table[i,jamo[i]]
            else
               jamo[i] := 0;
         end;
         if jamo[3] > 0 then begin
            if jamo[2] in [1..8,21] then font_number[1] := 95;
            if jamo[2] in [9,13,14,18,19] then font_number[1] := 114;
            if jamo[2] in [10..12,15..17,20] then font_number[1] := 133;
            if jamo[1] in [1,16] then font_number[2] := 194;
            if jamo[1] in [2..15,17..19] then font_number[2] := 215;
            if jamo[2] in [1,3,10] then font_number[3] := 236;
            if jamo[2] in [5,7,12,15,17,20,21] then font_number[3] := 263;
            if jamo[2] in [2,4,6,8,11,16] then font_number[3] := 290;
            if jamo[2] in [9,13,14,18,19] then font_number[3] := 317;
         end else begin
            if jamo[2] in [1..8,21] then font_number[1] := 0;
            if jamo[2] in [9,13,19] then font_number[1] := 19;
            if jamo[2] in [14,18] then font_number[1] := 38;
            if jamo[2] in [10..12,20] then font_number[1] := 57;
            if jamo[2] in [15..17] then font_number[1] := 76;
            if jamo[1] in [1,16] then font_number[2] := 152;
            if jamo[1] in [2..15,17..19] then font_number[2] := 173;
            font_number[3] := 0;
         end;

         move(han_font_data[font_number[1]+jamo[1]-1]^,buffer,32);
         buffer[32] := 0; buffer[33] := 0;
         pb1 := addr(buffer);
         pb2 := addr(han_font_data[font_number[2]+jamo[2]-1]^);
         for i := 0 to 31 do begin
            pb1^ := pb1^ or pb2^;
            inc(pb1); inc(pb2);
         end;
         if font_number[3] > 0 then begin
            pb1 := addr(buffer);
            pb2 := addr(han_font_data[font_number[3]+jamo[3]-1]^);
            for i := 0 to 31 do begin
               pb1^ := pb1^ or pb2^;
               inc(pb1); inc(pb2);
            end;
         end;

         if is_bold then begin

            temp_buffer := buffer;

            for j := 0 to 31 do begin
               buffer[j] := buffer[j] or buffer[j+2];
            end;
            for j := 31 downto 1 do begin
               buffer[j] := buffer[j] or buffer[j-2];
            end;
            for j := 0 to 31 do begin
               buffer[j] := buffer[j] or (buffer[j] shr 1);
            end;
            for j := 31 downto 0 do begin
               buffer[j] := buffer[j] or (buffer[j] shl 1);
            end;

            putUnitHangul(x+(s_ptr-1)*8,y,0);
            buffer := temp_buffer;

         end;

         putUnitHangul(x+(s_ptr-1)*8,y,color);
         inc(s_ptr,2);
      end
      else begin

         temp_b := ord(s[s_ptr]);
         if (temp_b < 30) then temp_b := 31;
         if (temp_b > 126) then temp_b := 32;
         dec(temp_b,31);

         fillChar(buffer,32,#0);
         pb1 := addr(buffer);
         pb2 := addr(ascii_font_data[temp_b]^);
         for i := 0 to 15 do begin
            pb1^ := pb2^;
            inc(pb1,2);
            inc(pb2);
         end;

         if is_bold then begin

            temp_buffer := buffer;

            for j := 0 to 15 do begin
               buffer[j*2] := buffer[j*2] or buffer[(j+1)*2];
            end;
            for j := 15 downto 1 do begin
               buffer[j*2] := buffer[j*2] or buffer[(j-1)*2];
            end;
            for j := 0 to 15 do begin
               buffer[j*2] := buffer[j*2] or (buffer[j*2] shr 1);
            end;
            for j := 15 downto 0 do begin
               buffer[j*2] := buffer[j*2] or (buffer[j*2] shl 1);
            end;

            putUnitHangul(x+(s_ptr-1)*8,y,0);
            buffer := temp_buffer;

         end;

         putUnitHangul(x+(s_ptr-1)*8,y,color);
         inc(s_ptr);
      end;
   end;
end;

procedure initHangul;
const
   ENGLISH_FONT_FILE = 'Ascii.Fnt';
   KOREAN_FONT_FILE  = 'GHan10.Fnt';
var
   f : file;
begin

   assign(f,ENGLISH_FONT_FILE);
  {$I-}
   reset(f,1);
  {$I+}
   if IOResult <> 0 then occurError('File not found : '+ENGLISH_FONT_FILE);

   max_font := -1;
   while not eof(f) do begin
      inc(max_font);
      new(ascii_font_data[max_font]);
      BlockRead(f,ascii_font_data[max_font]^,16);
   end;

   close(f);

   assign(f,KOREAN_FONT_FILE);
  {$I-}
   reset(f,1);
  {$I+}
   if IOResult <> 0 then occurError('File not found : '+KOREAN_FONT_FILE);

   max_font := -1;
   while not eof(f) do begin
      inc(max_font);
      new(han_font_data[max_font]);
      BlockRead(f,han_font_data[max_font]^,32);
   end;

   close(f);

   setHangulMode(Epixel);
end;

(* Hangul Unit End *)

(* Box Object Definition Begin *)

type
   PGraphicBox = ^TGraphicBox;
   TGraphicBox = object
      x1,y1,x2,y2,max_line,fore_color,back_color,hide_color : byte;
      message : array[1..MAX_BOX_STRING] of string;
      enable_message : array[1..MAX_BOX_STRING] of boolean;

      constructor init(_x1,_x2,_y1,_fore_color,_back_color,_hide_color,_max_line : byte);
      destructor  done;
      procedure enableMessage(number : byte);
      procedure disableMessage(number : byte);
      procedure setMessage(number : byte; s : string);
      procedure drawTextBox;
      function  selectTextBox(y_line : byte) : integer;
   end;

procedure printString(x,y : integer; s : string; fore_color, back_color : integer);
begin
   bar(x*8,y*16,length(s)*8-1,15,back_color);
   printHangul(x*8,y*16,s,fore_color,FALSE);
end;

constructor TGraphicBox.init;
var
   i : byte;
begin
   x1 := _x1; x2 := _x1 + _x2; y1 := _y1; y2 := _y1 + succ(_max_line);
   fore_color := _fore_color;
   back_color := _back_color;
   hide_color := _hide_color;
   max_line := _max_line;
   for i := 1 to max_line do begin
      message[i] := '';
      enable_message[i] := TRUE;
   end;
end;

destructor  TGraphicBox.done; begin end;

procedure TGraphicBox.enableMessage(number : byte);
begin
   if number in [1..MAX_BOX_STRING] then begin
      enable_message[number] := TRUE;
   end;
end;

procedure TGraphicBox.disableMessage(number : byte);
begin
   if number in [1..MAX_BOX_STRING] then begin
      enable_message[number] := FALSE;
   end;
end;

procedure TGraphicBox.setMessage(number : byte; s : string);
begin
   if number in [1..MAX_BOX_STRING] then begin
      message[number] := s;
   end;
end;

procedure TGraphicBox.drawTextBox;
var
   i : integer;
begin
   bar(x1*8,y1*16,(x2-x1)*8+7,(y2-y1)*16+15,back_color);
   lineX(x1*8,x2*8+7,y1*16,15);
   lineX(x1*8,x2*8+7,y1*16+1,15);
   lineX(x1*8,x2*8+7,y2*16+15,15);
   lineX(x1*8,x2*8+7,y2*16+14,15);
   lineY(x1*8,y1*16,y2*16+15,15);
   lineY(x1*8+1,y1*16,y2*16+15,15);
   lineY(x2*8+7,y1*16,y2*16+15,15);
   lineY(x2*8+6,y1*16,y2*16+15,15);
   for i := 1 to max_line do begin
      if message[i] <> '' then begin
         if enable_message[i] then
            printString(x1+2,(y1+i),message[i],fore_color,back_color)
         else
            printString(x1+2,(y1+i),message[i],hide_color,back_color);
      end else begin
         lineX(x1*8,x2*8+7,(y1+i)*16+7,15);
         lineX(x1*8,x2*8+7,(y1+i)*16+8,15);
      end;
   end;
end;

function  TGraphicBox.selectTextBox(y_line : byte) : integer;
const
   UP_KEY   = 256 + 72;
   DOWN_KEY = 256 + 80;
var
   i, key_code : integer;
begin
   key_code := 0;
   printString(x1+2,y1+y_line,message[y_line],fore_color,hide_color);
   flipPage;
   while not (key_code in [13,27]) do begin
      key_code := ord(readKey);
      if key_code = 0 then key_code := 256 + ord(readKey);
      printString(x1+2,y1+y_line,message[y_line],fore_color,back_color);
      case key_code of
         UP_KEY   :
         if y_line > 1 then begin
            repeat
               dec(y_line);
            until ((message[y_line] <> '') and enable_message[y_line]) or (y_line = 1);
            if y_line = 1 then begin
               while ((message[y_line] = '') or not enable_message[y_line]) do inc(y_line);
            end;
         end;
         DOWN_KEY : if y_line < max_line then begin
            repeat
               inc(y_line)
            until ((message[y_line] <> '') and enable_message[y_line]) or (y_line = max_line);
            if y_line = max_line then begin
               while ((message[y_line] = '') or not enable_message[y_line]) do dec(y_line);
            end;
         end;
      end;
      printString(x1+2,y1+y_line,message[y_line],fore_color,hide_color);
      flipPage;
   end;
   if key_code = 13 then selectTextBox := y_line else selectTextBox := 0;
end;

(* Box Object Definition End *)

(* Message Object Definition Begin *)

const
   MAX_MESSAGE = 10;
type
   TUnitMessage = record
      x, y, delay : word;
      color       : byte;
      is_bold     : boolean;
      s           : string[80];
   end;
   PMessage = ^TMessage;
   TMessage = object

      m_message : array[1..MAX_MESSAGE] of TUnitMessage;

      constructor init;
      destructor  done;
      procedure   registerMessage(_x,_y : word; _s : string; _color : byte; _is_bold : boolean; _delay : word);
      procedure   displayMessage;
   end;

var
   message : PMessage;

constructor TMessage.init;
var
   i : integer;
begin
   for i := 1 to MAX_MESSAGE do begin
      m_message[i].delay := 0;
   end;
end;

destructor  TMessage.done; begin end;

procedure   TMessage.registerMessage(_x,_y : word; _s : string; _color : byte; _is_bold : boolean; _delay : word);
var
   i : integer;
begin
   for i := 1 to MAX_MESSAGE do begin
      if m_message[i].delay = 0 then begin
         with m_message[i] do begin
            x := _x; y := _y;
            s := _s;
            color := _color;
            is_bold := _is_bold;
            delay := _delay;
         end;
         exit;
      end;
   end;
end;

procedure   TMessage.displayMessage;
var
   i : integer;
begin
   for i := 1 to MAX_MESSAGE do begin
      if m_message[i].delay > 0 then begin
         with m_message[i] do begin
            printHangul(x,y,s,color,is_bold);
            dec(delay);
         end;
      end;
   end;
end;

(* Message Object Definition End *)

(* Map Object Definition Begin *)

type

   TMapAttribute = (Emap_tile,Emap_height,Emap_person);
   TXaxisMap = array[0..MAX_MAP_X,Emap_tile..Emap_person] of byte;

   PMap = ^TMap;
   TMap = object

      m_map_data : array[0..MAX_MAP_Y] of ^TXaxisMap;

      constructor init;
      destructor  done;
      function    readMap(attribute : TMapAttribute; x, y : integer) : byte;
      procedure   writeMap(attribute : TMapAttribute; x, y : integer; value : byte);
   end;

var
   map : PMap;

constructor TMap.init;
var
   i, j, k : integer;
begin
   for j := 0 to MAX_MAP_Y do begin
      new(m_map_data[j]);
   end;
   randomize;
   for j := 0 to MAX_MAP_Y do begin
      for i := 0 to MAX_MAP_X do begin
         case random(100) of
             0..80 : k := 3;
            81..92 : k := 2;
            93..96 : k := 1;
            97..99 : k := 0;
         end;
         m_map_data[j]^[i][Emap_tile] := k;
         if random(100) < 5 then k := 4 else k := 0;
         m_map_data[j]^[i][Emap_height] := random(k+1);
         m_map_data[j]^[i][Emap_person] := 0;
      end;
   end;
end;

destructor TMap.done;
var
   j : integer;
begin
   for j := 0 to MAX_MAP_Y do begin
      dispose(m_map_data[j]);
   end;
end;

function  TMap.readMap(attribute : TMapAttribute; x, y : integer) : byte;

 function  isUnvalancedMap(x, y : integer; height : byte) : boolean;
 begin
    isUnvalancedMap := FALSE;
    if (x >= 0) and (y >= 0) and (x <= MAX_MAP_X) and (y <= MAX_MAP_Y) then begin
       if readMap(Emap_height,x,y) <> height then isUnvalancedMap := TRUE;
    end;
 end;

var
   tile_no : byte;
begin
   if (x >= 0) and (y >= 0) and (x <= MAX_MAP_X) and (y <= MAX_MAP_Y) then begin
      if attribute = Emap_tile then begin
         tile_no := m_map_data[y]^[x][Emap_tile] * 4;
         if isUnvalancedMap(x,y-1,readMap(Emap_height,x,y)) then inc(tile_no,1);
         if isUnvalancedMap(x+1,y,readMap(Emap_height,x,y)) then inc(tile_no,2);
         readMap := tile_no;
      end else begin
         readMap := m_map_data[y]^[x][attribute];
      end;
   end else readMap := 0;
end;

procedure TMap.writeMap(attribute : TMapAttribute; x, y : integer; value : byte);
begin
   if (x >= 0) and (y >= 0) and (x <= MAX_MAP_X) and (y <= MAX_MAP_Y) then begin
      m_map_data[y]^[x][attribute] := value;
   end;
end;

(* Map Object Definition End *)

(* Person Object Definition Begin *)

type

   PPerson = ^TPerson;
   TPerson = object

      m_number, m_x, m_y : integer;
      m_face, m_face_count : integer;
      m_name : string[16];
      m_hit_point, m_spell_point, m_level, m_experience : integer;

      constructor init(number, x, y : integer);
      destructor  done;
      procedure   setParameter(name : string; HP, SP, level, exp : integer);
      function    returnCharaTile : pointer;
      function    returnX : integer;
      function    returnY : integer;
      procedure   moveXY(x1, y1 : integer);
      procedure   warpXY(x, y : integer);
      procedure   displayAbility;
      function    doAction : integer;

   end;

var
   person : array[1..MAX_PERSON] of PPerson;

constructor TPerson.init(number, x, y : integer);
begin
   m_number := number;
   m_x := x;
   m_y := y;
   m_face := 0;
   m_face_count := 0;
   map^.writeMap(Emap_person,x,y,number);
end;

destructor  TPerson.done; begin end;

procedure   TPerson.setParameter(name : string; HP, SP, level, exp : integer);
begin
   m_name := name;
   m_hit_point := HP;
   m_spell_point := SP;
   m_level := level;
   m_experience := exp;
end;

function    TPerson.returnCharaTile : pointer;
begin
   returnCharaTile := object_data[(m_number-1)*12+m_face+m_face_count*4];
end;

function    TPerson.returnX : integer;
begin
   returnX := m_x;
end;

function    TPerson.returnY : integer;
begin
   returnY := m_y;
end;

procedure   TPerson.moveXY(x1, y1 : integer);
begin
   if y1 < 0 then m_face := 2;
   if y1 > 0 then m_face := 1;
   if x1 < 0 then m_face := 0;
   if x1 > 0 then m_face := 3;
   inc(m_face_count);
   m_face_count := m_face_count mod 3;

   if (m_x+x1 < 0) or (m_y+y1 < 0) or (m_x+x1 > MAX_MAP_X) or (m_y+y1 > MAX_MAP_Y) then exit;

   map^.writeMap(Emap_person,m_x,m_y,0);
   if map^.readMap(Emap_height,m_x,m_y) + 2 >= map^.readMap(Emap_height,m_x+x1,m_y+y1) then begin
      m_x := m_x + x1;
      m_y := m_y + y1;
      if map^.readMap(Emap_person,m_x,m_y) <> 0 then begin
         if m_number = MAIN_CHARACTER then begin
            person[map^.readMap(Emap_person,m_x,m_y)]^.moveXY(-x1,-y1);
         end else begin
            m_x := m_x - x1;
            m_y := m_y - y1;
         end;
      end;
   end;
   map^.writeMap(Emap_person,m_x,m_y,m_number);
end;

procedure TPerson.warpXY(x, y : integer);
begin
   if (x < 0) or (y < 0) or (x > MAX_MAP_X) or (y > MAX_MAP_Y) then exit;
   if map^.readMap(Emap_person,m_x,m_y) = 0 then begin
      map^.writeMap(Emap_person,m_x,m_y,0);
      m_x := x; m_y := y;
      map^.writeMap(Emap_person,m_x,m_y,m_number);
   end;
end;

procedure TPerson.displayAbility;
const
   DELAY_TIME = 50;
var
   s        : string;
   x, y     : integer;
begin
   x := MAX_X_LINE div 16 - 8;
   y := MAP_Y_WIDE * 2;
   message^.registerMessage(x*8,(y-1)*16,'    q : '+m_name,8*20,TRUE,DELAY_TIME);
   str(m_hit_point : 5,s);
   message^.registerMessage(x*8,(y+1)*16,'A    b : '+s+'%',8*7,TRUE,DELAY_TIME);
   str(m_spell_point : 5,s);
   message^.registerMessage(x*8,(y+2)*16,'a    b : '+s+'%',8*7,TRUE,DELAY_TIME);
   str(m_level : 6,s);
   message^.registerMessage(x*8,(y+3)*16,'A    I : '+s,8*7,TRUE,DELAY_TIME);
   str(m_experience : 6,s);
   message^.registerMessage(x*8,(y+4)*16,'w  á : '+s,8*7,TRUE,DELAY_TIME);
end;

function  TPerson.doAction : integer;
var
   i, j, x1, y1 : integer;
   box : PGraphicBox;
begin
   doAction := 0;
   if m_number = MAIN_CHARACTER then begin
      case getKey of
         _UP_KEY    : moveXY(0,-1);
         _DOWN_KEY  : moveXY(0, 1);
         _LEFT_KEY  : moveXY(-1,0);
         _RIGHT_KEY : moveXY( 1,0);
         _ESC_KEY, _SPACE_KEY :
         begin
            i := MAX_X_LINE div 16 - 10;
            j := MAP_Y_WIDE * 2;
            new(box,init(i,19,j,8*19,100,8*10,5));
            with box^ do begin
               setMessage(1,'  Ae a ');
               setMessage(2,' wbái  a ');
               setMessage(3,' ei  aa ');
               setMessage(4,' i  aa ');
               setMessage(5,' Ai  {a ');
               disableMessage(3);
               drawTextBox;
               flipPage;
               i := selectTextBox(1);
            end;
            dispose(box,done);
            case i of
               1 :
               begin
                  i := MAX_X_LINE div 16 - 20;
                  j := MAP_Y_WIDE * 2;
                  new(box,init(i,39,j,8*19,8*10-1,8*11,2));
                  with box^ do begin
                     setMessage(1,'< Aa a ơA iA aa >');
                     setMessage(2,'         ee aq : ew');
                     disableMessage(2);
                     drawTextBox;
                     flipPage;
                     readKey;
                  end;
                  dispose(box,done);
               end;
               2 :
               begin
                  displayAbility;
               end;
               4 :
               begin
                  if MAIN_CHARACTER < MAX_PERSON then inc(MAIN_CHARACTER)
                                                 else MAIN_CHARACTER := 1;
                  drawScroll;
               end;
               5 :
               begin
                  doAction := _ESC_KEY;
               end;
            end;
         end;
      end;
   end else begin
      x1 := sign(person[MAIN_CHARACTER]^.returnX-returnX);
      y1 := sign(person[MAIN_CHARACTER]^.returnY-returnY);
      if (x1 <> 0) and (y1 <> 0) then begin
         if random(2) = 0 then x1 := 0 else y1 := 0;
      end;
      moveXY(x1,y1);
   end;
end;

(* Person Object Definition End *)

procedure drawScroll;
var
   _number, _height, _person : byte;
   x,y,z : integer;
   map_x, map_y   : integer;
begin

   clearDevice(0);

   map_x := person[MAIN_CHARACTER]^.returnX;
   map_y := person[MAIN_CHARACTER]^.returnY;

   for y := -MAP_Y_WIDE to MAP_Y_WIDE do begin
      for x := MAP_X_WIDE downto -MAP_X_WIDE do begin
         _number := map^.readMap(Emap_tile,map_x+x,map_y+y);
         _height := map^.readMap(Emap_height,map_x+x,map_y+y);
         _person := map^.readMap(Emap_person,map_x+x,map_y+y);
         for z := 0 to _height do begin
            putImage(x*60+(y-x)*29+MAP_X_CENTER,y*(50-37)-x*14-z*13+MAP_Y_CENTER,tile_data[_number]^);
         end;
         if _person > 0 then begin
            putImage(x*60+(y-x)*29+MAP_X_CENTER,y*(50-37)-x*14+MAP_Y_CENTER-z*13-45,person[_person]^.returnCharaTile^);
         end;
      end;
   end;

   message^.displayMessage;

   flipPage;

end;

procedure initializeProgram;
const
   GRAPHIC_DATA_FILE = 'NETO.DAT';
var
   f          : file;
   i, j, k    : integer;
   image_size : word;
begin

   writeLn('Initializing Map...');
   new(map,init);
   new(message,init);

   for i := 0 to pred(MAX_BUFFER) do begin
      getMem(virtual_buffer[i],BUFFER_LENGTH);
      if not assigned(virtual_buffer[i]) then
         occurError('Virtual Buffer Allocation Error');
   end;

   initGraph;

   assign(f,GRAPHIC_DATA_FILE);
  {$I-}
   reset(f,1);
  {$I+}
   if IOResult <> 0 then occurError('File not found : '+GRAPHIC_DATA_FILE);

   BlockRead(f,palette_data,sizeof(palette_data));
   image_size := 60 * 45 + 4;
   for i := 0 to pred(MAX_TILE) do begin
      getMem(tile_data[i],image_size);
      BlockRead(f,tile_data[i]^,image_size);
   end;
   image_size := 54 * 70 + 4;
   for i := 0 to pred(MAX_CHARA) do begin
      getMem(object_data[i],image_size);
      BlockRead(f,object_data[i]^,image_size);
   end;
   close(f);
   port[$3C8] := 0;
   for i := 0 to 255 do begin
      port[$3C9] := palette_data[i]._RED;
      port[$3C9] := palette_data[i]._GREEN;
      port[$3C9] := palette_data[i]._BLUE;
   end;

   for i := 1 to MAX_PERSON do new(person[i],init(i,MAX_MAP_X div 2,MAX_MAP_X div 2+i));

   person[1]^.setParameter('Aaa',100,100,10,5000);
   person[2]^.setParameter('a', 20, 20, 3,500);
   person[3]^.setParameter('Aaa', 50, 80, 5,2000);

   initHangul;
   setHangulMode(EGradation);

   playTMF('NeTo');
end;

procedure closeProgram;
var
   i : integer;
begin
   endTMF;

   dispose(message,done);
   dispose(map,done);
   closeGraph;
   for i := 0 to pred(MAX_BUFFER) do freeMem(virtual_buffer[i],BUFFER_LENGTH);
end;

var
   i, code : integer;
begin

   initializeProgram;

   repeat
      drawScroll;
      if pressedKey then
      for i := 1 to MAX_PERSON do begin
         if person[i]^.doAction = _ESC_KEY then begin
            code := _ESC_KEY;
         end;
      end;
   until code = _ESC_KEY;

   closeProgram;

end.
