/*
    Module      :   3DS
    Version     :   1.0.0
    Description :   Autodesk 3DS file import for Watcom C/C++
    Author info :   Hoseo univ.  Computer eng. '94
                    E-Mail : sungksun@hitel.kol.co.kr
                   
    Revison history
        date        version     author          comment
    ----------------------------------------------------
    97-05-05        1.0.0       Digitalizer     3DS import stuff
*/

#include <stdio.h>
#include <mem.h>

#if     __WATCOMC__ > 1000
#pragma pack(push, 1)
#else
#pragma pack(1)
#endif

typedef struct{
	unsigned short id;
	unsigned int size;
}chunk;

#if     __WATCOMC__ > 1000
#pragma pack(pop)
#else
#pragma pack()
#endif

/*
** chunk types
*/
#define MAIN3DS                  0x4D4D                             // 3DS file ID
    #define EDIT3DS                  0x3D3D                         // object
        #define EDIT_MESH_VERSION        0x3d3e                     // mesh version
        #define EDIT_MATERIAL            0xAFFF                     // material object
            #define MAT_NAME                 0xa000                 // material name
            #define MAT_AMBIENT              0xa010                 // ambient
            #define MAT_DIFFUSE              0xa020                 // diffuse
            #define MAT_SPECULAR             0xa030                 // specular
            #define MAT_SHINESS              0xa040                 // shiness
            #define MAT_SHIN2PCT             0xa041                 // shiness strength
            #define MAT_SHIN3PCT             0xa042
            #define MAT_TRANSPARENCY         0xa050                 // transparency
            #define MAT_XPFALL               0xa052                 // trans falloff
            #define MAT_REFBLUR              0xa053                 // reflect blur
            #define MAT_SELF_ILLUM           0xa080                 // self illum
            #define MAT_TWO_SIDE             0xa081                 // two side
            #define MAT_DECAL                0xa082
            #define MAT_ADDITIVE             0xa083
            #define MAT_SELF_ILPCT           0xa084
            #define MAT_WIRE                 0xa085
            #define MAT_SUPERSMP             0xa086
            #define MAT_WIRESIZE             0xa087
            #define MAT_FACEMAP              0xa088
            #define MAT_XPFALLIN             0xa08a
            #define MAT_PHONGSOFT            0xa08c
            #define MAT_WIREABS              0xa08e
            #define MAT_SHADING              0xa100                 // shading type
            #define MAT_TEXMAP               0xa200                 // texture mapping
            #define MAT_SPECMAP              0xa204                 // specular mapping
            #define MAT_OPACMAP              0xa210                 // opacity mapping
            #define MAT_REFLMAP              0xa220                 // reflection mapping
            #define MAT_BUMPMAP              0xa230                 // bump mapping
            #define MAT_USE_XPFALL           0xa240
            #define MAT_USE_REFBLUR          0xa250
            #define MAT_BUMP_PERCENT         0xa252
            #define MAT_MAPNAME              0xa300
            #define MAT_ACUBIC               0xa310
            #define MAT_SXP_TEXT_DATA        0xa320
            #define MAT_SXP_TEXT2_DATA       0xa321
        #define EDIT_CONFIG1             0x0100
        #define EDIT_CONFIG2             0x3E3D
        #define EDIT_AMBIENT_LIGHT       0x2100
        #define EDIT_FOG                 0x2200
        #define EDIT_USE_FOG             0x2201
        #define EDIT_FOG_BGND            0x2210
        #define EDIT_DISTANCE_CUE        0x2300
        #define EDIT_LAYER_FOG           0x2302
        #define EDIT_USE_LAYER_FOG       0x2303
        #define EDIT_DCUE_BGND           0x2301
        #define EDIT_SMAGIC              0x2d2d
        #define EDIT_LMAGIC              0x2d3d
        #define VIEWPORT_LAYOUT_OLD      0x7000
        #define VIEWPORT_LAYOUT          0x7001
        #define VIEWPORT_DATA_OLD        0x7010
        #define VIEWPORT_DATA            0x7011
        #define VIEWPORT_DATA_3          0x7012
        #define VIEWPORT_SIZE            0x7020
        #define VIEWPORT_NETWORK_VIEW    0x7030
        #define EDIT_BITMAP              0x1100
        #define EDIT_BACKGR              0x1200
        #define EDIT_USE_SOLID_BGND      0x1201
        #define EDIT_V_GRADIENT          0x1300
        #define EDIT_LO_SHADOW_BIAS      0x1400
        #define EDIT_HI_SHADOW_BIAS      0x1410
        #define EDIT_SHADOW_MAP_SIZE     0x1420
        #define EDIT_SHADOW_SAMPLE       0x1430
        #define EDIT_SHADOW_RANGE        0x1440
        #define EDIT_SHADOW_FILTER       0x1450
        #define EDIT_RAY_BIAS            0x1460
        #define EDIT_O_CONSTS            0x1500
        #define EDIT_OBJECT              0x4000                     // object block
        #define OBJ_HIDDEN               0x4010
        #define OBJ_VIS_LOFTER           0x4011
        #define OBJ_DOESNT_CAST          0x4012
        #define OBJ_MATTE                0x4013
        #define OBJ_FAST                 0x4014
        #define OBJ_PROCEDURAL           0x4015
        #define OBJ_FROZEN               0x4016
        #define OBJ_DONT_RCVSHADOW       0x4017
            #define OBJ_TRIMESH              0x4100                 // mesh object
                #define TRI_VERTEXL              0x4110             // vertex list
                #define TRI_FACEL2               0x4111
                #define TRI_FACEL1               0x4120             // face list
                #define TRI_MSH_MAT_GROUP        0x4130             // face material
                #define TRI_OLD_MAT_GROUP        0x4131
                #define TRI_TEXTURE_VERTS        0x4140             // texture coordinate
                #define TRI_SMOOTH               0x4150             // smooth list
                #define TRI_LOCAL_AXIS           0x4160             // triangle matrix
                #define TRI_MESH_COLOR           0x4165
                #define TRI_MESH_TEXTURE_INFO    0x4170
            #define OBJ_CAMERA               0x4700                 // camera object
            #define OBJ_CAM_SEE_CONE         0x4710
            #define OBJ_CAM_RANGES           0x4720
            #define OBJ_DIRECT_LIGHT         0x4600                 // light object
            #define OBJ_DL_SPOTLIGHT         0x4610                 // spotlight info
            #define OBJ_DL_OFF               0x4620                 // on/off flag
            #define OBJ_DL_ATTENUATE         0x4625
            #define OBJ_DL_RAYSHAD           0x4627
            #define OBJ_DL_SHADOWED          0x4630
            #define OBJ_DL_LOCAL_SHADOW      0x4640
            #define OBJ_DL_LOCAL_SHADOW2     0x4641
            #define OBJ_DL_SEE_CONE          0x4650                 // cone flag
            #define OBJ_DL_SPOT_RECTANGULAR  0x4651
            #define OBJ_DL_SPOT_OVERSHOOT    0x4652
            #define OBJ_DL_SPOT_PROJECTOR    0x4653
            #define OBJ_DL_EXCLUDE           0x4654                 // exclude object
            #define OBJ_DL_RANGE             0x4655                 // light range
            #define OBJ_DL_SPOT_ROLL         0x4656                 // spotlight roll
            #define OBJ_DL_SPOT_ASPECT       0x4657                 // spotlight aspect
            #define OBJ_DL_RAY_BIAS          0x4658
            #define OBJ_DL_INNER_RANGE       0x4659
            #define OBJ_DL_OUTER_RANGE       0x465a
            #define OBJ_DL_MULTIPLIER        0x465b
            #define OBJ_N_AMBIENT_LIGHT      0x4680
    #define KEY3DS                      0xb000
        #define KEY_STAOBJ              0xb002                      // start object description
            #define KEY_NMEHIE              0xb010                  // name & Hierarchy
            #define KEY_NMEDMY              0xb011                  // name dummy object
            #define KEY_OBJPIV              0xb020                  // object pivot point?
        #define KEY_FRAMES              0xB008                      // chunk Frame

        #define OBJ_PROC_NAME            0x4181
        #define OBJ_PROC_DATA            0x4182
        #define OBJ_MSH_BOXMAP           0x4190

        #define OBJ_N_D_L_OLD            0x4400
        #define OBJ_N_CAM_OLD            0x4500

        #define OBJ_HIERARCHY            0x4f00
        #define OBJ_PARENT_OBJECT        0x4f10
        #define OBJ_PIVOT_OBJECT         0x4f30
        #define OBJ_PIVOT_LIMITS         0x4f40
        #define OBJ_XLATE_RANGE          0x4f50

        #define EDIT_DEFAULT_VIEW        0x3000
        #define EDIT_VIEW_TOP            0x3010
        #define EDIT_VIEW_BOTTOM         0x3020
        #define EDIT_VIEW_LEFT           0x3030
        #define EDIT_VIEW_RIGHT          0x3040
        #define EDIT_VIEW_FRONT          0x3050
        #define EDIT_VIEW_BACK           0x3060
        #define EDIT_VIEW_USER           0x3070
        #define EDIT_VIEW_CAMERA         0x3080
        #define EDIT_VIEW_WINDOW         0x3090

        #define EDIT_MLIB_MAGIC          0x3daa
        #define EDIT_PRJ_MAGIC           0x3dc2
        #define EDIT_MAT_MAGIC           0x3dff

/*
** common sub types
*/
// these define the different color chunk types
#define COL_RGB                 0x0010                              // RGB color in float
#define COL_TRU                 0x0011                              // true color in byte
#define COL_LIN_TRU 			0x0012
#define COL_LIN_RGB 			0x0013
// percentage
#define PERCENTAGE_INT          0x0030                              // percentage
#define PERCENTAGE_FLOAT        0x0031                              //

int chunkProcess(FILE *fp, unsigned int startPos, unsigned int size, int tab);

// dump function for analyze
int dump(FILE *fp, unsigned int startPos, unsigned int size){
    char buffer[256], tmp[3];
    char ch;
    int i;

    startPos-=sizeof(chunk);
    size+=sizeof(chunk);

    fseek(fp, startPos, SEEK_SET);

    for(i=0; i<size; i++){
        if(!i) memset(buffer, 32, 256);

        fread(&ch, 1, 1, fp);
        sprintf(tmp, "%02x", ch);

        buffer[(i%32)*3+0]=tmp[0];
        buffer[(i%32)*3+1]=tmp[1];
        buffer[(i%32)*3+2]=32;

        if(ch>31)   buffer[(i%32)+32*3+0]=ch;
        else        buffer[(i%32)+32*3+0]='.';
                    buffer[(i%32)+32*3+1]='\0';

        if((i+1)%32==0) printf("%s\n", buffer);
    }
    if(i%32) printf("%s\n", buffer);

    return(1);
}

// string input
int stringInput(FILE *fp, char *buffer){
    int count;
    char ch;

    count=0;
    while(fread(&ch, 1, 1, fp)){
        *(buffer+(count++))=ch;
        if(ch=='\0') break;
    }

    return(--count);
}

void chunkInfoPrint(int tab, char *description, int size, int currentPos){  // chunk information print
    int i;

    for(i=0; i<tab; i++) printf(" ");

    printf("[%s(size:%d, postion:%d)]\n", description, size+sizeof(chunk), currentPos-sizeof(chunk));
}

// unknown chunk
int CHUNK_UNKNOWN(FILE *fp, unsigned int currentPos, unsigned int size, char *description, int tab){
    int i;

    for(i=0; i<tab; i++) printf(" ");
    printf("[unknown chunk(size:%d, postion:%d)] - %s\n", size+sizeof(chunk), currentPos-sizeof(chunk), description);

    dump(fp, currentPos, size);

    return(1);
}

int CHUNK_MAIN3DS(FILE *fp, unsigned int currentPos, unsigned int size, char *description, int tab){
    chunkInfoPrint(tab, description, size, currentPos);

    chunkProcess(fp, currentPos, size, tab);

    return(1);
}

int CHUNK_EDIT3DS(FILE *fp, unsigned int currentPos, unsigned int size, char *description, int tab){
    chunkInfoPrint(tab, description, size, currentPos);

    chunkProcess(fp, currentPos, size, tab);

    return(1);
}

int CHUNK_EDIT_MESH_VERSION(FILE *fp, unsigned int currentPos, unsigned int size, char *description, int tab){
    unsigned int version;

    chunkInfoPrint(tab, description, size, currentPos);

    if(!fread(&version, 4, 1, fp)) return(0);

    printf("Version : %d\n", version);

    return(1);
}

int CHUNK_EDIT_MATERIAL(FILE *fp, unsigned int currentPos, unsigned int size, char *description, int tab){
    chunkInfoPrint(tab, description, size, currentPos);

    chunkProcess(fp, currentPos, size, tab);

    return(1);
}

int CHUNK_MAT_NAME(FILE *fp, unsigned int currentPos, unsigned int size, char *description, int tab){
    char ch;

    chunkInfoPrint(tab, description, size, currentPos);

	while(!feof(fp)){
		fread(&ch, 1, 1, fp);
		if(ch=='\0') break;
		printf("%c", ch);
	}
	printf("\n");

    return(1);
}

int CHUNK_MAT_AMBIENT(FILE *fp, unsigned int currentPos, unsigned int size, char *description, int tab){
    chunkInfoPrint(tab, description, size, currentPos);

    chunkProcess(fp, currentPos, size, tab);

    return(1);
}

int CHUNK_COL_RGB(FILE *fp, unsigned int currentPos, unsigned int size, char *description, int tab){
    float r, g, b;

    chunkInfoPrint(tab, description, size, currentPos);

    fread(&r, sizeof(float), 1, fp);    fread(&g, sizeof(float), 1, fp);    fread(&b, sizeof(float), 1, fp);
	printf("R - %f  G - %f  B - %f\n", r, g, b);

    return(1);
}

int CHUNK_COL_TRU(FILE *fp, unsigned int currentPos, unsigned int size, char *description, int tab){
    unsigned char r, g, b;

    chunkInfoPrint(tab, description, size, currentPos);

    fread(&r, sizeof(char), 1, fp);     fread(&g, sizeof(char), 1, fp);     fread(&b, sizeof(char), 1, fp);
    printf("R - %f  G - %f  B - %f\n", (float)r/255., (float)g/255, (float)b/255.);

    return(1);
}

int CHUNK_COL_LIN_TRU(FILE *fp, unsigned int currentPos, unsigned int size, char *description, int tab){
    unsigned char r, g, b;

    chunkInfoPrint(tab, description, size, currentPos);

    fread(&r, sizeof(char), 1, fp);     fread(&g, sizeof(char), 1, fp);     fread(&b, sizeof(char), 1, fp);
    printf("R - %f  G - %f  B - %f\n", (float)r/255., (float)g/255, (float)b/255.);

    return(1);
}

int CHUNK_COL_LIN_RGB(FILE *fp, unsigned int currentPos, unsigned int size, char *description, int tab){
    float r, g, b;

    chunkInfoPrint(tab, description, size, currentPos);

    fread(&r, sizeof(float), 1, fp);    fread(&g, sizeof(float), 1, fp);    fread(&b, sizeof(float), 1, fp);
	printf("R - %f  G - %f  B - %f\n", r, g, b);

    return(1);
}

int CHUNK_MAT_DIFFUSE(FILE *fp, unsigned int currentPos, unsigned int size, char *description, int tab){
    chunkInfoPrint(tab, description, size, currentPos);

    chunkProcess(fp, currentPos, size, tab);

    return(1);
}

int CHUNK_MAT_SPECULAR(FILE *fp, unsigned int currentPos, unsigned int size, char *description, int tab){
    chunkInfoPrint(tab, description, size, currentPos);

    chunkProcess(fp, currentPos, size, tab);

    return(1);
}
int CHUNK_MAT_SHINESS(FILE *fp, unsigned int currentPos, unsigned int size, char *description, int tab){
    chunkInfoPrint(tab, description, size, currentPos);

    chunkProcess(fp, currentPos, size, tab);

    return(1);
}

int CHUNK_PERCENTAGE_INT(FILE *fp, unsigned int currentPos, unsigned int size, char *description, int tab){
    short value;

    chunkInfoPrint(tab, description, size, currentPos);

    if(!fread(&value, 2, 1, fp)) return(0);
    printf("%f\n", (float)value/100.);

    return(1);
}

int CHUNK_PERCENTAGE_FLOAT(FILE *fp, unsigned int currentPos, unsigned int size, char *description, int tab){
    float value;

    chunkInfoPrint(tab, description, size, currentPos);

    if(!fread(&value, 4, 1, fp)) return(0);
    printf("%f\n", value);

    return(1);
}

int CHUNK_MAT_SHIN2PCT(FILE *fp, unsigned int currentPos, unsigned int size, char *description, int tab){
    chunkInfoPrint(tab, description, size, currentPos);

    chunkProcess(fp, currentPos, size, tab);

    return(1);
}

int CHUNK_MAT_TRANSPARENCY(FILE *fp, unsigned int currentPos, unsigned int size, char *description, int tab){
    chunkInfoPrint(tab, description, size, currentPos);

    chunkProcess(fp, currentPos, size, tab);

    return(1);
}

int CHUNK_MAT_XPFALL(FILE *fp, unsigned int currentPos, unsigned int size, char *description, int tab){
    chunkInfoPrint(tab, description, size, currentPos);

    chunkProcess(fp, currentPos, size, tab);

    return(1);
}

int CHUNK_MAT_REFBLUR(FILE *fp, unsigned int currentPos, unsigned int size, char *description, int tab){
    chunkInfoPrint(tab, description, size, currentPos);

    chunkProcess(fp, currentPos, size, tab);

    return(1);
}

int CHUNK_MAT_SELF_ILPCT(FILE *fp, unsigned int currentPos, unsigned int size, char *description, int tab){
    chunkInfoPrint(tab, description, size, currentPos);

    chunkProcess(fp, currentPos, size, tab);

    return(1);
}

int CHUNK_MAT_SHADING(FILE *fp, unsigned int currentPos, unsigned int size, char *description, int tab){
    short type;
    char *shadingTextList[]={
        "NONE", "FLAT", "GOURAUD", "PHONG", "METAL"
    };

    chunkInfoPrint(tab, description, size, currentPos);

    if(!fread(&type, 2, 1, fp)) return(0);

    printf("Shading type : %s\n", shadingTextList[type]);
    
    return(1);
}

int CHUNK_EDIT_AMBIENT_LIGHT(FILE *fp, unsigned int currentPos, unsigned int size, char *description, int tab){
    chunkInfoPrint(tab, description, size, currentPos);
    
    chunkProcess(fp, currentPos, size, tab);

    return(1);
}

int CHUNK_EDIT_OBJECT(FILE *fp, unsigned int currentPos, unsigned int size, char *description, int tab){
    char buffer[256];
    int skip;

    chunkInfoPrint(tab, description, size, currentPos);

    skip=stringInput(fp, buffer);
    printf("Object name : %s\n", buffer);

    chunkProcess(fp, currentPos+skip+1, size-(skip+1), tab);

    return(1);
}

int CHUNK_OBJ_CAMERA(FILE *fp, unsigned int currentPos, unsigned int size, char *description, int tab){
    float cx, cy, cz;
    float tx, ty, tz;
    float rotation, lens;

    chunkInfoPrint(tab, description, size, currentPos);

    fread(&cx, 4, 1, fp);
    fread(&cz, 4, 1, fp);
    fread(&cy, 4, 1, fp);
    printf("Camera eye - X : %f  Y : %f  Z : %f\n", cx, cy, cz);

    fread(&tx, 4, 1, fp);
    fread(&tz, 4, 1, fp);
    fread(&ty, 4, 1, fp);
    printf("Camera target - X : %f  Y : %f  Z : %f\n", tx, ty, tz);

    fread(&rotation, 4, 1, fp);
    fread(&lens, 4, 1, fp);
    printf("Camera rotation - %f\n", rotation);
    printf("Camera lens - %fmm\n", lens);

    chunkProcess(fp, currentPos+(4*8), size-(4*8), tab);

    return(1);
}

int CHUNK_OBJ_CAM_RANGES(FILE *fp, unsigned int currentPos, unsigned int size, char *description, int tab){
    float nearPlane, farPlane;

    chunkInfoPrint(tab, description, size, currentPos);

    fread(&nearPlane, 4, 1, fp);
    fread(&farPlane, 4,1, fp);

    printf("Near plane : %f\n", nearPlane);
    printf("Far plane : %f\n", farPlane);

    return(1);
}

int CHUNK_OBJ_DIRECT_LIGHT(FILE *fp, unsigned int currentPos, unsigned int size, char *description, int tab){
    float x, y, z;

    chunkInfoPrint(tab, description, size, currentPos);

    fread(&x, 4, 1, fp);
    fread(&y, 4, 1, fp);
    fread(&z, 4, 1, fp);
    printf("X : %f  Y : %f  Z : %f\n", x, y, z);

    chunkProcess(fp, currentPos+12, size-12, tab);

    return(1);
}

int CHUNK_OBJ_DL_SPOTLIGHT(FILE *fp, unsigned int currentPos, unsigned int size, char *description, int tab){
    float tx, ty, tz;
    float hotspot, falloff;

    chunkInfoPrint(tab, description, size, currentPos);

    fread(&tx, 4, 1, fp);
    fread(&tz, 4, 1, fp);
    fread(&ty, 4, 1, fp);
    fread(&hotspot, 4, 1, fp);
    fread(&falloff, 4, 1, fp);

    printf("The target of the spot is at - X : %f  Y : %f  Z : %f\n", tx, ty, tz);
    printf("The hotspot of this light is : %f\n", hotspot);
    printf("The falloff of this light is : %f\n", falloff);

    return(1);
}

int CHUNK_OBJ_DL_OFF(FILE *fp, unsigned int currentPos, unsigned int size, char *description, int tab){
    char value;

    chunkInfoPrint(tab, description, size, currentPos);

    fread(&value, 1, 1, fp);
    if(value)   printf("Light is on\n");
    else        printf("Light is off\n");

    return(1);
}

int CHUNK_OBJ_DL_INNER_RANGE(FILE *fp, unsigned int currentPos, unsigned int size, char *description, int tab){
    float value;

    chunkInfoPrint(tab, description, size, currentPos);

    fread(&value, 4, 1, fp);
    printf("Inner range : %f\n", value);

    return(1);
}

int CHUNK_OBJ_DL_OUTER_RANGE(FILE *fp, unsigned int currentPos, unsigned int size, char *description, int tab){
    float value;

    chunkInfoPrint(tab, description, size, currentPos);

    fread(&value, 4, 1, fp);
    printf("Outer range : %f\n", value);

    return(1);
}

int CHUNK_OBJ_DL_MULTIPLIER(FILE *fp, unsigned int currentPos, unsigned int size, char *description, int tab){
    float value;

    chunkInfoPrint(tab, description, size, currentPos);

    fread(&value, 4, 1, fp);
    printf("Multiplier : %f\n", value);

    return(1);
}

int CHUNK_OBJ_TRIMESH(FILE *fp, unsigned int currentPos, unsigned int size, char *description, int tab){
    chunkInfoPrint(tab, description, size, currentPos);

    chunkProcess(fp, currentPos, size, tab);

    return(1);
}

int CHUNK_TRI_VERTEXL(FILE *fp, unsigned int currentPos, unsigned int size, char *description, int tab){
    short numVertex;
    float x, y, z;
    int i;

    chunkInfoPrint(tab, description, size, currentPos);

    fread(&numVertex, 2, 1, fp);
    for(i=0; i<numVertex; i++){
        fread(&x, 4, 1, fp);
        fread(&z, 4, 1, fp);
        fread(&y, 4, 1, fp);
        printf("%5d - X : %10f  Y:%10f  Z:%10f\n", i, x, y, z);
    }

    return(1);
}

int CHUNK_TRI_FACEL1(FILE *fp, unsigned int currentPos, unsigned int size, char *description, int tab){
    short numFace;
    short a, b, c, v, ab, bc, ca, mapping;
    int chunkSize;
    int i;

    chunkInfoPrint(tab, description, size, currentPos);

    fread(&numFace, 2, 1, fp);
    for(i=0; i<numFace; i++){
        fread(&a, 2, 1, fp);
        fread(&b, 2, 1, fp);
        fread(&c, 2, 1, fp);
        fread(&v, 2, 1, fp);
        ab=(v&0x0004)>>2;
        bc=(v&0x0002)>>1;
        ca=(v&0x0001);
        mapping=(v&0x0008)>>3;

        printf("%5d - A : %-5d  B : %-5d  C : %-5d  VISIBLE : %04xH(AB:%1d BC:%1d CA:%1d MAPPING:%1d)\n", i, a, b, c, v, ab, bc, ca, mapping);
    }

    chunkSize=2+(numFace*(2+2+2+2));

    chunkProcess(fp, currentPos+chunkSize, size-chunkSize, tab);

    return(1);
}

int CHUNK_TRI_MSH_MAT_GROUP(FILE *fp, unsigned int currentPos, unsigned int size, char *description, int tab){
    char name[256];
    short numFace, count, data;
    int i;

    chunkInfoPrint(tab, description, size, currentPos);

    stringInput(fp, name);

    printf("Material name : %s\n", name);

    fread(&numFace, 2, 1, fp);

    for(i=0, count=0; i<numFace; i++){
        fread(&data, 2, 1, fp);
        printf("%5d, ", data);

        if((++count)%8==0) printf("\n");
    }

    if(count%8) printf("\n");

    return(1);
}

int CHUNK_TRI_TEXTURE_VERTS(FILE *fp, unsigned int currentPos, unsigned int size, char *description, int tab){
    float u, v;
    short count;
    int i;

    chunkInfoPrint(tab, description, size, currentPos);

    fread(&count, 2, 1, fp);
    for(i=0; i<count; i++){
        fread(&u, 4, 1, fp);
        fread(&v, 4, 1, fp);
        printf("%5d - U : %10f  V : %10f\n", i, u, v);
    }

    return(1);
}

int CHUNK_TRI_SMOOTH(FILE *fp, unsigned int currentPos, unsigned int size, char *description, int tab){
    int numSmooth, s;
    int i, j;

    chunkInfoPrint(tab, description, size, currentPos);

    numSmooth=size/4;
    for(i=0; i<numSmooth; i++){
        fread(&s, 4, 1, fp);
        printf("%5d - ", i);
        for(j=0; j<32; j++) if(s&(1<<j)) printf("%d, ", j+1);
        printf("\n");
    }

    return(1);
}

int CHUNK_TRI_LOCAL_AXIS(FILE *fp, unsigned int currentPos, unsigned int size, char *description, int tab){
    float matrix[4][4];
    float x, y, z;
    int i, j;

    chunkInfoPrint(tab, description, size, currentPos);

    for(i=0; i<3; i++){
        fread(&x, 4, 1, fp);    fread(&z, 4, 1, fp);    fread(&y, 4, 1, fp);

        if(i==1) j=2;
        else     if(i==2) j=1;
                 else     j=i;

        matrix[j][0]=x; matrix[j][1]=y; matrix[j][2]=z; matrix[j][3]=0.;
    }

    fread(&x, 4, 1, fp);    fread(&z, 4, 1, fp);    fread(&y, 4, 1, fp);

    matrix[3][0]=x; matrix[3][1]=y; matrix[3][2]=z; matrix[3][3]=0.;

    for(i=0; i<4; i++){
        for(j=0; j<4; j++) printf("%9.2f ", matrix[i][j]);
        printf("\n");
    }

    return(1);
}

int CHUNK_KEY3DS(FILE *fp, unsigned int currentPos, unsigned int size, char *description, int tab){
    chunkInfoPrint(tab, description, size, currentPos);

    chunkProcess(fp, currentPos, size, tab);

    return(1);
}

int CHUNK_KEY_FRAMES(FILE *fp, unsigned int currentPos, unsigned int size, char *description, int tab){
    int start, end;

    chunkInfoPrint(tab, description, size, currentPos);

    fread(&start, 4, 1, fp);
    fread(&end, 4, 1, fp);
    printf("Frames - start : %d  end : %d\n", start, end);

    return(1);
}

struct{
	unsigned short id;
	char *description;
    int (*funcPtr)(FILE *fp, unsigned int currentPos, unsigned int size, char *description, int tab);
}chunkList[]={
    {MAIN3DS,                   "MAIN3DS",                  CHUNK_MAIN3DS},
    {EDIT3DS,                   "EDIT3DS",                  CHUNK_EDIT3DS},
    {EDIT_MATERIAL,             "EDIT_MATERIAL",            CHUNK_EDIT_MATERIAL},
    {EDIT_CONFIG1,              "EDIT_CONFIG1",             CHUNK_UNKNOWN},
    {EDIT_CONFIG2,              "EDIT_CONFIG2",             CHUNK_UNKNOWN},
    {MAT_NAME,                  "MAT_NAME",                 CHUNK_MAT_NAME},
    {MAT_AMBIENT,               "MAT_AMBIENT",              CHUNK_MAT_AMBIENT},
    {COL_RGB,                   "COL_RGB",                  CHUNK_COL_RGB},
    {COL_TRU,                   "COL_TRU",                  CHUNK_COL_TRU},
    {COL_LIN_TRU,               "COL_LIN_TRU",              CHUNK_COL_LIN_TRU},
    {COL_LIN_RGB,               "COL_LIN_RGB",              CHUNK_COL_LIN_RGB},
    {MAT_DIFFUSE,               "MAT_DIFFUSE",              CHUNK_MAT_DIFFUSE},
    {MAT_SPECULAR,              "MAT_SPECULAR",             CHUNK_MAT_SPECULAR},
    {MAT_SHINESS,               "MAT_SHINESS",              CHUNK_MAT_SHINESS},
    {PERCENTAGE_INT,            "PERCENTAGE_INT",           CHUNK_PERCENTAGE_INT},
    {PERCENTAGE_FLOAT,          "PERCENTAGE_FLOAT",         CHUNK_PERCENTAGE_FLOAT},
    {MAT_SHIN2PCT,              "MAT_SHIN2PCT",             CHUNK_MAT_SHIN2PCT},
    {MAT_SHIN3PCT,              "MAT_SHIN3PCT",             CHUNK_UNKNOWN},
    {MAT_TRANSPARENCY,          "MAT_TRANSPARENCY",         CHUNK_MAT_TRANSPARENCY},
    {MAT_XPFALL,                "MAT_XPFALL",               CHUNK_MAT_XPFALL},
    {MAT_REFBLUR,               "MAT_REFBLUR",              CHUNK_MAT_REFBLUR},
    {MAT_SELF_ILLUM,            "MAT_SELF_ILLUM",           CHUNK_UNKNOWN},
    {MAT_TWO_SIDE,              "MAT_TWO_SIDE",             CHUNK_UNKNOWN},
    {MAT_DECAL,                 "MAT_DECAL",                CHUNK_UNKNOWN},
    {MAT_ADDITIVE,              "MAT_ADDITIVE",             CHUNK_UNKNOWN},
    {MAT_SELF_ILPCT,            "MAT_SELF_ILPCT",           CHUNK_MAT_SELF_ILPCT},
    {MAT_WIRE,                  "MAT_WIRE",                 CHUNK_UNKNOWN},
    {MAT_SUPERSMP,              "MAT_SUPERSMP",             CHUNK_UNKNOWN},
    {MAT_WIRESIZE,              "MAT_WIRESIZE",             CHUNK_UNKNOWN},
    {MAT_FACEMAP,               "MAT_FACEMAP",              CHUNK_UNKNOWN},
    {MAT_XPFALLIN,              "MAT_XPFALLIN",             CHUNK_UNKNOWN},
    {MAT_PHONGSOFT,             "MAT_PHONGSOFT",            CHUNK_UNKNOWN},
    {MAT_WIREABS,               "MAT_WIREABS",              CHUNK_UNKNOWN},
    {MAT_SHADING,               "MAT_SHADING",              CHUNK_MAT_SHADING},
    {MAT_TEXMAP,                "MAT_TEXMAP",               CHUNK_UNKNOWN},
    {MAT_SPECMAP,               "MAT_SPECMAP",              CHUNK_UNKNOWN},
    {MAT_OPACMAP,               "MAT_OPACMAP",              CHUNK_UNKNOWN},
    {MAT_REFLMAP,               "MAT_REFLMAP",              CHUNK_UNKNOWN},
    {MAT_BUMPMAP,               "MAT_BUMPMAP",              CHUNK_UNKNOWN},
    {MAT_USE_XPFALL,            "MAT_USE_XPFALL",           CHUNK_UNKNOWN},
    {MAT_USE_REFBLUR,           "MAT_USE_REFBLUR",          CHUNK_UNKNOWN},
    {MAT_BUMP_PERCENT,          "MAT_BUMP_PERCENT",         CHUNK_UNKNOWN},
    {MAT_MAPNAME,               "MAT_MAPNAME",              CHUNK_UNKNOWN},
    {MAT_ACUBIC,                "MAT_ACUBIC",               CHUNK_UNKNOWN},
    {MAT_SXP_TEXT_DATA,         "MAT_SXP_TEXT_DATA",        CHUNK_UNKNOWN},
    {MAT_SXP_TEXT2_DATA,        "MAT_SXP_TEXT2_DATA",       CHUNK_UNKNOWN},
    {EDIT_BITMAP,               "EDIT_BITMAP",              CHUNK_UNKNOWN},
    {EDIT_BACKGR,               "EDIT_BACKGR",              CHUNK_UNKNOWN},
    {EDIT_USE_SOLID_BGND,       "EDIT_USE_SOLID_BGND",      CHUNK_UNKNOWN},
    {EDIT_V_GRADIENT,           "EDIT_V_GRADIENT",          CHUNK_UNKNOWN},
    {EDIT_LO_SHADOW_BIAS,       "EDIT_LO_SHADOW_BIAS",      CHUNK_UNKNOWN},
    {EDIT_HI_SHADOW_BIAS,       "EDIT_HI_SHADOW_BIAS",      CHUNK_UNKNOWN},
    {EDIT_SHADOW_MAP_SIZE,      "EDIT_SHADOW_MAP_SIZ",      CHUNK_UNKNOWN},
    {EDIT_SHADOW_SAMPLE,        "EDIT_SHADOW_SAMPLE",       CHUNK_UNKNOWN},
    {EDIT_SHADOW_RANGE,         "EDIT_SHADOW_RANGE",        CHUNK_UNKNOWN},
    {EDIT_SHADOW_FILTER,        "EDIT_SHADOW_FILTER",       CHUNK_UNKNOWN},
    {EDIT_RAY_BIAS,             "EDIT_RAY_BIAS",            CHUNK_UNKNOWN},
    {EDIT_O_CONSTS,             "EDIT_O_CONSTS",            CHUNK_UNKNOWN},
    {EDIT_AMBIENT_LIGHT,        "EDIT_AMBIENT_LIGHT",       CHUNK_EDIT_AMBIENT_LIGHT},
    {EDIT_FOG,                  "EDIT_FOG",                 CHUNK_UNKNOWN},
    {EDIT_USE_FOG,              "EDIT_USE_FOG",             CHUNK_UNKNOWN},
    {EDIT_FOG_BGND,             "EDIT_FOG_BGND",            CHUNK_UNKNOWN},
    {EDIT_DISTANCE_CUE,         "EDIT_DISTANCE_CUE",        CHUNK_UNKNOWN},
    {EDIT_LAYER_FOG,            "EDIT_LAYER_FOG",           CHUNK_UNKNOWN},
    {EDIT_USE_LAYER_FOG,        "EDIT_USE_LAYER_FOG",       CHUNK_UNKNOWN},
    {EDIT_DCUE_BGND,            "EDIT_DCUE_BGND",           CHUNK_UNKNOWN},
    {EDIT_SMAGIC,               "EDIT_SMAGIC",              CHUNK_UNKNOWN},
    {EDIT_LMAGIC,               "EDIT_LMAGIC",              CHUNK_UNKNOWN},
    {EDIT_DEFAULT_VIEW,         "EDIT_DEFAULT_VIEW",        CHUNK_UNKNOWN},
    {EDIT_VIEW_TOP,             "EDIT_VIEW_TOP",            CHUNK_UNKNOWN},
    {EDIT_VIEW_BOTTOM,          "EDIT_VIEW_BOTTOM",         CHUNK_UNKNOWN},
    {EDIT_VIEW_LEFT,            "EDIT_VIEW_LEFT",           CHUNK_UNKNOWN},
    {EDIT_VIEW_RIGHT,           "EDIT_VIEW_RIGHT",          CHUNK_UNKNOWN},
    {EDIT_VIEW_FRONT,           "EDIT_VIEW_FRONT",          CHUNK_UNKNOWN},
    {EDIT_VIEW_BACK,            "EDIT_VIEW_BACK",           CHUNK_UNKNOWN},
    {EDIT_VIEW_USER,            "EDIT_VIEW_USER",           CHUNK_UNKNOWN},
    {EDIT_VIEW_CAMERA,          "EDIT_VIEW_CAMERA",         CHUNK_UNKNOWN},
    {EDIT_VIEW_WINDOW,          "EDIT_VIEW_WINDOW",         CHUNK_UNKNOWN},
    {EDIT_MESH_VERSION,         "EDIT_MESH_VERSION",        CHUNK_EDIT_MESH_VERSION},
    {EDIT_MLIB_MAGIC,           "EDIT_MLIB_MAGIC",          CHUNK_UNKNOWN},
    {EDIT_PRJ_MAGIC,            "EDIT_PRJ_MAGIC",           CHUNK_UNKNOWN},
    {EDIT_MAT_MAGIC,            "EDIT_MAT_MAGIC",           CHUNK_UNKNOWN},
    {EDIT_OBJECT,               "EDIT_OBJECT",              CHUNK_EDIT_OBJECT},
    {OBJ_HIDDEN,                "OBJ_HIDDEN",               CHUNK_UNKNOWN},
    {OBJ_VIS_LOFTER,            "OBJ_VIS_LOFTER",           CHUNK_UNKNOWN},
    {OBJ_DOESNT_CAST,           "OBJ_DOESNT_CAST",          CHUNK_UNKNOWN},
    {OBJ_MATTE,                 "OBJ_MATTE",                CHUNK_UNKNOWN},
    {OBJ_FAST,                  "OBJ_FAST",                 CHUNK_UNKNOWN},
    {OBJ_PROCEDURAL,            "OBJ_PROCEDURAL",           CHUNK_UNKNOWN},
    {OBJ_FROZEN,                "OBJ_FROZEN",               CHUNK_UNKNOWN},
    {OBJ_DONT_RCVSHADOW,        "OBJ_DONT_RCVSHADOW",       CHUNK_UNKNOWN},
    {OBJ_TRIMESH,               "OBJ_TRIMESH",              CHUNK_OBJ_TRIMESH},
    {TRI_VERTEXL,               "TRI_VERTEXL",              CHUNK_TRI_VERTEXL},
    {TRI_FACEL2,                "TRI_FACEL2",               CHUNK_UNKNOWN},
    {TRI_FACEL1,                "TRI_FACEL1",               CHUNK_TRI_FACEL1},
    {TRI_MSH_MAT_GROUP,         "TRI_MSH_MAT_GROUP",        CHUNK_TRI_MSH_MAT_GROUP},
    {TRI_OLD_MAT_GROUP,         "TRI_OLD_MAT_GROUP",        CHUNK_UNKNOWN},
    {TRI_TEXTURE_VERTS,         "TRI_TEXTURE_VERTS",        CHUNK_TRI_TEXTURE_VERTS},
    {TRI_SMOOTH,                "TRI_SMOOTH",               CHUNK_TRI_SMOOTH},
    {TRI_LOCAL_AXIS,            "TRI_LOCAL_AXIS",           CHUNK_TRI_LOCAL_AXIS},
    {TRI_MESH_COLOR,            "TRI_MESH_COLOR",           CHUNK_UNKNOWN},
    {TRI_MESH_TEXTURE_INFO,     "TRI_MESH_TEXTURE_INFO",    CHUNK_UNKNOWN},
    {OBJ_PROC_NAME,             "OBJ_PROC_NAME",            CHUNK_UNKNOWN},
    {OBJ_PROC_DATA,             "OBJ_PROC_DATA",            CHUNK_UNKNOWN},
    {OBJ_MSH_BOXMAP,            "OBJ_MSH_BOXMAP",           CHUNK_UNKNOWN},
    {OBJ_N_D_L_OLD,             "OBJ_N_D_L_OLD",            CHUNK_UNKNOWN},
    {OBJ_N_CAM_OLD,             "OBJ_N_CAM_OLD",            CHUNK_UNKNOWN},
    {OBJ_DIRECT_LIGHT,          "OBJ_DIRECT_LIGHT",         CHUNK_OBJ_DIRECT_LIGHT},
    {OBJ_DL_SPOTLIGHT,          "OBJ_DL_SPOTLIGHT",         CHUNK_OBJ_DL_SPOTLIGHT},
    {OBJ_DL_OFF,                "OBJ_DL_OFF",               CHUNK_OBJ_DL_OFF},
    {OBJ_DL_ATTENUATE,          "OBJ_DL_ATTENUATE",         CHUNK_UNKNOWN},
    {OBJ_DL_RAYSHAD,            "OBJ_DL_RAYSHAD",           CHUNK_UNKNOWN},
    {OBJ_DL_SHADOWED,           "OBJ_DL_SHADOWED",          CHUNK_UNKNOWN},
    {OBJ_DL_LOCAL_SHADOW,       "OBJ_DL_LOCAL_SHADOW",      CHUNK_UNKNOWN},
    {OBJ_DL_LOCAL_SHADOW2,      "OBJ_DL_LOCAL_SHADOW2",     CHUNK_UNKNOWN},
    {OBJ_DL_SEE_CONE,           "OBJ_DL_SEE_CONE",          CHUNK_UNKNOWN},
    {OBJ_DL_SPOT_RECTANGULAR,   "OBJ_DL_SPOT_RECTANGULAR",  CHUNK_UNKNOWN},
    {OBJ_DL_SPOT_OVERSHOOT,     "OBJ_DL_SPOT_OVERSHOOT",    CHUNK_UNKNOWN},
    {OBJ_DL_SPOT_PROJECTOR,     "OBJ_DL_SPOT_PROJECTOR",    CHUNK_UNKNOWN},
    {OBJ_DL_EXCLUDE,            "OBJ_DL_EXCLUDE",           CHUNK_UNKNOWN},
    {OBJ_DL_RANGE,              "OBJ_DL_RANGE",             CHUNK_UNKNOWN},
    {OBJ_DL_SPOT_ROLL,          "OBJ_DL_SPOT_ROLL",         CHUNK_UNKNOWN},
    {OBJ_DL_SPOT_ASPECT,        "OBJ_DL_SPOT_ASPECT",       CHUNK_UNKNOWN},
    {OBJ_DL_RAY_BIAS,           "OBJ_DL_RAY_BIAS",          CHUNK_UNKNOWN},
    {OBJ_DL_INNER_RANGE,        "OBJ_DL_INNER_RANGE",       CHUNK_OBJ_DL_INNER_RANGE},
    {OBJ_DL_OUTER_RANGE,        "OBJ_DL_OUTER_RANGE",       CHUNK_OBJ_DL_OUTER_RANGE},
    {OBJ_DL_MULTIPLIER,         "OBJ_DL_MULTIPLIER",        CHUNK_OBJ_DL_MULTIPLIER},
    {OBJ_N_AMBIENT_LIGHT,       "OBJ_N_AMBIENT_LIGHT",      CHUNK_UNKNOWN},
    {OBJ_CAMERA,                "OBJ_CAMERA",               CHUNK_OBJ_CAMERA},
    {OBJ_CAM_SEE_CONE,          "OBJ_CAM_SEE_CONE",         CHUNK_UNKNOWN},
    {OBJ_CAM_RANGES,            "OBJ_CAM_RANGES",           CHUNK_OBJ_CAM_RANGES},
    {OBJ_HIERARCHY,             "OBJ_HIERARCHY",            CHUNK_UNKNOWN},
    {OBJ_PARENT_OBJECT,         "OBJ_PARENT_OBJECT",        CHUNK_UNKNOWN},
    {OBJ_PIVOT_OBJECT,          "OBJ_PIVOT_OBJECT",         CHUNK_UNKNOWN},
    {OBJ_PIVOT_LIMITS,          "OBJ_PIVOT_LIMITS",         CHUNK_UNKNOWN},
    {OBJ_XLATE_RANGE,           "OBJ_XLATE_RANGE",          CHUNK_UNKNOWN},
    {VIEWPORT_LAYOUT_OLD,       "VIEWPORT_LAYOUT_OLD",      CHUNK_UNKNOWN},
    {VIEWPORT_LAYOUT,           "VIEWPORT_LAYOUT",          CHUNK_UNKNOWN},
    {VIEWPORT_DATA_OLD,         "VIEWPORT_DATA_OLD",        CHUNK_UNKNOWN},
    {VIEWPORT_DATA,             "VIEWPORT_DATA",            CHUNK_UNKNOWN},
    {VIEWPORT_DATA_3,           "VIEWPORT_DATA_3",          CHUNK_UNKNOWN},
    {VIEWPORT_SIZE,             "VIEWPORT_SIZE",            CHUNK_UNKNOWN},
    {VIEWPORT_NETWORK_VIEW,     "VIEWPORT_NETWORK_VIEW",    CHUNK_UNKNOWN},
    {KEY3DS,                    "KEY3DS",                   CHUNK_KEY3DS},
    {KEY_STAOBJ,                "KEY_STAOBJ",               CHUNK_UNKNOWN},
    {KEY_NMEHIE,                "KEY_NMEHIE",               CHUNK_UNKNOWN},
    {KEY_NMEDMY,                "KEY_NMEDMY",               CHUNK_UNKNOWN},
    {KEY_OBJPIV,                "KEY_OBJPIV",               CHUNK_UNKNOWN},
    {KEY_FRAMES,                "KEY_FRAMES",               CHUNK_KEY_FRAMES}
};

int chunkProcess(FILE *fp, unsigned int startPos, unsigned int size, int tab){
    chunk cnk;
    unsigned int currentPos;
    int i;

    currentPos=startPos;

    while(currentPos<startPos+size){
        if(fseek(fp, currentPos, SEEK_SET)) return(0);
        if(!fread(&cnk, sizeof(chunk), 1, fp)) return(0);

        for(i=0; i<sizeof(chunkList)/sizeof(chunkList[0]); i++)
            if(cnk.id==chunkList[i].id){
                if(!(chunkList[i].funcPtr(fp, currentPos+sizeof(chunk), cnk.size-sizeof(chunk), chunkList[i].description, tab+4))) return(0);
                break;
            }

        if(i==sizeof(chunkList)/sizeof(chunkList[0])) CHUNK_UNKNOWN(fp, currentPos+sizeof(chunk), cnk.size-sizeof(chunk), "unknown identifier", tab+4);

        currentPos+=cnk.size;
    }

    return(1);
}

void main(int argc, char *argv[]){
    FILE *fp;
    unsigned int size;

    if(argc!=2) return;

    if(!(fp=fopen(argv[1], "rb"))) return;
    fseek(fp, 0, SEEK_END);
    size=ftell(fp);
    fseek(fp, 0, SEEK_SET);

    if(chunkProcess(fp, 0, size, 0)) printf("Success...\n");
    else printf("Fail...\n");

    fclose(fp);
}
