{$DEFINE _OVERLAY}

unit DVHan20;
{$IFDEF __OVERLAY}
   {$O+,F+}
{$ENDIF}

(* ver 20.003 *)
INTERFACE

uses
   dos, graph;

type
   han_mode_T = (PIXEL_UNIT_MODE,QUICK_MODE,GRADATION_MODE,EXPAND_MODE);
   han_type_T = (SAMMUL,GOTHIC,MYUNGJO);
   han_gradation_T = (UP_TO_DOWN,DOWN_TO_UP,LEFT_TO_RIGHT,RIGHT_TO_LEFT,DOWN_SLASH,ROUND_SPREAD);

 procedure setRGB(Color, Red, Green, Blue : byte);
 procedure waitVerticalRetrace(count : integer);
 procedure setHanColor(color : byte);
 function  getHanColor : byte;
 procedure setHanMode(mode : han_mode_T);
 procedure setHanType(han_type : han_type_T);
 function  getHanType : han_type_T;
 procedure setGradationType(gradation : han_gradation_T);
 procedure setGradationColor(color : byte);
 procedure setHanBoundary(x1,y1,x2,y2 : integer);
 procedure setHanActivePage(page : byte);
 procedure setHanExpandRate(rate : real);
 procedure printHangul(x,y : integer; s : string);
 procedure printHangulOUTLINE(x,y : integer; s : string; back_color : byte);
 procedure printHangulOUTLINE_CLIPPING(x,y : integer; s : string; back_color : byte);
 procedure printHangulGRADATION(x,y : integer; s : string; delay : word);
 procedure fadeOutGradationColor(delay : word);
 procedure printNetImage(x1,y1,x2,y2 : integer; back_color : byte);
 procedure setMaxInputByte(max_byte : byte);

 function  inputHangul(x,y : integer; s : string; back_color : word; is_hangul : boolean) : string;
function inputHangulEXTENDED(x,y : integer; s : string; back_color : word; is_hangul : boolean; var result : integer) : string;


IMPLEMENTATION

type
   han_font_data_T = array[0..223,0..31] of byte;
   ascii_font_data_T = array[0..95,0..15] of byte;
   screen_T = array[0..60000] of byte;

const
   ASCII_FONT_SIZE = 96*16;
   HAN_FONT_SIZE = 224*32;
   _HAN_COLOR_S : byte = 15;
   _HAN_MODE_S : han_mode_T = PIXEL_UNIT_MODE;
   _HAN_GRADATION_S : han_gradation_T = ROUND_SPREAD;
   _HAN_GRADATION_COLOR : array[1..3] of byte = (63,42,21);
   SHIFT_CODE = 1000;
   SCAN_CODE = 256;


   han_table : array[0..31] of byte = (
	 $00,  $00,  $00,  $00,  $00,  $00,  $00,  $00,  $00,  $00,
	 $00,  $00,  $00,  $11,  $02,  $02,  $02,  $02,  $02,  $11,
	 $11,  $02,  $02,  $02,  $02,  $02,  $11,  $11,  $02,  $00,
	 $02,  $02 );

   key_table1 : array[0..25] of byte = (
                8,126,16,13,5,7,20,113,105,107,
                103,129,127,120,104,110,9,2,4,11,
                111,19,14,18,119,17);
   key_table2 : array[0..25] of byte = (
                8,126,16,13,6,7,20,113,105,107,
                103,129,127,120,106,112,10,3,4,12,
                111,19,15,18,119,17);
   third_table1 : array[0..20] of byte = (
                0,1,2,3,5,8,1,9,17,19,1,21,22,23,24,1,25,26,27,28,29);
   first_table : array[0..29] of byte = (
                0,1,2,3,100,4,101,102,5,7,
                103,104,105,106,107,108,109,8,1,9,
                110,11,12,13,14,16,17,18,19,20);
   third_table2 : array[0..10,0..2] of byte = (
                (2,11,4),(5,14,6),(5,20,7),(9,2,10),(9,8,11),(9,9,12),
                (9,11,13),(9,18,14),(9,19,15),(9,20,16),(19,11,20));
   second_table : array[0..6,0..2] of byte = (
                (13,3,14),(13,4,15),(13,29,18),(20,7,21),
                (20,10,22),(20,29,23),(27,29,28));
   init_hangul_code : array[1..3] of byte = (1,2,1);

   __SCAN_CODE : char = #0;

var
{   _SCREEN_ADDRESS_0 : screen_T absolute SegA000;
   _SCREEN_ADDRESS_1 : screen_T absolute SegA000:$8000;}
   scr : ^screen_T;

   han_buf : array[0..31] of byte;
   ascii_font_data : ^ascii_font_data_T;
   han_font_data : ^han_font_data_T;
   han_font_data_type : array[SAMMUL..MYUNGJO] of ^han_font_data_T;
   gradation_han_table : array[0..15,0..15] of byte;
   gradation_ascii_table : array[0..15,0..7] of byte;

   EXPAND_RATE : byte;
   MAX_INPUT_BYTE : byte;
   boundary_x1, boundary_x2, boundary_y1, boundary_y2 : integer;
   remember_x_cursor, remember_y_cursor : integer;

   is_hangul_mode,      (* if this flag is TRUE, you can use HANGUL *)
   is_inserting_mode    (* if this flag is TRUE, program sets INSERTING MODE *)
                      : boolean;
   input_x_cursor,      (* X and Y axis where is inputting strings *)
   input_y_cursor
                      : integer;
   hangul_code          (* the consonants and vowels of inputed hangul codes *)
                      : array[1..4] of byte;
   input_code,          (* hangul code that is inputed rately *)
   cursor_ptr           (* cursor pointer in strings *)
                      : byte;

procedure setRGB(Color, Red, Green, Blue : byte); assembler;
asm
   mov dx,$3c8;
   mov al,Color;
   out dx,al;
   inc dx;
   mov al,Red;
   out dx,al;
   mov al,Green;
   out dx,al;
   mov al,Blue;
   out dx,al;
end;

function KeyPressed : boolean; assembler;
asm
      cmp  __SCAN_CODE, 0
      jne  @JUMP00
      mov  ah, 1
      int  $16
      mov  al, FALSE
      je   @JUMP01
@JUMP00:
      mov  al, TRUE
@JUMP01:
end;

function readKey : char; assembler;
asm
      mov  al, __SCAN_CODE
      mov  __SCAN_CODE, 0
      or   al, al
      jne  @JUMP00
      xor  ah, ah
      int  $16
      or   al, al
      jne  @JUMP00
      mov  __SCAN_CODE, ah
      or   ah, ah
      jne  @JUMP00
      mov  al, 'C'-64
@JUMP00:
end;

procedure clearKeyBuffer; assembler;
asm
@START:
      mov  ah, 1
      int  $16
      je   @EXIT
      xor  ah, ah
      int  $16
      jmp  @START
@EXIT:
end;

procedure waitVerticalRetrace(count : integer);
begin
   while count > 0 do begin
      while (port[$03DA] and 8) > 0 do clearKeyBuffer;
      while (port[$03DA] and 8) = 0 do;
      dec(count);
   end;
end;

procedure setHanColor(color : byte);
begin
   _HAN_COLOR_S := color;
end;

function  getHanColor : byte;
begin
   getHanColor := _HAN_COLOR_S;
end;

procedure setHanMode(mode : han_mode_T);
begin
   _HAN_MODE_S := mode;
end;

procedure setHanType(han_type : han_type_T);
begin
   han_font_data := addr(han_font_data_type[han_type]^);
end;

procedure setGradationType(gradation : han_gradation_T);
const
   gradation_table1_1 : array[0..15,0..15] of byte = (
                (1,1,1,1,1,2,2,2,3,3,3,4,4,4,5,5),
                (1,1,1,1,2,2,2,3,3,3,4,4,4,5,5,5),
                (1,1,1,2,2,2,3,3,3,4,4,4,5,5,5,6),
                (1,1,2,2,2,3,3,3,4,4,4,5,5,5,6,6),
                (1,2,2,2,3,3,3,4,4,4,5,5,5,6,6,6),
                (2,2,2,3,3,3,4,4,4,5,5,5,6,6,6,7),
                (2,2,3,3,3,4,4,4,5,5,5,6,6,6,7,7),
                (2,3,3,3,4,4,4,5,5,5,6,6,6,7,7,7),
                (3,3,3,4,4,4,5,5,5,6,6,6,7,7,7,8),
                (3,3,4,4,4,5,5,5,6,6,6,7,7,7,8,8),
                (3,4,4,4,5,5,5,6,6,6,7,7,7,8,8,8),
                (4,4,4,5,5,5,6,6,6,7,7,7,8,8,8,9),
                (4,4,5,5,5,6,6,6,7,7,7,8,8,8,9,9),
                (4,5,5,5,6,6,6,7,7,7,8,8,8,9,9,9),
                (5,5,5,6,6,6,7,7,7,8,8,8,9,9,9,9),
                (5,5,6,6,6,7,7,7,8,8,8,9,9,9,9,9));
   gradation_table1_2 : array[0..15,0..7] of byte = (
                (1,1,1,1,1,2,2,3),
                (1,1,1,1,2,2,3,3),
                (1,1,1,2,2,3,3,4),
                (1,1,2,2,3,3,4,4),
                (1,2,2,3,3,4,4,5),
                (2,2,3,3,4,4,5,6),
                (2,3,3,4,4,5,6,6),
                (3,3,4,4,5,6,6,7),
                (3,4,4,5,6,6,7,7),
                (4,4,5,6,6,7,7,8),
                (4,5,6,6,7,7,8,8),
                (5,6,6,7,7,8,8,9),
                (6,6,7,7,8,8,9,9),
                (6,7,7,8,8,9,9,9),
                (7,7,8,8,9,9,9,9),
                (7,8,8,9,9,9,9,9));
   gradation_table2_1 : array[0..15,0..15] of byte = (
                (9,9,9,9,9,8,8,8,8,8,8,9,9,9,9,9),
                (9,9,9,9,8,8,7,7,7,7,8,8,9,9,9,9),
                (9,9,9,8,8,7,7,6,6,7,7,8,8,9,9,9),
                (9,9,8,7,7,6,6,5,5,6,6,7,7,8,9,9),
                (9,8,8,7,6,5,5,4,4,5,5,6,7,8,8,9),
                (8,8,7,6,5,4,4,3,3,4,4,5,6,7,8,8),
                (8,7,7,6,5,4,3,2,2,3,4,5,6,7,7,8),
                (8,7,6,5,4,3,2,1,1,2,3,4,5,6,7,8),
                (8,7,6,5,4,3,2,1,1,2,3,4,5,6,7,8),
                (8,7,7,6,5,4,3,2,2,3,4,5,6,7,7,8),
                (8,8,7,6,5,4,4,3,3,4,4,5,6,7,8,8),
                (9,8,8,7,6,5,5,4,4,5,5,6,7,8,8,9),
                (9,9,8,7,7,6,6,5,5,6,6,7,7,8,9,9),
                (9,9,9,8,8,7,7,6,6,7,7,8,8,9,9,9),
                (9,9,9,9,8,8,7,7,7,7,8,8,9,9,9,9),
                (9,9,9,9,9,8,8,8,8,8,8,9,9,9,9,9));
   gradation_table2_2 : array[0..15,0..7] of byte = (
                (9,9,9,9,9,9,9,9),
                (9,9,8,8,8,8,9,9),
                (9,8,7,6,6,7,8,9),
                (9,7,6,5,5,6,7,9),
                (8,7,5,4,4,5,7,8),
                (8,6,5,4,4,5,6,8),
                (8,6,4,3,3,4,6,8),
                (8,6,4,1,2,4,6,8),
                (8,6,4,1,2,4,6,8),
                (8,6,4,3,3,4,6,8),
                (8,6,5,4,4,5,6,8),
                (8,7,5,4,4,5,7,8),
                (9,7,6,5,5,6,7,9),
                (9,8,7,6,6,7,8,9),
                (9,9,8,8,8,8,9,9),
                (9,9,9,9,9,9,9,9));

const
   han_consult_table : array[0..15] of byte =
      (1,1,2,2,3,3,4,5,6,7,7,8,8,9,9,9);
   ascii_consult_table : array[0..15] of byte =
      (1,1,2,2,3,3,4,5,6,7,7,8,8,9,9,9);
var
   i,j : integer;
begin
   _HAN_GRADATION_S := gradation;
   case gradation of
      UP_TO_DOWN :
      begin
         for i := 0 to 15 do begin
            for j := 0 to 15 do begin
               gradation_han_table[i,j] := han_consult_table[i];
            end;
         end;
         for i := 0 to 15 do begin
            for j := 0 to 7 do begin
               gradation_ascii_table[i,j] := ascii_consult_table[i];
            end;
         end;
      end;
      DOWN_TO_UP :
      begin
         for i := 0 to 15 do begin
            for j := 0 to 15 do begin
               gradation_han_table[15-i,j] := han_consult_table[i];
            end;
         end;
         for i := 0 to 15 do begin
            for j := 0 to 7 do begin
               gradation_ascii_table[15-i,j] := ascii_consult_table[i];
            end;
         end;
      end;
      LEFT_TO_RIGHT :
      begin
         for i := 0 to 15 do begin
            for j := 0 to 15 do begin
               gradation_han_table[j,i] := han_consult_table[i];
            end;
         end;
         for i := 0 to 15 do begin
            for j := 0 to 7 do begin
               gradation_ascii_table[i,j] := ascii_consult_table[j * 2];
            end;
         end;
      end;
      RIGHT_TO_LEFT :
      begin
         for i := 0 to 15 do begin
            for j := 0 to 15 do begin
               gradation_han_table[j,15-i] := han_consult_table[i];
            end;
         end;
         for i := 0 to 15 do begin
            for j := 0 to 7 do begin
               gradation_ascii_table[i,7-j] := ascii_consult_table[j * 2];
            end;
         end;
      end;
      DOWN_SLASH :
      begin
         move(gradation_table1_1,gradation_han_table,sizeof(gradation_table1_1));
         move(gradation_table1_2,gradation_ascii_table,sizeof(gradation_table1_2));
      end;
      ROUND_SPREAD :
      begin
         move(gradation_table2_1,gradation_han_table,sizeof(gradation_table2_1));
         move(gradation_table2_2,gradation_ascii_table,sizeof(gradation_table2_2));
      end;
   end;

end;

procedure setGradationColor(color : byte);
const
   MAX_GRADATION_COLOR = 6;
   table : array[0..MAX_GRADATION_COLOR-1,1..3] of byte = (
      (63,42,21), (63,21,42), (42,63,21), (42,21,63), (21,42,63), (63,63,21));
var
   i : integer;
begin
   if color in [0..MAX_GRADATION_COLOR-1] then begin
      for i := 1 to 3 do
         _HAN_GRADATION_COLOR[i] := table[color][i];
      setPalette(10,10);
      setRGB(10,_HAN_GRADATION_COLOR[1],_HAN_GRADATION_COLOR[2],_HAN_GRADATION_COLOR[3]);
      setHanColor(10);
   end;
end;

function getHanType : han_type_T;
var
   HT : han_type_T;
begin
   for HT := SAMMUL to MYUNGJO do begin
      if han_font_data = addr(han_font_data_type[HT]^) then
         getHanType := HT;
   end;
end;

procedure setHanBoundary(x1,y1,x2,y2 : integer);
begin
   boundary_x1 := x1;
   boundary_x2 := x2;
   boundary_y1 := y1;
   boundary_y2 := y2;
end;

procedure setHanActivePage(page : byte);
begin
   if page = 0 then scr := ptr(SegA000,$0000);
   if page = 1 then scr := ptr(SegA000,$8000);
end;

procedure setHanExpandRate(rate : real);
begin
   if round(rate*10) > 0 then
      EXPAND_RATE := round(rate*10)
   else
      EXPAND_RATE := 10;
end;

procedure occurHanError(s : string);
begin
   asm
      mov ax, 3
      int $10
   end;
   writeln('ERROR : '+s+' not found.');
   halt;
end;

procedure putHan(x,y : integer);
var
   i,j,k : integer;
begin
   for j := 0 to 15 do begin
      for i := 0 to 7 do begin
         for k := 0 to 1 do begin
            if (((han_buf[j*2+k] shr i) and 1) > 0) then begin
               putpixel(x+k*8+(7-i), y, _HAN_COLOR_S);
            end;
         end;
      end;
      inc(y);
   end;
end;

procedure putAscii(x,y : integer);
var
   i,j : integer;
begin
   for j := 0 to 15 do begin
      for i := 0 to 7 do begin
         if (((han_buf[j] shr i) and 1) > 0) then begin
            putpixel(x+(7-i), y, _HAN_COLOR_S);
         end;
      end;
      inc(y);
   end;
end;

procedure QUICK_putHan(x,y : integer);
var
   i,j,k : integer;
   screen_address, dot_count : word;
begin
   port[$3C4] := 2;
   port[$3CE] := 4;
   for k := 0 to 3 do begin
      if (_HAN_COLOR_S and (1 shl k)) > 0 then begin
         dot_count := 0;
         screen_address := 80*y+(x shr 3);
         port[$3C5] := (1 shl k);
         port[$3CF] := k;
         for j := 0 to 15 do begin
            scr^[screen_address] := scr^[screen_address] or han_buf[dot_count];
            scr^[screen_address+1] := scr^[screen_address+1] or han_buf[dot_count+1];
            inc(dot_count,2);
            inc(screen_address,80);
         end;
      end else begin
         dot_count := 0;
         screen_address := 80*y+(x shr 3);
         port[$3C5] := (1 shl k);
         port[$3CF] := k;
         for j := 0 to 15 do begin
            scr^[screen_address] := scr^[screen_address] and (han_buf[dot_count] xor $FF);
            scr^[screen_address+1] := scr^[screen_address+1] and (not(han_buf[dot_count+1]));
            inc(dot_count,2);
            inc(screen_address,80);
         end;
      end;
   end;
   bar(0,0,0,0);
end;

procedure QUICK_putAscii(x,y : integer);
var
   i,j,k : integer;
   screen_address, dot_count : word;
begin
   port[$3C4] := 2;
   port[$3CE] := 4;
   for k := 0 to 3 do begin
      dot_count := 0;
      screen_address := 80*y+(x shr 3);
      port[$3C5] := (1 shl k);
      port[$3CF] := k;
      for j := 0 to 15 do begin
         if (_HAN_COLOR_S and (1 shl k)) > 0 then
            scr^[screen_address] := scr^[screen_address] or han_buf[dot_count]
         else
            scr^[screen_address] := scr^[screen_address] and (han_buf[dot_count] xor $FF);
         inc(dot_count);
         inc(screen_address,80);
      end;
   end;
   bar(0,0,0,0);
end;

procedure GRADATION_putHan(x,y : integer);
var
   i,j,k : integer;
begin
   for j := 0 to 15 do begin
      for i := 0 to 7 do begin
         for k := 0 to 1 do begin
            if (((han_buf[j*2+k] shr i) and 1) > 0) then begin
               putpixel(x+k*8+(7-i),y,gradation_han_table[j,k*8+7-i]);
            end;
         end;
      end;
      inc(y);
   end;
end;

procedure GRADATION_putAscii(x,y : integer);
var
   i,j : integer;
begin
   for j := 0 to 15 do begin
      for i := 0 to 7 do begin
         if (((han_buf[j] shr i) and 1) > 0) then begin
            putpixel(x+(7-i), y,gradation_ascii_table[j,7-i]);
         end;
      end;
      inc(y);
   end;
end;

procedure EXPAND_putHan(x,y : integer);
var
   i,j,k : integer;
   x_wide, y_wide, x_read, y_read, integer_rate : byte;
begin
   x := round(x*EXPAND_RATE/10);
   integer_rate := round(EXPAND_RATE/10);
   x_wide := round(8*EXPAND_RATE/10);
   y_wide := round(16*EXPAND_RATE/10);
   for j := 0 to y_wide-integer_rate do begin
      y_read := round(j*10/EXPAND_RATE);
      for i := 0 to x_wide do begin
         x_read := trunc(i*10/EXPAND_RATE);
         for k := 0 to 1 do begin
            if (((han_buf[y_read*2+k] shr x_read) and 1) > 0) then begin
               putpixel(x+(k+1)*x_wide-i, y+j, _HAN_COLOR_S);
            end;
         end;
      end;
   end;
end;

procedure EXPAND_putAscii(x,y : integer);
var
   i,j : integer;
   x_wide, y_wide, x_read, y_read, integer_rate : byte;
begin
   x := round(x*EXPAND_RATE/10);
   integer_rate := round(EXPAND_RATE/10);
   x_wide := round(8*EXPAND_RATE/10);
   y_wide := round(16*EXPAND_RATE/10);
   for j := 0 to y_wide-integer_rate do begin
      y_read := round(j*10/EXPAND_RATE);
      for i := 0 to x_wide-integer_rate do begin
         x_read := trunc(i*10/EXPAND_RATE);
         if (((han_buf[y_read] shr x_read) and 1) > 0) then begin
            putpixel(x+(x_wide-i), y+j, _HAN_COLOR_S);
         end;
      end;
   end;
end;

procedure printHangulSub(x,y : integer; var s : string; bold : boolean; must_clip : boolean);
var
   c : char;
   first, second, third, data1, data2 : byte;
   font_num : array[0..2] of byte;
   order,k : integer;
begin
   order := 1;
   while order <= length(s) do begin
      if must_clip then begin
         if y >= boundary_y2-16 then exit;
         if x >= boundary_x2-16 then exit;
      end else begin
         if y >= 479-16 then exit;
         if x >= 639-16 then exit;
      end;
      data1 := ord(s[order]);
      if ((data1 and $80) > 0) then begin

         inc(order,1);
         data2 := ord(s[order]);
         first := (data1 and $7C) shr 2;
         second := ((data1 and $03) shl 3) + (data2 shr 5);
         third := (data2 and $1F);
         fillchar(font_num,sizeof(font_num),0);
         if (first > 1) then begin
            font_num[0] := han_table[second] and $0F;
            if (third > 1) then inc(font_num[0],3);
            inc(font_num[0],(6*(first-2)));
         end;
         if (second > 2) then begin
            if (third > 1) then font_num[1] := 1
                           else font_num[1] := 0;
            inc(font_num[1],(2*(second-3)+114));
         end;
         if (third > 1) then begin
            font_num[2] := han_table[second] shr 4;
            font_num[2] := (2*(third-2)+168) + font_num[2];
         end;

         move(han_font_data^[font_num[0]],han_buf,32);
         if (third > 1) then begin
            for k := 0 to 31 do begin
               han_buf[k] := han_buf[k] or
                             (han_font_data^[font_num[1],k]) or
                             (han_font_data^[font_num[2],k]);
            end;
         end
         else if (second > 2) then begin
            for k := 0 to 31 do begin
               han_buf[k] := han_buf[k] or
                             (han_font_data^[font_num[1],k]);
            end;
         end;

         if bold then begin
            for k := 0 to 29 do
               han_buf[k] := han_buf[k] or han_buf[k+2];
            for k := 31 downto 2 do
               han_buf[k] := han_buf[k] or han_buf[k-2];
            for k := 0 to 31 do
               han_buf[k] := han_buf[k] or (han_buf[k] shr 1);
            for k := 31 downto 0 do
               han_buf[k] := han_buf[k] or (han_buf[k] shl 1);
         end;

         if must_clip then begin
            if (x >= boundary_x1) and (y >= boundary_y1) then
            case _HAN_MODE_S of
               PIXEL_UNIT_MODE : putHan(x, y);
               QUICK_MODE : QUICK_putHan(x, y);
               GRADATION_MODE : GRADATION_putHan(x,y);
               EXPAND_MODE : EXPAND_putHan(x,y);
            end;
         end else if (x >= 0) and (y >= 0) then begin
            case _HAN_MODE_S of
               PIXEL_UNIT_MODE : putHan(x, y);
               QUICK_MODE : QUICK_putHan(x, y);
               GRADATION_MODE : GRADATION_putHan(x,y);
               EXPAND_MODE : EXPAND_putHan(x,y);
            end;
         end;
         inc(x,16);
      end
      else begin

         if (data1 < 27) then data1 := 31;
         if (data1 >126) then data1 := 32;
         dec(data1,31);

         move(ascii_font_data^[data1],han_buf,16);

         if bold then begin
            for k := 0 to 14 do
               han_buf[k] := han_buf[k] or han_buf[k+1];
            for k := 15 downto 1 do
               han_buf[k] := han_buf[k] or han_buf[k-1];
            for k := 0 to 15 do
               han_buf[k] := han_buf[k] or (han_buf[k] shr 1);
            for k := 15 downto 0 do
               han_buf[k] := han_buf[k] or (han_buf[k] shl 1);
         end;

         if must_clip then begin
            if (x >= boundary_x1) and (y >= boundary_y1) then
            case _HAN_MODE_S of
               PIXEL_UNIT_MODE : putAscii(x, y);
               QUICK_MODE : QUICK_putAscii(x, y);
               GRADATION_MODE : GRADATION_putAscii(x,y);
               EXPAND_MODE : EXPAND_putAscii(x,y);
            end;
         end else if (x >= 0) and (y >= 0) then begin
            case _HAN_MODE_S of
               PIXEL_UNIT_MODE : putAscii(x, y);
               QUICK_MODE : QUICK_putAscii(x, y);
               GRADATION_MODE : GRADATION_putAscii(x,y);
               EXPAND_MODE : EXPAND_putAscii(x,y);
            end;
         end;
         inc(x,8);
      end;
      inc(order);
   end;
end;

procedure printHangul(x,y : integer; s : string);
begin
   printHangulSub(x,y,s,FALSE,FALSE);
end;

procedure printHangulOUTLINE(x,y : integer; s : string; back_color : byte);
var
   remember_color : byte;
begin
   remember_color := _HAN_COLOR_S;
   _HAN_COLOR_S := back_color;
   printHangulSub(x,y,s,TRUE,FALSE);
   _HAN_COLOR_S := remember_color;
   printHangulSub(x,y,s,FALSE,FALSE);
end;

procedure printHangulOUTLINE_CLIPPING(x,y : integer; s : string; back_color : byte);
var
   remember_color : byte;
begin
   remember_color := _HAN_COLOR_S;
   _HAN_COLOR_S := back_color;
   printHangulSub(x,y,s,TRUE,TRUE);
   _HAN_COLOR_S := remember_color;
   printHangulSub(x,y,s,FALSE,TRUE);
end;

procedure printHangulGRADATION(x,y : integer; s : string; delay : word);
var
   i, j, k : integer;
   remember_HAN_MODE_S : han_mode_T;
   colors : array[1..3] of byte;
begin
   remember_HAN_MODE_S := _HAN_MODE_S;

   setHanMode(GRADATION_MODE);
   for i := 1 to 9 do begin
      setPalette(i,i);
      setRGB(i,0,0,0);
   end;
   printHangul(x,y,s);
   j := 0;
   repeat
      for i := 1 to 9 do begin
         if ((j - i*10) > 0) and ((j - i*10) <= 63) then begin
            for k := 1 to 3 do begin
               colors[k] := j-i*10;
               if _HAN_GRADATION_COLOR[k] = 21 then colors[k] := colors[k] div 3
               else if _HAN_GRADATION_COLOR[k] = 42 then colors[k] := colors[k] * 2 div 3;
            end;
            setRGB(i,colors[1],colors[2],colors[3]);
         end;
      end;
      inc(j);
      if not KeyPressed then waitVerticalRetrace(delay);
   until (j = 90 + 63);
   setHanMode(PIXEL_UNIT_MODE);
   printHangul(x,y,s);

   _HAN_MODE_S := remember_HAN_MODE_S;
end;

procedure fadeOutGradationColor(delay : word);
var
   RGB_color : array[1..3] of byte;
   i, j : integer;
begin
   for i := 1 to 3 do
      RGB_color[i] := _HAN_GRADATION_COLOR[i];
   for i := 1 to 63 do begin
      for j := 1 to 3 do begin
         if RGB_color[j] > 0 then dec(RGB_color[j]);
      end;
      setRGB(10,RGB_color[1],RGB_color[2],RGB_color[3]);
      if not KeyPressed then waitVerticalRetrace(delay);
   end;
end;

procedure printNetImage(x1,y1,x2,y2 : integer; back_color : byte);
const
   net_image_data : array[0..15] of byte =
      ($AA,$55,$AA,$55,$AA,$55,$AA,$55,$AA,$55,$AA,$55,$AA,$55,$AA,$55);
var
   x, y : integer;
   color : byte;
begin
   color := getHanColor;
   setHanColor(back_color);
   move(net_image_data,han_buf,16);
   x := x1; y := y1;
   while y < y2 do begin
      while x < x2 do begin
         case _HAN_MODE_S of
            PIXEL_UNIT_MODE : putAscii(x, y);
            QUICK_MODE : QUICK_putAscii(x, y);
            GRADATION_MODE : GRADATION_putAscii(x,y);
            EXPAND_MODE : EXPAND_putAscii(x,y);
         end;
         x := x + 8;
      end;
      y := y + 16;
      x := x1;
   end;
   setHanColor(color);
end;

(*******   The follow is Hangul Inputting Routines    *******)

procedure setMaxInputByte(max_byte : byte);
begin
   MAX_INPUT_BYTE := max_byte;
end;

procedure printHangul_withDelete(x,y : integer; s : string);
var
   i : integer;
begin
   bar(x,y,x+(length(s)*8)-1,y+15);
   printHangul(x,y,s);
end;

function isAsciiCode(ch : char) : boolean;
begin
   isAsciiCode := ch in [#32..#127];
end;

function isAlphabet(ch : char) : boolean;
begin
   isAlphabet := ch in ['A'..'Z','a'..'z'];
end;

function isHangul(var str : string; n : integer) :boolean;
var
   i : integer;
begin
   i := 1;
   while (i < n) do begin
      if isAsciiCode(str[i]) then inc(i) else inc(i,2);
   end;
   if i = n then isHangul := FALSE else isHangul := TRUE;
end;

procedure insertCharacter(ch :char; var str : string; n : integer);
var
   i : integer;
begin
   i := length(str);
   while (i >= n) do begin
      str[i+1] := str[i];
      dec(i);
   end;
   str[n] := ch;
   inc(str[0]);
end;

procedure insertHangul(var str : string);
begin
   insertCharacter(' ',str,cursor_ptr+1);
   insertCharacter(' ',str,cursor_ptr+1);
   printHangul_withDelete(input_x_cursor,input_y_cursor,copy(str,cursor_ptr+1,length(str)-cursor_ptr));
end;

procedure deleteCharacter(var str : string; n : integer);
var
   i : integer;
begin
   i :=  n + 1;
   while (i <= length(str)) do begin
      str[i] := str[i+1];
      inc(i);
   end;
   dec(str[0]);
end;

procedure generateCode(var str : string);
begin
   str[1] := chr(128 or (hangul_code[1] shl 2) or (hangul_code[2] shr 3));
   str[2] := chr((hangul_code[2] shl 5) or hangul_code[3]);
   str[0] := chr(2);
end;

function isComplete : boolean;
begin
   if (is_hangul_mode and ((hangul_code[1] + hangul_code[2] + hangul_code[3]) <> 4)) then isComplete := FALSE
   else isComplete := TRUE;
end;

procedure initHangulCode;
var
   b : byte;
begin
   for b := 1 to 3 do hangul_code[b] := init_hangul_code[b];
end;

procedure completeEnglish(ch : char; var str : string);
begin
   if is_inserting_mode or (length(str) <= cursor_ptr) then insertCharacter(ch,str,cursor_ptr+1)
   else begin
      if isHangul(str,cursor_ptr+2) then str[cursor_ptr+2] := ' ';
      str[cursor_ptr+1] := ch;
   end;
   printHangul_withDelete(input_x_cursor,input_y_cursor,copy(str,cursor_ptr+1,length(str)-cursor_ptr));
   input_x_cursor := input_x_cursor + 8;
   inc(cursor_ptr);
   initHangulCode;
end;

procedure completeHangul(var str : string; is_increase_lenght : boolean);
var
   temp_s : string;
begin
   temp_s := '';
   generateCode(temp_s);
   str[cursor_ptr+1] := temp_s[1];
   str[cursor_ptr+2] := temp_s[2];
   cursor_ptr := cursor_ptr + 2;
   if is_increase_lenght and (is_inserting_mode or (length(str) <= cursor_ptr)) then insertHangul(str);
   printHangul_withDelete(input_x_cursor,input_y_cursor,copy(str,cursor_ptr-1,length(str)-cursor_ptr+2));
   input_x_cursor := input_x_cursor + 16;
   initHangulCode;
end;

procedure pressBackSpace(var str : string);
var
   i : integer;
begin
   if (cursor_ptr = 0) and isComplete then exit;
   if not isComplete then begin
      if hangul_code[3] <> init_hangul_code[3] then begin
         for i := 0 to 10 do begin
            if hangul_code[3] = third_table2[i][2] then begin
               hangul_code[3] := third_table2[i][0];
               exit;
            end;
         end;
         hangul_code[3] := init_hangul_code[3];
      end
      else if hangul_code[2] <> init_hangul_code[2] then begin
         for i := 0 to 6 do begin
            if hangul_code[2] = second_table[i][2] then begin
               hangul_code[2] := second_table[i][0];
               exit;
            end;
         end;
         hangul_code[2] := init_hangul_code[2];
      end
      else hangul_code[1] := init_hangul_code[1];
   end
   else if (length(str) > 1) and isHangul(str,cursor_ptr) then begin
      deleteCharacter(str,cursor_ptr-1);
      deleteCharacter(str,cursor_ptr-2);
      input_x_cursor := input_x_cursor - 16;
      cursor_ptr := cursor_ptr - 2;
      insertCharacter(' ',str,length(str)+1);
      insertCharacter(' ',str,length(str)+1);
      printHangul_withDelete(input_x_cursor,input_y_cursor,copy(str,cursor_ptr+1,length(str)-cursor_ptr));
      deleteCharacter(str,length(str)-1);
      deleteCharacter(str,length(str)-1);
   end
   else if length(str) > 0 then begin
      deleteCharacter(str,cursor_ptr-1);
      input_x_cursor := input_x_cursor - 8;
      dec(cursor_ptr);
      insertCharacter(' ',str,length(str)+1);
      printHangul_withDelete(input_x_cursor,input_y_cursor,copy(str,cursor_ptr+1,length(str)-cursor_ptr));
      deleteCharacter(str,length(str)-1);
   end;
end;

procedure pressRightArrow(var str : string);
begin
   if cursor_ptr < length(str) then begin
      if not isComplete then completeHangul(str,FALSE)
      else if isHangul(str,cursor_ptr+2) then begin
         cursor_ptr := cursor_ptr + 2;
         input_x_cursor := input_x_cursor + 16;
      end
      else begin
         inc(cursor_ptr);
         input_x_cursor := input_x_cursor + 8;
      end;
   end;
end;

procedure pressLeftArrow(var str : string);
begin
   if not isComplete then begin
      completeHangul(str,FALSE);
      cursor_ptr := cursor_ptr - 2;
      input_x_cursor := input_x_cursor - 16;
   end;
   if cursor_ptr > 0 then begin
      if isHangul(str,cursor_ptr) then begin
         cursor_ptr := cursor_ptr - 2;
         input_x_cursor := input_x_cursor - 16;
      end
      else begin
         dec(cursor_ptr);
         input_x_cursor := input_x_cursor - 8;
      end;
   end;
end;

procedure pressDelete(var str : string);
begin
   if (str = '') or (length(str) <= cursor_ptr) then exit;
   if not isComplete then begin
      completeHangul(str,TRUE);
      cursor_ptr := cursor_ptr - 2;
      input_x_cursor := input_x_cursor - 16;
   end;
   if isHangul(str,cursor_ptr+2) then begin
      deleteCharacter(str,cursor_ptr);
      deleteCharacter(str,cursor_ptr);
      insertCharacter(' ',str,length(str)+1);
      insertCharacter(' ',str,length(str)+1);
      printHangul_withDelete(input_x_cursor,input_y_cursor,copy(str,cursor_ptr+1,length(str)-cursor_ptr));
      deleteCharacter(str,length(str));
      deleteCharacter(str,length(str));
   end
   else begin
      deleteCharacter(str,cursor_ptr);
      insertCharacter(' ',str,length(str)+1);
      printHangul_withDelete(input_x_cursor,input_y_cursor,copy(str,cursor_ptr+1,length(str)-cursor_ptr));
      deleteCharacter(str,length(str));
   end;
end;

procedure processConsonant(var str : string);
var
   i : integer;
begin
   if hangul_code[2] = init_hangul_code[2] then begin
      if (hangul_code[1] = init_hangul_code[1]) and (is_inserting_mode or (length(str) <= cursor_ptr)) then
         insertHangul(str);
      if (not is_inserting_mode) and (not isHangul(str,cursor_ptr+1)) then str[cursor_ptr+3] := ' ';
      hangul_code[1] := input_code;
   end
   else if hangul_code[3] = init_hangul_code[3] then begin
      hangul_code[3] := third_table1[input_code];
      if hangul_code[3] = init_hangul_code[3] then begin
         completeHangul(str,FALSE);
         hangul_code[1] := input_code;
      end;
   end
   else begin
      for i := 0 to 10 do begin
         if ((third_table2[i][0] = hangul_code[3]) and (third_table2[i][1] = input_code)) then begin
            hangul_code[3] := third_table2[i][2];
            exit;
         end;
      end;
      completeHangul(str,TRUE);
      hangul_code[1] := input_code;
   end;
end;

procedure processVowel(var str : string);
var
   i : integer;
begin
   if hangul_code[1] = init_hangul_code[1] then begin
      initHangulCode;
   end
   else if hangul_code[3] = init_hangul_code[3] then begin
      if hangul_code[2] = init_hangul_code[2] then begin
         hangul_code[2] := input_code;
      end else begin
         for i := 0 to 6 do begin
            if ((second_table[i][0] = hangul_code[2]) and (second_table[i][1] = input_code)) then begin
               hangul_code[2] := second_table[i][2];
               exit;
            end;
         end;
         completeHangul(str,FALSE);
      end;
   end
   else begin
      hangul_code[4] := first_table[hangul_code[3]];
      hangul_code[3] := init_hangul_code[3];
      if hangul_code[4] >= 100 then begin
         hangul_code[4] := hangul_code[4] - 100;
         hangul_code[3] := third_table2[hangul_code[4]][0];
         hangul_code[4] := third_table2[hangul_code[4]][1];
      end;
      completeHangul(str,FALSE);
      hangul_code[1] := hangul_code[4];
      hangul_code[2] := input_code;
      if is_inserting_mode or (length(str) <= cursor_ptr) then insertHangul(str);
   end;
end;

procedure drawHangulCursor(var str : string; back_color : word);
var
   cursor_str : string;
begin
   cursor_str := ' ';
   if not isComplete then generateCode(cursor_str)
   else if length(str) <= cursor_ptr then cursor_str[0] := chr(1)
   else if isAsciiCode(str[cursor_ptr+1]) then begin
      cursor_str[1] := str[cursor_ptr+1];
      cursor_str[0] := chr(1);
   end else begin
      cursor_str[1] := str[cursor_ptr+1];
      cursor_str[2] := str[cursor_ptr+2];
      cursor_str[0] := chr(2);
   end;
   printHangul_withDelete(input_x_cursor,input_y_cursor,cursor_str);

   setfillstyle(SOLIDFILL,back_color);
   bar(remember_x_cursor,input_y_cursor+16,remember_x_cursor+8*MAX_INPUT_BYTE,input_y_cursor+16);
   setfillstyle(SOLIDFILL,back_color xor $15);
   bar(input_x_cursor,input_y_cursor+16,
        input_x_cursor+(length(cursor_str)*8)-1,input_y_cursor+16);
   setfillstyle(SOLIDFILL,back_color);
end;

function waitKeyIn(var str : string; back_color : word) : integer;
var
   i : integer;
begin
   drawHangulCursor(str,back_color);
   i := ord(readKey);
   if i = 0 then i := SCAN_CODE + ord(readKey);
   if (mem[Seg0040:$017] and $03) > 0 then i := i + SHIFT_CODE;
   waitKeyIn := i;
end;

function inputHangulSub(x,y : integer; s : string; back_color : word; is_hangul, extended_mode : boolean;
                        var result : integer) : string;
label
   EXIT_LOOP;
const
   ESCAPE_KEY = 27;
   ENTER_KEY = 13;
   BACK_SPACE_KEY = 8;
   MODE_CHANGING_KEY = 32 + SHIFT_CODE;
   UP_ARROW_KEY    = 72 + SCAN_CODE;
   DOWN_ARROW_KEY  = 80 + SCAN_CODE;
   LEFT_ARROW_KEY  = 75 + SCAN_CODE;
   RIGHT_ARROW_KEY = 77 + SCAN_CODE;
   INSERT_KEY = 82 + SCAN_CODE;
   DELETE_KEY = 83 + SCAN_CODE;
   HOME_KEY = 71 + SCAN_CODE;
   END_KEY = 79 + SCAN_CODE;

var
   i, j : integer;
   key_code : integer;
   ch : char;
   fill_information : FillSettingsType;

begin

   x := (x div 8) * 8;
   remember_x_cursor := x;
   remember_y_cursor := y;

   is_hangul_mode := is_hangul;
   is_inserting_mode := TRUE;

   getFillSettings(fill_information);
   setFillStyle(SOLIDFILL,back_color);

   if extended_mode then begin
      input_x_cursor := x;
      input_y_cursor := y;
      cursor_ptr := 0;
   end else begin
      input_x_cursor := x + length(s) * 8;
      input_y_cursor := y;
      cursor_ptr := length(s);
   end;
   printHangul_withDelete(x,y,s);
   initHangulCode;
   while TRUE do begin

      if (cursor_ptr >= MAX_INPUT_BYTE) or (length(s) > MAX_INPUT_BYTE) then begin
         key_code := ESCAPE_KEY;
      end else begin
         key_code := waitKeyIn(s,back_color);
      end;
      case key_code of
         MODE_CHANGING_KEY :
         begin
            if not isComplete then completeHangul(s,FALSE);
            is_hangul_mode := not is_hangul_mode;
         end;
         LEFT_ARROW_KEY : pressLeftArrow(s);
         RIGHT_ARROW_KEY : pressRightArrow(s);
         INSERT_KEY :
         begin
            if not isComplete then completeHangul(s,FALSE);
            is_inserting_mode := not is_inserting_mode;
         end;
         DELETE_KEY : pressDelete(s);
         HOME_KEY :
         begin
            if not isComplete then completeHangul(s,FALSE);
            input_x_cursor := x;
            input_y_cursor := y;
            cursor_ptr := 0;
         end;
         END_KEY :
         begin
            if not isComplete then completeHangul(s,FALSE);
            input_x_cursor := x + length(s) * 8;
            input_y_cursor := y;
            cursor_ptr := length(s);
         end;
         ENTER_KEY :
         if not extended_mode then begin
            goto EXIT_LOOP;
         end;
         UP_ARROW_KEY, DOWN_ARROW_KEY :
         if extended_mode then begin
            goto EXIT_LOOP;
         end;
         BACK_SPACE_KEY : pressBackSpace(s);
         ESCAPE_KEY :
         if extended_mode then begin
            goto EXIT_LOOP;
         end else begin
            i := length(s);
            bar(x,y,x+i*8,y+15);
            initHangulCode;
            s := '';
            input_x_cursor := x;
            input_y_cursor := y;
            cursor_ptr := 0;
         end
         else begin

            if key_code >= SHIFT_CODE then begin
               ch := chr(key_code - SHIFT_CODE);
               if not (ch in [#32..#127]) then ch := #0;
            end
            else if (key_code >= 32) and (key_code < 128) then ch := chr(key_code)
            else ch := #0;

            if is_hangul_mode and isAlphabet(ch) then begin
               ch := chr(ord(UpCase(ch))-ord('A'));
               if key_code >= SHIFT_CODE then input_code := key_table2[ord(ch)]
                                         else input_code := key_table1[ord(ch)];
               if input_code < 100 then begin
                  processConsonant(s);
               end
               else begin
                  input_code := input_code - 100;
                  processVowel(s);
               end;
            end
            else if ch in [#32..#127] then begin
               if not isComplete then completeHangul(s,FALSE);
               completeEnglish(ch,s);
            end;
         end;
      end;
   end;
EXIT_LOOP:
   with fill_information do setFillStyle(pattern, color);

   if not isComplete then completeHangul(s,FALSE);

   if (cursor_ptr >= MAX_INPUT_BYTE) or (length(s) > MAX_INPUT_BYTE) then begin
      result := 0; s[0] := chr(ord(s[0])-2);
   end else begin
      result := key_code;
   end;

   inputHangulSub := s;
end;

function inputHangul(x,y : integer; s : string; back_color : word; is_hangul : boolean) : string;
var
   result : integer;
begin
   inputHangul := inputHangulSub(x,y,s,back_color,is_hangul,FALSE,result);
end;

function inputHangulEXTENDED(x,y : integer; s : string; back_color : word; is_hangul : boolean; var result : integer) : string;
begin
   inputHangulEXTENDED := inputHangulSub(x,y,s,back_color,is_hangul_mode,TRUE,result);
end;


var
   fa : file of ascii_font_data_T;
   fh : file of han_font_data_T;
begin

   new(ascii_font_data);
   new(han_font_data_type[SAMMUL]);
   new(han_font_data_type[GOTHIC]);
   new(han_font_data_type[MYUNGJO]);

   {$I-}
   assign(fa,'ascii.fnt');
   reset(fa);
   read(fa,ascii_font_data^);
   close(fa);
   if IOResult <> 0 then occurHanError('Ascii.fnt');

   assign(fh,'Uhan20.ft1');
   reset(fh);
   read(fh,han_font_data_type[SAMMUL]^);
   close(fh);
   if IOResult <> 0 then occurHanError('UHan20.ft1');

   assign(fh,'UHan20.ft2');
   reset(fh);
   read(fh,han_font_data_type[GOTHIC]^);
   close(fh);
   if IOResult <> 0 then occurHanError('UHan20.ft2');

   assign(fh,'UHan20.ft3');
   reset(fh);
   read(fh,han_font_data_type[MYUNGJO]^);
   close(fh);
   if IOResult <> 0 then occurHanError('UHan20.ft3');

   {$I+}

   setHanMode(QUICK_MODE);
   setHanType(SAMMUL);
   setGradationType(_HAN_GRADATION_S);
   setHanBoundary(0,0,639,479);
   setHanExpandRate(1);
   setMaxInputByte(8);
   scr := ptr(SegA000,$0000);

end.
