// Note:  Must Define D3D_OVERLOADS to get C++ version of D3DMATRIX
#define D3D_OVERLOADS
#include <float.h>
#include <math.h>

#include "Matrix.H"

const float pi = 3.141592654f;

// sets D3D matrix to all 0's
D3DMATRIX ZeroMatrix()
{
    D3DMATRIX   m;

    m(0,0) = 0.0f; m(1,0) = 0.0f; m(2,0) = 0.0f; m(3,0) = 0.0f;
    m(0,1) = 0.0f; m(1,1) = 0.0f; m(2,1) = 0.0f; m(3,1) = 0.0f;
    m(0,2) = 0.0f; m(1,2) = 0.0f; m(2,2) = 0.0f; m(3,2) = 0.0f;
    m(0,3) = 0.0f; m(1,3) = 0.0f; m(2,3) = 0.0f; m(3,3) = 0.0f;

    return m;
}


// sets D3D matrix to Identiy (1's on diagonal, zero's elsewhere)
D3DMATRIX IdentityMatrix()
{
	return D3DMATRIX(1,0,0,0, 0,1,0,0, 0,0,1,0, 0,0,0,1);
}
 
 
// sets Projection matrix from fov, near and far planes
//	1. fov is in radians.
//	2. See Blinn, "A Trip Down the Graphics Pipeline" pg 188 for details.
D3DMATRIX ProjectionMatrix( const float near_plane, 
				            const float far_plane, 
				            const float fov )
{
	float c, s, Q;

	c = (float) cos(fov*0.5);
	s = (float) sin(fov*0.5);
	Q = s/(1.0f - near_plane/far_plane);

    D3DMATRIX m;

    m(0,0) = c;    m(1,0) = 0.0f; m(2,0) = 0.0f; m(3,0) = 0.0f;
    m(0,1) = 0.0f; m(1,1) = c;    m(2,1) = 0.0f; m(3,1) = 0.0f;
    m(0,2) = 0.0f; m(1,2) = 0.0f; m(2,2) = Q;    m(3,2) = -Q*near_plane;
    m(0,3) = 0.0f; m(1,3) = 0.0f; m(2,3) = s;    m(3,3) = 0.0f;

    return m;
}


// Controls where the camara is.
//	1. Note the roll parameter is in radians and rools the viewpoint
//		around the viewing direction
D3DMATRIX ViewMatrix( const D3DVECTOR& from, 
		              const D3DVECTOR& at, 
		              const D3DVECTOR& world_up, 
		              const float roll )
{
    D3DMATRIX view = IdentityMatrix();
    D3DVECTOR up, right, view_dir;

    view_dir = Normalize(at - from);

	right = CrossProduct(world_up, view_dir);
	up = CrossProduct(view_dir, right);

	right = Normalize(right);
	up = Normalize(up);
	
    view(0, 0) = right.x;
    view(1, 0) = right.y;
    view(2, 0) = right.z;
    view(0, 1) = up.x;
    view(1, 1) = up.y;
    view(2, 1) = up.z;
    view(0, 2) = view_dir.x;
    view(1, 2) = view_dir.y;
    view(2, 2) = view_dir.z;
	
    view(3, 0) = -DotProduct(right, from);
    view(3, 1) = -DotProduct(up, from);
    view(3, 2) = -DotProduct(view_dir, from);

	// Set roll
	if (roll != 0.0f) 
		view = MatrixMult(RotateZMatrix(-roll), view);

    return view;
}


// Rotate matrix about X axis
D3DMATRIX RotateXMatrix( const float rads )
{
	float	cosine, sine;

	cosine = (float) cos(rads);
    sine   = (float) sin(rads);

    D3DMATRIX   m;

    m(0,0) = 1.0f; m(1,0) = 0.0f;   m(2,0) = 0.0f;   m(3,0) = 0.0f;
    m(0,1) = 0.0f; m(1,1) = cosine; m(2,1) = sine;   m(3,1) = 0.0f;
    m(0,2) = 0.0f; m(1,2) = -sine;  m(2,2) = cosine; m(3,2) = 0.0f;
    m(0,3) = 0.0f; m(1,3) = 0.0f;   m(2,3) = 0.0f;   m(3,3) = 1.0f;

    return m;
}


// Rotate matrix about Y axis
D3DMATRIX RotateYMatrix( const float rads )
{
	float	cosine, sine;

	cosine = (float) cos(rads);
    sine   = (float) sin(rads);

    D3DMATRIX   m;

    m(0,0) = cosine; m(1,0) = 0.0f; m(2,0) = -sine;  m(3,0) = 0.0f;
    m(0,1) = 0.0f;   m(1,1) = 1.0f; m(2,1) = 0.0f;   m(3,1) = 0.0f;
    m(0,2) = sine;   m(1,2) = 0.0f; m(2,2) = cosine; m(3,2) = 0.0f;
    m(0,3) = 0.0f;   m(1,3) = 0.0f; m(2,3) = 0.0f;   m(3,3) = 1.0f;

    return m;
}
  

// Rotate matrix about Z axis
D3DMATRIX RotateZMatrix( const float rads )
{
	float	cosine, sine;

	cosine = (float) cos(rads);
    sine   = (float) sin(rads);

    D3DMATRIX   m;

    m(0,0) = cosine; m(1,0) = sine;   m(2,0) = 0.0f; m(3,0) = 0.0f;
    m(0,1) = -sine;  m(1,1) = cosine; m(2,1) = 0.0f; m(3,1) = 0.0f;
    m(0,2) = 0.0f;   m(1,2) = 0.0f;   m(2,2) = 1.0f; m(3,2) = 0.0f;
    m(0,3) = 0.0f;   m(1,3) = 0.0f;   m(2,3) = 0.0f; m(3,3) = 1.0f;
    
	return m;
}

D3DMATRIX TranslateXMatrix( const float dx )
{
    D3DMATRIX m;

    m(0,0) = 1.0f; m(1,0) = 0.0f; m(2,0) = 0.0f; m(3,0) = dx;
    m(0,1) = 0.0f; m(1,1) = 1.0f; m(2,1) = 0.0f; m(3,1) = 0.0f;
    m(0,2) = 0.0f; m(1,2) = 0.0f; m(2,2) = 1.0f; m(3,2) = 0.0f;
    m(0,3) = 0.0f; m(1,3) = 0.0f; m(2,3) = 0.0f; m(3,3) = 1.0f;

    return m;
}

D3DMATRIX TranslateYMatrix( const float dy )
{
    D3DMATRIX m;

    m(0,0) = 1.0f; m(1,0) = 0.0f; m(2,0) = 0.0f; m(3,0) = 0.0f;
    m(0,1) = 0.0f; m(1,1) = 1.0f; m(2,1) = 0.0f; m(3,1) = dy;
    m(0,2) = 0.0f; m(1,2) = 0.0f; m(2,2) = 1.0f; m(3,2) = 0.0f;
    m(0,3) = 0.0f; m(1,3) = 0.0f; m(2,3) = 0.0f; m(3,3) = 1.0f;

    return m;
}

D3DMATRIX TranslateZMatrix( const float dz )
{
    D3DMATRIX m;

    m(0,0) = 1.0f; m(1,0) = 0.0f; m(2,0) = 0.0f; m(3,0) = 0.0f;
    m(0,1) = 0.0f; m(1,1) = 1.0f; m(2,1) = 0.0f; m(3,1) = 0.0f;
    m(0,2) = 0.0f; m(1,2) = 0.0f; m(2,2) = 1.0f; m(3,2) = dz;
    m(0,3) = 0.0f; m(1,3) = 0.0f; m(2,3) = 0.0f; m(3,3) = 1.0f;

    return m;
}

// Returns matrix to translate by (dx, dy, dz)
D3DMATRIX TranslateMatrix( const float dx, const float dy, const float dz )
{
    D3DMATRIX m;

    m(0,0) = 1.0f; m(1,0) = 0.0f; m(2,0) = 0.0f; m(3,0) = dx;
    m(0,1) = 0.0f; m(1,1) = 1.0f; m(2,1) = 0.0f; m(3,1) = dy;
    m(0,2) = 0.0f; m(1,2) = 0.0f; m(2,2) = 1.0f; m(3,2) = dz;
    m(0,3) = 0.0f; m(1,3) = 0.0f; m(2,3) = 0.0f; m(3,3) = 1.0f;

    return m;
}


// Returns matrix to translate by v
D3DMATRIX TranslateMatrix( const D3DVECTOR& v )
{
    D3DMATRIX m;

    m(0,0) = 1.0f; m(1,0) = 0.0f; m(2,0) = 0.0f; m(3,0) = v.x;
    m(0,1) = 0.0f; m(1,1) = 1.0f; m(2,1) = 0.0f; m(3,1) = v.y;
    m(0,2) = 0.0f; m(1,2) = 0.0f; m(2,2) = 1.0f; m(3,2) = v.z;
    m(0,3) = 0.0f; m(1,3) = 0.0f; m(2,3) = 0.0f; m(3,3) = 1.0f;

    return m;
}


// scale matrix (uniform)
D3DMATRIX ScaleMatrix( const float size )
{
    D3DMATRIX m;

    m(0,0) = size; m(1,0) = 0.0f; m(2,0) = 0.0f; m(3,0) = 0.0f;
    m(0,1) = 0.0f; m(1,1) = size; m(2,1) = 0.0f; m(3,1) = 0.0f;
    m(0,2) = 0.0f; m(1,2) = 0.0f; m(2,2) = size; m(3,2) = 0.0f;
    m(0,3) = 0.0f; m(1,3) = 0.0f; m(2,3) = 0.0f; m(3,3) = 1.0f;

    return m;
}

  
// scale matrix
D3DMATRIX ScaleMatrix( const float a, const float b, const float c )
{
    D3DMATRIX m;

    m(0,0) = a;    m(1,0) = 0.0f; m(2,0) = 0.0f; m(3,0) = 0.0f;
    m(0,1) = 0.0f; m(1,1) = b;    m(2,1) = 0.0f; m(3,1) = 0.0f;
    m(0,2) = 0.0f; m(1,2) = 0.0f; m(2,2) = c;    m(3,2) = 0.0f;
    m(0,3) = 0.0f; m(1,3) = 0.0f; m(2,3) = 0.0f; m(3,3) = 1.0f;

    return m;
}
  

// scale matrix
D3DMATRIX ScaleMatrix( const D3DVECTOR& v )
{
    D3DMATRIX m;

    m(0,0) = v.x;  m(1,0) = 0.0f; m(2,0) = 0.0f; m(3,0) = 0.0f;
    m(0,1) = 0.0f; m(1,1) = v.y;  m(2,1) = 0.0f; m(3,1) = 0.0f;
    m(0,2) = 0.0f; m(1,2) = 0.0f; m(2,2) = v.z;  m(3,2) = 0.0f;
    m(0,3) = 0.0f; m(1,3) = 0.0f; m(2,3) = 0.0f; m(3,3) = 1.0f;

    return m;
}


// [C] = [A] * [B]
D3DMATRIX MatrixMult( const D3DMATRIX & a, const D3DMATRIX & b )
{
	D3DMATRIX m = ZeroMatrix();

	for (int i=0; i<4; i++) {
		for (int j=0; j<4; j++) {
			for (int k=0; k<4; k++) {
				m(i, j) += a(k, j) * b(i, k);
			}
		}
	}

	return m;
}


// V' = V * [M]
D3DVECTOR TransformVector( const D3DVECTOR& v, const D3DMATRIX & m )
{
	float	hvec[4];

	for (int i=0; i<4; i++) {
		hvec[i] = 0.0f;
		for (int j=0; j<4; j++) {
			if (j==3) {
				hvec[i] += m(j, i);
			} else {
				hvec[i] += v[j] * m(j, i);
			}
		}
	}
	D3DVECTOR ret(hvec[0]/hvec[3], hvec[1]/hvec[3], hvec[2]/hvec[3]);

	return ret;
}


// N' = N * [M]
D3DVECTOR TransformNormal( const D3DVECTOR& v, const D3DMATRIX & mat )
{
	D3DMATRIX	m;

	m = MatrixInverse(mat);
	m = MatrixTranspose(m);
	return TransformVector(v, m);
}


// Creates the inverse of a 4x4 matrix
static void	lubksb( D3DMATRIX & a, int *indx, float *b );
static void ludcmp( D3DMATRIX & a, int *indx, float *d );

D3DMATRIX MatrixInverse( const D3DMATRIX & m )
{
	D3DMATRIX	n, y;
	int			i, j, indx[4];
	float		d, col[4];

	n = m;
	ludcmp(n, indx, &d);

	for (j=0; j<4; j++) {
		for (i=0; i<4; i++) {
			col[i] = 0.0f;
		}
		col[j] = 1.0f;
		lubksb(n, indx, col);
		for (i=0; i<4; i++) {
			y(i, j) = col[i];
		}
	}
	return y;
}


// backward subsitution
static void lubksb( D3DMATRIX & a, int *indx, float *b )
{
	int		i, j, ii=-1, ip;
	float	sum;

	for (i=0; i<4; i++) {
		ip = indx[i];
		sum = b[ip];
		b[ip] = b[i];
		if (ii>=0) {
			for (j=ii; j<=i-1; j++) {
				sum -= a(i, j) * b[j];
			}
		} else if (sum != 0.0) {
			ii = i;
		}
		b[i] = sum;
	}
	for (i=3; i>=0; i--) {
		sum = b[i];
		for (j=i+1; j<4; j++) {
			sum -= a(i, j) * b[j];
		}
		b[i] = sum/a(i, i);
	}
}


// LU decomposition
static void ludcmp( D3DMATRIX & a, int *indx, float *d )
{
	float	vv[4];               /* implicit scale for each row */
	float	big, dum, sum, tmp;
	int		i, imax, j, k;

	*d = 1.0f;
	for (i=0; i<4; i++) {
		big = 0.0f;
		for (j=0; j<4; j++) {
			if ((tmp = (float) fabs(a(i, j))) > big) {
				big = tmp;
			}
		}
		/*
		if (big == 0.0f) {
			printf("ludcmp(): singular matrix found...\n");
			exit(1);
		}
		*/
		vv[i] = 1.0f/big;
	}
	for (j=0; j<4; j++) {
		for (i=0; i<j; i++) {
			sum = a(i, j);
			for (k=0; k<i; k++) {
				sum -= a(i, k) * a(k, j);
			}
			a(i, j) = sum;
		}
		big = 0.0f;
		for (i=j; i<4; i++) {
			sum = a(i, j);
			for (k=0; k<j; k++) {
				sum -= a(i, k)*a(k, j);
			}
			a(i, j) = sum;
			if ((dum = vv[i] * (float)fabs(sum)) >= big) {
				big = dum;
				imax = i;
			}
		}
		if (j != imax) {
			for (k=0; k<4; k++) {
				dum = a(imax, k);
				a(imax, k) = a(j, k);
				a(j, k) = dum;
			}
			*d = -(*d);
			vv[imax] = vv[j];
		}
		indx[j] = imax;
		if (a(j, j) == 0.0f) {
			a(j, j) = 1.0e-20f;      /* can be 0.0 also... */
		}
		if (j != 3) {
			dum = 1.0f/a(j, j);
			for (i=j+1; i<4; i++) {
				a(i, j) *= dum;
			}
		}
	}
}
  

// [M] = [M]'
D3DMATRIX MatrixTranspose( const D3DMATRIX & m )
{
	D3DMATRIX	ret;
	int			i, j;

	for (i=0; i<4; i++) {
		for (j=0; j<4; j++) {
			ret(i, j) = m(j, i);
		}
	}

	return ret;
}



