#include "Texture.h"


// Texture List
///////////////////////////////////////////////////////////////////////////////
LPTEXTURE FindTexture( LPTEXTURE lpTexture, char* lpszName )
{
	LPTEXTURE texture = lpTexture;

	while ( texture )
	{
		if ( !strcmp( lpszName, texture->szName ) )
			break;

		texture = texture->lpNext;
	}

	return texture;
}

LPTEXTURE CreateTexture( char* lpszName )
{
	LPTEXTURE texture = new TEXTURE;
	if ( !texture )
		return NULL;

	ZeroMemory( texture, sizeof(TEXTURE) );
	strcpy( texture->szName, lpszName );

	return texture;
}

LPTEXTURE CreateTexture( LPTEXTURE lpTexture, char* lpszName )
{
	if ( !lpTexture )
		return CreateTexture( lpszName );

	LPTEXTURE texture = FindTexture( lpTexture, lpszName );
	if ( !texture )
	{
		texture = new TEXTURE;
		if ( !texture )
			return NULL;

		ZeroMemory( texture, sizeof(TEXTURE) );
		strcpy( texture->szName, lpszName );


		LPTEXTURE lastTexture = lpTexture;
		while ( lastTexture->lpNext )
			lastTexture = lastTexture->lpNext;

		lastTexture->lpNext = texture;
	}

	return texture;
}

void DestroyTexture( LPTEXTURE lpTexture )
{
	LPTEXTURE texture;

	while ( lpTexture->lpNext )
	{
		texture = lpTexture->lpNext;
		lpTexture->lpNext = lpTexture->lpNext->lpNext;

		if ( texture->lpDDSurface )
		{
			texture->lpDDSurface->Release();
			texture->lpDDSurface = NULL;
		}

		delete [] texture;
	}

	delete [] lpTexture;
}

void AddTexture( LPTEXTURE lpTexture, LPTEXTURE lpChildTexture )
{
	LPTEXTURE texture = FindTexture( lpTexture, lpChildTexture->szName );
	if ( !texture )
	{
		texture = lpTexture;
		while ( texture->lpNext )
			texture = texture->lpNext;

		texture->lpNext = lpChildTexture;
	}
}


// Texture Func
///////////////////////////////////////////////////////////////////////////////

typedef struct {
    DWORD           bpp;        // we want a texture format of this bpp
    DDPIXELFORMAT   ddpf;       // place the format here
} TEXTUREFORMAT, *LPTEXTUREFORMAT;

// Texture Format EnumCallback Լ
HRESULT WINAPI TextureFormatEnumCallback( LPDDSURFACEDESC lpDdsd,
                                          LPVOID          lpUserArg )
{
    if ( !lpUserArg )
        return DDENUMRET_OK;


    DDPIXELFORMAT ddpf = lpDdsd->ddpfPixelFormat;

    // we use GetDC/BitBlt to init textures so we only
    // want to use formats that GetDC will support.
    if ( ddpf.dwFlags & (DDPF_ALPHA|DDPF_ALPHAPIXELS) )
        return DDENUMRET_OK;

    if ( ddpf.dwRGBBitCount <= 8 && !(ddpf.dwFlags & (DDPF_PALETTEINDEXED8 | DDPF_PALETTEINDEXED4)) )
        return DDENUMRET_OK;

    if ( ddpf.dwRGBBitCount > 8 && !(ddpf.dwFlags & DDPF_RGB) )
        return DDENUMRET_OK;

    // BUGBUG GetDC does not work for 1 or 4bpp YET!
    if ( ddpf.dwRGBBitCount < 8 )
        return DDENUMRET_OK;

    // keep the texture format that is nearest to the bitmap we have
    LPTEXTUREFORMAT lpTextureFormat = (LPTEXTUREFORMAT)lpUserArg;
    if ( lpTextureFormat->ddpf.dwRGBBitCount == 0 ||
         (ddpf.dwRGBBitCount >= lpTextureFormat->bpp &&
         (UINT)(ddpf.dwRGBBitCount - lpTextureFormat->bpp) < (UINT)(lpTextureFormat->ddpf.dwRGBBitCount - lpTextureFormat->bpp)) )
        lpTextureFormat->ddpf = ddpf;

    return DDENUMRET_OK;
}

static BOOL IsPowerOfTwo( int n )
{
    int i;
    int po2 = 1;

    if (!n) 
		return FALSE; // bail if n is zero
    
    for ( i=0; i<32; i++, po2<<=1 )
        if (n==po2) 
			return TRUE;

    return FALSE;
}
     
static int FindNearestPowerOf2( int n )
{
    // Clamp *down* to nearest power of two.

    if (n>256) return 256; // Value is above our limit
    if (n>128) return 128;
    if (n>64) return 64;
    if (n>32) return 32;
    if (n>16) return 16;
    if (n>8) return 8;
    if (n>4) return 4;
    if (n>2) return 2;
    
    return 1;
}

static void ScaleImage( int width_in, int height_in, void *texin,
                        int width_out, int height_out, void *texout )
{
    int i, j;
    float u, v, du, dv;
    BYTE *lpTexin = (BYTE *)texin;
    BYTE *lpTexout = (BYTE *)texout;

    // Performs very simple scaling.
    // Linear interpolation of the byte values CANNOT be used,
    // as the bytes are indexes into the CLUT (Colour Look Up Table)
    du = (float)width_in / (float)width_out;
    dv = (float)height_in / (float)height_out;
    
    for ( i=0; i<height_out; i++ ) 
	{
        v = (float)i * dv;
        for (j=0; j<width_out; j++) 
		{
            u = (float)j * du;
            *lpTexout++ = lpTexin[(int)u+(int)v*width_in];
        }
    }
}



LPTEXTURE Image2Texture( LPDIRECTDRAW2 lpDD, LPDIRECT3DDEVICE2 lpDev, LPTEXTURE *lpTexture, LPIMAGE lpImage, char *lpszName )
{
	LPTEXTURE texture = FindTexture( *lpTexture, lpszName );
	if ( texture )
		return texture;

	int                 i, j;
    WORD                width  = lpImage->Width;
    WORD                height = lpImage->Height;
    LPBYTE              data   = lpImage->lpData;
	LPDIRECTDRAWPALETTE lpDDPalette = NULL;

	// find the best texture format to use.
	TEXTUREFORMAT lpTextureFormat;
    ZeroMemory( &lpTextureFormat, sizeof(TEXTUREFORMAT) );

    lpTextureFormat.bpp = lpImage->BitsPerPixel;
    HRESULT hresult = lpDev->EnumTextureFormats( TextureFormatEnumCallback, (LPVOID)&lpTextureFormat );
	if ( hresult != D3D_OK )
        goto __error;

    // Check that the pixel format is what we need
    if ( lpTextureFormat.ddpf.dwFlags & DDPF_PALETTEINDEXED4 )
        goto __error;
	

	// Palette conversion tables for 8BPP, 15BPP, 16BPP and 32BPP
	BYTE  pal24to8bpp[256];  // RGB 332 non-palettised
	WORD  pal24to16bpp[256]; // RGB 555 & RGB 565
	DWORD pal24to24bpp[256]; // RGB 888

    if ( lpTextureFormat.ddpf.dwFlags & DDPF_PALETTEINDEXED8 )
	{
		PALETTEENTRY palEntry[256];

		for ( i=0; i<256; i++ )
		{
			palEntry[i].peRed   = lpImage->rgbEntry[i].Red;
			palEntry[i].peGreen = lpImage->rgbEntry[i].Green;
			palEntry[i].peBlue  = lpImage->rgbEntry[i].Blue;

			palEntry[i].peFlags = PC_NOCOLLAPSE;
		}

        // Now make our DDraw palettes for later use:
        hresult = lpDD->CreatePalette( DDPCAPS_INITIALIZE |
			                           DDPCAPS_8BIT | 
									   DDPCAPS_ALLOW256,
                                       palEntry,
									   &lpDDPalette, 
									   NULL );
        if ( hresult != DD_OK )
            goto __error;
    } 
	else 
	{
		int s;
		DWORD m;

        // Determine the red, green and blue masks' shift and scale.
        for ( s=0, m=lpTextureFormat.ddpf.dwRBitMask; !(m&1); s++, m>>=1 );
        int red_shift = s;
        int red_scale = 255 / (lpTextureFormat.ddpf.dwRBitMask >> s);
        
        for ( s=0, m=lpTextureFormat.ddpf.dwGBitMask; !(m&1); s++, m>>=1 );
        int green_shift = s;
        int green_scale = 255 / (lpTextureFormat.ddpf.dwGBitMask >> s);

        for ( s=0, m=lpTextureFormat.ddpf.dwBBitMask; !(m&1); s++, m>>=1 );
        int blue_shift = s;
        int blue_scale = 255 / (lpTextureFormat.ddpf.dwBBitMask >> s);


        // Compile the relavent table
        switch ( lpTextureFormat.ddpf.dwRGBBitCount )
		{
            int r,g,b;

            case 32:
                for ( i=0; i<256; i++ ) 
				{
                    r = lpImage->rgbEntry[i].Red / red_scale;
                    g = lpImage->rgbEntry[i].Green / green_scale;
                    b = lpImage->rgbEntry[i].Blue / blue_scale;
                    pal24to24bpp[i] = (r << red_shift) | (g << green_shift) | (b << blue_shift);
                }
                break;

            case 16:
                for ( i=0; i<256; i++ ) 
				{
                    r = lpImage->rgbEntry[i].Red / red_scale;
					g = lpImage->rgbEntry[i].Green / green_scale;
                    b = lpImage->rgbEntry[i].Blue / blue_scale;
                    pal24to16bpp[i] = (r << red_shift) | (g << green_shift) | (b << blue_shift);
                }
                break;

            case 8:
                for ( i=0; i<256; i++ ) 
				{
                    r = lpImage->rgbEntry[i].Red / red_scale;
                    g = lpImage->rgbEntry[i].Green / green_scale;
                    b = lpImage->rgbEntry[i].Blue / blue_scale;
                    pal24to8bpp[i] = (r << red_shift) | (g << green_shift) | (b << blue_shift);
                }
                break;
        }
    }


    // Check that width and height are a PO2 (power-of-two)
    if ( !IsPowerOfTwo(width) || !IsPowerOfTwo(height) ) 
	{
        // One, or both, of width and height are not a PO2. Find nearest fit.
        int bestwidth  = FindNearestPowerOf2(width);
        int bestheight = FindNearestPowerOf2(height);
		
		#define MAX_TEXTURE_SIZE 256 // Largest texture that the 3dfx can take
		BYTE scaletex[MAX_TEXTURE_SIZE][MAX_TEXTURE_SIZE]; // Used for texture scaling
        
		// Now rescale the texture
		ScaleImage( width, height, data, bestwidth, bestheight, scaletex );

		// Alter texture info to new, scaled texture.
		data = &scaletex[0][0];

		width  = bestwidth;
		height = bestheight;
	}


	if ( !*lpTexture )
		texture = *lpTexture = CreateTexture( lpszName );
	else
		texture = CreateTexture( *lpTexture, lpszName );

	if ( !texture )
		goto __error;

	// Now create surface
	DDSURFACEDESC ddsd;
	ZeroMemory( &ddsd, sizeof(DDSURFACEDESC) );

	ddsd.dwSize		= sizeof(DDSURFACEDESC);
	ddsd.dwFlags	= DDSD_CAPS |
					  DDSD_HEIGHT |
					  DDSD_WIDTH |
					  DDSD_PIXELFORMAT;
	ddsd.dwWidth	= width;
    ddsd.dwHeight	= height;

    memcpy( &ddsd.ddpfPixelFormat, &lpTextureFormat.ddpf, sizeof(DDPIXELFORMAT) );

	if ( lpD3DDeviceDesc->bIsHardware )
        ddsd.ddsCaps.dwCaps = DDSCAPS_VIDEOMEMORY | DDSCAPS_TEXTURE;
    else
        ddsd.ddsCaps.dwCaps = DDSCAPS_SYSTEMMEMORY| DDSCAPS_TEXTURE;


    hresult = lpDD->CreateSurface( &ddsd, &texture->lpDDSurface, NULL );
    if ( hresult != DD_OK )
        goto __error;

    // Lock the surface so it can be filled
    ZeroMemory( &ddsd, sizeof(DDSURFACEDESC) );
    ddsd.dwSize = sizeof(DDSURFACEDESC);
    hresult = texture->lpDDSurface->Lock( NULL, &ddsd, 0, NULL );
    if ( hresult != DD_OK )
        goto __error;


    // Fill the texture
    if ( lpTextureFormat.ddpf.dwFlags & DDPF_PALETTEINDEXED8 )
	{
		for ( i=0; i<height; i++ ) 
		{
			LPBYTE lpCP = (LPBYTE)(((LPBYTE)ddsd.lpSurface) + ddsd.lPitch * i);
            for ( j=0; j<width; j++ )
                *lpCP++ = *data++;
		}
    } 
	else 
	{
        switch ( lpTextureFormat.ddpf.dwRGBBitCount )
		{
            case 32:
                for ( i=0; i<height; i++ ) 
				{
					LPDWORD lpLP = (LPDWORD)(((LPBYTE)ddsd.lpSurface) + ddsd.lPitch * i);
                    for ( j=0; j<width; j++ )
                        *lpLP++ = pal24to24bpp[*data++];
                }
                break;

            case 16:
                for ( i=0; i<height; i++ ) 
				{
                    LPWORD lpSP = (LPWORD)(((LPBYTE)ddsd.lpSurface) + ddsd.lPitch * i);
                    for ( j=0; j<width; j++ )
                        *lpSP++ = pal24to16bpp[*data++];
                }
                break;
            
			case 8:
                for ( i=0; i<height; i++ ) 
				{
                    LPBYTE lpCP = (LPBYTE)(((LPBYTE)ddsd.lpSurface) + ddsd.lPitch * i);
                    for ( j=0; j<width; j++ )
                        *lpCP++ = pal24to8bpp[*data++];
                }
                break;
        }
    }

    // Unlock the surface
    texture->lpDDSurface->Unlock( ddsd.lpSurface );


    if ( lpTextureFormat.ddpf.dwFlags & DDPF_PALETTEINDEXED8 )
	{
		// Bind the palette to the texture
        hresult = texture->lpDDSurface->SetPalette( lpDDPalette );
        if ( hresult != DD_OK )
            goto __error;
	}


    // get the texture handle
    LPDIRECT3DTEXTURE2 lpD3DTexture2;
    hresult = texture->lpDDSurface->QueryInterface( IID_IDirect3DTexture2, (LPVOID*)&lpD3DTexture2 );
	if ( hresult != D3D_OK )
        goto __error;

    lpD3DTexture2->GetHandle( lpDev, &texture->Handle );
    lpD3DTexture2->Release();


    // Clean up - check everything to be on the safe side...
	if ( lpDDPalette )
		lpDDPalette->Release();

    return texture;


__error:
    // Clean up - check everything to be on the safe side...
    if ( lpDDPalette )
        lpDDPalette->Release();

    if ( texture )
    {
        if ( texture->lpDDSurface )
        {
            texture->lpDDSurface->Release();
            texture->lpDDSurface = NULL;
        }

        delete texture;

		if ( texture == *lpTexture )
			*lpTexture = NULL;
    }

    return NULL;
}


LPTEXTURE LoadTexture( LPDIRECTDRAW2 lpDD, LPDIRECT3DDEVICE2 lpDev, LPTEXTURE *lpTexture, char *lpszFileName )
{
	LPTEXTURE texture = FindTexture( *lpTexture, lpszFileName );
	if ( texture )
		return texture;

	// compare PCX, GIF, JPG, TGA, CEL, BMP
	char *ptr = strchr( lpszFileName, '.' );

	LPIMAGE lpImage;
	if ( !strcmp( ptr, ".cel" ) || !strcmp( ptr, ".CEL" ) )
		lpImage = LoadCEL( lpszFileName );
	else //if ( !strcmp( ptr, ".pcx" ) || !strcmp( ptr, ".PCX" ) )
	{
		strcpy( ptr, ".PCX" );
		lpImage = LoadPCX( lpszFileName );
	}

	if ( !lpImage )
		return NULL;

	texture = Image2Texture( lpDD, lpDev, lpTexture, lpImage, lpszFileName );

    DestroyImage( lpImage );

	return texture;
}


// Image Func
///////////////////////////////////////////////////////////////////////////////

LPIMAGE CreateImage( WORD Width, WORD Height, BYTE BPP )
{
    LPIMAGE lpImage = new IMAGE;
    if ( !lpImage )
        return NULL;

    lpImage->Width        = Width;
    lpImage->Height       = Height;
	lpImage->BitsPerPixel = BPP;
    lpImage->lpData       = new BYTE [Width*Height];
    if ( !lpImage->lpData )
	{
		delete lpImage;
		return NULL;
	}

	return lpImage;
}

void DestroyImage( LPIMAGE lpImage )
{
    if ( lpImage->lpData )
        delete [] lpImage->lpData;

    delete lpImage;
}


