//////////////////////////////////////////////////////////////////////////////////
// Project Name: [ CDXRM Class Library - CDXRM.lib ]
// Source File:  [ CDXRM Main header file ]
// Author:       [ Danny Farley - danny@jags.co.uk ]
// Revision:     [ 1.0 ]
//////////////////////////////////////////////////////////////////////////////////
#ifndef CDXRM_H
#define CDXRM_H

#pragma warning (disable : 4244)

#include <d3drmwin.h>
#include <CDX.h>

//////////////////////////////////////////////////////////////////////////////////
// GLOBALS
//////////////////////////////////////////////////////////////////////////////////
extern void _Assert (const char *file, int line, const char *msg);

class CDXRMScene;
class CDXRMCamera;
class CDXRMLight;
class CDXRMMatrix;

//////////////////////////////////////////////////////////////////////////////////
// CDXRMEngine Class
//////////////////////////////////////////////////////////////////////////////////
class CDXRMEngine
{
public:
	CDXRMEngine(void);
	~CDXRMEngine(void);

	BOOL CreateFullScreen(CDXScreen* pScreen, void *hWnd,	int Width, int Height, int BPP,
												D3DCOLORMODEL cm = (D3DCOLORMODEL)NULL);
	BOOL CreateWindowed(CDXScreen* pScreen, void *hWnd, int Width, int Height,
											D3DCOLORMODEL cm = (D3DCOLORMODEL)NULL);

	BOOL CreateFrame(LPDIRECT3DRMFRAME pParent, LPDIRECT3DRMFRAME* pFrame);
	BOOL CreateLight(D3DRMLIGHTTYPE type, double r, double g, double b,
                   LPDIRECT3DRMLIGHT* pLight);
	BOOL CreateMeshBuilder(LPDIRECT3DRMMESHBUILDER* pMeshBld);

	HRESULT Update(void);
	HRESULT Tick(double Tick);
	HRESULT SetDither(BOOL Dither);
	HRESULT SetQuality(D3DRMRENDERQUALITY Quality);
	HRESULT SetShades(DWORD Shades);
	HRESULT SetTextureQuality(D3DRMTEXTUREQUALITY TextureQuality);

public:
	LPDIRECT3D m_D3D;
	LPDIRECT3DRM m_D3DRM;
	LPDIRECT3DRMDEVICE m_Device;

	BOOL m_bHardware3D;
	DWORD m_ZBufferBitDepth;
	DWORD m_ZBufferMemType;
	D3DCOLORMODEL m_ColorModel;

	CDXScreen* m_Screen;
	HRESULT rval;
};

//////////////////////////////////////////////////////////////////////////////////
// CDXRMFrame Class
//////////////////////////////////////////////////////////////////////////////////
class CDXRMFrame
{
public:
	CDXRMFrame(void);
	CDXRMFrame(CDXRMEngine* pEngine);
	~CDXRMFrame(void);

	BOOL Create(CDXRMFrame* Parent);
	void AddChild(CDXRMFrame* pChild);
	void AddLight(CDXRMLight* pLight);
	void SetDirection(double dx, double dy, double dz,
										double ux, double uy, double uz, CDXRMFrame* pRef = NULL);
	void GetDirection(double &dx, double &dy, double &dz,
										double &ux, double &uy, double &uz, CDXRMFrame* pRef = NULL);
	void SetPosition(double x, double y, double z, CDXRMFrame* pRef = NULL);
	void GetPosition(double &x, double &y, double &z, CDXRMFrame* pRef = NULL);
	void SetRotation(double x, double y, double z, double t, CDXRMFrame* pRef = NULL);
	void GetRotation(double &x, double &y, double &z, float t, CDXRMFrame* pRef = NULL);
	LPDIRECT3DRMFRAME GetRef(CDXRMFrame* pRef);
	HRESULT Move(double delta);
	HRESULT AddTransform(CDXRMMatrix& m, D3DRMCOMBINETYPE ct = D3DRMCOMBINE_AFTER);
	HRESULT AddRotation(double x, double y, double z, double t,
		                  D3DRMCOMBINETYPE ct = D3DRMCOMBINE_AFTER);

public:
	CDXRMEngine* m_Engine;
	LPDIRECT3DRMFRAME m_Frame;
};

//////////////////////////////////////////////////////////////////////////////////
// CDXRMScene Class
//////////////////////////////////////////////////////////////////////////////////
class CDXRMScene : public CDXRMFrame
{
public:
	CDXRMScene(CDXRMEngine* pEngine);
	~CDXRMScene(void);

	BOOL Create(void);
	HRESULT SetAmbientLight(double r, double g, double b);

public:
	LPDIRECT3DRMLIGHT m_Light;
};

//////////////////////////////////////////////////////////////////////////////////
// CDXRMCamera Class
//////////////////////////////////////////////////////////////////////////////////
class CDXRMCamera : public CDXRMFrame
{
public:
	CDXRMCamera(CDXRMEngine* pEngine);
	~CDXRMCamera(void);

	BOOL Create(CDXRMScene* pScene);
	HRESULT Clear(void);
	HRESULT Render(CDXRMScene* pScene);
	HRESULT ForceUpdate(DWORD X1, DWORD Y1, DWORD X2, DWORD Y2);

public:
	LPDIRECT3DRMVIEWPORT m_View;
};

//////////////////////////////////////////////////////////////////////////////////
// CDXRMShape Class
//////////////////////////////////////////////////////////////////////////////////
class CDXRMShape : public CDXRMFrame
{
public:
	CDXRMShape(CDXRMEngine* pEngine);
	~CDXRMShape(void);

	BOOL Create(D3DVECTOR* pVectors, int nVectors,
							D3DVECTOR* pNormals, int nNormals,
							int* pFaceData, BOOL bAutoGen = FALSE);
	BOOL CreateCube(double x, double y, double z);
	BOOL CreateSphere(double r, int nBands);
	BOOL Load(const char* szFilename);

	HRESULT SetColor(double r, double g, double b);

public:
	LPDIRECT3DRMMESHBUILDER m_MeshBld;
};

//////////////////////////////////////////////////////////////////////////////////
// CDXRMLight Class
//////////////////////////////////////////////////////////////////////////////////
class CDXRMLight : public CDXRMFrame
{
public:
	CDXRMLight(CDXRMEngine* pEngine);
	~CDXRMLight(void);

	BOOL Create(D3DRMLIGHTTYPE Type, double r, double g, double b);
	BOOL CreateAmbient(double r, double g, double b);
	BOOL CreateDirectional(double r, double g, double b);
	BOOL CreatePoint(double r, double g, double b);
	BOOL CreateParallelPoint(double r, double g, double b);
	BOOL CreateSpot(double r, double g, double b);

	void SetColor(double r, double g, double b);

public:
	LPDIRECT3DRMLIGHT m_Light;
};

//////////////////////////////////////////////////////////////////////////////////
// CDXRMMatrix Class
//////////////////////////////////////////////////////////////////////////////////
class CDXRMMatrix
{
public:
	CDXRMMatrix(void);
	CDXRMMatrix(double v00, double v01, double v02, double v03,
		          double v10, double v11, double v12, double v13,
              double v20, double v21, double v22, double v23,
              double v30, double v31, double v32, double v33);

	CDXRMMatrix& operator = (const CDXRMMatrix& r);
	friend CDXRMMatrix operator + (const CDXRMMatrix& a, const CDXRMMatrix& b);
	CDXRMMatrix& operator += (const CDXRMMatrix& r);
	friend CDXRMMatrix operator * (const CDXRMMatrix& a, const CDXRMMatrix& b);
	CDXRMMatrix& operator *= (const CDXRMMatrix& r);

	void Rotate(double rx, double ry, double rz);
	void Translate(double dx, double dy, double dz);
	void Scale(double sx, double sy, double sz);
	void Scale(double s) { Scale(s, s, s); }
	void Create(D3DRMMATRIX4D& rlm);

public:
	double m_00, m_01, m_02, m_03;
	double m_10, m_11, m_12, m_13;
	double m_20, m_21, m_22, m_23;
	double m_30, m_31, m_32, m_33;
};

//////////////////////////////////////////////////////////////////////////////////
// MACROS
//////////////////////////////////////////////////////////////////////////////////
#define ASSERT(x)  if(!(x)) _Assert( __FILE__, __LINE__, #x)
#define RELEASE(x) if(x != NULL) { x->Release(); x = NULL; }

#endif