//////////////////////////////////////////////////////////////////////////////////
// Project Name: [ CDXRM Class Library - CDXRM.lib ]
// Source File:  [ CDXRMEngine Implementation ]
// Author:       [ Danny Farley - danny@jags.co.uk ]
// Revision:     [ 1.0 ]
//////////////////////////////////////////////////////////////////////////////////
#include "CDXRM.h"

typedef struct _D3DDeviceInfo
{
	D3DCOLORMODEL cm;
	D3DDEVICEDESC HWDeviceDesc;
	D3DDEVICEDESC SWDeviceDesc;
	LPGUID        lpHWGuid;
	LPGUID        lpSWGuid;
} D3DDeviceInfo;

typedef struct _VideoMode
{
	DWORD width;
	DWORD height;
	DWORD bpp;
} VideoMode;

static D3DDeviceInfo DeviceInfo;
static VideoMode VidModes[20];
static int CurrMode = 0;
static int NumModes = 0;
static LPGUID D3DDeviceGuid = NULL;
static DWORD GDIMem = 0;
static DDCAPS HALCaps, HELCaps;
static DWORD ModeX, ModeY, ModeBPP;

//////////////////////////////////////////////////////////////////////////////////
// CDXRMEngine Constructor
//////////////////////////////////////////////////////////////////////////////////
CDXRMEngine::CDXRMEngine(void)
{
	m_D3D = NULL;
	m_D3DRM = NULL;
	m_Screen = NULL;
	m_Device = NULL;

	m_bHardware3D = FALSE;
	m_ZBufferBitDepth = 16;
	m_ZBufferMemType = DDSCAPS_SYSTEMMEMORY;
}

//////////////////////////////////////////////////////////////////////////////////
// CDXRMEngine Destructor
//////////////////////////////////////////////////////////////////////////////////
CDXRMEngine::~CDXRMEngine(void)
{
	RELEASE(m_D3D);
	RELEASE(m_D3DRM);
	RELEASE(m_Device);
}

//////////////////////////////////////////////////////////////////////////////////
// EnumDD - enumerates DD drivers, chooses hardware if available
//////////////////////////////////////////////////////////////////////////////////
BOOL FAR PASCAL EnumDD(GUID FAR* lpGUID, LPSTR lpDriverDesc, LPSTR lpDriverName, LPVOID lpContext)
{
	LPDIRECTDRAW lpDD;
	DDCAPS HALCaps, HELCaps;

	// Make sure the guid is valid
	if (lpGUID)
	{
		// Try to create a DirectDraw object
		DirectDrawCreate(lpGUID, &lpDD, NULL);

		// Get the DirectDraw capabilities
		memset(&HALCaps, 0, sizeof(DDCAPS));
		HALCaps.dwSize = sizeof(DDCAPS);

		memset(&HELCaps, 0, sizeof(DDCAPS));
		HELCaps.dwSize = sizeof(DDCAPS);

		lpDD->GetCaps(&HALCaps, &HELCaps);

		// Does this driver have 3D hardware capabilites?
		if (HALCaps.dwCaps & DDCAPS_3D)
		{
			*(LPDIRECTDRAW*)lpContext = lpDD;
			return DDENUMRET_CANCEL;
		}

		*(LPDIRECTDRAW*)lpContext = NULL;
		lpDD->Release();
	}
	return DDENUMRET_OK;
}

//////////////////////////////////////////////////////////////////////////////////
// EnumModes - enumerates video modes
//////////////////////////////////////////////////////////////////////////////////
HRESULT CALLBACK EnumModes(LPDDSURFACEDESC pddsd, LPVOID Context)
{
	// Are the mode and the device compatible
	if(DeviceInfo.lpHWGuid)
	{
		// Make sure there is enough video ram to support this mode
		DWORD dwBitDepthMultiplier = pddsd->ddpfPixelFormat.dwRGBBitCount / 8;
		DWORD dwVidRamNeeded = ((pddsd->dwWidth * pddsd->dwHeight) * dwBitDepthMultiplier) * 3;

		if(dwVidRamNeeded > (HALCaps.dwVidMemFree + GDIMem)) return DDENUMRET_OK;

		// Make sure the device can render at a given bit depth
		switch (pddsd->ddpfPixelFormat.dwRGBBitCount)
		{
		case 8 :
			if (!(DeviceInfo.HWDeviceDesc.dwDeviceRenderBitDepth & DDBD_8)) return DDENUMRET_OK;
			break;
		
		case 16 :
			if (!(DeviceInfo.HWDeviceDesc.dwDeviceRenderBitDepth & DDBD_16)) return DDENUMRET_OK;
			break;
		
		case 24 :
			if (!(DeviceInfo.HWDeviceDesc.dwDeviceRenderBitDepth & DDBD_24)) return DDENUMRET_OK;
			break;
		
		case 32 :
			if (!(DeviceInfo.HWDeviceDesc.dwDeviceRenderBitDepth & DDBD_32)) return DDENUMRET_OK;
			break;

			// If we have hardware, start up in requested mode if possible
			if((pddsd->dwWidth == ModeX) && (pddsd->dwHeight == ModeY) &&
				 (pddsd->ddpfPixelFormat.dwRGBBitCount == ModeBPP))
				 CurrMode = NumModes;

			// At least try to get the size right
			else if ((pddsd->dwWidth == ModeX) && (pddsd->dwHeight == ModeY))
							 CurrMode = NumModes;
		}
	}

	// Record the video mode information
	VidModes[NumModes].width  = pddsd->dwWidth;
	VidModes[NumModes].height = pddsd->dwHeight;
	VidModes[NumModes].bpp    = pddsd->ddpfPixelFormat.dwRGBBitCount;
	
	NumModes++;
	
	return DDENUMRET_OK;
}

//////////////////////////////////////////////////////////////////////////////////
// EnumD3D - enumerates D3D drivers, chooses hardware if available
//////////////////////////////////////////////////////////////////////////////////
HRESULT WINAPI EnumD3D(LPGUID lpGuid, LPSTR lpDeviceDescription, LPSTR lpDeviceName,
											 LPD3DDEVICEDESC lpHWDesc, LPD3DDEVICEDESC lpHELDesc, LPVOID lpContext)
{
	static BOOL bFoundHW = FALSE;

	// No need to enumerate if we already found the device that supports what we want
	if(bFoundHW) return D3DENUMRET_OK;

	D3DDeviceInfo* pInfo = (D3DDeviceInfo *)lpContext;

	// Is this a hardware device?
	if(lpHWDesc->dcmColorModel & pInfo->cm)
	{
		// Make sure the driver has Z buffering and texturing capabilities
		if(((lpHWDesc->dwDeviceZBufferBitDepth & DDBD_16) ||
        (lpHWDesc->dwDeviceZBufferBitDepth & DDBD_24) ||
        (lpHWDesc->dwDeviceZBufferBitDepth & DDBD_32)) &&
       ((lpHWDesc->dwDevCaps & D3DDEVCAPS_TEXTURESYSTEMMEMORY) ||
			  (lpHWDesc->dwDevCaps & D3DDEVCAPS_TEXTUREVIDEOMEMORY)))
		{
			// Record the HAL description for later use
			memcpy(&pInfo->HWDeviceDesc, lpHWDesc, sizeof(D3DDEVICEDESC));

			// Record the guid for later use
			pInfo->lpHWGuid = lpGuid;

			// No need to keep looking for any more devices
			bFoundHW = TRUE;
		}
		return D3DENUMRET_OK;
	}

	// Is this a software device?
	if (lpHELDesc->dcmColorModel & pInfo->cm)
	{
		// Record the HEL description for later use
		memcpy(&pInfo->SWDeviceDesc, lpHELDesc, sizeof(D3DDEVICEDESC));

		// Record the guid for later use
		pInfo->lpSWGuid = lpGuid;
		D3DDeviceGuid = lpGuid;
	}
	return D3DENUMRET_OK;
}

//////////////////////////////////////////////////////////////////////////////////
// CDXRMEngine CreateFullScreen - creates a full screen application
//////////////////////////////////////////////////////////////////////////////////
BOOL CDXRMEngine::CreateFullScreen(CDXScreen* pScreen, void *hWnd, int Width,
																	 int Height, int BPP,	D3DCOLORMODEL cm)
{
	DWORD dwFlags;
	DDSURFACEDESC ddsd;
	DDSCAPS ddscaps;

	m_Screen = pScreen;
	if(m_Screen == NULL) return FALSE;

	m_Screen->m_bFullScreen = TRUE;
	m_Screen->m_dwPixelWidth = Width;
	m_Screen->m_dwPixelHeight = Height;
	m_Screen->m_BPP = BPP;
	m_Screen->m_hWnd = hWnd;
	if(cm) m_ColorModel = cm;

	ModeX = Width;
	ModeY = Height;
	ModeBPP = BPP;

	dwFlags = DDSCL_EXCLUSIVE | DDSCL_FULLSCREEN | DDSCL_ALLOWREBOOT | DDSCL_ALLOWMODEX;

	// Enumerate available DD drivers
	rval = DirectDrawEnumerate(EnumDD, &m_Screen->m_lpDD);
	ASSERT(rval == DD_OK);

	// If no hardware found then use a software driver
	if(!m_Screen->m_lpDD)
	{
		rval = DirectDrawCreate(NULL, &m_Screen->m_lpDD, NULL);
		ASSERT(rval == DD_OK);
	}

	// Zero out caps structure
  memset(&HALCaps, 0, sizeof(DDCAPS));
	HALCaps.dwSize = sizeof(DDCAPS);
	memset(&HELCaps, 0, sizeof(DDCAPS));
	HELCaps.dwSize = sizeof(DDCAPS);

	// How much video memory is Windows using?
	memset(&ddsd, 0, sizeof(ddsd));
	ddsd.dwSize = sizeof(ddsd);

	rval = m_Screen->m_lpDD->GetDisplayMode(&ddsd);
	ASSERT(rval == DD_OK);

	GDIMem = ddsd.lPitch * ddsd.dwHeight * (ddsd.ddpfPixelFormat.dwRGBBitCount / 8);

	// Get hardware capabilities
	rval = m_Screen->m_lpDD->GetCaps(&HALCaps, &HELCaps);
	ASSERT(rval == DD_OK);

	// Are we using hardware?
	m_bHardware3D = HALCaps.dwCaps & DDCAPS_3D;

	memset(&DeviceInfo, 0, sizeof(D3DDeviceInfo));

	// If a color model is specified try to get it
	if(m_ColorModel) DeviceInfo.cm = m_ColorModel;

	// Otherwise get RGB if hardware, MONO if software
	else DeviceInfo.cm = m_bHardware3D ? D3DCOLOR_RGB : D3DCOLOR_MONO;

	// Create Direct3D object
	rval = m_Screen->m_lpDD->QueryInterface(IID_IDirect3D, (LPVOID *)&m_D3D);
	ASSERT(rval == DD_OK);

	// Enumerate the drivers
	m_D3D->EnumDevices(EnumD3D, &DeviceInfo);

	// If the driver we found is software and RGB, reenumerate for MONO
	if(DeviceInfo.lpSWGuid && DeviceInfo.cm == D3DCOLOR_RGB)
	{
		// Unless we said we wanted RGB no matter what
		if(m_ColorModel != D3DCOLOR_RGB)
		{
			DeviceInfo.cm = D3DCOLOR_MONO;
			m_D3D->EnumDevices(EnumD3D, &DeviceInfo);
		}
	}

	// Now do some final cleanup with the driver we selected
	if(DeviceInfo.lpHWGuid)
	{
		// We have a hardware driver

		// Make sure to use a video memory based ZBuffer
		m_ZBufferMemType = DDSCAPS_VIDEOMEMORY;

		// Use 16 bit Z buffering if possible, higher if not
		if(DeviceInfo.HWDeviceDesc.dwDeviceZBufferBitDepth & DDBD_16)
			m_ZBufferBitDepth = 16;

		else if(DeviceInfo.HWDeviceDesc.dwDeviceZBufferBitDepth & DDBD_24)
			m_ZBufferBitDepth = 24;

    else if(DeviceInfo.HWDeviceDesc.dwDeviceZBufferBitDepth & DDBD_32)
			m_ZBufferBitDepth = 32;

		else m_ZBufferBitDepth = 0;

    // Use Hardware device
		D3DDeviceGuid = DeviceInfo.lpHWGuid;
	}
	else
	{
		// We have a software driver

		// Use a system memory based ZBuffer
		m_ZBufferMemType = DDSCAPS_SYSTEMMEMORY;

		// And force the bit depth to 16
		m_ZBufferBitDepth = 16;

		// Default to the software device
		D3DDeviceGuid = DeviceInfo.lpSWGuid;

		// Indicate we are not using hardware
		m_bHardware3D = FALSE;
	}
	
	// Set our final color model
	m_ColorModel = DeviceInfo.cm;

	// Create Direct3DRM object
	rval = Direct3DRMCreate(&m_D3DRM);
	ASSERT(rval == D3DRM_OK);

	// Set some defualts
	m_D3DRM->SetDefaultTextureColors(16);
	m_D3DRM->SetDefaultTextureShades(16);

	rval = m_Screen->m_lpDD->SetCooperativeLevel(hWnd, dwFlags);
	ASSERT(rval == DD_OK);

	// Enumerate all video modes
	rval = m_Screen->m_lpDD->EnumDisplayModes(0, NULL, NULL, EnumModes);
	ASSERT(rval == DD_OK);

 // Try to get the specified video mode
 for(int i = 0; i < NumModes; i++)
 {
	 if(VidModes[i].width == m_Screen->m_dwPixelWidth && 
			VidModes[i].height == m_Screen->m_dwPixelHeight &&
			VidModes[i].bpp == m_Screen->m_BPP)
			CurrMode = i;
 }

	m_Screen->m_dwPixelWidth = VidModes[CurrMode].width;
	m_Screen->m_dwPixelHeight = VidModes[CurrMode].height;
	m_Screen->m_BPP = VidModes[CurrMode].bpp;

	rval = m_Screen->m_lpDD->SetDisplayMode(m_Screen->GetWidth(), m_Screen->GetHeight(), m_Screen->m_BPP);
	ASSERT(rval == DD_OK);

	ddsd.dwSize = sizeof(ddsd);
	ddsd.dwFlags = DDSD_CAPS | DDSD_BACKBUFFERCOUNT;
	ddsd.ddsCaps.dwCaps = DDSCAPS_PRIMARYSURFACE |
												DDSCAPS_FLIP |
												DDSCAPS_COMPLEX | DDSCAPS_3DDEVICE;
	ddsd.dwBackBufferCount = 1;

	m_Screen->m_lpDDSFront = new CDXSurface();
	m_Screen->m_lpDDSBack = new CDXSurface();

	rval = m_Screen->m_lpDD->CreateSurface(&ddsd, &m_Screen->m_lpDDSFront->m_lpDDS, NULL);
	ASSERT(rval == DD_OK);

	ddscaps.dwCaps = DDSCAPS_BACKBUFFER;

	rval = m_Screen->m_lpDDSFront->m_lpDDS->GetAttachedSurface(&ddscaps, &m_Screen->m_lpDDSBack->m_lpDDS);
	ASSERT(rval == DD_OK);

	// Only create a Z buffer if m_ZBufferBitDepth > 0
	if(m_ZBufferBitDepth)
	{
		// Create a Z buffer
		memset(&ddsd, sizeof(DDSURFACEDESC), 0);
		ddsd.dwSize             = sizeof(DDSURFACEDESC);
		ddsd.dwFlags            = DDSD_WIDTH | DDSD_HEIGHT | DDSD_CAPS | DDSD_ZBUFFERBITDEPTH;
		ddsd.dwWidth            = m_Screen->m_dwPixelWidth;
		ddsd.dwHeight           = m_Screen->m_dwPixelHeight;
		ddsd.ddsCaps.dwCaps     = DDSCAPS_ZBUFFER | m_ZBufferMemType;
		ddsd.dwZBufferBitDepth  = m_ZBufferBitDepth;

		// Create the Z buffer surface
		rval = m_Screen->m_lpDD->CreateSurface(&ddsd, &m_Screen->m_ZBuffer, NULL);
		ASSERT(rval == DD_OK);

		// Attach Z buffer to the back buffer
		m_Screen->m_lpDDSBack->m_lpDDS->AddAttachedSurface(m_Screen->m_ZBuffer);
	}

	if(m_Screen->m_BPP == 8)
	{
		BYTE Pal[768];
		PALETTEENTRY Pe[768];

		// Setup the palette
		Pe[0].peFlags		= D3DPAL_READONLY;
		Pe[254].peFlags = D3DPAL_READONLY;
		Pe[255].peFlags = D3DPAL_READONLY;

		// These entries are free for Direct3D to change
		for(int i = 1; i < 254; i++)
		{
			Pe[i].peRed		= Pal[i * 3];
			Pe[i].peGreen = Pal[(i * 3) + 1];
			Pe[i].peBlue	= Pal[(i * 3) + 2];
			Pe[i].peFlags = D3DPAL_FREE;
		}

		m_Screen->m_lpDD->CreatePalette(DDPCAPS_8BIT | DDPCAPS_INITIALIZE, Pe, &m_Screen->m_lpDDPalette, NULL);
		m_Screen->m_lpDDSFront->m_lpDDS->SetPalette(m_Screen->m_lpDDPalette);
		m_Screen->m_lpDDSBack->m_lpDDS->SetPalette(m_Screen->m_lpDDPalette);
	}

	// Create the RM device
	rval = m_D3DRM->CreateDeviceFromSurface(D3DDeviceGuid, m_Screen->m_lpDD,
																						m_Screen->m_lpDDSBack->m_lpDDS,
																						&m_Device );
	ASSERT(rval == D3DRM_OK);

	// Set the buffer count to 2
	rval = m_Device->SetBufferCount(2);
	ASSERT(rval == D3DRM_OK);

	// Set up some defaults
	m_Device->SetQuality(D3DRMRENDER_GOURAUD);
	m_Device->SetDither(FALSE);
	if(m_bHardware3D) m_Device->SetTextureQuality(D3DRMTEXTURE_LINEAR);
	else m_Device->SetTextureQuality(D3DRMTEXTURE_NEAREST);
	m_Device->SetShades(8);

	return TRUE;
}

//////////////////////////////////////////////////////////////////////////////////
// CDXRMEngine CreateWindowed - creates a windowed application
//////////////////////////////////////////////////////////////////////////////////
BOOL CDXRMEngine::CreateWindowed(CDXScreen* pScreen, void *hWnd, int Width,
																 int Height, D3DCOLORMODEL cm)
{
	DWORD dwFlags;
	DDSURFACEDESC ddsd;
	HDC hDC;

	m_Screen = pScreen;
	if(m_Screen == NULL) return FALSE;

	m_Screen->m_bFullScreen = FALSE;
	m_Screen->m_dwPixelWidth = Width;
	m_Screen->m_dwPixelHeight = Height;
	m_Screen->m_hWnd = hWnd;
	if(cm) m_ColorModel = cm;

	// Get the current Windows BPP
	hDC = GetDC(NULL);
	m_Screen->m_BPP = GetDeviceCaps(hDC, PLANES) * GetDeviceCaps(hDC, BITSPIXEL);
	ReleaseDC(NULL, hDC);

	dwFlags = DDSCL_NORMAL;

	//
	// Enumerate available DD drivers
	// rval = DirectDrawEnumerate(EnumDD, &m_Screen->m_lpDD);
	// ASSERT(rval == DD_OK);
	//

	// If no hardware found then use a software driver
	if(!m_Screen->m_lpDD)
	{
		rval = DirectDrawCreate(NULL, &m_Screen->m_lpDD, NULL);
		ASSERT(rval == DD_OK);
	}

	// Zero out caps structure
  memset(&HALCaps, 0, sizeof(DDCAPS));
	HALCaps.dwSize = sizeof(DDCAPS);
	memset(&HELCaps, 0, sizeof(DDCAPS));
	HELCaps.dwSize = sizeof(DDCAPS);

	// Get hardware capabilities
	rval = m_Screen->m_lpDD->GetCaps(&HALCaps, &HELCaps);
	ASSERT(rval == DD_OK);

	// Are we using hardware?
	m_bHardware3D = HALCaps.dwCaps & DDCAPS_3D;

	memset(&DeviceInfo, 0, sizeof(D3DDeviceInfo));

	// If a color model is specified try to get it
	if(m_ColorModel) DeviceInfo.cm = m_ColorModel;

	// Otherwise get RGB if hardware, MONO if software
	else DeviceInfo.cm = m_bHardware3D ? D3DCOLOR_RGB : D3DCOLOR_MONO;

	// Create Direct3D object
	rval = m_Screen->m_lpDD->QueryInterface(IID_IDirect3D, (LPVOID *)&m_D3D);
	ASSERT(rval == DD_OK);

	//
	// Enumerate the drivers
	// m_D3D->EnumDevices(EnumD3D, &DeviceInfo);
	//

	// If the driver we found is software and RGB, reenumerate for MONO
	if(DeviceInfo.lpSWGuid && DeviceInfo.cm == D3DCOLOR_RGB)
	{
		// Unless we said we wanted RGB no matter what
		if(m_ColorModel != D3DCOLOR_RGB)
		{
			DeviceInfo.cm = D3DCOLOR_MONO;
			m_D3D->EnumDevices(EnumD3D, &DeviceInfo);
		}
	}

	// Now do some final cleanup with the driver we selected
	if(DeviceInfo.lpHWGuid)
	{
		// We have a hardware driver

		// Make sure to use a video memory based ZBuffer
		m_ZBufferMemType = DDSCAPS_VIDEOMEMORY;

		// Use 16 bit Z buffering if possible, higher if not
		if(DeviceInfo.HWDeviceDesc.dwDeviceZBufferBitDepth & DDBD_16)
			m_ZBufferBitDepth = 16;

		else if(DeviceInfo.HWDeviceDesc.dwDeviceZBufferBitDepth & DDBD_24)
			m_ZBufferBitDepth = 24;

    else if(DeviceInfo.HWDeviceDesc.dwDeviceZBufferBitDepth & DDBD_32)
			m_ZBufferBitDepth = 32;

		else m_ZBufferBitDepth = 0;

    // Use Hardware device
		D3DDeviceGuid = DeviceInfo.lpHWGuid;
	}
	else
	{
		// We have a software driver

		// Use a system memory based ZBuffer
		m_ZBufferMemType = DDSCAPS_SYSTEMMEMORY;

		// And force the bit depth to 16
		m_ZBufferBitDepth = 16;

		// Default to the software device
		D3DDeviceGuid = DeviceInfo.lpSWGuid;

		// Indicate we are not using hardware
		m_bHardware3D = FALSE;
	}
	
	// Set our final color model
	m_ColorModel = DeviceInfo.cm;

	// Create Direct3DRM object
	rval = Direct3DRMCreate(&m_D3DRM);
	ASSERT(rval == D3DRM_OK);

	// Set some defualts
	m_D3DRM->SetDefaultTextureColors(16);
	m_D3DRM->SetDefaultTextureShades(16);

	// Allow ModeX so we detect these modes when enumerating
	// rval = m_Screen->m_lpDD->SetCooperativeLevel(hWnd, DDSCL_EXCLUSIVE | DDSCL_FULLSCREEN | DDSCL_ALLOWMODEX | DDSCL_NOWINDOWCHANGES);
	// ASSERT(rval == DD_OK);

	//
	// Enumerate all video modes
	// rval = m_Screen->m_lpDD->EnumDisplayModes(0, NULL, NULL, EnumModes);
	// ASSERT(rval == DD_OK);
	//

	rval = m_Screen->m_lpDD->SetCooperativeLevel(hWnd, dwFlags);
	ASSERT(rval == DD_OK);

	ddsd.dwSize = sizeof(ddsd);
	ddsd.dwFlags = DDSD_CAPS;
	ddsd.ddsCaps.dwCaps = DDSCAPS_PRIMARYSURFACE;

	m_Screen->m_lpDDSFront = new CDXSurface();
	m_Screen->m_lpDDSBack = new CDXSurface();

	rval = m_Screen->m_lpDD->CreateSurface(&ddsd, &m_Screen->m_lpDDSFront->m_lpDDS, NULL);
	ASSERT(rval == DD_OK);

	ddsd.dwFlags = DDSD_CAPS | DDSD_WIDTH | DDSD_HEIGHT;    
	ddsd.ddsCaps.dwCaps = DDSCAPS_OFFSCREENPLAIN | DDSCAPS_3DDEVICE;
	ddsd.dwWidth = Width;
	ddsd.dwHeight = Height;

	rval = m_Screen->m_lpDD->CreateSurface( &ddsd, &m_Screen->m_lpDDSBack->m_lpDDS, NULL );
	ASSERT(rval == DD_OK);

	// Create the window clipper
	rval = m_Screen->m_lpDD->CreateClipper(0, &m_Screen->m_lpClipper, NULL);
	ASSERT(rval == DD_OK);

	rval = m_Screen->m_lpClipper->SetHWnd(0, hWnd);
	ASSERT(rval == DD_OK);

	rval = m_Screen->m_lpDDSFront->m_lpDDS->SetClipper(m_Screen->m_lpClipper);
	ASSERT(rval == DD_OK);

	// Only create a Z buffer if m_ZBufferBitDepth > 0
	if(m_ZBufferBitDepth)
	{
		// Create a Z buffer
		memset(&ddsd, sizeof(DDSURFACEDESC), 0);
		ddsd.dwSize             = sizeof(DDSURFACEDESC);
		ddsd.dwFlags            = DDSD_WIDTH | DDSD_HEIGHT | DDSD_CAPS | DDSD_ZBUFFERBITDEPTH;
		ddsd.dwWidth            = m_Screen->m_dwPixelWidth;
		ddsd.dwHeight           = m_Screen->m_dwPixelHeight;
		ddsd.ddsCaps.dwCaps     = DDSCAPS_ZBUFFER | m_ZBufferMemType;
		ddsd.dwZBufferBitDepth  = m_ZBufferBitDepth;

		// Create the Z buffer surface
		rval = m_Screen->m_lpDD->CreateSurface(&ddsd, &m_Screen->m_ZBuffer, NULL);
		ASSERT(rval == DD_OK);

		// Attach Z buffer to the back buffer
		m_Screen->m_lpDDSBack->m_lpDDS->AddAttachedSurface(m_Screen->m_ZBuffer);
	}

	if(m_Screen->m_BPP == 8)
	{
		BYTE Pal[768];
		PALETTEENTRY Pe[768];

		// Setup the palette
		Pe[0].peFlags		= D3DPAL_READONLY;
		Pe[254].peFlags = D3DPAL_READONLY;
		Pe[255].peFlags = D3DPAL_READONLY;

		// These entries are free for Direct3D to change
		for(int i = 1; i < 254; i++)
		{
			Pe[i].peRed		= Pal[i * 3];
			Pe[i].peGreen = Pal[(i * 3) + 1];
			Pe[i].peBlue	= Pal[(i * 3) + 2];
			Pe[i].peFlags = D3DPAL_FREE;
		}

		m_Screen->m_lpDD->CreatePalette(DDPCAPS_8BIT | DDPCAPS_INITIALIZE, Pe, &m_Screen->m_lpDDPalette, NULL);
		m_Screen->m_lpDDSFront->m_lpDDS->SetPalette(m_Screen->m_lpDDPalette);
		m_Screen->m_lpDDSBack->m_lpDDS->SetPalette(m_Screen->m_lpDDPalette);
	}

	// Create the RM device
	rval = m_D3DRM->CreateDeviceFromClipper(m_Screen->m_lpClipper, NULL,
																					m_Screen->m_dwPixelWidth,
																					m_Screen->m_dwPixelHeight,
																					&m_Device);
	ASSERT(rval == D3DRM_OK);

	// Set up some defaults
	m_Device->SetQuality(D3DRMRENDER_GOURAUD);
	m_Device->SetDither(FALSE);
	m_Device->SetTextureQuality(D3DRMTEXTURE_NEAREST);
	m_Device->SetShades(8);

	return TRUE;
}

//////////////////////////////////////////////////////////////////////////////////
// CDXRMEngine Update
//////////////////////////////////////////////////////////////////////////////////
HRESULT CDXRMEngine::Update(void)
{
	return m_Device->Update();
}

//////////////////////////////////////////////////////////////////////////////////
// CDXRMEngine CreateFrame
//////////////////////////////////////////////////////////////////////////////////
BOOL CDXRMEngine::CreateFrame(LPDIRECT3DRMFRAME pParent,
															LPDIRECT3DRMFRAME* pFrame)
{
	rval = m_D3DRM->CreateFrame(pParent, pFrame);
	if(rval != D3DRM_OK) return FALSE;

	return TRUE;
}

//////////////////////////////////////////////////////////////////////////////////
// CDXRMEngine CreateLight
//////////////////////////////////////////////////////////////////////////////////
BOOL CDXRMEngine::CreateLight(D3DRMLIGHTTYPE type, double r, double g, double b,
                              LPDIRECT3DRMLIGHT* pLight)
{
	rval = m_D3DRM->CreateLightRGB(type, D3DVAL(r), D3DVAL(g), D3DVAL(b), pLight);
	if(rval != D3DRM_OK) return FALSE;

	return TRUE;
}

//////////////////////////////////////////////////////////////////////////////////
// CDXRMEngine CreateMeshBuilder
//////////////////////////////////////////////////////////////////////////////////
BOOL CDXRMEngine::CreateMeshBuilder(LPDIRECT3DRMMESHBUILDER* pMeshBld)
{
	rval = m_D3DRM->CreateMeshBuilder(pMeshBld);
	if(rval != D3DRM_OK) return FALSE;

	return TRUE;
}

//////////////////////////////////////////////////////////////////////////////////
// CDXRMEngine SetDither
//////////////////////////////////////////////////////////////////////////////////
HRESULT CDXRMEngine::SetDither(BOOL Dither)
{
	return m_Device->SetDither(Dither);
}

//////////////////////////////////////////////////////////////////////////////////
// CDXRMEngine SetQuality
//////////////////////////////////////////////////////////////////////////////////
HRESULT CDXRMEngine::SetQuality(D3DRMRENDERQUALITY Quality)
{
	return m_Device->SetQuality(Quality);
}

//////////////////////////////////////////////////////////////////////////////////
// CDXRMEngine SetShades
//////////////////////////////////////////////////////////////////////////////////
HRESULT CDXRMEngine::SetShades(DWORD Shades)
{
	return m_Device->SetShades(Shades);
}

//////////////////////////////////////////////////////////////////////////////////
// CDXRMEngine SetTextureQuality
//////////////////////////////////////////////////////////////////////////////////
HRESULT CDXRMEngine::SetTextureQuality(D3DRMTEXTUREQUALITY TextureQuality)
{
	return m_Device->SetTextureQuality(TextureQuality);
}

//////////////////////////////////////////////////////////////////////////////////
// CDXRMEngine Tick
//////////////////////////////////////////////////////////////////////////////////
HRESULT CDXRMEngine::Tick(double Tick)
{
	return m_D3DRM->Tick(Tick);
}

//////////////////////////////////////////////////////////////////////////////////
// _Assert - internal error handling
//////////////////////////////////////////////////////////////////////////////////
static void _Assert(const char *file, int line, const char *msg)
{
	int result;	
	static char buf[1024];

	sprintf(buf, "Assertion Failed %s at %d:  %s", file, line, msg);

	result = MessageBox(NULL, buf, "Assertion Failure", MB_OKCANCEL | 
										  MB_APPLMODAL | MB_ICONERROR);

	if(result == IDCANCEL) PostQuitMessage(0);
}
