/**************************************************************************
  ReadASE.cpp

  Description:
    Routines to fully read an ASE file and store it in a generic
    object format (GenObject3d, defined in [ReadASE.h])

      File Types Supported          Ext    Read  Write
      ----------------------------  ---    ----  -----
      3D Studio ASCII Scene Export  ASE    Yes   No

  --------- Copyright (c) 1998 Daniel Dunham | crow@tconl.com ---------

**************************************************************************/

// ################################################### Includes
#include <stdio.h>  // for FILE interface
#include <stdlib.h> // for atoi(), malloc()
#include <string.h> // for strcmp

#include "ReadASE.h" // GenObject3d format, and prototypes

// ################################################### Prototypes
// ---------- ASE Prototypes ----------
void readln(FILE *infile,char *line);
void RA_GetNode(char *curline, char *node);
void RA_GetArg(char *curline, char *arg);
void RA_CmdInterpret(GenObject3d *obj,FILE *inp,char *node, char *arg);
void RA_GO_ReadGeomObject(GenObject3d *obj,FILE *inp, char *node, char *arg);
void RA_GO_ReadMesh(GenObject3d *obj,FILE *inp, char *node, char *arg);
void RA_GO_ReadMeshVertexList(GenObject3d *obj,FILE *inp, char *node, char *arg);
void RA_GO_ReadMeshFaceList(GenObject3d *obj,FILE *inp, char *node, char *arg);
void RA_GO_ReadMeshTexVertexList(GenObject3d *obj,FILE *inp, char *node, char *arg);
void RA_GO_ReadMeshTexFaceList(GenObject3d *obj,FILE *inp, char *node, char *arg);
bool RA_IsSection(char *arg);
void RA_SkipSection(FILE *inp);

bool ReadASE(char *filename,GenObject3d *obj); // Our only visible function

// ################################################### Variables
float s[4096], t[4096]; // temporary texture coordinate holders

// ################################################### Implementation of ReadASE
// readln 
//    Purpose: Read a single line from a file
//    Params : infile      File to read from
//             line        Buffer in which to store line          
void readln(FILE *infile,char *line)
{
    fgets(line,255,infile);
}

// RA_GetNode
//    Purpose: Read the node from the current line
//    Params : curline     Current line from file
//             node        Result of node extraction
//    Example: if [curline] == *3DSMAX_ASCIIEXPORT 200 then
//                [node]    =  *3DSMAX_ASCIIEXPORT
void RA_GetNode(char *curline, char *node)
{
    char *p, *q;

      p = (char *)curline;
      q = (char *)node;

      // Skip any leading whitespace
      while( (*p == 32) || (*p == 9) ) {
        p++;
      } // Should now be at asterisk before node

      // Read until encountering space
      while( (*p != 32) && (*p != 9) ) {
        *q++ = *p++;
      }
      *q = 0; // Zero-terminate string
}

// RA_GetArg 
//    Purpose: Extract arguments from current line, stripping quotes if necessary
//    Params : curline     Current line from file
//             arg         Result of argument extraction
//    Example: if [curline] == *3DSMAX_ASCIIEXPORT 200 then
//                [arg]     =  200
void RA_GetArg(char *curline, char *arg)
{
  char *p, *q;

    p = (char *)curline;
    q = (char *)arg;

    // Skip any leading whitespace
    while( (*p == 32) || (*p == 9) ) {
      p++;
    } // Should now be at asterisk before node

    // Skip over node
    while( (*p != 32) && (*p != 9) ) {
      p++;
    }

    // Skip any more leading whitespace
    while( (*p == 32) || (*p == 9) ) {
      p++;
    } // Should now be at beginning of argument

    if(*p == '"') { // If we are dealing with a quoted argument
      p++;
      while( (*p != '"') ) { // go until quotes
        *q++ = *p++;
      }
    }
    else {
      while( (*p != '\n') && (*p != '\r') ) { // Else go to EOL
        if(*p == 9) {
          *q++ = 32;
          p++;
        }
        else
          *q++ = *p++;
      }
    }

    *q = 0; // Zero-terminate string
}

// RA_CmdInterpret
//    Purpose: Interpret current line of file and take appropriate action
//    Params : obj         GenObject3d object to change
//             inp         Input file
//             node        Current line's node
//             arg         Current line's arguments
void RA_CmdInterpret(GenObject3d *obj,FILE *inp,char *node, char *arg)
{
    // Line format: *3DSMAX_ASCIIEXPORT [integer version]
    // Description: Lists the current version number of the 
    //              application that wrote this ASE file
    if(strcmp(node,"*3DSMAX_ASCIIEXPORT") == 0) {
#ifdef READASE_PRINTSTATUS
      printf("Version: %s\n",arg);
#endif
      return;
    }
    // Line format: *COMMENT [string comment]
    // Description: Optional comment inserted by the 
    //              application that wrote this ASE file
    if(strcmp(node,"*COMMENT") == 0) {
#ifdef READASE_PRINTSTATUS
      printf("Comment: %s\n",arg);
#endif
      return;
    }
    // Line format: *GEOMOBJECT {
    // Description: Beginning of a Geometric Object specification
    if(strcmp(node,"*GEOMOBJECT") == 0) {
#ifdef READASE_PRINTSTATUS
      printf("Reading GeomObject...\n");
#endif
      RA_GO_ReadGeomObject(obj,inp,node,arg); // Spawn reader of GEOMOBJECT node
    }

    // If we don't interpret the section, then we need to jump over it
    if(RA_IsSection(arg))
      RA_SkipSection(inp);
}

// RA_CmdInterpret
//    Purpose: Interpret current line of file and take appropriate action
//    Params : obj         GenObject3d object to change
//             inp         Input file
//             node        Current line's node
//             arg         Current line's arguments
void RA_GO_ReadGeomObject(GenObject3d *obj,FILE *inp, char *node, char *arg)
{
  char curline[256];

    while(1) {
      readln(inp,curline);
      if(curline[0] == '}')
        break;
      if(feof(inp))
        break;
      RA_GetNode(curline,node);
      RA_GetArg(curline,arg);

      // Line format: *NODE_NAME [string name]
      // Description: Name of the geometric object
      if(strcmp(node,"*NODE_NAME") == 0) {
#ifdef READASE_PRINTSTATUS
        printf("GO-> Name : %s\n",arg);
#endif
        continue;
      }

      // Line format: *MESH {
      // Description: Start of mesh section
      if(strcmp(node,"*MESH") == 0) {
#ifdef READASE_PRINTSTATUS
        printf("GO-> Reading Mesh...\n");
#endif
        RA_GO_ReadMesh(obj,inp,node,arg); // Read Mesh section
        continue;
      }

      // If we don't interpret the section, then we need to jump over it
      if(RA_IsSection(curline))
        RA_SkipSection(inp);
    }
}

// RA_GO_ReadMesh
//    Purpose: Read a *MESH tree
//    Params : obj         GenObject3d object to change
//             inp         Input file
//             node        Current line's node
//             arg         Current line's arguments
void RA_GO_ReadMesh(GenObject3d *obj,FILE *inp, char *node, char *arg)
{
  char curline[256];
  int  i;

    while(1) {
      readln(inp,curline);
      RA_GetNode(curline,node);
      if(node[0] == '}')
        break;
      if(feof(inp))
        break;
      RA_GetArg(curline,arg);

      // Line format: *MESH_NUMBVERTEX [integer #]
      // Description: Number of vertices in this mesh
      if(strcmp(node,"*MESH_NUMVERTEX") == 0) {
#ifdef READASE_PRINTSTATUS
        printf("GO-MESH-> NumVerts: %s\n",arg);
#endif

        // Allocate memory for the vertices
        obj->NumVertices = atoi(arg);
        obj->VertexList = (GenPoint3d *)malloc(sizeof(GenPoint3d)*obj->NumVertices);
        for(i=0;i<obj->NumVertices;i++) {
          // Default the vertices
          obj->VertexList[i].x = 0.f;
          obj->VertexList[i].y = 0.f;
          obj->VertexList[i].z = 0.f;
        }
        continue;
      }

      // Line format: *MESH_NUMFACES [integer #]
      // Description: Number of faces in this mesh
      if(strcmp(node,"*MESH_NUMFACES") == 0) {
#ifdef READASE_PRINTSTATUS
        printf("GO-MESH-> NumFaces: %s\n",arg);
#endif

        // Allocate memory for the faces
        obj->NumFaces = atoi(arg);
        obj->FaceList = (GenFace3d *)malloc(sizeof(GenFace3d)*obj->NumFaces);
        for(i=0;i<obj->NumFaces;i++) {
          // Default the faces
          obj->FaceList[i].f1 = 0;
          obj->FaceList[i].f2 = 0;
          obj->FaceList[i].f3 = 0;
          obj->FaceList[i].n1 = 0;
          obj->FaceList[i].n2 = 0;
          obj->FaceList[i].n3 = 0;
          obj->FaceList[i].s1 = 0.f;
          obj->FaceList[i].t1 = 0.f;
          obj->FaceList[i].s2 = 0.f;
          obj->FaceList[i].t2 = 0.f;
          obj->FaceList[i].s3 = 0.f;
          obj->FaceList[i].t3 = 0.f;
        }
        continue;
      }

      // Line format: *MESH_VERTEX_LIST {
      // Description: Beginning of a vertex list
      if(strcmp(node,"*MESH_VERTEX_LIST") == 0) {
#ifdef READASE_PRINTSTATUS
        printf("GO-MESH-> Reading Mesh Vertex List...\n",arg);
#endif
        RA_GO_ReadMeshVertexList(obj,inp,node,arg); // Read the vertex list
        continue;
      }

      // Line format: *MESH_FACE_LIST {
      // Description: Beginning of face list
      if(strcmp(node,"*MESH_FACE_LIST") == 0) {
#ifdef READASE_PRINTSTATUS
        printf("GO-MESH-> Reading Mesh Face List...\n",arg); // Read the face list
#endif
        RA_GO_ReadMeshFaceList(obj,inp,node,arg);
        continue;
      }

      // Line format: *MESH_TVERT_LIST {
      // Description: Beginning of texture vertex list
      if(strcmp(node,"*MESH_TVERTLIST") == 0) {
#ifdef READASE_PRINTSTATUS    
        printf("GO-MESH-> Reading Texture Vertex List...\n",arg);
#endif
        RA_GO_ReadMeshTexVertexList(obj,inp,node,arg); // Read the texture vertex list
        continue;
      }

      // Line format: *MESH_TFACE_LIST {
      // Description: Beginning of texture face list
      if(strcmp(node,"*MESH_TFACELIST") == 0) {
#ifdef READASE_PRINTSTATUS    
        printf("GO-MESH-> Reading Texture Face List...\n",arg);
#endif
        RA_GO_ReadMeshTexFaceList(obj,inp,node,arg); // Read the texture vertex list
        continue;
      }

      // If we don't interpret the section, then we need to jump over it
      if(RA_IsSection(curline))
        RA_SkipSection(inp);
    }
}

// RA_GO_ReadMeshVertexList
//    Purpose: Read a *MESH_VERTEX_LIST tree
//    Params : obj         GenObject3d object to change
//             inp         Input file
//             node        Current line's node
//             arg         Current line's arguments
void RA_GO_ReadMeshVertexList(GenObject3d *obj,FILE *inp, char *node, char *arg)
{
  char curline[256];
  int   num;
  float x,y,z;

    while(1) {
      readln(inp,curline);
      RA_GetNode(curline,node);
      if(node[0] == '}')
        break;
      if(feof(inp))
        break;
      RA_GetNode(curline,node);
      RA_GetArg(curline,arg);
      sscanf(arg,"%d %f %f %f",&num,&x,&y,&z);
#ifdef READASE_PRINTINFO
      printf("GO-MESH-VTXLIST-> Vertex %d (%f,%f,%f)\n",num,x,y,z);
#endif
      obj->VertexList[num].x = x;
      obj->VertexList[num].y = y;
      obj->VertexList[num].z = z;
    }
}

// RA_GO_ReadMeshFaceList
//    Purpose: Read a *MESH_FACE_LIST tree
//    Params : obj         GenObject3d object to change
//             inp         Input file
//             node        Current line's node
//             arg         Current line's arguments
void RA_GO_ReadMeshFaceList(GenObject3d *obj,FILE *inp, char *node, char *arg)
{
  char curline[256];
  int   num, f1, f2, f3, n1, n2, n3;
  
    while(1) {
      readln(inp,curline);
      RA_GetNode(curline,node);
      if(node[0] == '}')
        break;
      if(feof(inp))
        break;
      RA_GetNode(curline,node);
      RA_GetArg(curline,arg);
      sscanf(arg,"%d: A: %d B: %d C: %d AB: %d BC: %d CA: %d",&num,&f1,&f2,&f3,&n1,&n2,&n3);
      obj->FaceList[num].f1 = f1; // Face Vertices
      obj->FaceList[num].f2 = f2;
      obj->FaceList[num].f3 = f3;
      obj->FaceList[num].n1 = n1; // Normals
      obj->FaceList[num].n2 = n2;
      obj->FaceList[num].n3 = n3;
#ifdef READASE_PRINTINFO
      printf("GO-MESH-FACELIST-> Face %d (%d,%d,%d)\n",num,f1,f2,f3);
#endif
    }
}

// RA_GO_ReadMeshTexVertexList
//    Purpose: Read a *MESH_TVERT_LIST tree
//    Params : obj         GenObject3d object to change
//             inp         Input file
//             node        Current line's node
//             arg         Current line's arguments
void RA_GO_ReadMeshTexVertexList(GenObject3d *obj,FILE *inp, char *node, char *arg)
{
  char curline[256];
  int   num;
  float u,v;

    while(1) {
      readln(inp,curline);
      RA_GetNode(curline,node);
      if(node[0] == '}')
        break;
      if(feof(inp))
        break;
      RA_GetNode(curline,node);
      RA_GetArg(curline,arg);
      sscanf(arg,"%d %f %f",&num,&u,&v);
#ifdef READASE_PRINTINFO
      printf("GO-MESH-TEXVTXLIST-> Texture Vertex %d (%f,%f)\n",num,u,v);
#endif
      s[num] = u;
      t[num] = v;
    }
}

// RA_GO_ReadMeshTexFaceList
//    Purpose: Read a *MESH_TFACE_LIST tree
//    Params : obj         GenObject3d object to change
//             inp         Input file
//             node        Current line's node
//             arg         Current line's arguments
void RA_GO_ReadMeshTexFaceList(GenObject3d *obj,FILE *inp, char *node, char *arg)
{
  char curline[256];
  int   num;
  int   f1,f2,f3;

    while(1) {
      readln(inp,curline);
      RA_GetNode(curline,node);
      if(node[0] == '}')
        break;
      if(feof(inp))
        break;
      RA_GetNode(curline,node);
      RA_GetArg(curline,arg);
      sscanf(arg,"%d %d %d %d",&num,&f1,&f2,&f3);
#ifdef READASE_PRINTINFO
      printf("GO-MESH-TEXFACELIST-> Texture Face %d (%d,%d,%d)\n",num,f1,f2,f3);
#endif
      obj->FaceList[num].s1 = s[f1];
      obj->FaceList[num].t1 = t[f1];

      obj->FaceList[num].s2 = s[f2];
      obj->FaceList[num].t2 = t[f2];

      obj->FaceList[num].s3 = s[f3];
      obj->FaceList[num].t3 = t[f3];
    }
}

// RA_IsSection
//    Purpose: Determine if the current line is the start of a section
//    Params : arg         Current line's arguments
//    Returns: Is Section (true) or is not (false)
bool RA_IsSection(char *arg)
{
    if(strchr(arg,'{') != NULL)
      return true;
    return false;
}

// RA_SkipSection
//    Purpose: Read lines until reaching the end of the entire section
//    Params : inp         Input file
void RA_SkipSection(FILE *inp)
{
  char curline[256], node[256];
  int  level = 1;

    while(1) {
      readln(inp,curline);
      if(RA_IsSection(curline)) {
        level++;
        continue;
      }

      RA_GetNode(curline,node);
      if(node[0] == '}') {
        if( (--level) == 0)
          return;
        continue;
      }
    }
    
}

// ReadASE
//    Purpose: Fully parse an ASE file and create an GenObject3d object
//    Params : filename    ASE Filename
//             obj         Object to read data into
//    Returns: Success (true) or failure (false)
bool ReadASE(char *filename,GenObject3d *obj)
{
  FILE *inp;
  char curline[256], node[256], arg[256];

    // Open data for reading
    inp = NULL;
    inp = fopen(filename,"rb");

    if(inp == NULL) // If file could not be opened
      return false;

    while(1) {
      // Read next line from ASE file
      readln(inp,curline);
      if(feof(inp)) // If at end of file, stop
        break;

      // Break down line
      RA_GetNode(curline,node); 
      RA_GetArg(curline,arg);

      // Interpret this line
      RA_CmdInterpret(obj,inp,node,arg);
    }
#ifdef READASE_PRINTSTATUS
    printf("ASE Read Complete.\n");
#endif
    fclose(inp);
    return true;
}
