// THE
//     ##   ## ##   ## ##  ## #####    #####
//    #### ## ##   ## ##  ## ##      ##  ##
//   ## #### ##   ## #####  #####      ##
//  ##   ## ##   ## ##  ## ##        ##
// ##   ##  #####  ##  ## #####   #######
//                                        LIBRARY 
//
// (C)1998 Odin Jensen and Nuke Software, All rights reserved.
//
// File          : NUKE2.H 
// Description   : Header file for all modules of the library
// Programming   : Odin Jensen


// Only include this header file once
#ifndef _NUKE2_HEADER_
#define _NUKE2_HEADER_

// +------------------------+
// | External Include Files |
// +------------------------+

#include <windows.h>
#include <windowsx.h>
#include <stdlib.h>
#include <stdio.h>
#include <malloc.h>
#include <math.h>
#include <mmsystem.h>
#include <vfw.h>
#include <ddraw.h>
#include <dsound.h>
#include <dinput.h>
#include <dplay.h>

// +--------------------+
// | Nuke2 Definitons   |
// +--------------------+

// Nuke2 version string
#define N2_Version "v0.1.1"

// Nuke2 error type 
#define N2_ERR int

// Nuke2 surface type
#define N2_SURFTYPE int

// Nuke2 open file type
#define N2_OPENSTATUS int

// Nuke2 surface types
#define N2_SYSMEM  0x100
#define N2_VIDMEM  0x101
#define N2_BESTFIT 0x102

// Nuke2 file open types
#define N2_OPEN 0x800
#define N2_CREATE 0x801
#define N2_APPEND 0x802

// Define RELEASE macro. Just call on any object whether allocated or not. Quite secure :)
#define RELEASE(x) if(x!=NULL) {x->Release();x=NULL;}
#define FREE(x) if (x!=NULL) {free(x);x=NULL;}

// +-----------------------------+
// | Nuke2 External Definitons   |
// +-----------------------------+

// DirectDraw2 Object
extern LPDIRECTDRAW2 N2_lpDD;

// Width of screen
extern int N2_Width;

// Height of screen
extern int N2_Height;

// Bitdepth of screen
extern int N2_BitDepth;

// DirectSound Object
extern LPDIRECTSOUND N2_lpDS;

// +------------------------+
// | Nuke2 Type Definitions |
// +------------------------+

// N2_File directory structure
typedef struct 
{
 char Name[256]; // File name w/o path.
 __int64 Offset; // Offset is file position, Size is size.
 long Size;
} DirEntry;

// +-------------------+
// | Nuke2 Error codes |
// +-------------------+

// Success
#define N2_SUCCESS             0x220

// N2_File errors
#define N2_FILENOTFOUND          0x221
#define N2_NOFILEDIR             0x222
#define N2_NOMEMFILEDIR          0x223
#define N2_BADFILEHEAD           0x224
#define N2_FILECREATEFAILED      0x225
#define N2_FILENOTBMP            0x226
#define N2_BADBMPINFOHEADER      0x227
#define N2_BMPINFOALLOCFAILED    0x228
#define N2_LOADPALETTEFAILED     0x229
#define N2_BITMAPDATAALLOCFAILED 0x230
#define N2_BADIMAGEDATA          0x231

// N2_Screen errors
#define N2_DDCREATEFAILED        0x250
#define N2_DD2QUERYFAILED        0x251
#define N2_SETCOOPFAILED         0x252
#define N2_INVALIDVMODE	         0x253
#define N2_FRONTCREATEFAILED     0x254
#define N2_BACKCREATEFAILED      0x255
#define N2_CREATECLIPPERFAILED   0x256
#define N2_CLIPWITHHWNDFAILED    0x257
#define N2_CLIPATTACHFAILED      0x258

// N2_MIDI errors
#define N2_MIDICLOSEFAILED       0x270
#define N2_MIDIOPENFAILED	     0x271
#define N2_MIDIPLAYFAILED        0x272
#define N2_MIDISTOPFAILED        0x273
#define N2_MIDIRESUMEFAILED      0x274
#define N2_MIDIRESTARTFAILED     0x275

// N2_Input errors
#define N2_DICREATEFAILED          0x290
#define N2_DIMOUSEAQUIREFAILED     0x291
#define N2_DIKEYSAQUIREFAILED      0x292
#define N2_DI2QUERYFAILED          0x293
#define N2_DISETJOYXRANGEFAILED    0x294
#define N2_DISETJOYYRANGEFAILED    0x295
#define N2_DISETJOYDEADZONEXFAILED 0x296
#define N2_DISETJOYDEADZONEYFAILED 0x297
#define N2_DIJOYAQUIREFAILED       0x298

// N2_Surface errors
#define N2ERR_CREATESURFACE		   0x320

// N2_Sound errors

#define N2_DSOUNDCREATEFAILED      0x360
#define N2_DSOUNDSETCOOPFAILED     0x361

// <------------------------------------------- Graphic Objects ------------------------------------------------>

// +----------------------+
// | Nuke2 Surface Object |
// +----------------------+

class N2_Surface
{
private:
  	
  // Surface position and dimension.
  int XPos, YPos, Width, Height;
  
  // Bitmap info structure
  LPBITMAPINFO BitmapInfo;
  
  // RGB structure to hold palette
  LPRGBQUAD RGB;

  // Bitmap data for local copy
  BYTE * Data;
  
  // Bool to keep track of local copy. Only used with system memory surfaces
  // Set automatically by create's SURFTYPE parameter
  BOOL LocalCopy;
  
  // Rect to clip against
  RECT ClipRect;

  // Surface area rect
  RECT OwnRect;

  // Surface destination rect
  RECT DestOwnRect;
  
  // Surface type
  N2_SURFTYPE SurfType;

  // Place to copy data upon restore
  int Copy_X, Copy_Y;

  // Device context
  HDC N2_DC;

  // X, Y offset for scrolling within self (Woah. Deep)
  int OffsetX, OffsetY;

public:
  
  // Local DirectDraw palette
  LPDIRECTDRAWPALETTE lpDDPal;

	// Surface pointer
  LPDIRECTDRAWSURFACE Surface;	  
  
  // DD Descriptor
  DDSURFACEDESC N2_DDSD;

  // Constructor
  N2_Surface();

  // Destructor
  ~N2_Surface();

  // Create normal surface
  N2_ERR Create(int width, int height, N2_SURFTYPE Type);

   // Create surface of known w,h 
  N2_ERR N2_Surface::Create(N2_SURFTYPE Type);
  
  // Load from file onto existing surface
  N2_ERR Load(char * FileName);
  
  // Load from data file onto existing surface
  N2_ERR Load(FILE * Handle);
  
  // Draw normal
  void Draw(int x, int y, BOOL Trans, N2_Surface * Dest);
  
  // Draw normal with internal x,y
  void Draw(BOOL Trans, N2_Surface * Dest);
  
  // Draw clipped normal  
  void DrawClip(int x, int y, BOOL Trans, N2_Surface * Dest);
  
  // Draw clipped normal with internal x,y
  void DrawClip(BOOL Trans, N2_Surface * Dest);
      
  // Restore surface
  void Restore();
  
  // Release surface
  void Release(){RELEASE(Surface);};
  
  // Set color key of surface
  void ColorKey(DWORD Color);
  
  // Check collision of area (SPRITE)
  BOOL CheckHit(N2_Surface * Target);
  
  // Check collision of point (MOUSE)
  BOOL CheckPointHit(POINT Point, N2_Surface * Target);
  
  // Set position
  void SetPos(int x, int y){XPos=x;YPos=y;};
  
  // Set clip rect
  void SetClip(RECT CRect){ClipRect.top=CRect.top;ClipRect.bottom=CRect.bottom;ClipRect.left=CRect.left;ClipRect.right=CRect.right;};
  
  // Set Surface rect
  void SetRect(RECT CRect){OwnRect.top=CRect.top;OwnRect.bottom=CRect.bottom;OwnRect.left=CRect.left;OwnRect.right=CRect.right;};
  
  // Set Surface rect2
  void SetRect(int left, int top, int right, int bottom){OwnRect.top=top;OwnRect.bottom=bottom;OwnRect.left=left;OwnRect.right=right;};
  
  // Set Dest rect
  void SetDRect(RECT CRect){DestOwnRect.top=CRect.top;DestOwnRect.bottom=CRect.bottom;DestOwnRect.left=CRect.left;DestOwnRect.right=CRect.right;};
  
  // Get X position
  int GetX(){return XPos;};
  
  // Get Y position
  int GetY(){return YPos;};
  
  // Get width
  int GetWidth(){return Width;};
  
  // Get height
  int GetHeight(){return Height;};  

  // Lock surface
  void Lock();

  // Unlock surface
  void Unlock();
	
  // Erase surface
  void Erase(DWORD color);
  
  // Copy bitmap data to surface
  void N2_Surface::Data2Surface(int x, int y);

  // Set source x,y for bitmap restore to surface
  void SetCopy(int x, int y){Copy_X=x; Copy_Y=y;};
   
  // Disable local copy
  void LocalCopyOn();
  
  // Disable local copy
  void LocalCopyOff();

  // Load 256 color palette
  N2_ERR LoadPalette(char * FileName);

 // Load 256 color palette for N2_File
  N2_ERR LoadPalette(FILE * Handle);

 // Draw text on surface
 void Text(int x, int y, char * text, DWORD color);

 // Capture DC
 void GetDC();

 // Release DC
 void ReleaseDC();

};

// +--------------------+
// | Nuke2 Screen Object|
// +--------------------+

class N2_Screen
{
private:
    
  // DirectDraw clipper for windowed mode
  LPDIRECTDRAWCLIPPER lpDDClipper; 
  
  // Primary surface pointer
  N2_Surface * lpDDSFront; 
  
  // Back surface pointer
  N2_Surface * lpDDSBack; 
  
  // Fullscreen or not? (Used by flip and draw to determine how to render)
  BOOL FullScreen;

  // Windows handle
  HWND N2_SCRHWND;
public:
  // Constructor
  N2_Screen();
  
  // Deconstructor
  ~N2_Screen();
  
  // Create full screen in any bitdepth
  N2_ERR CreateFull(HWND hwnd, DWORD Width, DWORD Height, DWORD BitDepth, BOOL StandardVGA);
  
  // Create windowed screen with current screen bit depth
  N2_ERR CreateWindowed(HWND hwnd, int Width, int Height);
      
  // Restore
  void Restore();
  
  // Flip screen
  void Flip(BOOL WaitVB);

  // Return back buffer
  N2_Surface * GetBack(){return lpDDSBack;};

  // Return primary surface
  N2_Surface * GetFront(){return lpDDSFront;};

  // Is in fullscreen?
  BOOL IsFullscreen(){return FullScreen;};
  
};

// <------------------------------------------- File Objects --------------------------------------------------->


// +------------------+
// | Nuke2 file class |
// +------------------+

class N2_File
{
protected:
 // Bool to keep check of fileopen
 BOOL FILE_OPEN;
 
 // File pointer
 FILE * FP;
 
 // Dir offset
 __int64 DIR_OFFSET;
 
 // Number of files in N2_File
 UINT NumFiles;
 
 // Directory memory pointer
 DirEntry * Dir;
public:
 
 // Constructor	
 N2_File();
 
 // Destructor
 ~N2_File();
 
 // Try to open file	
 N2_ERR Open(LPSTR FileName); 

 // Close file
 void Close();
 
 // Return handle of given filename
 FILE * Seek(LPSTR FileName);
 
 // Get size of file
 long GetSize(LPSTR FileName); 
 
 // Return number of files
 UINT GetNumFiles(){return NumFiles;};
 
 // Get a DirEntry by it's name
 DirEntry * GetEntryByName(LPSTR FileName);
 
 // Get a DirEntry by it's number (Good for DirList :) 
 DirEntry * GetEntryByNumber(int FileNum);
};

// +----------------------------------------------------+
// | Nuke2 INI file object. (Can be used as log also :) |
// +----------------------------------------------------+

class N2_INI
{
 private:
  // File handle
  FILE * fh;
 
  // File open?
  BOOL FOpen;
  
  // System info structure
  SYSTEM_INFO SystemInfo;
  
  // Memory info structure
  MEMORYSTATUS MStatus;
  
 public:
  
  // Constructor
  N2_INI();
  
  // Destructor
  ~N2_INI();
  
  // Open INI file
  N2_ERR Open(char * name, N2_OPENSTATUS Status);
  
  // Close INI file
  void Close();
  
  // Write string to INI
  void WriteString(char * string);
  
  // Write 2 strings to INI
  void Write2Strings(char * string, char * string2);  
  
  // Write string and dword to INI  
  void WriteStringDWORD(char * string, DWORD value);
  
  // Read string from INI  
  void ReadString(char * string);
  
  // Read string and dword from INI    
  void ReadStringDWORD(char * string, DWORD * value);
  
  // Read 2 strings from INI
  void Read2Strings(char * string, char * string2);  
  
  // Dump Nuke2 version
  void DumpNuke2Ver();
  
  // Dump system info
  void DumpSysInfo();
  
  // Dump memory info
  void DumpMemInfo();

  // Check if file is open
  BOOL FileOpen(){return FOpen;};
  
  // Get file pointer for direct access.
  FILE * GetFP(){return fh;};
};

// <------------------------------------------- Music Objects -------------------------------------------------->

// +-----------------+
// | NUKE MIDI class |
// +-----------------+

class N2_MIDI
{
private:
 // Windows handle
 HWND Hwnd;

 // Was MIDI loaded from WAD?
 BOOL IsFromWAD;
 
 // Are we ready?
 BOOL IsReady;
 
 // Filename
 LPSTR TFileName;
 public:
 
 // Constructor
 N2_MIDI();
 
 // Destructor
 ~N2_MIDI();
 
 // Create MIDI session
 void Create(HWND hwnd);
 
 // Load from data file
 N2_ERR Load(LPSTR Name, FILE * fh, long size);
 
 // Load from single file
 N2_ERR Load(LPSTR FileName);
 
 // Play
 N2_ERR Play();
 
 // Stop
 N2_ERR Stop();
 
 // Pause
 N2_ERR Pause();
 
 // Resume
 N2_ERR Resume();
 
 // Restart
 N2_ERR Restart(); 
};


// <------------------------------------------- Timer Objects -------------------------------------------------->


// +------------------+
// | Nuke2 tick timer |
// +------------------+

class N2_TickTimer
{
protected:
 
 // Counters
 long TimeNow,TimeOld;
public:
 
 // Query timer for 'value' millisecs
 BOOL Query(int value);
};

// +--------------------+
// | Frame time counter |
// +--------------------+

class N2_FPS
{
protected:
 
  // Lot of show FPS counters
 int FrameRate;
 int FrameCount;
 int FrameCount0;
 DWORD FrameTime;
 DWORD FrameTime0; 
public :

 // Update FPS counter
 void Update();
 
 // Draw FPS onto surface
 void Show(int x, int y, DWORD color, N2_Surface * Dest);
};
 
 // <------------------------------------------- Input Objects ------------------------------------------------->


// +---------------+
// | Input handler |
// +---------------+

class N2_Input
{
private:
 // DirectInput object
 LPDIRECTINPUT N2_lpDI;
 
 // DirectKeyboard device
 LPDIRECTINPUTDEVICE N2_lpDIDKeys;
 
 // DirectMouse device
 LPDIRECTINPUTDEVICE N2_lpDIDMouse;
 
 // DirectJoy device
 LPDIRECTINPUTDEVICE2 N2_lpDIDJoy;
 
 public:
 // Mouse coords
 POINT Mouse;
 
 // Joy coords
 POINT Joystick;
 
 // Joystick B1 and B2
 BOOL JoyB1, JoyB2;
 
 // Mouse button status
 BOOL MouseLB, MouseMB, MouseRB;
 
 // Key status for all keys
 BYTE Keys[256];
 
 // Is mouse active
 BOOL MouseActive;
 
 // Is Keyboard active
 BOOL KeysActive;
 
 // Is Joystick active
 BOOL JoyActive;

 // Constructor	
 N2_Input();
 
 // Destructor
 ~N2_Input();
 
 // Create input object
 N2_ERR Create(HINSTANCE inst, HWND hwnd);
 
 // Release access to input devices
 void Release();
 
 // Get access to input devices
 void Capture();
 
 // Update input devices
 void Refresh();
 
 // Set wich devices you want to be active in this session
 void SetActiveDevices(BOOL Mouse, BOOL Keys, BOOL Joy);
 
 // Set mouse return values to absolute instead of normal.
 void SetMouseAbs();
 
 // Set joystick return values to absolute instead of normal.
 void SetJoyAbs();
 
 // Run Mouse control panel
 void RunMouseCPL(HWND hwnd=NULL);
 
 // Run Joystick control panel
 void RunJoyCPL(HWND hwnd=NULL);

};


// <------------------------------------------- Sound Objects -------------------------------------------------->

// +----------------+
// | N2_Sound class |
// +----------------+

class N2_Sound
{
public:
 // Constructor
 N2_Sound();
 
 // Destructor
 ~N2_Sound();

 // Release sound
 void Release();
 
 // Create sound object
 N2_ERR Create(HWND hwnd);
};

// <------------------------------------------- Non-object functions ------------------------------------------->


// +----------------------------+
// | N2_ERR to string converter |
// +----------------------------+

char * N2_Err2Str(N2_ERR ErrorValue);

// +--------------------------+
// | Error box. Returns FALSE |
// +--------------------------+

BOOL N2_ErrBox(char * Header, N2_ERR Error);

// +-----------------------------------+
// | Error box overload. Returns FALSE |
// +-----------------------------------+

BOOL N2_ErrBox(char * Header, char * ErrorMsg);

// +---------------------------------------+
// | Clip coords and rectangle to DestRect |
// +---------------------------------------+

void N2_ClipRect(int *DestX, int *DestY, RECT *SrcRect, RECT *DestRect);

// +-----------------+
// | Get desktop BPP |
// +-----------------+

int N2_GetDesktopBPP();

// +---------------------+
// | Compare two strings |
// +---------------------+

BOOL N2_CompareStrings(char * str1, char * str2);

#endif