#include <windows.h>
#include <GL/gl.h>
#include <GL/glu.h>
#include <math.h>

#include "gldd.h"

#define glRGB(x, y, z)  glColor3ub((GLubyte)x, (GLubyte)y, (GLubyte)z)

const GLfloat cRotateSpeed = 0.1f;
GLfloat roll,pitch,yaw;

char *className = "OpenGL";
char *windowName = "Arrow";
int winX = 0, winY = 0;

// DirectDraw display mode sizes
const int ddWidth = 320, ddHeight = 200, ddBPP = 16;

HDC hDC;
HGLRC hGLRC;
HPALETTE hPalette;

void init(void)
{
    GLfloat aspect;

    aspect = (GLfloat)ddWidth / (GLfloat)ddHeight;
    glMatrixMode(GL_PROJECTION);
    gluPerspective(45.0, aspect, 1.0f, 3000.0);
    glMatrixMode(GL_MODELVIEW);

    glEnable(GL_DEPTH_TEST);
    glCullFace (GL_BACK);
    glEnable (GL_CULL_FACE);
    glClearColor (0.5f, 0.5f, 1.0f, 0.0f);
    glDisable (GL_DITHER);
}

void drawstuff(void)
{
    /* clear color and depth buffers */
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

    /* draw six faces of a cube */
    glBegin(GL_QUADS);
        glRGB (50, 50, 50);
        
        glVertex3f( 0.5f, 1.0f,-5.0f);
        glVertex3f( 0.5f, 1.0f, 2.0f);
        glVertex3f( 0.5f,-1.0f, 2.0f);
        glVertex3f( 0.5f,-1.0f,-5.0f);

        glVertex3f(-0.5f,-1.0f,-5.0f);
        glVertex3f(-0.5f,-1.0f, 2.0f);
        glVertex3f(-0.5f, 1.0f, 2.0f);
        glVertex3f(-0.5f, 1.0f,-5.0f);

        glRGB (40, 40, 40);

        glVertex3f( 0.5f, 1.0f,-5.0f);
        glVertex3f( 0.5f,-1.0f,-5.0f);
        glVertex3f(-0.5f,-1.0f,-5.0f);
        glVertex3f(-0.5f, 1.0f,-5.0f);

        glRGB (60, 60, 60);

        glVertex3f( 0.5f, 1.0f,-5.0f);
        glVertex3f(-0.5f, 1.0f,-5.0f);
        glVertex3f(-0.5f, 1.0f, 2.0f);
        glVertex3f( 0.5f, 1.0f, 2.0f);

        glVertex3f(-0.5f,-1.0f,-5.0f);
        glVertex3f( 0.5f,-1.0f,-5.0f);
        glVertex3f( 0.5f,-1.0f, 2.0f);
        glVertex3f(-0.5f,-1.0f, 2.0f);

        glRGB (70, 70, 70);

        glVertex3f(-0.5f,-1.0f, 2.0f);
        glVertex3f( 0.5f,-1.0f, 2.0f);
        glVertex3f( 0.5f,-2.0f, 2.0f);
        glVertex3f(-0.5f,-2.0f, 2.0f);

        glVertex3f(-0.5f, 2.0f, 2.0f);
        glVertex3f( 0.5f, 2.0f, 2.0f);
        glVertex3f( 0.5f, 1.0f, 2.0f);
        glVertex3f(-0.5f, 1.0f, 2.0f);

        glRGB (80, 80, 80);

        glVertex3f(-0.5f,-2.0f, 2.0f);
        glVertex3f( 0.5f,-2.0f, 2.0f);
        glVertex3f( 0.5f, 0.0f, 5.0f);
        glVertex3f(-0.5f, 0.0f, 5.0f);

        glVertex3f( 0.5f, 2.0f, 2.0f);
        glVertex3f(-0.5f, 2.0f, 2.0f);
        glVertex3f(-0.5f, 0.0f, 5.0f);
        glVertex3f( 0.5f, 0.0f, 5.0f);

    glEnd();
    glBegin(GL_TRIANGLES);
        glRGB (50, 50, 50);
        
        glVertex3f( 0.5f, 0.0f, 5.0f);
        glVertex3f( 0.5f,-2.0f, 2.0f);
        glVertex3f( 0.5f, 2.0f, 2.0f);

        glVertex3f(-0.5f, 0.0f, 5.0f);
        glVertex3f(-0.5f, 2.0f, 2.0f);
        glVertex3f(-0.5f,-2.0f, 2.0f);
        
    glEnd();
}

void resize(void)
{
    /* set viewport to cover the window */
    glViewport(0, 0, ddWidth, ddHeight);
}

void setupPixelFormat(HDC hDC)
{
    PIXELFORMATDESCRIPTOR pfd = {
        sizeof(PIXELFORMATDESCRIPTOR),  /* size */
        1,                              /* version */
        PFD_SUPPORT_OPENGL |
        PFD_DRAW_TO_WINDOW |
        PFD_DOUBLEBUFFER,               /* support double-buffering */
        PFD_TYPE_RGBA,                  /* color type */
        ddBPP,                             /* prefered color depth */
        0, 0, 0, 0, 0, 0,               /* color bits (ignored) */
        0,                              /* no alpha buffer */
        0,                              /* alpha bits (ignored) */
        0,                              /* no accumulation buffer */
        0, 0, 0, 0,                     /* accum bits (ignored) */
        16,                             /* depth buffer */
        0,                              /* no stencil buffer */
        0,                              /* no auxiliary buffers */
        PFD_MAIN_PLANE,                 /* main layer */
        0,                              /* reserved */
        0, 0, 0,                        /* no layer, visible, damage masks */
    };
    int pixelFormat;

    pixelFormat = ChoosePixelFormat(hDC, &pfd);
    if (pixelFormat == 0) {
        MessageBox(WindowFromDC(hDC), "ChoosePixelFormat failed.", "Error",
                MB_OK);
        exit(1);
    }

    if (SetPixelFormat(hDC, pixelFormat, &pfd) != TRUE) {
        MessageBox(WindowFromDC(hDC), "SetPixelFormat failed.", "Error",
                MB_OK);
        exit(1);
    }
}

void setupPalette(HDC hDC)
{
    int pixelFormat = GetPixelFormat(hDC);
    PIXELFORMATDESCRIPTOR pfd;
    LOGPALETTE* pPal;
    int paletteSize;

    DescribePixelFormat(hDC, pixelFormat, sizeof(PIXELFORMATDESCRIPTOR), &pfd);

    if (pfd.dwFlags & PFD_NEED_PALETTE) {
        paletteSize = 1 << pfd.cColorBits;
    } else {
        return;
    }

    pPal = (LOGPALETTE*)
        malloc(sizeof(LOGPALETTE) + paletteSize * sizeof(PALETTEENTRY));
    pPal->palVersion = 0x300;
    pPal->palNumEntries = paletteSize;

    /* build a simple RGB color palette */
    {
        int redMask = (1 << pfd.cRedBits) - 1;
        int greenMask = (1 << pfd.cGreenBits) - 1;
        int blueMask = (1 << pfd.cBlueBits) - 1;
        int i;

        for (i=0; i<paletteSize; ++i) {
            pPal->palPalEntry[i].peRed =
                    (((i >> pfd.cRedShift) & redMask) * 255) / redMask;
            pPal->palPalEntry[i].peGreen =
                    (((i >> pfd.cGreenShift) & greenMask) * 255) / greenMask;
            pPal->palPalEntry[i].peBlue =
                    (((i >> pfd.cBlueShift) & blueMask) * 255) / blueMask;
            pPal->palPalEntry[i].peFlags = 0;
        }
    }

    hPalette = CreatePalette(pPal);
    free(pPal);

    if (hPalette) {
        SelectPalette(hDC, hPalette, FALSE);
        RealizePalette(hDC);
    }
}

LRESULT APIENTRY 
WndProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
    switch (message) {
    case WM_CREATE:
        /* initialize OpenGL rendering */
        hDC = GetDC(hWnd);
        setupPixelFormat(hDC);
        setupPalette(hDC);
        hGLRC = wglCreateContext(hDC);
        wglMakeCurrent(hDC, hGLRC);
        init();
        return 0;
    case WM_DESTROY:
        /* finish OpenGL rendering */
        if (hGLRC) {
            wglMakeCurrent(NULL, NULL);
            wglDeleteContext(hGLRC);
        }
        if (hPalette) {
            DeleteObject(hPalette);
        }
        ReleaseDC(hWnd, hDC);
        
        // Close DirectDraw, restoring display mode if needed.
        glddCloseDirectDraw();
        
        PostQuitMessage(0);
        return 0;
    case WM_SIZE:
        // Ignore window size changes
/*
        if (hGLRC) {
            winWidth = (int) LOWORD(lParam);
            winHeight = (int) HIWORD(lParam);
            resize();
            return 0;
        }
*/
    case WM_PALETTECHANGED:
        /* realize palette if this is *not* the current window */
        if (hGLRC && hPalette && (HWND) wParam != hWnd) {
            UnrealizeObject(hPalette);
            SelectPalette(hDC, hPalette, FALSE);
            RealizePalette(hDC);
            drawstuff();
            break;
        }
        break;
    case WM_QUERYNEWPALETTE:
        /* realize palette if this is the current window */
        if (hGLRC && hPalette) {
            UnrealizeObject(hPalette);
            SelectPalette(hDC, hPalette, FALSE);
            RealizePalette(hDC);
            drawstuff();
            return TRUE;
        }
        break;
    case WM_PAINT:
        {
            PAINTSTRUCT ps;
            BeginPaint(hWnd, &ps);
            if (hGLRC) {
                drawstuff();
            }
            EndPaint(hWnd, &ps);
            return 0;
        }
        break;
    case WM_CHAR:
        /* handle keyboard input */
        switch ((int)wParam) {
        case VK_ESCAPE:
            DestroyWindow(hWnd);
            return 0;
        default:
            break;
        }
        break;
    default:
        break;
    }
    return DefWindowProc(hWnd, message, wParam, lParam);
}

int APIENTRY
WinMain(HINSTANCE hCurrentInst, HINSTANCE hPreviousInst, LPSTR lpszCmdLine,
    int nCmdShow)
{
    WNDCLASS wndClass;
    HWND hWnd;
    MSG msg;
    BOOL stop;
    DWORD lastTimeGetTime, thisTimeGetTime;
    GLfloat deltaTime, rotatedelta;
    GLuint ArrowDisplayList = 1;

    /* register window class */
    wndClass.style = 0; //CS_HREDRAW | CS_VREDRAW;
    wndClass.lpfnWndProc = WndProc;
    wndClass.cbClsExtra = 0;
    wndClass.cbWndExtra = 0;
    wndClass.hInstance = hCurrentInst;
    wndClass.hIcon = LoadIcon(NULL, IDI_APPLICATION);
    wndClass.hCursor = LoadCursor(NULL, IDC_ARROW);
    wndClass.hbrBackground = NULL;
    wndClass.lpszMenuName = NULL;
    wndClass.lpszClassName = className;
    RegisterClass(&wndClass);

    /* create window */
    hWnd = CreateWindow(
        className, windowName,
            WS_POPUP | WS_CLIPCHILDREN | WS_CLIPSIBLINGS,
        0, 0, ddWidth, ddHeight,
        NULL, NULL, hCurrentInst, NULL);

    if (!hWnd)
        return FALSE;

    // Create DirectDraw
    if (!glddInitDirectDraw(hWnd)) {
        MessageBox(NULL, "Could not init DirectDraw.", "Error", MB_OK);
        return FALSE;
    };

    // Set the display mode that we require    
    if (!glddChangeDisplayMode(ddWidth, ddHeight, ddBPP)) {
        MessageBox(NULL, "Could not change display mode.", "Error", MB_OK);
        return FALSE;
    };
        
    /* display window */
    ShowWindow(hWnd, SW_NORMAL);
    UpdateWindow(hWnd);
    
    glMatrixMode(GL_MODELVIEW);

    // Lets make a display list
    glNewList(ArrowDisplayList, GL_COMPILE);
    drawstuff(); // Draw the primitives into our display list
    glEndList();
         
    stop=FALSE;
    roll = pitch = yaw = 0.0f;
    lastTimeGetTime = timeGetTime();
    
    while (!stop){
        if (PeekMessage(&msg, 0, 0, 0, PM_REMOVE) == TRUE) {
            TranslateMessage(&msg);
            DispatchMessage(&msg);
            if (msg.message==WM_QUIT){
                stop=TRUE;
                PostQuitMessage(0);
            }
        }
            
        thisTimeGetTime = timeGetTime();
        deltaTime = (GLfloat)(thisTimeGetTime - lastTimeGetTime);
        lastTimeGetTime = thisTimeGetTime;

        rotatedelta = cRotateSpeed * deltaTime;
        if (GetAsyncKeyState(VK_UP))    pitch += rotatedelta;
        if (GetAsyncKeyState(VK_DOWN))  pitch -= rotatedelta;
        
        if (GetAsyncKeyState(VK_RIGHT)) roll += rotatedelta;
        if (GetAsyncKeyState(VK_LEFT))  roll -= rotatedelta;
        
        if (GetAsyncKeyState('Z'))     yaw  -= rotatedelta;
        if (GetAsyncKeyState('X'))     yaw  += rotatedelta;
        
        glPushMatrix();
            glTranslatef(0.0f, 0.0f, -10.0f);
            glRotatef(pitch, 1.0f, 0.0f, 0.0f);
            glRotatef(roll, 0.0f, 1.0f, 0.0f);
            glRotatef(yaw, 0.0f, 0.0f, 1.0f);
            glCallList(ArrowDisplayList);
        glPopMatrix();        
    
        wglSwapLayerBuffers(hDC, WGL_SWAP_MAIN_PLANE);  //For Microsoft's OpenGL
        // SwapBuffers(hDC); // Use this for CosmoGL
    }
    return msg.wParam;
}
