#include <dxsound.h>

CWave::CWave()
	: m_dwImageLen(0), m_pImageData(NULL)
{
}

CWave::CWave(LPCSTR lpFileName)
	: m_dwImageLen(0), m_pImageData(NULL)
{
  Create(lpFileName);
}

CWave::~CWave()
{
	// Free the wave image data
	Free();
}

BOOL CWave::Create(LPCSTR lpFileName)
{
	// Free any previous wave image data
	Free();

	// Open the wave file
	int hFile = _lopen(lpFileName, OF_READ);
	if(-1 == hFile)
		return FALSE;

	// Get the file length
	m_dwImageLen = _llseek(hFile, 0, FILE_END);
	_llseek(hFile, 0, FILE_BEGIN);

	// Allocate and lock memory for the image data
	m_pImageData = (BYTE*)GlobalLock(GlobalAlloc(GMEM_MOVEABLE | GMEM_SHARE, m_dwImageLen));
	if(!m_pImageData)
		return FALSE;

	// Read the image data from the file
	_lread(hFile, m_pImageData, m_dwImageLen);
	_lclose(hFile);

	return TRUE;
}

BOOL CWave::GetFormat(WAVEFORMATEX& wfFormat) const
{
	// Check validity
	if(!IsValid())
		return FALSE;

	// Setup and open the MMINFO structure
	CMMMemoryIOInfo mmioInfo((HPSTR)m_pImageData, m_dwImageLen);
	CMMIO 			 mmio(mmioInfo);

	// Find the WAVE chunk
	CMMTypeChunk mmckParent('W','A','V','E');
	mmio.Descend(mmckParent, MMIO_FINDRIFF);

	// Find and read the format subchunk
	CMMIdChunk mmckSubchunk('f','m','t',' ');
	mmio.Descend(mmckSubchunk, mmckParent, MMIO_FINDCHUNK);
	mmio.Read((HPSTR)&wfFormat, sizeof(WAVEFORMATEX));
	mmio.Ascend(mmckSubchunk);

	return TRUE;
}

DWORD CWave::GetDataLen() const
{
	// Check validity
	if(!IsValid())
		return 0;

	// Setup and open the MMINFO structure
	CMMMemoryIOInfo mmioInfo((HPSTR)m_pImageData, m_dwImageLen);
	CMMIO 			 mmio(mmioInfo);

	// Find the WAVE chunk
	CMMTypeChunk mmckParent('W','A','V','E');
	mmio.Descend(mmckParent, MMIO_FINDRIFF);

	// Find and get the size of the data subchunk
	CMMIdChunk mmckSubchunk('d','a','t','a');
	mmio.Descend(mmckSubchunk, mmckParent, MMIO_FINDCHUNK);

	return mmckSubchunk.cksize;
}

DWORD CWave::GetData(BYTE*& pWaveData, DWORD dwMaxLen) const
{
	// Check validity
	if(!IsValid())
		return 0;

	// Setup and open the MMINFO structure
	CMMMemoryIOInfo mmioInfo((HPSTR)m_pImageData, m_dwImageLen);
	CMMIO 			 mmio(mmioInfo);

	// Find the WAVE chunk
	CMMTypeChunk mmckParent('W','A','V','E');
	mmio.Descend(mmckParent, MMIO_FINDRIFF);

	// Find and get the size of the data subchunk
	CMMIdChunk mmckSubchunk('d','a','t','a');
	mmio.Descend(mmckSubchunk, mmckParent, MMIO_FINDCHUNK);
	DWORD dwLenToCopy = mmckSubchunk.cksize;

	// Allocate memory if the passed in pWaveData was NULL
	if(NULL == pWaveData)
		pWaveData = (BYTE*)GlobalLock(GlobalAlloc(GMEM_MOVEABLE, dwLenToCopy));
	else
		// If we didn't allocate our own memory, honor dwMaxLen
		if(dwMaxLen < dwLenToCopy)
			dwLenToCopy = dwMaxLen;

	if(pWaveData)
		// Read waveform data into the buffer
		mmio.Read((HPSTR)pWaveData, dwLenToCopy);

	return dwLenToCopy;
}

BOOL CWave::Free()
{
	// Free any previous wave data
	if(m_pImageData)
	{
		HGLOBAL	hgmemWave = GlobalHandle(m_pImageData);

		if(hgmemWave)
		{
			// Unlock and free memory
			GlobalUnlock(hgmemWave);
			GlobalFree(hgmemWave);

			m_pImageData = NULL;
			m_dwImageLen = 0;

			return TRUE;
		}
	}

	return FALSE;
}

BOOL CWave::IsValid() const
{
	return (m_pImageData ? TRUE : FALSE);
}

