#include <dx3d.h>

typedef struct _FindMaterialData
{
	DWORD 		  bpp;					// we want a material format of this bpp
	DDPIXELFORMAT ddpf;					// place the format here
} FindMaterialData;

HRESULT CALLBACK EnumTextureFormatsCallback (LPDDSURFACEDESC lpDdsd, LPVOID lParam);

HRESULT CreateMaterial(D3DTEXTUREMATERIAL &material, D3DCOLORVALUE diffuse, D3DCOLORVALUE ambient, D3DCOLORVALUE specular, LPCSTR lpFileName, int x, int y, int dx, int dy)
{
	HRESULT ddrval;

	// Create material handle
	if(NULL != lpFileName)
	{
		HBITMAP hBitmap = (HBITMAP)LoadImage(NULL, lpFileName, IMAGE_BITMAP, 0, 0, LR_LOADFROMFILE | LR_CREATEDIBSECTION);
		if(NULL == hBitmap)
			return DDERR_NOTFOUND;

		BITMAP bitmap;

		// Get size of bitmap
		GetObject(hBitmap, sizeof(BITMAP), &bitmap);

		// Use the passed size, unless zero
		dx = 0 == dx ? bitmap.bmWidth  : dx;
		dy = 0 == dy ? bitmap.bmHeight : dy;

		// Find the best material format to use.
		FindMaterialData findData;
		memset(&findData, 0, sizeof(FindMaterialData));

		findData.bpp = bitmap.bmBitsPixel;

		// Enumerate all material formats for this device
		lpD3DDevice->EnumTextureFormats(EnumTextureFormatsCallback, (LPVOID)&findData);

		DDSURFACEDESC ddsd;
		memset(&ddsd, 0, sizeof(DDSURFACEDESC));

		ddsd.dwSize 			= sizeof(DDSURFACEDESC);
		ddsd.dwFlags			= DDSD_CAPS | DDSD_HEIGHT | DDSD_WIDTH | DDSD_PIXELFORMAT;
		ddsd.ddsCaps.dwCaps	= DDSCAPS_SYSTEMMEMORY | DDSCAPS_TEXTURE;
		ddsd.dwWidth			= dx;
		ddsd.dwHeight			= dy;
		ddsd.ddpfPixelFormat = findData.ddpf;

		ddrval = lpDD->CreateSurface(&ddsd, &material.lpDDSurfaceSystem, NULL);
		if(DD_OK != ddrval)
			return ddrval;

		if(0 != d3dHALDeviceDesc.dcmColorModel)
		{
			ddsd.ddsCaps.dwCaps	= DDSCAPS_VIDEOMEMORY | DDSCAPS_TEXTURE | DDSCAPS_ALLOCONLOAD;

			ddrval = lpDD->CreateSurface(&ddsd, &material.lpDDSurfaceVideo, NULL);
			if(DD_OK != ddrval)
				return ddrval;
		}

		HDC hBitmapDC = CreateCompatibleDC(NULL);
		SelectObject(hBitmapDC, hBitmap);

		DWORD adw[256];
		GetDIBColorTable(hBitmapDC, 0, 256, (RGBQUAD *)adw);

		for(int i = 0; i < 256; i++)
			adw[i] = RGB(GetBValue(adw[i]), GetGValue(adw[i]), GetRValue(adw[i]));

		// Create a DirectDraw palette for the material.
		LPDIRECTDRAWPALETTE lpDDPalette = NULL;
		ddrval = lpDD->CreatePalette(DDPCAPS_8BIT, (PALETTEENTRY *)adw, &lpDDPalette, NULL);
		if(DD_OK != ddrval)
			return ddrval;

		material.lpDDSurfaceSystem->SetPalette(lpDDPalette);

		if(NULL != material.lpDDSurfaceVideo)
			material.lpDDSurfaceVideo->SetPalette(lpDDPalette);

		Release(lpDDPalette);

		memset(&ddsd, 0, sizeof(DDSURFACEDESC));
		ddsd.dwSize  = sizeof(DDSURFACEDESC);
		ddsd.dwFlags = DDSD_HEIGHT | DDSD_WIDTH;

		// Get size of surface.
		material.lpDDSurfaceSystem->GetSurfaceDesc(&ddsd);

		HDC hdc;
		ddrval = material.lpDDSurfaceSystem->GetDC(&hdc);
		if(DD_OK != ddrval)
			return ddrval;

		BitBlt(hdc, 0, 0, ddsd.dwWidth, ddsd.dwHeight, hBitmapDC, x, y, SRCCOPY);
		material.lpDDSurfaceSystem->ReleaseDC(hdc);

		DeleteDC(hBitmapDC);
		DeleteObject(hBitmap);

		// Get the material handle
		if(NULL != material.lpDDSurfaceVideo)
		{
			LPDIRECT3DTEXTURE2 lpD3DTextureVideo = NULL;
			ddrval = material.lpDDSurfaceVideo->QueryInterface(IID_IDirect3DTexture2, (LPVOID *)&lpD3DTextureVideo);
			if(DD_OK != ddrval)
				return ddrval;

			LPDIRECT3DTEXTURE2 lpD3DTextureSystem = NULL;
			ddrval = material.lpDDSurfaceSystem->QueryInterface(IID_IDirect3DTexture2, (LPVOID *)&lpD3DTextureSystem);
			if(DD_OK != ddrval)
				return ddrval;

			ddrval = lpD3DTextureVideo->Load(lpD3DTextureSystem);
			if(DD_OK != ddrval)
				return ddrval;

			ddrval = lpD3DTextureVideo->GetHandle(lpD3DDevice, &material.hD3DTexture);
			if(D3D_OK != ddrval)
				return ddrval;

			Release(lpD3DTextureSystem);
			Release(lpD3DTextureVideo);

			// Set of color key(default is black color)
			DDCOLORKEY ddck;
			ddck.dwColorSpaceHighValue = RGB_MAKE(0, 0, 0);
			ddck.dwColorSpaceLowValue	= ddck.dwColorSpaceHighValue;

			ddrval = material.lpDDSurfaceVideo->SetColorKey(DDCKEY_SRCBLT, &ddck);
			if(DD_OK != ddrval)
				return ddrval;
		}
		else
		{
			LPDIRECT3DTEXTURE2 lpD3DTextureSystem = NULL;
			ddrval = material.lpDDSurfaceSystem->QueryInterface(IID_IDirect3DTexture2, (LPVOID *)&lpD3DTextureSystem);
			if(DD_OK != ddrval)
				return ddrval;

			ddrval = lpD3DTextureSystem->GetHandle(lpD3DDevice, &material.hD3DTexture);
			if(D3D_OK != ddrval)
				return ddrval;

			Release(lpD3DTextureSystem);

			// Set of color key (default is black color)
			DDCOLORKEY ddck;
			ddck.dwColorSpaceHighValue = RGB_MAKE(0, 0, 0);
			ddck.dwColorSpaceLowValue	= ddck.dwColorSpaceHighValue;

			ddrval = material.lpDDSurfaceSystem->SetColorKey(DDCKEY_SRCBLT, &ddck);
			if(DD_OK != ddrval)
				return ddrval;
		}
	}

	// Create material
	ddrval = lpD3D->CreateMaterial(&material.lpD3DMaterial, NULL);
	if(D3D_OK != ddrval)
		return ddrval;

	D3DMATERIAL d3dMaterial;
	memset(&d3dMaterial, 0, sizeof(D3DMATERIAL));

	d3dMaterial.dwSize	  = sizeof(D3DMATERIAL);
	d3dMaterial.diffuse	  = diffuse;
	d3dMaterial.ambient	  = ambient;
	d3dMaterial.specular   = specular;
	d3dMaterial.power 	  = D3DVAL(40.0);
	d3dMaterial.dwRampSize = 16;

	if(NULL != material.hD3DTexture)
		d3dMaterial.hTexture = material.hD3DTexture;

	ddrval = material.lpD3DMaterial->SetMaterial(&d3dMaterial);
	if(D3D_OK != ddrval)
		return ddrval;

	ddrval = material.lpD3DMaterial->GetHandle(lpD3DDevice, &material.hD3DMaterial);
	if(D3D_OK != ddrval)
		return ddrval;

	return D3D_OK;
}

HRESULT CALLBACK EnumTextureFormatsCallback(LPDDSURFACEDESC lpDdsd, LPVOID lParam)
{
	FindMaterialData *findData = (FindMaterialData *)lParam;

	DDPIXELFORMAT ddpf = lpDdsd->ddpfPixelFormat;

	// We use GetDC/BitBlt to init material so we only want to use formats that GetDC will support.
	if(ddpf.dwFlags & (DDPF_ALPHA | DDPF_ALPHAPIXELS))
		return DDENUMRET_OK;

	if((8 >= ddpf.dwRGBBitCount) && !(ddpf.dwFlags & (DDPF_PALETTEINDEXED8 | DDPF_PALETTEINDEXED4)))
		return DDENUMRET_OK;

	if((8 < ddpf.dwRGBBitCount) && !(ddpf.dwFlags & DDPF_RGB))
		return DDENUMRET_OK;

	// BUGBUG GetDC does not work for 1 or 4bpp YET!
	if(8 > ddpf.dwRGBBitCount)
		return DDENUMRET_OK;

	// Keep the material format that is nearest to the bitmap we have
	if((0 == findData->ddpf.dwRGBBitCount) ||
		((ddpf.dwRGBBitCount >= findData->bpp) &&
		((UINT)(ddpf.dwRGBBitCount - findData->bpp) < (UINT)(findData->ddpf.dwRGBBitCount - findData->bpp))))
	{
		findData->ddpf = ddpf;

		return DDENUMRET_CANCEL;
	}

	return DDENUMRET_OK;
}

void ReleaseMaterial(D3DTEXTUREMATERIAL &material)
{
	Release(material.lpD3DMaterial);
	Release(material.lpDDSurfaceSystem);
	Release(material.lpDDSurfaceVideo);
}

HRESULT RestoreMaterial(D3DTEXTUREMATERIAL &material)
{
	HRESULT ddrval = material.lpDDSurfaceSystem->Restore();
	if(DD_OK != ddrval)
		return ddrval;

	if(NULL != material.lpDDSurfaceVideo)
	{
		// Get the material handle
		LPDIRECT3DTEXTURE2 lpD3DTextureSystem = NULL;
		ddrval = material.lpDDSurfaceSystem->QueryInterface(IID_IDirect3DTexture2, (LPVOID *)&lpD3DTextureSystem);
		if(DD_OK != ddrval)
			return ddrval;

		LPDIRECT3DTEXTURE2 lpD3DTextureVideo = NULL;
		ddrval = material.lpDDSurfaceVideo->QueryInterface(IID_IDirect3DTexture2, (LPVOID *)&lpD3DTextureVideo);
		if(DD_OK != ddrval)
			return ddrval;

		ddrval = lpD3DTextureVideo->Load(lpD3DTextureSystem);
		if(DD_OK != ddrval)
			return ddrval;

		Release(lpD3DTextureSystem);
		Release(lpD3DTextureVideo);
	}

	return DD_OK;
}

