//////////////////////////////////////////////////////////////////////////////////
// Project Name: [ C3D Class Library - C3D.lib ]
// Author:       [ Dan Farley - 97308096@brookes.ac.uk ]
// Source File:  [ C3D_Matrix Implementation ]
// Revision:     [ 1.6 ]
//////////////////////////////////////////////////////////////////////////////////
#include "C3D.h"
#include <math.h>

//////////////////////////////////////////////////////////////////////////////////
// Multiply
//////////////////////////////////////////////////////////////////////////////////
C3D_Matrix Multiply(const C3D_Matrix a, const C3D_Matrix b)
{
	C3D_Matrix ret;

	for(int i = 0; i < 4; i++)
		for(int j = 0; j < 4; j++)
			for(int k = 0; k < 4; k++)
				ret(i, j) += a(k, j) * b(i, k);

	return ret;
}

//////////////////////////////////////////////////////////////////////////////////
// C3D_Matrix constructor
//////////////////////////////////////////////////////////////////////////////////
C3D_Matrix::C3D_Matrix(void)
{
	_11 = _12 = _13 = _14 = 0.0f;
	_21 = _22 = _23 = _24 = 0.0f;
	_31 = _32 = _33 = _34 = 0.0f;
	_41 = _42 = _43 = _44 = 0.0f;
}

//////////////////////////////////////////////////////////////////////////////////
// C3D_Matrix = operator
//////////////////////////////////////////////////////////////////////////////////
C3D_Matrix C3D_Matrix::operator = (const C3D_Matrix &m)
{
	_11 = m._11; _12 = m._12; _13 = m._13; _14 = m._14;
	_21 = m._21; _22 = m._22; _23 = m._23; _24 = m._24;
	_31 = m._31; _32 = m._32; _33 = m._33; _34 = m._34;
	_41 = m._41; _42 = m._42; _43 = m._43; _44 = m._44;

	return *this;
}

//////////////////////////////////////////////////////////////////////////////////
// C3D_Matrix + operator
//////////////////////////////////////////////////////////////////////////////////
C3D_Matrix C3D_Matrix::operator + (const C3D_Matrix &m)
{
	_11 += m._11; _12 += m._12; _13 += m._13; _14 += m._14;
	_21 += m._21; _22 += m._22; _23 += m._23; _24 += m._24;
	_31 += m._31; _32 += m._32; _33 += m._33; _34 += m._34;
	_41 += m._41; _42 += m._42; _43 += m._43; _44 += m._44;

	return *this;
}

//////////////////////////////////////////////////////////////////////////////////
// C3D_Matrix - operator
//////////////////////////////////////////////////////////////////////////////////
C3D_Matrix C3D_Matrix::operator - (const C3D_Matrix &m)
{
	_11 -= m._11; _12 -= m._12; _13 -= m._13; _14 -= m._14;
	_21 -= m._21; _22 -= m._22; _23 -= m._23; _24 -= m._24;
	_31 -= m._31; _32 -= m._32; _33 -= m._33; _34 -= m._34;
	_41 -= m._41; _42 -= m._42; _43 -= m._43; _44 -= m._44;

	return *this;
}

//////////////////////////////////////////////////////////////////////////////////
// C3D_Matrix Zero
//////////////////////////////////////////////////////////////////////////////////
void C3D_Matrix::Zero(void)
{
	_11 = _12 = _13 = _14 = 0.0f;
	_21 = _22 = _23 = _24 = 0.0f;
	_31 = _32 = _33 = _34 = 0.0f;
	_41 = _42 = _43 = _44 = 0.0f;
}

//////////////////////////////////////////////////////////////////////////////////
// C3D_Matrix Identity
//////////////////////////////////////////////////////////////////////////////////
void C3D_Matrix::Identity(void)
{
	Zero();
	_11 = _22 = _33 = _44 = 1.0f;
}

//////////////////////////////////////////////////////////////////////////////////
// C3D_Matrix RotateX
//////////////////////////////////////////////////////////////////////////////////
void C3D_Matrix::RotateX(float rx)
{
	float cosine = cos(rx);
	float sine = sin(rx);

	_22 = cosine;
	_33 = cosine;
	_32 = -sine;
	_23 = sine;
}

//////////////////////////////////////////////////////////////////////////////////
// C3D_Matrix RotateY
//////////////////////////////////////////////////////////////////////////////////
void C3D_Matrix::RotateY(float ry)
{
	float cosine = cos(ry);
	float sine = sin(ry);

	_11 = cosine;
	_33 = cosine;
	_13 = -sine;
	_31 = sine;
}

//////////////////////////////////////////////////////////////////////////////////
// C3D_Matrix RotateZ
//////////////////////////////////////////////////////////////////////////////////
void C3D_Matrix::RotateZ(float rz)
{
	float cosine = cos(rz);
	float sine = sin(rz);

	_11 = cosine;
	_22 = cosine;
	_21 = -sine;
	_12 = sine;
}

//////////////////////////////////////////////////////////////////////////////////
// C3D_Matrix Translate
//////////////////////////////////////////////////////////////////////////////////
void C3D_Matrix::Translate(float dx, float dy, float dz)
{
	_41 = dx;
	_42 = dy;
	_43 = dz;
}

//////////////////////////////////////////////////////////////////////////////////
// C3D_Matrix Scale
//////////////////////////////////////////////////////////////////////////////////
void C3D_Matrix::Scale(float s)
{
	_11 = s;
	_22 = s;
	_33 = s;
}

//////////////////////////////////////////////////////////////////////////////////
// C3D_Matrix Scale
//////////////////////////////////////////////////////////////////////////////////
void C3D_Matrix::Scale(float sx, float sy, float sz)
{
	_11 = sx;
	_22 = sy;
	_33 = sz;
}

//////////////////////////////////////////////////////////////////////////////////
// C3D_Matrix Print
//////////////////////////////////////////////////////////////////////////////////
void C3D_Matrix::Print(void)
{
	printf("%f, %f, %f, %f\n", _11, _12, _13, 14);
	printf("%f, %f, %f, %f\n", _21, _22, _23, 24);
	printf("%f, %f, %f, %f\n", _31, _32, _33, 34);
	printf("%f, %f, %f, %f\n", _41, _42, _43, 44);
}

//////////////////////////////////////////////////////////////////////////////////
// C3D_Matrix SetView
//////////////////////////////////////////////////////////////////////////////////
void C3D_Matrix::SetView(D3DVECTOR from, D3DVECTOR at, D3DVECTOR world_up, float roll)
{
	D3DVECTOR up, right, view;
	Identity();

	view = Normalize(at - from);
	right = CrossProduct(world_up, view);
	up = CrossProduct(view, right);
	right = Normalize(right);
	up = Normalize(up);

	_11	= right.x;
	_21	= right.y;
	_31	= right.z;
	_12	= up.x;
	_22	= up.y;
	_32	= up.z;
	_13	= view.x;
	_23	= view.y;
	_33	= view.z;
	_41	= -DotProduct(right, from);
	_42	= -DotProduct(up, from);
	_43	= -DotProduct(view, from);

	if(roll != 0.0f)
	{
//		C3D_Matrix mat.Rotate(-roll);
//		this = Multiply(mat, this);
	}
}

//////////////////////////////////////////////////////////////////////////////////
// C3D_Matrix SetProjection
//////////////////////////////////////////////////////////////////////////////////
void C3D_Matrix::SetProjection(float near_plane, float far_plane, float fov)
{
	float c = (float)cos(fov * 0.5);
	float s = (float)sin(fov * 0.5);
	float Q = s / (1.0f - near_plane / far_plane);

	Zero();
	_11 = c;
	_22 = c;
	_33 = Q;
	_43	= -Q * near_plane;
	_34	= s;
}
