// 3dplus.h
//
// Copyright (c) Nigel Thompson 1996
//
// This file should be included in your application's main
// include file so that it is available to all modules that
// need access the the 3D classes
//

#ifndef _3DPLUS_H_
#define _3DPLUS_H_

// NOTE: This set of C++ classes assumes the Reality Lab APIs
// are defined using their floating point format so an RLValue is
// a double.
#ifdef FIXED_POINT_API
#pragma message("Invalid use of fixed-point API in rendering engine")
ERROR - Rendering API
#endif

// disable warning C4244 which is issued when a const double is converted
// to a float. This happens when we initialise RLValues
#pragma warning ( disable : 4244)
#pragma warning ( disable : 4305)

// include the Direct Draw classes
#include "3dDirect.h"

// include the resource symbols
#include "3dRes.h"

// global helper functions
extern CString Get3dError(HRESULT hr);

// File dialog filter strings
#define _3DOBJ_LOADFILTER "Shape Files (*.x)|*.x|All Files (*.*)|*.*||"
#define _3DOBJ_SAVEFILTER "Shape Files (*.x)|*.x|All files (*.*)|*.*||"

///////////////////////////////////////////////////////////////////
// C3dObject

class C3dObject : public CObject
{
public:
	DECLARE_DYNAMIC(C3dObject);
    C3dObject();
    virtual ~C3dObject();
	HRESULT GetResult() {return m_hr;}

protected:
	HRESULT m_hr; // result of last system call

};

///////////////////////////////////////////////////////////////////
// C3dEngine

class C3dDevice;
class C3dCamera;

class C3dEngine : public C3dObject
{
public:
	DECLARE_DYNAMIC(C3dEngine);
    C3dEngine();
    virtual ~C3dEngine();
    BOOL CreateFrame(IDirect3DRMFrame* pIParentFrame, IDirect3DRMFrame** pIFrame);
	IDirect3DRM* GetInterface() {return m_pIRM;}
    BOOL CreateViewport(C3dDevice* pDevice,
						C3dCamera* pCamera,
						int x, int y,
						int cx, int cy,
						IDirect3DRMViewport** pIViewport);
    BOOL CreateLight(D3DRMLIGHTTYPE type,
                     double r, double g, double b,
                     IDirect3DRMLight** piLight);
    BOOL CreateMeshBuilder(IDirect3DRMMeshBuilder** pIBld);
    BOOL CreateTexture(D3DRMIMAGE* pImage, IDirect3DRMTexture** pITexture);
    BOOL CreateWrap(D3DRMWRAPTYPE type,
                    IDirect3DRMFrame* pIRefFrame,
                    double ox, double oy, double oz,
                    double dx, double dy, double dz,
                    double ux, double uy, double uz,
                    double ou, double ov,
                    double su, double sv,
                    IDirect3DRMWrap** pIWrap);
	BOOL CreateMaterial(double sharp, IDirect3DRMMaterial** pIMat);

protected:
	IDirect3DRM* m_pIRM; // WinRealityLab interface 


private:
    static BOOL s_bInitialized;
    static IDirect3DRM* s_pRL; // main Reality Lab object

    static BOOL Initialize();
};

// the one and only 3dEngine object
extern C3dEngine the3dEngine;   

///////////////////////////////////////////////////////////////////
// C3dVisual

class C3dVisual : public C3dObject
{
public:
	DECLARE_DYNAMIC(C3dVisual);
	C3dVisual();
	const char* GetName() {return m_strName;}
	void SetName(const char* pszName) {m_strName = pszName;}
	IDirect3DRMVisual* GetInterface()
	{return _GetVisualInterface();}

protected:
	virtual IDirect3DRMVisual* _GetVisualInterface()
		{return NULL;}

	CString m_strName;
};

///////////////////////////////////////////////////////////////////
// C3dVector

class C3dVector	: public _D3DVECTOR

{
public:
	C3dVector();
	virtual ~C3dVector();
	C3dVector(const D3DVECTOR& r);
	C3dVector(double x, double y, double z);
    C3dVector& operator = (const D3DVECTOR& r);
    friend C3dVector operator + (const D3DVECTOR& a, const D3DVECTOR& b);
    C3dVector& operator += (const D3DVECTOR& r);
    friend C3dVector operator - (const D3DVECTOR& a, const D3DVECTOR& b);
    C3dVector& operator -= (const D3DVECTOR& r);
    friend C3dVector operator * (const D3DVECTOR& a, const D3DVECTOR& b);
    C3dVector& operator *= (const D3DVECTOR& r);
    friend C3dVector operator * (const D3DVECTOR& a, const double s);
    C3dVector& operator *= (const double s);
	C3dVector operator - ();
	double Dot(const D3DVECTOR& a);
	double Mag();
	BOOL Normalize();
	BOOL Coincident(const D3DVECTOR& r);
	C3dVector GenerateUp();
	BOOL IsNull();

public:
	//double m_x, m_y, m_z;
};

///////////////////////////////////////////////////////////////////
// C3dMatrix

class C3dMatrix	: public C3dObject

{
public:
	DECLARE_DYNAMIC(C3dMatrix);
    C3dMatrix();
    virtual ~C3dMatrix();
    C3dMatrix(const C3dMatrix& r);
    C3dMatrix(double v00, double v01, double v02, double v03,
              double v10, double v11, double v12, double v13,
              double v20, double v21, double v22, double v23,
              double v30, double v31, double v32, double v33);
    C3dMatrix& operator = (const C3dMatrix& r);
    friend C3dMatrix operator + (const C3dMatrix& a, const C3dMatrix& b);
    C3dMatrix& operator += (const C3dMatrix& r);
    friend C3dMatrix operator * (const C3dMatrix& a, const C3dMatrix& b);
    C3dMatrix& operator *= (const C3dMatrix& r);
    friend C3dVector operator * (const C3dMatrix& m, const D3DVECTOR& v);
    void Rotate(double rx, double ry, double rz);
    void Translate(double dx, double dy, double dz);
	void Scale(double sx, double sy, double sz);
	void Scale(double s) {Scale(s, s, s);}
	void MakeUnit();
	void Initialize(D3DRMMATRIX4D& rlm);

	// elements
	double m_00, m_01, m_02, m_03;
	double m_10, m_11, m_12, m_13;
	double m_20, m_21, m_22, m_23;
	double m_30, m_31, m_32, m_33;
};

#if 0
///////////////////////////////////////////////////////////////////
// C3dQuaternion

class C3dQuaternion : public _D3DRMQUATERNION
{
public:
	C3dQuaternion();
	C3dQuaternion(const D3DVECTOR& v, const double theta);
	C3dQuaternion(const D3DVECTOR& vDir, const D3DVECTOR& vUp);
	virtual ~C3dQuaternion();
    C3dQuaternion& operator = (const D3DRMQUATERNION& r);
};
#endif

////////////////////////////////////////////////////////////////////
// C3dMaterial

class C3dMaterial :	public C3dObject
{
public:
	DECLARE_DYNAMIC(C3dMaterial);
	C3dMaterial();
	virtual ~C3dMaterial();
	void SetEmissiveColor(double r, double g, double b);
	void SetSpecularPower(double p);
	void SetSpecularColor(double r, double g, double b);
	IDirect3DRMMaterial* GetInterface() {return m_pIMat;}

protected:
	IDirect3DRMMaterial* m_pIMat;
};

///////////////////////////////////////////////////////////////////
// C3dFrame
//
// Note: if pRef is NULL, this implies the frame's parent is to be used
// as the reference frame
//

class C3dTexture;

class C3dFrame : public C3dVisual
{
public:
	DECLARE_DYNAMIC(C3dFrame);
    C3dFrame();
    virtual ~C3dFrame();
    IDirect3DRMFrame* GetInterface() {return this == NULL ? NULL : m_pIFrame;}
    BOOL Create(C3dFrame* pParent);
    void SetPosition(double x, double y, double z, C3dFrame* pRef = NULL);
    void SetPosition(D3DVECTOR& p, C3dFrame* pRef = NULL)
		{SetPosition(p.x, p.y, p.z, pRef);}
    void GetPosition(double& x, double& y, double& z, C3dFrame* pRef = NULL);
    void GetPosition(D3DVECTOR& p, C3dFrame* pRef = NULL);
    void SetRotation(double x, double y, double z, double t, C3dFrame* pRef = NULL);
    void SetDirection(double dx, double dy, double dz,
                      double ux, double uy, double uz, C3dFrame* pRef = NULL);
    void GetDirection(double& dx, double& dy, double& dz,
                      double& ux, double& uy, double& uz, C3dFrame* pRef = NULL);
    void GetDirection(D3DVECTOR& d,D3DVECTOR& u, C3dFrame* pRef = NULL);
    void SetDirection(double dx, double dy, double dz, C3dFrame* pRef = NULL);
    void GetDirection(double& dx, double& dy, double& dz, C3dFrame* pRef = NULL);
    void SetDirection(D3DVECTOR& d, D3DVECTOR& u, C3dFrame* pRef = NULL);
    void SetDirection(D3DVECTOR& d, C3dFrame* pRef = NULL)
		{SetDirection(d.x, d.y, d.z, pRef);}
	void AddTransform(C3dMatrix& m, D3DRMCOMBINETYPE ct = D3DRMCOMBINE_BEFORE);
	void AddRotation(double x, double y, double z,
					 double t, D3DRMCOMBINETYPE ct = D3DRMCOMBINE_BEFORE);
    BOOL AddChild(C3dFrame* pChild);
    BOOL RemoveChild(C3dFrame* pChild);
	void RemoveAllChildren();
    void SetVelocity(double x, double y, double z, C3dFrame* pRef = NULL);
    BOOL SetTexture(C3dTexture* pTexture);
	void Attach(IDirect3DRMFrame* pIFrame);
	D3DVECTOR Transform(const D3DVECTOR& vLocal);
	BOOL IsChildOf(C3dFrame* pFrame);
	BOOL IsPartOf(C3dFrame* pFrame);
	void SetMaterialMode(D3DRMMATERIALMODE mode);
	BOOL AddVisual(C3dVisual* pVisual);
	BOOL RemoveVisual(C3dVisual* pVisual);

protected:
    IDirect3DRMFrame* m_pIFrame;

	IDirect3DRMFrame* _GetRef(C3dFrame* pRef);

};

class C3dFrameList : protected CObList
{
public:
	C3dFrameList();
	virtual ~C3dFrameList();
	void Append(C3dFrame* pFrame) {CObList::AddTail(pFrame);}
	void Remove(C3dFrame* pFrame);
	void Delete(C3dFrame* pFrame);
	void DeleteAll();
	C3dFrame* GetNext(POSITION& rPosition)
		{return (C3dFrame*) CObList::GetNext(rPosition);}

};

///////////////////////////////////////////////////////////////////
// C3dCamera

class C3dCamera : public C3dFrame
{
public:
	DECLARE_DYNAMIC(C3dCamera);

protected:

};

///////////////////////////////////////////////////////////////////
// C3dLight

class C3dLight : public C3dFrame
{
public:
	DECLARE_DYNAMIC(C3dLight);
    C3dLight();
    virtual ~C3dLight();
    BOOL Create(D3DRMLIGHTTYPE type, double r, double g, double b);
    BOOL SetColor(double r, double g, double b);
	void GetColor(double& r, double& g, double& b);
	void GetAttenuation(double& c, double& l, double& q);
	double GetRange();
	void SetAttenuation(double c, double l, double q);
	void SetRange(double r);
	IDirect3DRMLight* GetLight() {return m_pILight;}

protected:
   IDirect3DRMLight* m_pILight;

};

///////////////////////////////////////////////////////////////////
// C3dAmbLight

class C3dAmbLight : public C3dLight
{
public:
	DECLARE_DYNAMIC(C3dAmbLight);
    BOOL Create(double r, double g, double b);
};

///////////////////////////////////////////////////////////////////
// C3dDirLight

class C3dDirLight : public C3dLight
{
public:
	DECLARE_DYNAMIC(C3dDirLight);
    BOOL Create(double r, double g, double b);
};

///////////////////////////////////////////////////////////////////
// C3dParPtLight

class C3dParPtLight : public C3dLight
{
public:
	DECLARE_DYNAMIC(C3dParPtLight);
    BOOL Create(double r, double g, double b);
};

///////////////////////////////////////////////////////////////////
// C3dPtLight

class C3dPtLight : public C3dLight
{
public:
	DECLARE_DYNAMIC(C3dPtLight);
    BOOL Create(double r, double g, double b);
};

///////////////////////////////////////////////////////////////////
// C3dSpotLight

class C3dSpotLight : public C3dLight
{
public:
	DECLARE_DYNAMIC(C3dSpotLight);
    BOOL Create(double r, double g, double b);
};

///////////////////////////////////////////////////////////////////
// C3dImage and CImageList

class C3dImage : public C3dVisual
{
public:
	DECLARE_DYNAMIC(C3dImage);
    C3dImage();
    virtual ~C3dImage();
    BOOL Load(CFile* fp);
    BOOL Load(const char* pszFilename);
	BOOL Load() {return Load((const char*) NULL);}
    BOOL Load(UINT uiResid)
		{return LoadResource(MAKEINTRESOURCE(uiResid));}
	BOOL LoadResource(const char* pszResname);
    D3DRMIMAGE* GetObject() {return &m_rlimg;}
	int C3dImage::GetWidth() {return m_rlimg.width;}
	int C3dImage::GetHeight() {return m_rlimg.height;}
	BOOL SetNumPhases(int iPhases);
	int GetNumPhases() {return m_iPhases;}
	BOOL SetPhase(int iPhase);
	int GetPhase() {return m_iCurPhase;}

protected:
    void _Create(BITMAPINFOHEADER* pBMI, RGBQUAD* pRGB, BYTE* pBits);
    void _ReleaseMemory();
	virtual void _OnImageChanged() {;}

    D3DRMIMAGE m_rlimg;
	BYTE* m_pBits;
	int m_iTotalHeight;
	int m_iPhases;
	int m_iCurPhase;
};

class C3dImageList : protected CObList
{
public:
	C3dImageList();
	virtual ~C3dImageList();
	void Append(C3dImage* pImage) {CObList::AddTail(pImage);}
	void Remove(C3dImage* pImage);
	void Delete(C3dImage* pImage);
	void DeleteAll();
	C3dImage* GetNext(POSITION& rPosition)
		{return (C3dImage*) CObList::GetNext(rPosition);}
	POSITION GetHeadPosition()
		{return CObList::GetHeadPosition();}
	C3dImage* Find(const char* pszName);

};

// handy helper
extern BOOL DIBSave(const char* pszFilename, BITMAPINFO* pBMI, void* pBits);
extern int DIBColorEntries(BITMAPINFOHEADER* pBIH); 
extern int DIBStorageWidth(BITMAPINFOHEADER* pBIH);

///////////////////////////////////////////////////////////////////
// C3dShape

typedef double (* SURFHTFN)(double x, double z, void* pArg);
typedef double (* SOLIDRFN)(double z, void* pArg);

class C3dViewport;

class C3dShape : public C3dFrame
{
public:
	DECLARE_DYNAMIC(C3dShape);
    C3dShape();
    virtual ~C3dShape();
	void New();
    BOOL Create(D3DVECTOR* pVectors, int iVectors,
				D3DVECTOR* pNormals, int iNormals,
				int* pFaceData, BOOL bAutoGen = FALSE);
    BOOL Create(D3DVECTOR* pVectors, int iVectors, int* pFaceData, BOOL bAutoGen = FALSE)
        {return Create(pVectors, iVectors, NULL, 0, pFaceData, bAutoGen);}
    BOOL AddFaces(D3DVECTOR* pVectors, int iVectors,
				  D3DVECTOR* pNormals, int iNormals,
				  int* pFaceData);
	BOOL CreateCuboid(double x, double y, double z);
    BOOL CreateCube(double side)
		{return CreateCuboid(side, side, side);}
	const char* Load(const char* pszFileName);
	const char* Load() {return Load("");}
	BOOL Load(UINT uiResid);
	BOOL Save(const char* pszFileName = NULL);
    int GetFaceCount();
    BOOL SetFaceColor(int iFace, double r, double g, double b);
    BOOL SetColor(double r, double g, double b);
    BOOL SetFrameColor(double r, double g, double b);
	BOOL CreateSphere(double r, int nBands = 16);
    BOOL CreateSurface(double x1, double x2, double dx,
                       double z1, double z2, double dz,
                       SURFHTFN pfnHeight, void* pArg);
	BOOL CreateRSolid(double z1, double z2, double dz,
					  BOOL bClosed1, BOOL bClosed2,
					  SOLIDRFN pfnRad, void* pArg,
					  int nFacets = 8);
	BOOL CreateCone(double x1, double y1, double z1,
					double r1, BOOL bClosed1,
					double x2, double y2, double z2,
					double r2, BOOL bClosed2,
					int nFacets = 8);
	BOOL CreateRod(double x1, double y1, double z1,
					double x2, double y2, double z2,
					double r, int nFacets = 8)
		{return CreateCone(x1, y1, z1, r, TRUE, x2, y2, z2, r, TRUE, nFacets);}
    IDirect3DRMVisual* GetVisual() {return m_pIVisual;}
    void AttachVisual(IDirect3DRMVisual* pIVisual);
    BOOL GetBox(double& x1, double& y1, double& z1,
                double& x2, double& y2, double& z2);
	BOOL SetMaterial(C3dMaterial* pMaterial);
    BOOL SetTexture(C3dTexture* pTexture);
    BOOL SetFaceTexture(int iFirst, int iCount, C3dTexture* pTexture);
    BOOL SetFaceTexture(int iFace, C3dTexture* pTexture)
		{return SetFaceTexture(iFace, 1, pTexture);}
	IDirect3DRMMeshBuilder* GetMeshBuilder() {return m_pIMeshBld;}
	C3dVector GetFaceNormal(int iFace);
	C3dShape* Clone();
#if 0
	void Add(C3dShape* pShape);
#endif
	BOOL C3dShape::HitTest(CPoint pt,
						   C3dViewport* pViewport,
						   int* piFace = NULL,
						   D3DVECTOR* pvHit = NULL);
	BOOL CreateShadow(C3dLight*	pLight,
					  D3DVECTOR& vPt,
					  D3DVECTOR& vNormal);


	C3dImageList m_ImgList;		// list of associated images and tex maps

protected:
	virtual IDirect3DRMVisual* _GetVisualInterface()
		{return m_pIVisual;}

    IDirect3DRMVisual* m_pIVisual;
	IDirect3DRMMeshBuilder* m_pIMeshBld;
};

class C3dShapeList : protected CObList
{
public:
	C3dShapeList();
	virtual ~C3dShapeList();
	void Append(C3dShape* pShape) {CObList::AddTail(pShape);}
	void Remove(C3dShape* pShape);
	void Delete(C3dShape* pShape);
	void DeleteAll();
#if 0 // not used
	C3dShape* Find(C3dShape* pShape);
#endif
	C3dShape* GetNext(POSITION& rPosition)
		{return (C3dShape*) CObList::GetNext(rPosition);}

};

////////////////////////////////////////////////////////////////////
// C3dAnimKey, C3dAnimation

class C3dAnimKey : public C3dObject
{
public:
	C3dAnimKey(double time,
			   const D3DVECTOR& pos,
			   const D3DVECTOR& dir,
			   const D3DVECTOR& up)
	: m_vPos(pos), m_vDir(dir), m_vUp(up), m_dTime(time)
	{
	}

public:
	C3dVector m_vPos;
	C3dVector m_vDir;
	C3dVector m_vUp;
	double m_dTime;
};

class C3dAnimation : private CObList
{
public:
	C3dAnimation();
	virtual ~C3dAnimation();
	void Attach(C3dFrame* pFrame);
	BOOL AddKey(C3dAnimKey* pKey);
	BOOL AddKey(double time);
	BOOL SetTime(double time);
	double GetLength();
	void DeleteAll();
	int GetNumKeys() {return GetCount();}
	double GetCurTime() {return m_dCurTime;}

protected:
	C3dFrame* m_pFrame;
	double m_dCurTime;
};

///////////////////////////////////////////////////////////////////
// C3dDevice

class C3dDevice : public C3dObject
{
public:
	DECLARE_DYNAMIC(C3dDevice);
    C3dDevice();
    virtual ~C3dDevice();
	BOOL Create(CDirect3D* pD3D);
	int GetWidth();
	int GetHeight();
    void Update();
    IDirect3DRMDevice* GetInterface() {return m_pIDevice;} 
	void SetQuality(D3DRMRENDERQUALITY quality);

protected:
    IDirect3DRMDevice* m_pIDevice;
};

///////////////////////////////////////////////////////////////////
// C3dViewport

class C3dViewport : public C3dObject
{
public:
	DECLARE_DYNAMIC(C3dViewport);
    C3dViewport();
    virtual ~C3dViewport();
    void Clear();
    BOOL Create(C3dDevice* pDevice,
                C3dCamera* pCamera,
                int x, int y,
                int cx, int cy);
	void ForceUpdate(RECT* pRect);
    BOOL Render(C3dFrame* pFrame);
    IDirect3DRMViewport* GetInterface() {return m_pIViewport;}
	C3dShape* HitTest(CPoint pt);
	BOOL SetFieldOfView(double f);

protected:
    IDirect3DRMViewport* m_pIViewport;
};

///////////////////////////////////////////////////////////////////
// C3dStage

class C3dScene;

class C3dStage : public C3dFrame
{
public:
	DECLARE_DYNAMIC(C3dStage);
    C3dStage();
    virtual ~C3dStage();
	BOOL Create(CDirect3D* pD3D);
	void Clear()
		{m_Viewport.Clear();}
	void ForceUpdate(RECT* pRect)
		{m_Viewport.ForceUpdate(pRect);}
    void Render();
	void SetScene(C3dScene* pScene);
    void SetBackground(C3dScene* pScene);
	void SetQuality(D3DRMRENDERQUALITY quality);
    C3dCamera* GetCamera() {return &m_Camera;}
	C3dViewport* GetViewport() {return &m_Viewport;}


protected:
    C3dDevice m_Device;     // the device we render to
    C3dViewport m_Viewport; // the viewport (TODO: multiple vps)
    C3dCamera m_Camera;     // the camera
    C3dScene* m_pScene;     // current scene
	D3DRMRENDERQUALITY m_Quality;
};

///////////////////////////////////////////////////////////////////
// C3dTexture

class C3dTexture : public C3dImage
{
public:
    DECLARE_DYNAMIC(C3dTexture);
    C3dTexture();
    virtual ~C3dTexture();
	BOOL Create();
    BOOL Load(CFile* fp);
    BOOL Load(const char* pszFilename);
    BOOL LoadResource(const char* pszResname);
    BOOL Load(UINT uiResid);
    IDirect3DRMTexture* GetInterface() {return m_pITexture;}
	BOOL SetTransparency(double r, double g, double b);
	BOOL SetColors(int iColors);
	BOOL SetShades(int iShades);

protected:
	virtual IDirect3DRMVisual* _GetVisualInterface()
		{return m_pITexture;}
	virtual void _OnImageChanged();

    IDirect3DRMTexture* m_pITexture;
};

///////////////////////////////////////////////////////////////////
// C3dWrap

class C3dWrap : public C3dObject
{
public:
    DECLARE_DYNAMIC(C3dWrap);
    C3dWrap();
    virtual ~C3dWrap();
    IDirect3DRMWrap* GetInterface() {return m_pIWrap;}
    BOOL Create(D3DRMWRAPTYPE type,
                C3dFrame* pRefFrame,
                double ox, double oy, double oz,
                double dx, double dy, double dz,
                double ux, double uy, double uz,
                double ou, double ov,
                double su, double sv);
    BOOL Apply(C3dShape* pShape);
	BOOL ApplyRelative(C3dFrame* pFrame, C3dShape* pShape);
    BOOL Apply(C3dShape* pShape, int iFace);

protected:
    IDirect3DRMWrap* m_pIWrap;

};

///////////////////////////////////////////////////////////////////
// C3dChromeWrap

class C3dChromeWrap : public C3dWrap
{
public:
    DECLARE_DYNAMIC(C3dChromeWrap);
    C3dChromeWrap();
    virtual ~C3dChromeWrap();
	BOOL Create(C3dShape* pShape, C3dCamera* pCamera);

	C3dShape* m_pShape;
};
	
///////////////////////////////////////////////////////////////////
// C3dSprite

class C3dSprite : public C3dFrame
{
public:
	DECLARE_DYNAMIC(C3dSprite);
	C3dSprite();
	virtual ~C3dSprite();
	BOOL Create(C3dScene* pScene,
				double x, double y, double z,
				double scale,
				UINT uiIDBitmap,
				int iPhases = 1);
	BOOL SetPhase(int iPhase);
	int GetNumPhases() {return m_Tex.GetNumPhases();}
	int GetPhase() {return m_Tex.GetPhase();}

protected:
	C3dTexture m_Tex;
};
	
///////////////////////////////////////////////////////////////////
// C3dScene

class C3dScene : public C3dFrame
{
public:
	DECLARE_DYNAMIC(C3dScene);
    C3dScene();
    virtual ~C3dScene();
    BOOL Create();
    void SetAmbientLight(double r, double g, double b)
        {m_AmbLight.SetColor(r, g, b);}
    BOOL AddLight(C3dLight* pLight);
    void Move(double delta);
	void SetBackground(double r, double g, double b);
	void SetBackground(C3dImage* pImage);
	void SetCameraPosition(D3DVECTOR& p);
	void SetCameraDirection(D3DVECTOR& d);
	void SetCameraDirection(D3DVECTOR& d, D3DVECTOR& u);
	void SetCameraField(double f);
	C3dAmbLight* GetAmbLight() {return &m_AmbLight;}

public:
	C3dFrameList m_ShapeList;	// list of shapes
	C3dImageList m_ImgList;		// list of associated images
	D3DCOLOR m_clrBkgnd;		// background color
	C3dImage* m_pBkgndImg;		// background image
	C3dStage* m_pStage;			// stage to which we are attached
	C3dVector m_vCamPos;		// camera position
	C3dVector m_vCamDir;		// camera direction
	C3dVector m_vCamUp;			// camera up vector
	double m_dCamField;			// camera field of view

protected:
    C3dAmbLight m_AmbLight; // the ambient light
};

/////////////////////////////////////////////////////////////////////////////
// C3dWnd window

// Callback function used to notify selection changes
typedef void (CALLBACK * SELCHANGEFN)(C3dShape* pShape, CPoint pt, void *pArg);

class C3dController;

class C3dWnd : public CWnd
{
// Construction
public:
	C3dWnd();
	BOOL Create(const char* pszCaption,
				DWORD dwStyle,
				int x, int y,
				int cx, int cy,
				CWnd* pParent = NULL);
	BOOL Create(CWnd* pParent, UINT uiID, D3DCOLORMODEL cm = D3DCOLOR_MONO);

// Attributes
public:

// Operations
public:
	BOOL Update(double delta);
	BOOL SetScene(C3dScene* pScene);
	BOOL AttachController(C3dController* pController);
	C3dCamera* GetCamera()
		{return m_pStage->GetCamera();}
	C3dShape* HitTest(CPoint pt);
	void EnableMouseSelection(SELCHANGEFN pNotifyFn,
							  void* pArg,
							  BOOL bEnable = TRUE);
	C3dStage* GetStage() {return m_pStage;}
	CDirectDraw* GetDD() {return m_pDD;}

// Overrides
	// ClassWizard generated virtual function overrides
	//{{AFX_VIRTUAL(C3dWnd)
	protected:
	virtual void PostNcDestroy( );
	virtual BOOL OnWndMsg( UINT message, WPARAM wParam, LPARAM lParam, LRESULT* pResult );
	//}}AFX_VIRTUAL

// Implementation
public:
	virtual ~C3dWnd();

	// Generated message map functions
protected:
	//{{AFX_MSG(C3dWnd)
	afx_msg void OnActivateApp(BOOL bActive, HTASK hTask);
	//afx_msg void OnActivate(UINT nState, CWnd* pWndOther, BOOL bMinimized);
	afx_msg void OnPaletteChanged(CWnd* pFocusWnd);
	afx_msg int OnCreate(LPCREATESTRUCT lpCreateStruct);
	afx_msg void OnDestroy();
	afx_msg void OnMove(int x, int y);
	afx_msg void OnSize(UINT nType, int cx, int cy);
	afx_msg void OnPaint();
	afx_msg BOOL OnEraseBkgnd( CDC* pDC );
	//}}AFX_MSG
	DECLARE_MESSAGE_MAP()

protected:
	CDirectDraw* m_pDD;
	CDirect3D* m_pD3D;
	BOOL m_bAutoDelete;
	C3dStage* m_pStage;
	C3dScene* m_pScene;
	BOOL m_bEnableUpdates;
	int m_iWidth;		// client width
	int m_iHeight;		// client height
	CRect m_rcClient;	// client area in screen coords
	BOOL m_bRepaintAll;	// window size changed etc.
	BOOL m_bAppActive;
	CBitmap* m_pGrabImg;	// snapshot of the image
	CPalette* m_pGrabPal;
	C3dController* m_pController;
	BOOL m_bEnableMouseSelection;
	SELCHANGEFN m_pSelChangeFn;
	void* m_pSelChangeArg;
	D3DCOLORMODEL m_ColorModel;

	BOOL CreateStage();
	void ReleaseStage();
	void ReleaseDD();
};

// include the input device classes
#include "3dInpDev.h"

// include the dialog,property page and property sheet classes
#include "3dDlg.h"

// from 3dMath.cpp
extern BOOL _3dPointInPolygon(POINT* Points, int nPoints, CPoint pt);


#endif // _3DPLUS_H_