// 3dAnim.cpp
//
// Copyright (c) Nigel Thompson 1996
//
// Implementation for:
// C3dAnimation
//

#include "stdafx.h"
#include "3dplus.h"

C3dAnimation::C3dAnimation()
{
	m_pFrame = NULL;
	m_dCurTime = 0;
}

// virtual
C3dAnimation::~C3dAnimation()
{
	// Delete all the keys
	DeleteAll();
}

// Attach a frame
void C3dAnimation::Attach(C3dFrame* pFrame)
{
	m_pFrame = pFrame;
}

void C3dAnimation::DeleteAll()
{
	while (!IsEmpty()) {
		C3dAnimKey* pKey = (C3dAnimKey*) RemoveHead();
		ASSERT(pKey);
		delete pKey;
	}
}

// Add a key value based on the current frame's position
BOOL C3dAnimation::AddKey(double time)
{
	if (!m_pFrame) return FALSE;

	// get the frame's position and orientation
	C3dVector p, d, u;
	m_pFrame->GetPosition(p);
	m_pFrame->GetDirection(d, u);

	// Create a new key
	C3dAnimKey* pKey = new C3dAnimKey(time, p, d, u);

	// Add the key to the list
	return AddKey(pKey);
}

BOOL C3dAnimation::AddKey(C3dAnimKey* pNewKey)
{
	if (!pNewKey) return FALSE;

	// update the current time
	m_dCurTime = pNewKey->m_dTime;

	// insert the key in the list
	if (IsEmpty()) {

		AddTail(pNewKey);
		return TRUE;

	}

	// walk the list backwards
	POSITION pos = GetTailPosition();
	ASSERT(pos);
	do {
		POSITION thispos = pos;
		C3dAnimKey* pKey = (C3dAnimKey*) GetPrev(pos);
		if (pKey->m_dTime <= pNewKey->m_dTime) {

			// insert new key after this one
			InsertAfter(thispos, pNewKey);
			return TRUE;
		}
	} while (pos);

	// put it in at the start
	AddHead(pNewKey);

	return TRUE;
}

double C3dAnimation::GetLength()
{
	if (IsEmpty()) return 0;

	C3dAnimKey* pKey = (C3dAnimKey*) GetTail();
	ASSERT(pKey);
	return pKey->m_dTime;
}

BOOL C3dAnimation::SetTime(double time)
{
	m_dCurTime = time;

	if (!m_pFrame) return FALSE;
	if (IsEmpty()) return FALSE;

	// walk the list looking for a pair of keys which bracket
	// this value (or an exact match)
	POSITION pos = GetHeadPosition();
	ASSERT(pos);
	C3dAnimKey* pBefore = (C3dAnimKey*) GetNext(pos);
	ASSERT(pBefore);
	if (pBefore->m_dTime > time) return FALSE; // no key that early

	C3dAnimKey* pAfter = NULL;
	while (pos) {

		pAfter = (C3dAnimKey*) GetNext(pos);
		ASSERT(pAfter);
		if ((pBefore->m_dTime <= time) && (pAfter->m_dTime >= time)) break;
		pBefore = pAfter;
	}

	// Compute the interpolated values
	C3dVector p, d, u;
	double dt;
	if (pAfter != NULL) {
		dt = pAfter->m_dTime - pBefore->m_dTime;
	} else {
		dt = pBefore->m_dTime;
	}
	if ((pAfter == NULL) || (dt == 0)) {
		p = pBefore->m_vPos;
		d = pBefore->m_vDir;
		u = pBefore->m_vUp;
	} else {
		double r = (time - pBefore->m_dTime) / dt;
		p = pBefore->m_vPos + (pAfter->m_vPos - pBefore->m_vPos) * r;
		d = pBefore->m_vDir + (pAfter->m_vDir - pBefore->m_vDir) * r;
		u = pBefore->m_vUp + (pAfter->m_vUp - pBefore->m_vUp) * r;
	}

	// Set the new position and direction
	m_pFrame->SetPosition(p);
	m_pFrame->SetDirection(d, u);

	return TRUE;
}

