// 3dDirDrw.cpp
//
// Copyright (c) Nigel Thompson 1996
//
// Implementation for:
// CDirectDraw
//

#include "stdafx.h"
#include "3dDirect.h"

//////////////////////////////////////////////////////////////////
// CDDObject

IMPLEMENT_DYNAMIC(CDDObject, CObject)

CDDObject::CDDObject()
{
	m_hr = 0;
}

CDDObject::~CDDObject()
{
}

//////////////////////////////////////////////////////////////////
// CDDEngine

IMPLEMENT_DYNAMIC(CDDEngine, CDDObject)

CDDEngine::CDDEngine()
{
}

CDDEngine::~CDDEngine()
{
}

// Device enumeration data structure
typedef struct _EnumDevInfo {
	int iDeviceCount;
	int iDevice;
	CString strName;
	CString strDesc;
} EnumDevInfo;

// callback function for device enumeration
static BOOL FAR PASCAL EnumDevFn(GUID FAR * pGuid,
								 LPSTR pszDriver,
								 LPSTR pszDesc,
								 LPVOID pArg)
{
	EnumDevInfo* pInfo = (EnumDevInfo*) pArg;
	ASSERT(pInfo);
	
	// see if it's the one wanted
	if (pInfo->iDevice == pInfo->iDeviceCount) {
		pInfo->strName = pszDriver;
		pInfo->strDesc = pszDesc;
		return DDENUMRET_CANCEL; // stop enumerating
	}

	pInfo->iDeviceCount++;
	return DDENUMRET_OK;
}

// Get the number of Direct draw devices
int CDDEngine::GetNumDevices()
{
	EnumDevInfo di;
	di.iDeviceCount = 0;
	di.iDevice = -1;
	m_hr = ::DirectDrawEnumerate(EnumDevFn, &di);
	ASSERT(SUCCEEDED(m_hr));
	return di.iDeviceCount;
}

// Get info for a given device
BOOL CDDEngine::GetDeviceInfo(int iDev,
								CString& strName,
								CString& strDesc)
{
	int iNum = GetNumDevices();
	if (iDev >= iNum) return FALSE;

	EnumDevInfo di;
	di.iDeviceCount = 0;
	di.iDevice = iDev;
	m_hr = ::DirectDrawEnumerate(EnumDevFn, &di);
	ASSERT(SUCCEEDED(m_hr));
	strName = di.strName;
	strDesc = di.strDesc;
	return TRUE;
}

HRESULT CDDEngine::CreateDirectDraw(IDirectDraw** ppIDD)
{
	ASSERT(ppIDD);
	return ::DirectDrawCreate(NULL, ppIDD, NULL);
}

// the one and only DDEngine
CDDEngine theDDEngine;

//////////////////////////////////////////////////////////////////
// CDirectDraw

IMPLEMENT_DYNAMIC(CDirectDraw, CDDObject)

CDirectDraw::CDirectDraw()
{
	m_pFrontBuffer = NULL;
	m_pBackBuffer = NULL;
	m_pClipper = NULL;
	m_pPalette = NULL;
	m_bSWRender = FALSE;
	m_pIDD = NULL;
	m_bRestore = FALSE;
}

CDirectDraw::~CDirectDraw()
{
	// free all the buffers
	_ReleaseAll();

	// release the interface
	if (m_pIDD) {
		m_pIDD->Release();
		m_pIDD = NULL;
	}
}

void CDirectDraw::_ReleaseAll()
{
	if (m_bRestore) {
		RestoreMode();
	}
	if (m_pFrontBuffer) {
		delete m_pFrontBuffer;
		m_pFrontBuffer = NULL;
	}
	if (m_pBackBuffer) {
		delete m_pBackBuffer;
		m_pBackBuffer = NULL;
	}
	if (m_pClipper) {
		delete m_pClipper;
		m_pClipper = NULL;
	}
	if (m_pPalette) {
		delete m_pPalette;
		m_pPalette = NULL;
	}
	m_bSWRender = FALSE;
}

BOOL CDirectDraw::Create()
{
	_ReleaseAll();
	if (m_pIDD == NULL) {
		m_hr = theDDEngine.CreateDirectDraw(&m_pIDD);
		if (FAILED(m_hr)) return FALSE;
		ASSERT(m_pIDD);
	}
	return TRUE;
}

BOOL CDirectDraw::GetCaps(DDCAPS* pDrvCaps, DDCAPS* pHelCaps)
{
	ASSERT(m_pIDD);
	if (pDrvCaps) {
		memset(pDrvCaps, 0, sizeof(DDCAPS));
		pDrvCaps->dwSize = sizeof(DDCAPS);
	}
	if (pHelCaps) {
		memset(pHelCaps, 0, sizeof(DDCAPS));
		pHelCaps->dwSize = sizeof(DDCAPS);
	}
	m_hr = m_pIDD->GetCaps(pDrvCaps, pHelCaps);
	return SUCCEEDED(m_hr);
}

// mode enumeration info structure
typedef struct _EnumModeInfo {
	int iModeCount;
	int iMode;
	DDSURFACEDESC ddSurf;
} EnumModeInfo;

// callback function for device mode enumeration
static HRESULT FAR PASCAL EnumModesFn(LPDDSURFACEDESC psd,
						              LPVOID pArg)

{
	EnumModeInfo* pInfo = (EnumModeInfo*) pArg;
	ASSERT(pInfo);
	
	// see if it's the one wanted
	if (pInfo->iMode == pInfo->iModeCount) {
		pInfo->ddSurf = *psd;
		return DDENUMRET_CANCEL; // stop enumerating
	}

	pInfo->iModeCount++;
	return DDENUMRET_OK;
}

// Get the number of supported display modes
int CDirectDraw::GetNumModes()
{
	ASSERT(m_pIDD);
	EnumModeInfo mi;
	mi.iModeCount = 0;
	mi.iMode = -1;
	m_hr = m_pIDD->EnumDisplayModes(0, NULL, &mi, EnumModesFn);
	ASSERT(SUCCEEDED(m_hr));
	return mi.iModeCount;
}

// Get data for a given mode
BOOL CDirectDraw::GetModeInfo(int iMode, DDSURFACEDESC* pDesc)
{
	int iModes =  GetNumModes();
	if (iMode >= iModes) return FALSE;
	ASSERT(m_pIDD);
	EnumModeInfo mi;
	mi.iModeCount = 0;
	mi.iMode = iMode;
	m_hr = m_pIDD->EnumDisplayModes(0, NULL, &mi, EnumModesFn);
	ASSERT(SUCCEEDED(m_hr));
	*pDesc = mi.ddSurf; 
	return TRUE;
}

BOOL CDirectDraw::SetCooperativeLevel(HWND hWnd, DWORD dwFlags)
{
	ASSERT(m_pIDD);
	ASSERT(hWnd);
	m_hr = m_pIDD->SetCooperativeLevel(hWnd, dwFlags);
	return SUCCEEDED(m_hr);
}

// Set a given mode. If hWnd is NULL, we try to go full screen
BOOL CDirectDraw::_SetMode(HWND hWnd, int cx, int cy, int bpp, BOOL bFullScreen)
{
	ASSERT(m_pIDD);

	// release any existing buffers
	_ReleaseAll();

	// Set the cooperative level
	if (bFullScreen) {
		if (!SetCooperativeLevel(hWnd, DDSCL_EXCLUSIVE | DDSCL_FULLSCREEN)) {
			return FALSE;
		}
		m_hr = m_pIDD->SetDisplayMode(cx, cy, bpp);
		if (FAILED(m_hr)) {
			return FALSE;
		}
		m_bRestore = TRUE;
	} else {
		if (!SetCooperativeLevel(hWnd, DDSCL_NORMAL)) {
			return FALSE;
		}
	}

	// Create the front and back buffer surfaces

	m_iWidth = cx;
	m_iHeight = cy;
	
	DDSURFACEDESC sd;
	memset(&sd, 0, sizeof(sd));
	sd.dwSize = sizeof(sd);

	if (bFullScreen) {

		// Create a complex flipping surface with a front and
		// a back buffer
		sd.dwFlags = DDSD_CAPS
				   | DDSD_BACKBUFFERCOUNT;
		sd.ddsCaps.dwCaps = DDSCAPS_PRIMARYSURFACE
						  | DDSCAPS_FLIP
						  | DDSCAPS_COMPLEX
						  | DDSCAPS_3DDEVICE;
		sd.dwBackBufferCount = 1;

		// Create the front and back buffer surfaces
		m_pFrontBuffer = new CDDSurface;
		if (!m_pFrontBuffer->Create(this, &sd)) {
			return FALSE;
		}
		ASSERT(m_pFrontBuffer);

		// Get a pointer to the attached back buffer
		DDSCAPS caps;
		memset(&caps, 0, sizeof(caps));
		caps.dwCaps = DDSCAPS_BACKBUFFER;
		m_pBackBuffer = m_pFrontBuffer->GetAttachedSurface(&caps);
		if (!m_pBackBuffer) {
			delete m_pFrontBuffer;
			m_pFrontBuffer = NULL;
			return FALSE;
		}

	} else {

		// Create two surfaces for the windowed case
		// a primary surface we share with GDI and
		// a back buffer we render into

		// Create the front buffer surface
		// Note: since this is the primary (existing)
		// surface, we don't specify height and width
		sd.dwFlags = DDSD_CAPS;
		sd.ddsCaps.dwCaps = DDSCAPS_PRIMARYSURFACE;
		m_pFrontBuffer = new CDDSurface;
		if (!m_pFrontBuffer->Create(this, &sd)) {
			return FALSE;
		}

		// Create the back buffer surface
		sd.dwFlags = DDSD_WIDTH 
				   | DDSD_HEIGHT
				   | DDSD_CAPS;
		sd.dwWidth = cx;
		sd.dwHeight = cy;
		sd.ddsCaps.dwCaps = DDSCAPS_OFFSCREENPLAIN | DDSCAPS_3DDEVICE;
		m_pBackBuffer = new CDDSurface;
		if (!m_pBackBuffer->Create(this, &sd)) {
			delete m_pFrontBuffer;
			m_pFrontBuffer = NULL;
			return FALSE;
		}

		// Create a clipper object for the front buffer
		// so drawing is clipped to the window
		m_pClipper = new CDDClipper;
		if (!m_pClipper->Create(this, hWnd)) {
			return FALSE;
		}

		// attach the clipper to the front buffer
		if (!m_pFrontBuffer->SetClipper(m_pClipper)) {
			return FALSE;
		}

	}

	// See if the back buffer is in video memory and if not
	// set the software-only rendering flag so we won't try to use
	// a hardware renderer on it.
	m_pBackBuffer->GetDescription(&sd);
	if (!(sd.ddsCaps.dwCaps & DDSCAPS_VIDEOMEMORY))	{
		m_bSWRender = TRUE;
	}

	// Create and setup a palette if we are less than 16 bpp
	// find the pixel format needed to access a locked buffer
#if 0
	memset(&sd, 0, sizeof(sd));
	sd.dwSize = sizeof(sd);
	m_hr = m_pBackBuffer->GetInterface()->Lock(NULL, &sd, DDLOCK_WAIT, NULL);
	if (FAILED(m_hr)) return FALSE;
	m_hr = m_pBackBuffer->GetInterface()->Unlock(NULL);
#else
	m_pBackBuffer->GetDescription(&sd);
#endif
	int i;
	if (sd.ddpfPixelFormat.dwRGBBitCount < 16) {

		// yes, we need a palette
		// get the current hardware palette
		PALETTEENTRY pe[256];
		HDC hdc = ::GetDC(NULL);
		::GetSystemPaletteEntries(hdc, 0, 256, pe);
		::ReleaseDC(NULL, hdc);

		// use 254 entries if full screen, 236 if windowed
		if (bFullScreen) {
			pe[0].peFlags = D3DPAL_READONLY;
			for (i = 1; i < 255; i++) pe[i].peFlags = PC_RESERVED;
			pe[255].peFlags = D3DPAL_READONLY;
		} else {
			for (i = 0; i < 10; i++) pe[i].peFlags = D3DPAL_READONLY;
#ifdef _DEBUG
			// for debugging, set all the usable entries to gray
			// so we can see which ones get set by the renderer
			for (i = 10; i < 246; i++) {
				pe[i].peFlags = PC_RESERVED;
				pe[i].peRed = 128;
				pe[i].peGreen = 128;
				pe[i].peBlue = 128;
			}
#else
			for (i = 10; i < 246; i++) pe[i].peFlags = PC_RESERVED;
#endif
			for (i = 246; i < 256; i++) pe[i].peFlags = D3DPAL_READONLY;
		}

		m_pPalette = new CDDPalette;
		if (!m_pPalette->Create(this,
						   DDPCAPS_8BIT | DDPCAPS_INITIALIZE,
						   pe)) {
			return FALSE;
		}

		// Attach the palette to the buffers
		if (!m_pFrontBuffer->SetPalette(m_pPalette)) return FALSE;
		if (!m_pBackBuffer->SetPalette(m_pPalette)) return FALSE;

	}

	return TRUE;
}

void CDirectDraw::RestoreMode()
{
	if (m_bRestore) {
		ASSERT(m_pIDD);
		m_hr = m_pIDD->RestoreDisplayMode();
		ASSERT(SUCCEEDED(m_hr));
		m_bRestore = FALSE;
	}
}

void CDirectDraw::OnActivate(BOOL bActive)
{
	if (bActive) {
		// the window is being activated
		// if we are using a palette, restore it to the
		// front buffer
		if (m_pPalette && m_pFrontBuffer) {
			m_pFrontBuffer->SetPalette(m_pPalette);
		}
	}
}


//////////////////////////////////////////////////////////////////
// CDDSurface

IMPLEMENT_DYNAMIC(CDDSurface, CDDObject)

CDDSurface::CDDSurface()
{
	m_pISurf = NULL;
	m_bDontDestroy = FALSE;
	memset(&m_SurfDesc, 0, sizeof(m_SurfDesc));
}

CDDSurface::CDDSurface(IDirectDrawSurface* ps)
{
	ASSERT(ps);
	m_pISurf = ps;
	ASSERT(m_pISurf);
	// BUGBUG: AddRef'ing the surface doesn't stop it being
	// destroyed so just set a flag to say we don't destroy it in
	// the destructor
	//m_pISurf->AddRef();
	m_bDontDestroy = TRUE;
	_UpdateDescription();
}

CDDSurface::~CDDSurface()
{
	// release the interface
	if ((m_pISurf != NULL) && (!m_bDontDestroy)) {
		m_pISurf->Release();
	}
}

BOOL CDDSurface::Create(CDirectDraw* pDD, DDSURFACEDESC* pDesc)
{
	if (m_pISurf != NULL) {
		m_pISurf->Release();
		m_pISurf = NULL;
	}
	ASSERT(pDD);
	ASSERT(pDesc);
	m_hr = pDD->GetInterface()->CreateSurface(pDesc, &m_pISurf, NULL);
	if (FAILED(m_hr)) {
		m_pISurf = NULL;
		return FALSE;
	}
	ASSERT(m_pISurf);
	_UpdateDescription();
	return TRUE;
}

void CDDSurface::_UpdateDescription()
{
	memset(&m_SurfDesc, 0, sizeof(m_SurfDesc));
	m_SurfDesc.dwSize = sizeof(DDSURFACEDESC);
	ASSERT(m_pISurf);
	m_hr = m_pISurf->GetSurfaceDesc(&m_SurfDesc);
}

BOOL CDDSurface::GetDescription(DDSURFACEDESC* pDesc)
{
	if (!m_pISurf) return FALSE;
	_UpdateDescription();
	ASSERT(pDesc);
	memcpy(pDesc, &m_SurfDesc, sizeof(DDSURFACEDESC));
	return TRUE;
}

CDDSurface* CDDSurface::GetAttachedSurface(DDSCAPS* pCaps)
{
	ASSERT(m_pISurf);
	ASSERT(pCaps);
	IDirectDrawSurface* ps = NULL;
	m_hr = m_pISurf->GetAttachedSurface(pCaps, &ps);
	if (FAILED(m_hr)) {
		return NULL;
	}
	ASSERT(ps);
	CDDSurface* pds = new CDDSurface(ps);
	return pds;
}

BOOL CDDSurface::SetClipper(CDDClipper* pClipper)
{
	ASSERT(m_pISurf);
	ASSERT(pClipper);
	m_hr = m_pISurf->SetClipper(pClipper->GetInterface());
	if (FAILED(m_hr)) return FALSE;
	_UpdateDescription();
	return TRUE;
}

BOOL CDDSurface::SetPalette(CDDPalette* pPalette)
{
	ASSERT(m_pISurf);
	ASSERT(pPalette);
	m_hr = m_pISurf->SetPalette(pPalette->GetInterface());
	return SUCCEEDED(m_hr);
}

CDDPalette* CDDSurface::GetPalette()
{
	ASSERT(m_pISurf);
	IDirectDrawPalette* pIPal = 0;
	m_hr = m_pISurf->GetPalette(&pIPal);
	if (FAILED(m_hr)) return NULL;

	// Create a CDDPalette object to return
	ASSERT(pIPal);
	CDDPalette* pPal = new CDDPalette(pIPal);
	pIPal->Release();

	return pPal;
}

BOOL CDDSurface::AddAttachedSurface(CDDSurface* pSurface)
{
	ASSERT(m_pISurf);
	ASSERT(pSurface);
	m_hr = m_pISurf->AddAttachedSurface(pSurface->GetInterface());
	if (FAILED(m_hr)) return FALSE;
	_UpdateDescription();
	return TRUE;
}

BOOL CDDSurface::Blt(RECT* prcTo, CDDSurface* pSurFrom, RECT* prcFrom,
			         DWORD dwFlags,	LPDDBLTFX lpDDBltFx)
{
	ASSERT(m_pISurf);
	ASSERT(prcTo);
	ASSERT(pSurFrom);
	ASSERT(prcFrom);
	m_hr = m_pISurf->Blt(prcTo, pSurFrom->GetInterface(), prcFrom,
						dwFlags, lpDDBltFx);
	return SUCCEEDED(m_hr);
}

BOOL CDDSurface::Flip(DWORD dwFlags)
{
	ASSERT(m_pISurf);
	m_hr = m_pISurf->Flip(NULL, dwFlags);
	return SUCCEEDED(m_hr);
}

CDC* CDDSurface::GetDC()
{
	HDC hdc = NULL;
	ASSERT(m_pISurf);
	m_hr = m_pISurf->GetDC(&hdc);
	ASSERT(SUCCEEDED(m_hr));
	CDC* pDC = new CDC;
	pDC->Attach(hdc);
	return pDC;
}

void CDDSurface::ReleaseDC(CDC* pDC)
{
	ASSERT(pDC);
	HDC hdc = pDC->Detach();
	delete pDC;
	ASSERT(m_pISurf);
	m_hr = m_pISurf->ReleaseDC(hdc);
	ASSERT(SUCCEEDED(m_hr));
}

void CDDSurface::SetColorKey(DWORD dwFlags, DDCOLORKEY* pDDColorKey)
{
	ASSERT(m_pISurf);
	m_hr = m_pISurf->SetColorKey(dwFlags, pDDColorKey);
	ASSERT(SUCCEEDED(m_hr));
	_UpdateDescription();
}

void CDDSurface::GetRect(RECT& rc)
{
	rc.left = rc.top = 0;
	rc.right = GetWidth();
	rc.bottom = GetHeight();
}

int CDDSurface::GetBitsPerPixel()
{
	// Note: only valid for RGB formats
	if (!(m_SurfDesc.ddpfPixelFormat.dwFlags & DDPF_RGB)) {
		return 0; // not RGB
	}

	return m_SurfDesc.ddpfPixelFormat.dwRGBBitCount;
}

void CDDSurface::GetRGBMasks(DWORD& r, DWORD& g, DWORD &b)
{
	// Note: only valid for RGB formats
	if (!(m_SurfDesc.ddpfPixelFormat.dwFlags & DDPF_RGB)) {
		r = g = b = 0;
		return; // not RGB
	}

	r = m_SurfDesc.ddpfPixelFormat.dwRBitMask;
	g = m_SurfDesc.ddpfPixelFormat.dwGBitMask;
	b = m_SurfDesc.ddpfPixelFormat.dwBBitMask;
}

void* CDDSurface::Lock()
{
	ASSERT(m_pISurf);

	// Lock the entire surface
	m_hr = m_pISurf->Lock(NULL,
					      &m_SurfDesc,
						  DDLOCK_SURFACEMEMORYPTR|DDLOCK_WAIT,
						  NULL);
	if (FAILED(m_hr)) return NULL;
	return m_SurfDesc.lpSurface;
}

void CDDSurface::Unlock()
{
	if (!m_SurfDesc.lpSurface) return; // not locked
	m_hr = m_pISurf->Unlock(m_SurfDesc.lpSurface);
	ASSERT(SUCCEEDED(m_hr));
	m_SurfDesc.lpSurface = NULL;
}

//////////////////////////////////////////////////////////////////
// CDDClipper

IMPLEMENT_DYNAMIC(CDDClipper, CDDObject)

CDDClipper::CDDClipper()
{
	m_pIClip = NULL;
}

CDDClipper::~CDDClipper()
{
	// release the interface
	if (m_pIClip != NULL) {
		m_pIClip->Release();
	}
}

BOOL CDDClipper::Create(CDirectDraw* pDD, HWND hWnd)
{
	ASSERT(!m_pIClip);
	ASSERT(pDD);
	ASSERT(hWnd);
	m_hr = pDD->GetInterface()->CreateClipper(0, &m_pIClip, NULL);
	if (FAILED(m_hr)) return FALSE;
	ASSERT(m_pIClip);
	m_hr = m_pIClip->SetHWnd(0, hWnd);
	return SUCCEEDED(m_hr);
}

//////////////////////////////////////////////////////////////////
// CDDPalette

IMPLEMENT_DYNAMIC(CDDPalette, CDDObject)

CDDPalette::CDDPalette()
{
	m_pIPal = NULL;
}

CDDPalette::CDDPalette(IDirectDrawPalette* pIPal)
{
	m_pIPal = pIPal;
	if (m_pIPal) {
		m_pIPal->AddRef();
	}
}

CDDPalette::~CDDPalette()
{
	// release the interface
	if (m_pIPal != NULL) {
		m_pIPal->Release();
	}
}

BOOL CDDPalette::Create(CDirectDraw* pDD, DWORD dwFlags, LPPALETTEENTRY pColorTable)
{
	if (m_pIPal != NULL) {
		m_pIPal->Release();
		m_pIPal = NULL;
	}
	ASSERT(pDD);
	m_hr = pDD->GetInterface()->CreatePalette(dwFlags, pColorTable, &m_pIPal, NULL);
	if (FAILED(m_hr)) return FALSE;
	ASSERT(m_pIPal);
	return TRUE;
}

BOOL CDDPalette::GetEntries(int iStart, int iCount, PALETTEENTRY* pColorTable)
{
	ASSERT(m_pIPal);
	m_hr = m_pIPal->GetEntries(0, iStart, iCount, pColorTable);
	return SUCCEEDED(m_hr);
}

BOOL CDDPalette::SetEntries(int iStart, int iCount, PALETTEENTRY* pColorTable)
{
	ASSERT(m_pIPal);
	m_hr = m_pIPal->SetEntries(0, iStart, iCount, pColorTable);
	return SUCCEEDED(m_hr);
}


//////////////////////////////////////////////////////////////////
// CDirect3D

IMPLEMENT_DYNAMIC(CDirect3D, CDDObject)

CDirect3D::CDirect3D()
{
	m_pDD = NULL;
	m_pID3D = NULL;
	m_pID3DDevice = NULL;
	m_pZBuffer = NULL;
}

CDirect3D::~CDirect3D()
{
	// release the interfaces
	if (m_pZBuffer) {
		delete m_pZBuffer;
	}
	if (m_pID3DDevice) {
		m_pID3DDevice->Release();
	}
	if (m_pID3D != NULL) {
		m_pID3D->Release();
	}
}

BOOL CDirect3D::Create(CDirectDraw* pDD)
{
	ASSERT(pDD);
	if (m_pZBuffer) {
		delete m_pZBuffer;
		m_pZBuffer = NULL;
	}
	if (m_pID3DDevice) {
		m_pID3DDevice->Release();
		m_pID3DDevice = NULL;
	}
	if (m_pID3D != NULL) {
		m_pID3D->Release();
		m_pID3D = NULL;
	}
	m_pDD = pDD;
	m_hr = m_pDD->GetInterface()->QueryInterface(IID_IDirect3D,
											  (VOID**)&m_pID3D);
	if (FAILED(m_hr)) {
		m_pID3D = NULL;
		return FALSE;
	}
	ASSERT(m_pID3D);
	return TRUE;
}

// structure for device enumeration
typedef struct _ENUMDEVINFO {
	D3DCOLORMODEL cm;
	LPGUID pHWGuid;
	D3DDEVICEDESC HWDesc;
	LPGUID pSWGuid;
	D3DDEVICEDESC SWDesc;
} ENUMDEVINFO;

// Device enumeration function
HRESULT FAR PASCAL EnumDevFn(LPGUID pGuid,
							 LPSTR pDevDesc,
							 LPSTR pDevName,
							 LPD3DDEVICEDESC pHWDesc,
							 LPD3DDEVICEDESC pSWDesc,
							 LPVOID pArg)
{
	ENUMDEVINFO* pdi = (ENUMDEVINFO*) pArg;
	ASSERT(pdi);

	// see if we have a suitable hardware device
	if (pHWDesc->dcmColorModel == pdi->cm) {
		pdi->HWDesc = *pHWDesc;
		pdi->pHWGuid = pGuid;
	}

	// see if we have a suitable software device
	if (pSWDesc->dcmColorModel == pdi->cm) {
		pdi->SWDesc = *pSWDesc;
		pdi->pSWGuid = pGuid;
	}

	return D3DENUMRET_OK;
}

static int GetBppFromFlags(DWORD dwFlags)
{
	if (dwFlags & DDBD_32) {
		return 32;
	} else if (dwFlags & DDBD_24) {
		return 24;
	} else if (dwFlags & DDBD_16) {
		return 16;
	} else if (dwFlags & DDBD_8) {
		return 8;
	} else if (dwFlags & DDBD_4) {
		return 4;
	} else if (dwFlags & DDBD_2) {
		return 2;
	} else if (dwFlags & DDBD_1) {
		return 1;
	}
	return 0;
}

BOOL CDirect3D::_CreateZBuffer(int bpp, DWORD dwMemFlags)
{
	// Get some info on the back buffer
	ASSERT(m_pDD);
	CDDSurface* pBB = m_pDD->GetBackBuffer();
	if(!pBB) return FALSE;
	DDSURFACEDESC ds;
	pBB->GetDescription(&ds);
	int cx = ds.dwWidth;
	int cy = ds.dwHeight;

	// Build a descriptor for the Z surface
	memset(&ds, 0, sizeof(ds));
	ds.dwSize = sizeof(ds);
	ds.dwFlags = DDSD_WIDTH | DDSD_HEIGHT | DDSD_CAPS | DDSD_ZBUFFERBITDEPTH;
	ds.dwHeight = cy;
	ds.dwWidth = cx;
	ds.ddsCaps.dwCaps = DDSCAPS_ZBUFFER | dwMemFlags;
	ds.dwZBufferBitDepth = bpp;

	// create the z buffer surface
	m_pZBuffer = new CDDSurface;
	if (!m_pZBuffer->Create(m_pDD, &ds)) {
		return FALSE;
	}

	// attach the z buffer to the back buffer
	if (!pBB->AddAttachedSurface(m_pZBuffer)) {
		return FALSE;
	}

	return TRUE;
}

BOOL CDirect3D::SetMode(D3DCOLORMODEL cm)
{
	// Find a hardware or software device that supports the model
	ENUMDEVINFO edi;
	memset(&edi, 0, sizeof(edi));
	edi.cm = cm;
	ASSERT(m_pID3D);
	m_hr = m_pID3D->EnumDevices(EnumDevFn, &edi);
	if (!edi.pHWGuid && !edi.pSWGuid) {
		return FALSE; // no suitable device
	}

	// if there was no hardware device or the direct draw
	// device is forced to use software only rendering we set
	// up the software version
	LPGUID pGuid = NULL;
	BOOL b;
	if (!edi.pHWGuid || m_pDD->IsSWRenderOnly()) {
		pGuid = edi.pSWGuid;
		b = _CreateZBuffer(16, DDSCAPS_SYSTEMMEMORY);
	} else {
		pGuid = edi.pHWGuid;

		// Get the device caps to see what bit depth we need for the buf.
		DDCAPS hw;
		m_pDD->GetCaps(&hw);
		int bpp = GetBppFromFlags(hw.dwZBufferBitDepths);
		if (bpp == 0) bpp = 16; 
		b = _CreateZBuffer(bpp, DDSCAPS_VIDEOMEMORY);
	}
	if (!b) return FALSE;

	// get the interface from the GUID for the device
	ASSERT(!m_pID3DDevice);
	m_hr = m_pDD->GetBackBuffer()->GetInterface()->QueryInterface(*pGuid,
											(LPVOID*)&m_pID3DDevice);
	if (FAILED(m_hr)) return FALSE;
	ASSERT(m_pID3DDevice);

	return TRUE;
}