// 3dDirGrb.cpp
//
// Copyright (c) Nigel Thompson 1996
//
// Implementation for:
// CDirectDraw::GrabPalette
// CDirectDraw::GrabImage
//

#include "stdafx.h"
#include "3dDirect.h"

// define USE_HW_PAL  if we want to grab the hardware palette
// rather than the current palette for this DirectDraw object

// Grab the current palette
CPalette* CDirectDraw::GrabPalette()
{
	if (!m_pPalette) return NULL;
	LOGPALETTE* plp = (LOGPALETTE*) new BYTE[sizeof(LOGPALETTE) 
											 + 256 * sizeof(PALETTEENTRY)];
	plp->palVersion	= 0x300;
	plp->palNumEntries = 256;
	m_pPalette->GetInterface()->GetEntries(0,
										0,
									    256,
										plp->palPalEntry);

	// Set the palette entry flags
	for (int i = 0; i < 256; i++) plp->palPalEntry[i].peFlags = 0;

	// Create a palette from the color set
	CPalette* pPal = new CPalette;
	pPal->CreatePalette(plp);
	delete plp;

	return pPal;
}

// Grab the current back buffer image
// if the bitmap info header or bits are returned then the caller
// must free the associated memory (using delete pBMI and 
// ::DeleteObject(hBmp). Don't try to delete the bits as they
// belong to the bitmap object.
HBITMAP CDirectDraw::GrabImage(BITMAPINFO** ppBMI, void** ppBits)
{
	// Set the initial return results
	if (ppBMI) *ppBMI= NULL;
	if (ppBits) *ppBits = NULL;
	if (!m_pBackBuffer) return NULL;

	// Lock the back buffer to get its description
	// WARNING: you can't step through this code with the
	// debugger - the GDI surface will be locked
	LPDIRECTDRAWSURFACE iS = m_pBackBuffer->GetInterface();
	ASSERT(iS);
	DDSURFACEDESC ds;
	ds.dwSize = sizeof(ds);
	m_hr = iS->Lock(NULL,
					&ds,
					DDLOCK_WAIT,
					NULL);
	if (m_hr != DD_OK) {
		TRACE("Failed to lock surface\n");
		return NULL; // failed to lock surface
	}

	// Unlock the surface again so we can use the debugger
	// and also exit if we need to
	m_hr = m_pBackBuffer->GetInterface()->Unlock(ds.lpSurface);

	// make sure the surface is of a type we can deal with
	// Note: we only handle 8 bpp pal, 16 bpp RGB and 24 bpp RGB here
	if (!(ds.ddpfPixelFormat.dwFlags & DDPF_PALETTEINDEXED8)
	&& !(ds.ddpfPixelFormat.dwFlags & DDPF_RGB)) {
		return NULL; // don't handle this format
	}
	int iBitCount;
	if (ds.ddpfPixelFormat.dwFlags & DDPF_PALETTEINDEXED8) {
		iBitCount = 8;
	} else if (ds.ddpfPixelFormat.dwFlags & DDPF_RGB) {
		// check it's a bit depth we can deal with
		iBitCount = ds.ddpfPixelFormat.dwRGBBitCount;
		if ((iBitCount != 16) && (iBitCount != 24)) {
			return NULL; // not handled
		}
	}

	ASSERT(ds.dwFlags & DDSD_WIDTH);
	int iWidth = ds.dwWidth;
	ASSERT(ds.dwFlags & DDSD_HEIGHT);
	int iHeight = ds.dwHeight;

	// see if we need to create a color table or not
	int iClrTabEntries = 0;
	if (ds.ddpfPixelFormat.dwFlags & DDPF_PALETTEINDEXED8) {

		// Yes we need to build a color table
		iClrTabEntries = 256;
		iBitCount = 8;
	}

	// Create a BITMAPINFO structure to describe the bitmap
	int iSize = sizeof(BITMAPINFO) + iClrTabEntries * sizeof(RGBQUAD);
	BITMAPINFO* pBMI = (BITMAPINFO*) new BYTE[iSize];
	memset(pBMI, 0, iSize);
	pBMI->bmiHeader.biSize = sizeof(BITMAPINFOHEADER);
	pBMI->bmiHeader.biWidth = iWidth; 
	pBMI->bmiHeader.biHeight = iHeight;
	pBMI->bmiHeader.biPlanes = 1;
	pBMI->bmiHeader.biBitCount = iBitCount;
	pBMI->bmiHeader.biClrUsed = iClrTabEntries;

	HDC hdcScreen = ::GetDC(NULL);
	// create the color table if needed
	if (iClrTabEntries > 0) {
		ASSERT(iClrTabEntries <= 256);
		PALETTEENTRY pe[256];
		ASSERT(m_pPalette);
		m_pPalette->GetInterface()->GetEntries(0,
											0,
											iClrTabEntries,
											pe);
		for (int i = 0; i < iClrTabEntries; i++) {
			pBMI->bmiColors[i].rgbRed = pe[i].peRed;
			pBMI->bmiColors[i].rgbGreen = pe[i].peGreen;
			pBMI->bmiColors[i].rgbBlue = pe[i].peBlue;
		}
	}

	// Create a DIB section the same size as the back buffer
	BYTE* pBits = NULL;
	HBITMAP hBmp = ::CreateDIBSection(hdcScreen,
									  pBMI,
									  DIB_RGB_COLORS,
									  (VOID**)&pBits,
									  NULL,
									  0);
   	::ReleaseDC(NULL, hdcScreen);
	if (!hBmp) {
		delete pBMI;
		return NULL;
	}
	ASSERT(pBits);

	// Copy the bits to the DIB surface

	int iDIBScan = 
        (((pBMI->bmiHeader.biWidth * pBMI->bmiHeader.biBitCount) + 31) & ~31) >> 3;
	int iSurfScan = ds.lPitch;
	BYTE* pDIBLine = pBits + (iHeight - 1) * iDIBScan;
	BYTE* pSurfLine = (BYTE*)ds.lpSurface;

	// Build the shift masks
	// we shift down until the lsb of the source maps to the lsb of the destination
	// then we shift again until we only have 5 bits of precision
	DWORD dwRShift = 0;
	DWORD dwGShift = 0;
	DWORD dwBShift = 0;
	DWORD dwNotMask;
	if ((ds.ddpfPixelFormat.dwFlags & DDPF_RGB) && (iBitCount >= 16)) {
		if (iBitCount == 16) {
			dwNotMask = 0xFFFFFFE0;
		} else {
			dwNotMask = 0xFFFFFF00;
		}
		DWORD dwMask = ds.ddpfPixelFormat.dwRBitMask;
		ASSERT(dwMask);
		while ((dwMask & 0x01) == 0) {
			dwRShift++;
			dwMask = dwMask >> 1;
		}
		while ((dwMask & dwNotMask) != 0) {
			dwRShift++;
			dwMask = dwMask >> 1;
		}
		dwMask = ds.ddpfPixelFormat.dwGBitMask;
		ASSERT(dwMask);
		while ((dwMask & 0x01) == 0) {
			dwGShift++;
			dwMask = dwMask >> 1;
		}
		while ((dwMask & dwNotMask) != 0) {
			dwGShift++;
			dwMask = dwMask >> 1;
		}
		dwMask = ds.ddpfPixelFormat.dwBBitMask;
		ASSERT(dwMask);
		while ((dwMask & 0x01) == 0) {
			dwBShift++;
			dwMask = dwMask >> 1;
		}
		while ((dwMask & dwNotMask) != 0) {
			dwBShift++;
			dwMask = dwMask >> 1;
		}
	}

	// lock the surface again so we can get the bits
	m_hr = iS->Lock(NULL,
					&ds,
					DDLOCK_SURFACEMEMORYPTR | DDLOCK_WAIT,
					NULL);
	ASSERT(m_hr == DD_OK);

	for (int y = 0; y < iHeight; y++) {
		switch (iBitCount) {
		case 8:	{
			BYTE* pDIBPix = pDIBLine;
			BYTE* pSurfPix = pSurfLine;
			for (int x = 0; x < iWidth; x++) {
				*pDIBPix++ = *pSurfPix++;
			}
			} break;

		case 16: {
			WORD* pDIBPix = (WORD*)pDIBLine;
			WORD* pSurfPix = (WORD*)pSurfLine;
			WORD r, g, b;
			for (int x = 0; x < iWidth; x++) {
				r = (*pSurfPix & (WORD) ds.ddpfPixelFormat.dwRBitMask) >> dwRShift;
				g = (*pSurfPix & (WORD) ds.ddpfPixelFormat.dwGBitMask) >> dwGShift;
				b = (*pSurfPix & (WORD) ds.ddpfPixelFormat.dwBBitMask) >> dwBShift;
				*pDIBPix++ = ((r & 0x1F) << 10)
					       | ((g & 0x1F) << 5)
						   | (b & 0x1F);
				pSurfPix++;
			}
			} break;

		case 24: {
			BYTE* pDIBPix = pDIBLine;
			BYTE* pSurfPix = pSurfLine;

			for (int x = 0; x < iWidth; x++) {
				// WARNING: I'm assuming the same RGB masks
				// for the surface and the DIB which is not really
				// valid: we should look at ds.ddpfPixelFormat.dwRGBBitMask etc.
				*pDIBPix++ = *pSurfPix++;
				*pDIBPix++ = *pSurfPix++;
				*pDIBPix++ = *pSurfPix++;
			}
			} break;

		default:
			// shouldn't be able to get here
			break;
		}
		pDIBLine -= iDIBScan;
		pSurfLine += iSurfScan;
	}

	// Unlock the buffer
	m_hr = m_pBackBuffer->GetInterface()->Unlock(ds.lpSurface);

	// see if we're returning the bitmap info header
	if (ppBMI) {
		*ppBMI = pBMI;
	} else {
		delete pBMI;
	}

	// See if we're returning the bits pointer
	if (ppBits) *ppBits = pBits;

	// return the bitmap handle
	return hBmp;
}

// grab the current image and convert it to a CBitmap object
CBitmap* CDirectDraw::GrabImage()
{
	// Try to get the image
	HBITMAP hBmp = GrabImage(NULL, NULL);
	if (!hBmp) {
		return NULL;
	}

	// Create a CBitmap object to return
	CBitmap* pBmp = new CBitmap;
	pBmp->Attach(hBmp);

	return pBmp;
}