// 3dFrame.cpp
//
// Copyright (c) Nigel Thompson 1996
//
// Implementation for:
// C3dFrame	
//

#include "stdafx.h"
#include "3dplus.h"

//////////////////////////////////////////////////////////////////
// C3dFrame

IMPLEMENT_DYNAMIC(C3dFrame, C3dVisual)

C3dFrame::C3dFrame()
{
    m_pIFrame = NULL;
	m_strName = "3D Frame";
}

C3dFrame::~C3dFrame()
{
    // release any objects we are holding
    if (m_pIFrame) {
		m_pIFrame->SetAppData(0); // not in a C++ object now
		m_pIFrame->Release();
	}
}

// Get the reference frame
IDirect3DRMFrame* C3dFrame::_GetRef(C3dFrame* pRef)
{
	if (pRef) {
		return pRef->GetInterface();
	} else {
		if (!m_pIFrame) return NULL;
		// get the parent of this frame
		IDirect3DRMFrame* pIParent = NULL;
		m_hr = m_pIFrame->GetParent(&pIParent);
		//ASSERT(SUCCEEDED(m_hr));		  // BUGBUG this fails during object creation
		return pIParent;
	}
}

BOOL C3dFrame::Create(C3dFrame* pParent)
{
	if (m_pIFrame) {
		m_pIFrame->Release();
		m_pIFrame = NULL;
	}
    if (!the3dEngine.CreateFrame(_GetRef(pParent), &m_pIFrame)) {
        TRACE("Frame create failed\n");
        m_pIFrame = NULL;
        return FALSE;
    }
    ASSERT(m_pIFrame);
	m_pIFrame->SetAppData((ULONG)this);

    return TRUE;
}

void C3dFrame::SetPosition(double x, double y, double z, C3dFrame* pRef)
{
    ASSERT(m_pIFrame);
    m_hr = m_pIFrame->SetPosition(_GetRef(pRef), D3DVAL(x), D3DVAL(y), D3DVAL(z));
    ASSERT(SUCCEEDED(m_hr));
}

void C3dFrame::GetPosition(double& x, double& y, double& z, C3dFrame* pRef)
{
    ASSERT(m_pIFrame);
    D3DVECTOR v;

    m_hr = m_pIFrame->GetPosition(_GetRef(pRef), &v);
    ASSERT(SUCCEEDED(m_hr));
    x = v.x; 
    y = v.y;
    z = v.z;
}

void C3dFrame::GetPosition(D3DVECTOR& p, C3dFrame* pRef)
{
    ASSERT(m_pIFrame);

    m_hr = m_pIFrame->GetPosition(_GetRef(pRef), &p);
    ASSERT(SUCCEEDED(m_hr));
}

void C3dFrame::SetRotation(double x, double y, double z, double t, C3dFrame* pRef)
{
    ASSERT(m_pIFrame);

    m_hr = m_pIFrame->SetRotation(_GetRef(pRef), 
                               D3DVAL(x), D3DVAL(y), D3DVAL(z),
                               D3DVAL(t));
    ASSERT(SUCCEEDED(m_hr));
}

void C3dFrame::SetDirection(double dx, double dy, double dz,
                            double ux, double uy, double uz, C3dFrame* pRef)
{
    ASSERT(m_pIFrame);
    m_hr = m_pIFrame->SetOrientation(_GetRef(pRef),
                                  D3DVAL(dx), D3DVAL(dy), D3DVAL(dz),
                                  D3DVAL(ux), D3DVAL(uy), D3DVAL(uz));
    ASSERT(SUCCEEDED(m_hr));
}

void C3dFrame::SetDirection(D3DVECTOR& d,D3DVECTOR& u, C3dFrame* pRef)
{
    ASSERT(m_pIFrame);
    m_hr = m_pIFrame->SetOrientation(_GetRef(pRef),
                                  D3DVAL(d.x), D3DVAL(d.y), D3DVAL(d.z),
                                  D3DVAL(u.x), D3DVAL(u.y), D3DVAL(u.z));
    ASSERT(SUCCEEDED(m_hr));
}

// Set the direction with a default 'up' vector
void C3dFrame::SetDirection(double dx, double dy, double dz, C3dFrame* pRef)
{
    ASSERT(m_pIFrame);

	// Create a vector for the forward direction
	C3dVector d(dx, dy, dz);

	// generate an arb. up vector
	C3dVector u = d.GenerateUp();

	SetDirection(d.x, d.y, d.z, u.x, u.y, u.z, pRef);
}

void C3dFrame::GetDirection(double& dx, double& dy, double& dz,
                            double& ux, double& uy, double& uz, C3dFrame* pRef)
{
    ASSERT(m_pIFrame);
    D3DVECTOR dir;
    D3DVECTOR up;

    m_hr = m_pIFrame->GetOrientation(_GetRef(pRef), &dir, &up);
    ASSERT(SUCCEEDED(m_hr));
    dx = dir.x;
    dy = dir.y;
    dz = dir.z;
    ux = up.x;
    uy = up.y;
    uz = up.z;
}

void C3dFrame::GetDirection(D3DVECTOR& d, D3DVECTOR& u, C3dFrame* pRef)
{
    ASSERT(m_pIFrame);

    m_hr = m_pIFrame->GetOrientation(_GetRef(pRef), &d, &u);
    ASSERT(SUCCEEDED(m_hr));
}

void C3dFrame::GetDirection(double& dx, double& dy, double& dz, C3dFrame* pRef)
{
    ASSERT(m_pIFrame);
    D3DVECTOR dir;
    D3DVECTOR up;

    m_hr = m_pIFrame->GetOrientation(_GetRef(pRef), &dir, &up);
    ASSERT(SUCCEEDED(m_hr));
    dx = dir.x;
    dy = dir.y;
    dz = dir.z;
}

void C3dFrame::SetVelocity(double x, double y, double z, C3dFrame* pRef)
{
    ASSERT(m_pIFrame);

    m_hr = m_pIFrame->SetVelocity(_GetRef(pRef),
								  D3DVAL(x), D3DVAL(y), D3DVAL(z),
								  FALSE);
    ASSERT(SUCCEEDED(m_hr));
}

BOOL C3dFrame::AddChild(C3dFrame* pChild)
{
    ASSERT(pChild);
    ASSERT(m_pIFrame);

    m_hr = m_pIFrame->AddChild(pChild->GetInterface());
    return SUCCEEDED(m_hr);
}

BOOL C3dFrame::RemoveChild(C3dFrame* pChild)
{
    ASSERT(pChild);
    ASSERT(m_pIFrame);

    m_hr = m_pIFrame->DeleteChild(pChild->GetInterface());
    return SUCCEEDED(m_hr);
}

void C3dFrame::RemoveAllChildren()
{
	// get the child list
	IDirect3DRMFrameArray* pIfa = NULL;
	ASSERT(m_pIFrame);
	m_hr = m_pIFrame->GetChildren(&pIfa);
	ASSERT(SUCCEEDED(m_hr));
	if (pIfa) {
		ULONG uSize = pIfa->GetSize();

		// walk the list
		IDirect3DRMFrame* pIfr;
		for (ULONG u = 0; u < uSize; u++) {
			m_hr = pIfa->GetElement(u, &pIfr);
			ASSERT(SUCCEEDED(m_hr));
			ASSERT(pIfr);
			m_hr = m_pIFrame->DeleteChild(pIfr);
			ASSERT(SUCCEEDED(m_hr));
		}
	}
}

BOOL C3dFrame::SetTexture(C3dTexture* pTexture)
{
    ASSERT(pTexture);
    ASSERT(m_pIFrame);

    m_hr = m_pIFrame->SetTexture(pTexture->GetInterface());
    return SUCCEEDED(m_hr);
}

void C3dFrame::AddTransform(C3dMatrix& m, D3DRMCOMBINETYPE ct)
{
    D3DRMMATRIX4D rlm;
	m.Initialize(rlm);
	m_hr = m_pIFrame->AddTransform(ct, rlm);
    ASSERT(SUCCEEDED(m_hr));
}

void C3dFrame::AddRotation(double x, double y, double z,
					       double t, D3DRMCOMBINETYPE ct)
{
	m_hr = m_pIFrame->AddRotation(ct, D3DVAL(x), D3DVAL(y), D3DVAL(z), D3DVAL(t));
    ASSERT(SUCCEEDED(m_hr));
}

// Attach a frame interface to this frame
void C3dFrame::Attach(IDirect3DRMFrame* pIFrame)
{
	ASSERT(pIFrame);
	if (m_pIFrame) {
		m_pIFrame->Release();
	}
	m_pIFrame = pIFrame;
	if (m_pIFrame) {
		m_pIFrame->AddRef();
	}
}

// Transform a vector from local to world coordinates
D3DVECTOR C3dFrame::Transform(const D3DVECTOR& vLocal)
{
	ASSERT(m_pIFrame);
	D3DVECTOR vw;
	m_hr = m_pIFrame->Transform(&vw, (D3DVECTOR*)&vLocal);
    ASSERT(SUCCEEDED(m_hr));
	return vw;
}

// Test if this frame is a child of a given frame
BOOL C3dFrame::IsChildOf(C3dFrame* pFrame)
{
	if(pFrame == NULL) return FALSE;
	ASSERT(m_pIFrame);
	IDirect3DRMFrame* pIThisFrame = m_pIFrame;
	ASSERT(pIThisFrame);
	pIThisFrame->AddRef();

	do {

		IDirect3DRMFrame* pIParent = NULL;
		m_hr = pIThisFrame->GetParent(&pIParent);
		pIThisFrame->Release();
		if (FAILED(m_hr)) {
			return FALSE;
		}

		if (pIParent == pFrame->m_pIFrame) {
			pIParent->Release();
			return TRUE;
		}

		pIThisFrame = pIParent;
	} while (pIThisFrame != NULL);

	return FALSE;
}

// Test if this frame is the same as, or the child of
// a given frame
BOOL C3dFrame::IsPartOf(C3dFrame* pFrame)
{
	if (pFrame == NULL) return FALSE;
	if (pFrame->m_pIFrame == m_pIFrame) return TRUE;
	return IsChildOf(pFrame);
}

void C3dFrame::SetMaterialMode(D3DRMMATERIALMODE mode)
{
    ASSERT(m_pIFrame);
    m_hr = m_pIFrame->SetMaterialMode(mode);
    ASSERT(SUCCEEDED(m_hr));
}

BOOL C3dFrame::AddVisual(C3dVisual* pVisual)
{
    ASSERT(m_pIFrame);
	ASSERT(pVisual);
	m_hr = m_pIFrame->AddVisual(pVisual->GetInterface());
	return SUCCEEDED(m_hr);
}

BOOL C3dFrame::RemoveVisual(C3dVisual* pVisual)
{
    ASSERT(m_pIFrame);
	ASSERT(pVisual);
	m_hr = m_pIFrame->DeleteVisual(pVisual->GetInterface());
	return SUCCEEDED(m_hr);
}


////////////////////////////////////////////////////////////////////////////
// C3dFrameList

C3dFrameList::C3dFrameList()
{
}

C3dFrameList::~C3dFrameList()
{
	DeleteAll();
}

void C3dFrameList::Remove(C3dFrame* pFrame)
{
	if (!pFrame) return;
	POSITION pos = CObList::Find(pFrame);
	if (pos) {
		RemoveAt(pos);
	}
}

void C3dFrameList::Delete(C3dFrame* pFrame)
{
	if (!pFrame) return;
	Remove(pFrame);
	delete pFrame;
}

void C3dFrameList::DeleteAll()
{
	while (!IsEmpty()) {
		C3dFrame* pFrame = (C3dFrame*) RemoveHead();
		delete pFrame;
	}
}
