// 3dImage.cpp
//
// Copyright (c) Nigel Thompson 1996
//
// Implementation for:
// C3dImage, C3dTexture, C3dWrap
//

#include "stdafx.h"
#include "3dplus.h"

//////////////////////////////////////////////////////////////////
// Windows bitmap helper functions

static BOOL IsWinDIB(BITMAPINFOHEADER* pBIH)
{
    ASSERT(pBIH);
    if (((BITMAPCOREHEADER *)pBIH)->bcSize == sizeof(BITMAPCOREHEADER)) {
        return FALSE;
    }
    return TRUE;
}

int DIBColorEntries(BITMAPINFOHEADER* pBIH) 
{
    BITMAPCOREHEADER *pBCH;
    int iColors, iBitCount;

    ASSERT(pBIH);
    pBCH = (BITMAPCOREHEADER *) pBIH;

    // start off by assuming the color table size from
    // the bit per pixel field
    if (IsWinDIB(pBIH)) {
        iBitCount = pBIH->biBitCount;
    } else {
        iBitCount = pBCH->bcBitCount;
    }

    switch (iBitCount) {
    case 1:
        iColors = 2;
        break;
    case 4:
        iColors = 16;
        break;
    case 8:
        iColors = 256;
        break;
    default:
        iColors = 0;
        break;
    }

    // If this is a Windows DIB, then the color table length
    // is determined by the biClrUsed field if it is non-zero
    if (IsWinDIB(pBIH) && (pBIH->biClrUsed != 0)) {
        iColors = pBIH->biClrUsed;
    }

    // Make sure the value is reasonable since some products
    // will write out more then 256 colors for an 8 bpp DIB!!!
    int iMax = 0;
    switch (iBitCount) {
    case 1:
        iMax = 2;
        break;
    case 4:
        iMax = 16;
        break;
    case 8:
        iMax = 256;
        break;
    default:
        iMax = 0;
        break;
    }
    if (iMax) {
        if (iColors > iMax) {
            iColors = iMax;
        }
    }

    return iColors;
}

int DIBStorageWidth(BITMAPINFOHEADER* pBIH)
{
    return (((pBIH->biWidth * pBIH->biBitCount)
        + 31) & ~31) >> 3;
}

// Save a bitmap to a disk file
BOOL DIBSave(const char* pszFilename, BITMAPINFO* pBMI, void* pBits)
{
    ASSERT(pszFilename);
	ASSERT(pBMI);
	ASSERT(pBits);
	
	// Try to open the file for write access.
    CFile file;
    if (!file.Open(pszFilename,
                   CFile::modeReadWrite
                     | CFile::modeCreate
                     | CFile::shareExclusive)) {
        AfxMessageBox("Failed to open file");
        return FALSE;
    }

    int iColors = DIBColorEntries((BITMAPINFOHEADER*)pBMI);
	int iBits = DIBStorageWidth((BITMAPINFOHEADER*)pBMI) * pBMI->bmiHeader.biHeight;

	BITMAPFILEHEADER bfh;

    // Construct the file header.
    bfh.bfType = 0x4D42; // 'BM'
    bfh.bfSize = 
        sizeof(BITMAPFILEHEADER) +
        sizeof(BITMAPINFOHEADER) +
        iColors * sizeof(RGBQUAD) +
        iBits;
    bfh.bfReserved1 = 0;
    bfh.bfReserved2 = 0;
    bfh.bfOffBits =
        sizeof(BITMAPFILEHEADER) +
        sizeof(BITMAPINFOHEADER) +
        iColors * sizeof(RGBQUAD);

    // Write the file header.
    int iSize = sizeof(bfh);
    TRY {
        file.Write(&bfh, iSize);
    } CATCH(CFileException, e) {
        TRACE("Failed to write file header");
        return FALSE;
    } END_CATCH

    // Write the BITMAPINFO structure.
    iSize = 
        sizeof(BITMAPINFOHEADER) +
        iColors * sizeof(RGBQUAD);
    TRY {
        file.Write(pBMI, iSize);
    } CATCH(CFileException, e) {
        TRACE("Failed to write BITMAPINFO");
        return FALSE;
    } END_CATCH

    // Write the bits.
    iSize = iBits;
    TRY {
        file.Write(pBits, iSize);
    } CATCH(CFileException, e) {
        TRACE("Failed to write bits");
        return FALSE;
    } END_CATCH

    file.Close();

	return TRUE;
}


//////////////////////////////////////////////////////////////////
// C3dImage

IMPLEMENT_DYNAMIC(C3dImage, C3dVisual)

C3dImage::C3dImage()
{
    m_strName = "Image";
	memset(&m_rlimg, 0, sizeof(m_rlimg));
	m_iPhases = 1;
	m_iCurPhase = 0;
	m_pBits = NULL;
	m_iTotalHeight = 0;
}

C3dImage::~C3dImage()
{
    // destroy all the memory objects we own
    _ReleaseMemory();
}

void C3dImage::_ReleaseMemory()
{
    if (m_pBits) {
        delete m_pBits;
		m_pBits = NULL;
    }
    if (m_rlimg.buffer2) {
        delete m_rlimg.buffer2;
    }
    if (m_rlimg.palette) {
        delete m_rlimg.palette;
    }
    memset(&m_rlimg, 0, sizeof(m_rlimg));
}

// Create from a Windows bitmap
// Note: we have to copy the bits because the scan lines are the
// wrong way up in the Windows bitmap
void C3dImage::_Create(BITMAPINFOHEADER* pBMI, RGBQUAD* pRGB, BYTE* pBits)
{
    // now set up all the RLImage parameters
    m_rlimg.width = pBMI->biWidth;
    m_rlimg.height = pBMI->biHeight;
    m_rlimg.aspectx = 1;
    m_rlimg.aspecty = 1;
    m_rlimg.depth = pBMI->biBitCount;
    m_rlimg.rgb = (pBMI->biBitCount > 8); // TRUE if no palette
    m_rlimg.bytes_per_line = DIBStorageWidth(pBMI);
    m_rlimg.buffer1 = new BYTE [m_rlimg.bytes_per_line * m_rlimg.height];
    m_rlimg.buffer2 = NULL;

	// Set up the phase data
	m_pBits = (BYTE*) m_rlimg.buffer1;
	m_iTotalHeight = m_rlimg.height;
	m_iPhases = 1;
	m_iCurPhase = 0;

	// copy the bits, inverting the image
	BYTE* ps = pBits + m_rlimg.bytes_per_line * (m_rlimg.height - 1);
	BYTE* pd = (BYTE*) m_rlimg.buffer1;
	for (int i = 0; i < m_rlimg.height; i++) {
		memcpy(pd, ps, m_rlimg.bytes_per_line);
		ps -= m_rlimg.bytes_per_line;
		pd += m_rlimg.bytes_per_line;
	}

    switch (pBMI->biBitCount) {
    case 16:
        m_rlimg.red_mask = 0x7C00;
        m_rlimg.green_mask = 0x03E0;
        m_rlimg.blue_mask = 0x001F;
        m_rlimg.alpha_mask = 0x0000;
        m_rlimg.palette_size = 0;
        break;

    case 24:
        m_rlimg.red_mask = 0xFF0000;
        m_rlimg.green_mask = 0x00FF00;
        m_rlimg.blue_mask = 0x0000FF;
        m_rlimg.alpha_mask = 0x000000;
        m_rlimg.palette_size = 0;
        break;

    case 32:
        m_rlimg.red_mask = 0x00FF0000;
        m_rlimg.green_mask = 0x0000FF00;
        m_rlimg.blue_mask = 0x000000FF;
        m_rlimg.alpha_mask = 0x00000000;
        m_rlimg.palette_size = 0;
        break;

    default:
        m_rlimg.red_mask = 0xFC;
        m_rlimg.green_mask = 0xFC;
        m_rlimg.blue_mask = 0xFC;
        m_rlimg.alpha_mask = 0xFC;
        m_rlimg.palette_size = DIBColorEntries(pBMI);
        break;
    }
    if (m_rlimg.palette_size > 0) {
        // copy the color table
        m_rlimg.palette = new D3DRMPALETTEENTRY [m_rlimg.palette_size];
        for (int i = 0; i < m_rlimg.palette_size; i++) {
            m_rlimg.palette[i].red = pRGB[i].rgbRed;
            m_rlimg.palette[i].green = pRGB[i].rgbGreen;
            m_rlimg.palette[i].blue = pRGB[i].rgbBlue;
            m_rlimg.palette[i].flags = D3DRMPALETTE_FREE;
        }
    }
}

BOOL C3dImage::Load(CFile *fp)
{
    BOOL bIsPM = FALSE;
    BITMAPINFOHEADER bmi;
    RGBQUAD clrtab[256];
    BYTE *pBits = NULL;
    CDC dc;
	int iBitsSize, iColorTableSize;

    // get the current file position
	// (we might not be at the start)
    DWORD dwFileStart = fp->GetPosition();

    // delete any existing image
    _ReleaseMemory();
    
    // set up to read the new image
    memset(&bmi, 0, sizeof(bmi));
	bmi.biSize = sizeof(BITMAPINFOHEADER);

	// read the file header to get the file size and to
    // find where the bits start in the file
    BITMAPFILEHEADER BmpFileHdr;
    int iBytes;
    iBytes = fp->Read(&BmpFileHdr, sizeof(BmpFileHdr));
    if (iBytes != sizeof(BmpFileHdr)) {
        TRACE("Failed to read file header");
        goto $abort;
    }

    // check we have the magic 'BM' at the start
    if (BmpFileHdr.bfType != 0x4D42) {
        TRACE("Not a bitmap file");
        goto $abort;
    }

    // make a wild guess that the file is in Windows DIB
    // format and read the BITMAPINFOHEADER.  If it turns
    // out to be a PM DIB file we'll convert it later.
    iBytes = fp->Read(&bmi, sizeof(BITMAPINFOHEADER)); 
    if (iBytes != sizeof(BITMAPINFOHEADER)) {
        TRACE("Failed to read BITMAPINFOHEADER");
        goto $abort;
    }

    // check we got a real Windows DIB file
    if (bmi.biSize != sizeof(BITMAPINFOHEADER)) {
        if (bmi.biSize != sizeof(BITMAPCOREHEADER)) {
            TRACE(" File is not Windows or PM DIB format");
            goto $abort;
        }

        // set a flag to convert PM file to Win format later
        bIsPM = TRUE;

        // back up the file pointer and read the BITMAPCOREHEADER
        // and create the BITMAPINFOHEADER from it
        fp->Seek(dwFileStart + sizeof(BITMAPFILEHEADER), CFile::begin);
        BITMAPCOREHEADER BmpCoreHdr;
        iBytes = fp->Read(&BmpCoreHdr, sizeof(BmpCoreHdr)); 
        if (iBytes != sizeof(BmpCoreHdr)) {
            TRACE("Failed to read BITMAPCOREHEADER");
            goto $abort;
        }

        bmi.biSize = sizeof(BITMAPINFOHEADER);
        bmi.biWidth = (int) BmpCoreHdr.bcWidth;
        bmi.biHeight = (int) BmpCoreHdr.bcHeight;
        bmi.biPlanes = BmpCoreHdr.bcPlanes;
        bmi.biBitCount = BmpCoreHdr.bcBitCount;
        bmi.biCompression = BI_RGB;
        bmi.biSizeImage = 0;
        bmi.biXPelsPerMeter = 0;
        bmi.biYPelsPerMeter = 0;
        bmi.biClrUsed = 0;
        bmi.biClrImportant = 0;
    }

    // read in the color table from the file.
    bmi.biClrUsed = DIBColorEntries(&bmi);
    iColorTableSize = bmi.biClrUsed * sizeof(RGBQUAD);
    iBitsSize = bmi.biHeight * DIBStorageWidth(&bmi); 

	memset(&clrtab, 0, sizeof(clrtab));
    if (bIsPM == FALSE) {
        // read the color table from the file
        iBytes = fp->Read(clrtab, iColorTableSize);
        if (iBytes != iColorTableSize) {
            TRACE("Failed to read color table");
            goto $abort;
        }
    } else {
        // read each PM color table entry in turn and convert it
        // to Win DIB format as we go
        RGBQUAD* pRGB = clrtab;
        int i;
        RGBTRIPLE rgbt;
        for (i = 0; i < (int) bmi.biClrUsed; i++) {
            iBytes = fp->Read(&rgbt, sizeof(RGBTRIPLE));
            if (iBytes != sizeof(RGBTRIPLE)) {
                TRACE("Failed to read RGBTRIPLE");
                goto $abort;
            }
            pRGB->rgbBlue = rgbt.rgbtBlue;
            pRGB->rgbGreen = rgbt.rgbtGreen;
            pRGB->rgbRed = rgbt.rgbtRed;
            pRGB->rgbReserved = 0;
            pRGB++;
        }
    }

    // Create the memory for the bits
	pBits = new BYTE [iBitsSize];

	// see what we got
	if (pBits == NULL) {
		goto $abort;
	}

	// make all the bits black
	memset(pBits, 0, iBitsSize);

    // seek to the bits in the file
    fp->Seek(dwFileStart + BmpFileHdr.bfOffBits, CFile::begin);

    // read the bits
    iBytes = fp->Read(pBits, iBitsSize);
    if (iBytes != iBitsSize) {
        TRACE("Failed to read bits");
        goto $abort;
    }

    _Create(&bmi, clrtab, pBits);
	delete pBits;

	m_strName = "File image";
	return TRUE; 
                
$abort: // something went wrong

	
	return FALSE;    
}

// Load the image from a Windows bitmap file
BOOL C3dImage::Load(const char* pszFilename)
{
    CString strFile;    

    if ((pszFilename == NULL) 
    ||  (strlen(pszFilename) == 0)) {

        // Show an open file dialog to get the name
        CFileDialog dlg   (TRUE,    // open
                           NULL,    // no default extension
                           NULL,    // no initial file name
                           OFN_FILEMUSTEXIST
                             | OFN_HIDEREADONLY,
                           "Windows bitmaps (*.BMP)|*.BMP|All files (*.*)|*.*||");
        if (dlg.DoModal() != IDOK) {
            return FALSE;
        }

        strFile = dlg.GetPathName();

    } else {
        // copy the supplied file path
        strFile = pszFilename;                    
    }

    // Try to open the file for read access
    CFile file;
    if (! file.Open(strFile,
                    CFile::modeRead | CFile::shareDenyWrite)) {
        return FALSE;
    }

    if (!Load(&file)) return FALSE;

	m_strName = strFile;
	return TRUE;
}

BOOL C3dImage::LoadResource(const char* pszResname)
{
    ASSERT(pszResname);
    HINSTANCE hInst = AfxGetResourceHandle();
    HRSRC hrsrc = ::FindResource(hInst, pszResname, RT_BITMAP);
    if (!hrsrc) {
        TRACE("BITMAP resource %s not found\n", pszResname);
        return FALSE;
    }
    HGLOBAL hg = ::LoadResource(hInst, hrsrc);
    if (!hg) {
        TRACE("Failed to load BITMAP resource");
        return FALSE;
    }
    BITMAPINFO* pBMI = (BITMAPINFO*) LockResource(hg);
    ASSERT(pBMI);

	// create the new image from the packed DIB structure
    RGBQUAD* pRGB = (RGBQUAD*)(((BYTE*)pBMI)+sizeof(BITMAPINFOHEADER));
	BYTE* pBits = (BYTE*) pBMI
				+ sizeof(BITMAPINFOHEADER)
				+ DIBColorEntries((BITMAPINFOHEADER*)pBMI) * sizeof(RGBQUAD);

    _Create((BITMAPINFOHEADER*)pBMI, pRGB, pBits);

	m_strName = "Image resource";
	return TRUE;
}

// Divide the total image into smaller blocks
BOOL C3dImage::SetNumPhases(int iPhases)
{
	// Ensure the blocks will be at least one line high
	if ((iPhases+1) > m_iTotalHeight) return FALSE;

	m_iPhases = iPhases;
	m_rlimg.height = m_iTotalHeight / (m_iPhases+1);

	// Reset to phase 0
	return SetPhase(0);
}

BOOL C3dImage::SetPhase(int iPhase)
{
	if ((iPhase < 0) || (iPhase >= m_iPhases)) {
		return FALSE;
	}
	m_iCurPhase = iPhase;

	// Copy the bits for the requested phase to the work area
	int iBytes = m_rlimg.bytes_per_line * m_rlimg.height;
	memcpy(m_rlimg.buffer1, 
		   (BYTE*)m_rlimg.buffer1 + (m_iCurPhase + 1) * iBytes,
		   iBytes);

	// Notify any derived class of the change
	_OnImageChanged();
	return TRUE;
}

#if 0 // not used
BOOL C3dImage::MakeOpacityMask()
{
	if (m_rlimg.depth != 8) return FALSE; // must be 8 bpp

    m_rlimg.red_mask = 0x0000;
    m_rlimg.green_mask = 0x0000;
    m_rlimg.blue_mask = 0x0000;
    m_rlimg.alpha_mask = 0x00FF;
	m_rlimg.rgb = TRUE;
	return TRUE;
}
#endif


////////////////////////////////////////////////////////////////////////////
// C3dImageList

C3dImageList::C3dImageList()
{
}

C3dImageList::~C3dImageList()
{
	DeleteAll();
}

void C3dImageList::Remove(C3dImage* pImage)
{
	if (!pImage) return;
	POSITION pos = CObList::Find(pImage);
	if (pos) {
		RemoveAt(pos);
	}
}

void C3dImageList::Delete(C3dImage* pImage)
{
	if (!pImage) return;
	Remove(pImage);
	delete pImage;
}

void C3dImageList::DeleteAll()
{
	while (!IsEmpty()) {
		C3dImage* pImage = (C3dImage*) RemoveHead();
		delete pImage;
	}
}

C3dImage* C3dImageList::Find(const char* pszName)
{
	if (!pszName) return NULL;
	POSITION pos = GetHeadPosition();
	while (pos) {
		C3dImage* pImg = GetNext(pos);
		ASSERT(pImg);
		if (!strcmp(pszName, pImg->GetName())) {
			return pImg;
		}
	}
	return NULL;
}


//////////////////////////////////////////////////////////////////
// C3dTexture

IMPLEMENT_DYNAMIC(C3dTexture, C3dVisual)

C3dTexture::C3dTexture()
{
    m_pITexture = NULL;
	m_strName = "3D Texture";
}

C3dTexture::~C3dTexture()
{
    if (m_pITexture) m_pITexture->Release();
}

// Create a texture from a loaded image
BOOL C3dTexture::Create()
{
    // Release any exisitng texture
	if (m_pITexture) {
		m_pITexture->Release();
		m_pITexture = NULL;
	}

#ifdef _DEBUG
	// validate that the image is suitable I.e. that its sizes are powers of 2
	// Note: texture that don't meet this test can still be used as background images
	for (int i = 0; (1 << i) < GetWidth(); i++);
	for (int j = 0; (1 << j) < GetHeight(); j++);
	if (GetWidth() != (1 << i) || GetHeight() != (1 << j)) {
		TRACE("This image can't be used as a texture. Its sides are not exact powers of 2\n");
	}
#endif // _DEBUG

    if (!the3dEngine.CreateTexture(GetObject(), &m_pITexture)) {
        TRACE("Texture create failed\n");
        m_pITexture = NULL;
        return FALSE;
    }

    ASSERT(m_pITexture);
    return TRUE;
}

BOOL C3dTexture::Load(CFile* fp)
{
	if (!C3dImage::Load(fp)) {
		return FALSE;
	}
	return Create();
}

BOOL C3dTexture::Load(const char* pszFilename)
{
	if (!C3dImage::Load(pszFilename)) {
		return FALSE;
	}
	return Create();
}
		   
BOOL C3dTexture::LoadResource(const char* pszResname)
{
	if (!C3dImage::LoadResource(pszResname)) {
		return FALSE;
	}
	return Create();
}

BOOL C3dTexture::Load(UINT uiResid)
{
	if (!C3dImage::Load(uiResid)) {
		return FALSE;
	}
	return Create();
}

BOOL C3dTexture::SetTransparency(double r, double g, double b)
{
    ASSERT(m_pITexture);

	m_hr = m_pITexture->SetDecalTransparentColor(D3DRMCreateColorRGB(r, g, b));
	return SUCCEEDED(m_hr);
}

BOOL C3dTexture::SetColors(int iColors)
{
    ASSERT(m_pITexture);
	m_hr = m_pITexture->SetColors(iColors);
	return SUCCEEDED(m_hr);
}

BOOL C3dTexture::SetShades(int iShades)
{
    ASSERT(m_pITexture);
	m_hr = m_pITexture->SetShades(iShades);
	return SUCCEEDED(m_hr);
}

// virtual
void C3dTexture::_OnImageChanged()
{
	if (!m_pITexture) return; // might not be created yet

	// The image has changed so notify the renderer
	// that the pixels have changed but not the palette
	m_hr = m_pITexture->Changed(TRUE, FALSE);
	ASSERT(SUCCEEDED(m_hr));
}


//////////////////////////////////////////////////////////////////
// C3dWrap

IMPLEMENT_DYNAMIC(C3dWrap, C3dObject)

C3dWrap::C3dWrap()
{
    m_pIWrap = NULL;
}

C3dWrap::~C3dWrap()
{
    if (m_pIWrap) m_pIWrap->Release();
}

BOOL C3dWrap::Create(D3DRMWRAPTYPE type,
                     C3dFrame* pRefFrame,
                     double ox, double oy, double oz,
                     double dx, double dy, double dz,
                     double ux, double uy, double uz,
                     double ou, double ov,
                     double su, double sv)
{
    ASSERT(m_pIWrap == NULL); // only create once

    IDirect3DRMFrame* pIRefFrame = NULL;
    if (pRefFrame) {
        pIRefFrame = pRefFrame->GetInterface();
    }
    if (!the3dEngine.CreateWrap(type,
                                pIRefFrame,
                                ox, oy, oz,
                                dx, dy, dz,
                                ux, uy, uz,
                                ou, ov,
                                su, sv,
                                &m_pIWrap)) {
        TRACE("Wrap create failed\n");
        m_pIWrap = NULL;
        return FALSE;
    }

    ASSERT(m_pIWrap);
    return TRUE;
}

// Apply the wrap to an entire shape
BOOL C3dWrap::Apply(C3dShape* pShape)
{
    ASSERT(pShape);
    ASSERT(m_pIWrap);
    HRESULT hr;

    hr = m_pIWrap->Apply(pShape->GetVisual());
    return SUCCEEDED(hr);
}

// Apply the wrap to an entire shape
BOOL C3dWrap::ApplyRelative(C3dFrame* pFrame, C3dShape* pShape)
{
    ASSERT(pFrame);
	ASSERT(pShape);
    ASSERT(m_pIWrap);

    m_hr = m_pIWrap->ApplyRelative(pFrame->GetInterface(),
								   pShape->GetVisual());
    return SUCCEEDED(m_hr);
}

// Apply the wrap to a single face of a shape
BOOL C3dWrap::Apply(C3dShape* pShape, int nFace)
{
    ASSERT(pShape);
    ASSERT(m_pIWrap);
    if (nFace >= pShape->GetFaceCount()) return FALSE;

    // Get the face list
	IDirect3DRMMeshBuilder* pIBld = pShape->GetMeshBuilder();
	ASSERT(pIBld);
	IDirect3DRMFaceArray* pIFaces = NULL;
	m_hr = pIBld->GetFaces(&pIFaces);
	ASSERT(SUCCEEDED(m_hr));

	// Get the requested face
	IDirect3DRMFace* pIFace = NULL;
	m_hr = pIFaces->GetElement(nFace, &pIFace);
	ASSERT(SUCCEEDED(m_hr));

	// Apply the wrap to the face
    m_hr = m_pIWrap->Apply(pIFace);
	ASSERT(SUCCEEDED(m_hr));

	// Release the interfaces
	pIFace->Release();
	pIFaces->Release();

    return SUCCEEDED(m_hr);
}

//////////////////////////////////////////////////////////////////
// C3dChromeWrap

IMPLEMENT_DYNAMIC(C3dChromeWrap, C3dWrap)

static void C3dChromeWrapCallback(IDirect3DRMFrame* pIFrame,
                                  void* pArg, D3DVALUE delta)
{
	C3dChromeWrap* pThis = (C3dChromeWrap*) pArg;
	ASSERT(pThis);
	ASSERT(pThis->IsKindOf(RUNTIME_CLASS(C3dChromeWrap)));
    pThis->ApplyRelative(pThis->m_pShape,
						 pThis->m_pShape);
}

C3dChromeWrap::C3dChromeWrap()
{
	m_pShape = NULL;
}

C3dChromeWrap::~C3dChromeWrap()
{
}

BOOL C3dChromeWrap::Create(C3dShape* pShape, C3dCamera* pCamera)
{
	if (!C3dWrap::Create(D3DRMWRAP_CHROME,
						 pCamera,
						 0, 0, 0,
						 0, 1, 0, 
						 0, 0, -1, 
						 0, 0,	
						 1, -1)) {
		return FALSE;
	}

	// Install the movement callback
	ASSERT(pShape);
	m_pShape = pShape;
	IDirect3DRMFrame* pIFrame = pShape->GetInterface();
	ASSERT(pIFrame);
	m_hr = pIFrame->AddMoveCallback(C3dChromeWrapCallback, this);
	return SUCCEEDED(m_hr);
}

