// 3dInCtlr.cpp
//
// Copyright (c) Nigel Thompson 1996
//
// Implementation for:
// C3dController, C3dPosCtlr, C3dFlyCtlr
//

#include "stdafx.h"
#include "3dplus.h"

//////////////////////////////////////////////////////////////////
// C3dController
//

IMPLEMENT_SERIAL(C3dController, CObject, 0)

C3dController::C3dController()
{
	m_iDevs = 3;
	m_pWnd = NULL;
	m_iCurDevice = 0;
}

BOOL C3dController::Create(C3dWnd* pWnd,
						   GET3DFRAMEFN pGet3dFrameFn,
						   void* pArg/*= NULL*/,
						   CONTROLNOTIFYFN pNotifyFn/*= NULL*/,
						   DWORD dwNotify/*= CONTROL_NOTIFY_ALL*/)
{
	if (!pGet3dFrameFn || !pWnd) {
		return FALSE; // must have these
	}
	m_pGet3dFrameFn = pGet3dFrameFn;
	m_pArg = pArg;
	m_pNotifyFn = pNotifyFn;
	m_dwNotify = dwNotify;
	ASSERT(pWnd);
  	pWnd->AttachController(this);
	m_pWnd = pWnd;
	memset(&m_PrevState, 0, sizeof(m_PrevState));
	SelectDevice(0);
	return TRUE;
}

C3dController::~C3dController()
{
}

// virtual 
BOOL C3dController::ShowConfigDlg()
{
	// Show the current input device config dlg
	if (m_pInpDev) {
		return m_pInpDev->ShowConfigDlg();
	}
	return FALSE;
}

// virtual 
void C3dController::Update()
{
	if (!m_pInpDev) return; // no input device set

	// get a pointer to the shape
	ASSERT(m_pGet3dFrameFn);
	C3dFrame* pFrame = m_pGet3dFrameFn(m_pArg);
	if (!pFrame) return; // nothing to control :)

	// get new control value and update the shape's parameters
	_3DINPUTSTATE NewState;
	if (!m_pInpDev->GetState(NewState)) {
		return; // failed to read device
	}

	// call the derived class to modify the frame's parameters
	OnUpdate(NewState, pFrame);

	// make any change notifications
	DWORD dwFlags = 0;
	if (m_pNotifyFn) {
		if ((m_dwNotify & CONTROL_X_CHANGE) && (NewState.dX != m_PrevState.dX)) {
			dwFlags |= CONTROL_X_CHANGE;
		}
		if ((m_dwNotify & CONTROL_Y_CHANGE) && (NewState.dY != m_PrevState.dY)) {
			dwFlags |= CONTROL_Y_CHANGE;
		}
		if ((m_dwNotify & CONTROL_Z_CHANGE) && (NewState.dZ != m_PrevState.dZ)) {
			dwFlags |= CONTROL_Z_CHANGE;
		}
		if ((m_dwNotify & CONTROL_R_CHANGE) && (NewState.dR != m_PrevState.dR)) {
			dwFlags |= CONTROL_R_CHANGE;
		}
		if ((m_dwNotify & CONTROL_U_CHANGE) && (NewState.dU != m_PrevState.dU)) {
			dwFlags |= CONTROL_U_CHANGE;
		}
		if ((m_dwNotify & CONTROL_POV_CHANGE) && (NewState.dPov != m_PrevState.dPov)) {
			dwFlags |= CONTROL_POV_CHANGE;
		}
		if ((m_dwNotify & CONTROL_BTN_CHANGE) && (NewState.dwButtons != m_PrevState.dwButtons)) {
			dwFlags |= CONTROL_BTN_CHANGE;
		}

		if (dwFlags != 0) {
			m_pNotifyFn(m_pArg, dwFlags, &NewState);
		}
	}

	// save the new state
	m_PrevState = NewState;
}

// virtual
void C3dController::OnUpdate(_3DINPUTSTATE& st, C3dFrame* pFrame)
{
	ASSERT(0); // implement in derived class
}

// virtual
void C3dController::OnUserEvent(HWND hWnd, UINT uiMsg, WPARAM wParam, LPARAM lParam)
{
	// send the message to the input device
	if (m_pInpDev) {
		m_pInpDev->OnUserEvent(hWnd, uiMsg, wParam, lParam);
	}
}

// virtual
BOOL C3dController::SelectDevice(int i)
{
	switch (i) {
	case 0:
		m_pInpDev = &m_KeyDev;
		break;
	case 1:
		m_pInpDev = &m_MouseDev;
		break;
	case 2:
		if (m_JoyDev.IsPresent()) {
			m_pInpDev = &m_JoyDev;
		} else {
			m_pInpDev = &m_KeyDev;
			i = 0;
		}
		break;
	default:
		return FALSE;
	}
	m_iCurDevice = i;
	return TRUE;
}

// virtual
const char* C3dController::GetDeviceName(int i)
{
	switch (i) {
	case 0:
		return m_KeyDev.GetName();
		break;
	case 1:
		return m_MouseDev.GetName();
		break;
	case 2:
		return m_JoyDev.GetName();
		break;
	default:
		return FALSE;
	}
	return NULL;
}

//////////////////////////////////////////////////////////////////
// C3dPosCtlr - positioning controller
//

IMPLEMENT_SERIAL(C3dPosCtlr, C3dController, 0)

// virtual
void C3dPosCtlr::OnUpdate(_3DINPUTSTATE& st, C3dFrame* pFrame)
{
	// get the stage to use as a reference frame for the positions etc
	ASSERT(m_pWnd);
	C3dStage* pStage = m_pWnd->GetStage();
	ASSERT(pStage);

	double x, y, z;
	pFrame->GetPosition(x, y, z, pStage);
	x += st.dX * 0.1 ;
	y += st.dY * 0.1;
	z += st.dZ * 0.1;
	pFrame->SetPosition(x, y, z, pStage);

	C3dVector d, u;
	pFrame->GetDirection(d, u, pStage);
	// rotate the dir and up vectors
	double a = 3.0;
	C3dMatrix r;
	r.Rotate(-st.dR * a, -st.dU * a, -st.dV * a);
	d = r * d;
	u = r * u;
	pFrame->SetDirection(d, u, pStage);
}

//////////////////////////////////////////////////////////////////
// C3dFlyCtlr - flying controller
//

IMPLEMENT_SERIAL(C3dFlyCtlr, C3dController, 0)

// virtual
void C3dFlyCtlr::OnUpdate(_3DINPUTSTATE& st, C3dFrame* pFrame)
{
	// get the velocity (we use Z axis for this)
	double v = st.dZ / 10;

	// get the roll, pitch and yaw from the x, y and u axes
	double pitch = st.dY / 3;
	double roll = -st.dX / 3;
	double yaw = st.dU / 5;

	// modify the pitch and roll by the velocity for a bit of
	// extra realism
	// pitch *= v;
	// roll *= v;

	pFrame->AddRotation(1, 0, 0, pitch, D3DRMCOMBINE_BEFORE);
    pFrame->AddRotation(0, 0, 1, roll, D3DRMCOMBINE_BEFORE);
	pFrame->AddRotation(0, 1, 0, yaw, D3DRMCOMBINE_BEFORE);

    // Get the current direction vector
    double x1, y1, z1;
    pFrame->GetDirection(x1, y1, z1);

    // multiply the direction vector by the velocity
    x1 *= v;
    y1 *= v;
    z1 *= v;

    // Get the current position
    double x, y, z;
    pFrame->GetPosition(x, y, z);

    // Update the position
    x += x1;
    y += y1;
    z += z1;
    pFrame->SetPosition(x, y, z);

}

