// 3dInpDev.cpp
//
// Copyright (c) Nigel Thompson 1996
//
// Implementation for:
// C3dInputDevice
//

#include "stdafx.h"
#include "3dplus.h"

//////////////////////////////////////////////////////////////////
// C3dInputDevice

IMPLEMENT_SERIAL(C3dInputDevice, CObject, 0)

C3dInputDevice::C3dInputDevice()
{
    m_strName = "Generic 3D Input Device";
}

C3dInputDevice::~C3dInputDevice()
{
}

// virtual
BOOL C3dInputDevice::ShowConfigDlg()
{
	AfxMessageBox("The device is not configurable");
	return FALSE;
}

// virtual
BOOL C3dInputDevice::GetState(_3DINPUTSTATE& st)
{
	memset(&st, 0, sizeof(_3DINPUTSTATE));
	return FALSE;
}

// virtual
void C3dInputDevice::OnUserEvent(HWND hWnd, UINT uiMsg, WPARAM wParam, LPARAM lParam)
{
}

//////////////////////////////////////////////////////////////////
// C3dKeyInDev

IMPLEMENT_SERIAL(C3dKeyInDev, C3dInputDevice, 0)

C3dKeyInDev::C3dKeyInDev()
{
	memset(&m_st, 0, sizeof(m_st));
	m_st.dPov = -1;
    m_strName = "Keyboard";
	m_bShift = FALSE;
	m_bControl = FALSE;
}

C3dKeyInDev::~C3dKeyInDev()
{
}

// virtual
BOOL C3dKeyInDev::GetState(_3DINPUTSTATE& st)
{
	st = m_st;
	return TRUE;
}

// virtual
void C3dKeyInDev::OnUserEvent(HWND hWnd, UINT uiMsg, WPARAM wParam, LPARAM lParam)
{
	switch (uiMsg) {
	case WM_KEYDOWN:
		OnKeyDown(wParam, LOWORD(lParam), HIWORD(lParam));
		break;

	case WM_KEYUP:
		OnKeyUp(wParam, LOWORD(lParam), HIWORD(lParam));
		break;

	default:
		break;
	}
}

static void Inc(double& d)
{
	if (d < 0) {
		d = 0;
	} else {
		d += 0.02;
	}
}

static void Dec(double& d)
{
	if (d > 0) {
		d = 0;
	} else {
		d -= 0.02;
	}
}

// virtual
void C3dKeyInDev::OnKeyDown(UINT nChar, UINT nRepCnt, UINT nFlags)
{
	double dInc = 0.02;
	switch(nChar) {
	case VK_SHIFT:
		m_bShift = TRUE;
		break;
	case VK_CONTROL:
		m_bControl = TRUE;
		break;
	case VK_RIGHT:
		if (m_bShift) {
			Inc(m_st.dV);
		} else if (m_bControl) {
			Inc(m_st.dU);
		} else {
			Inc(m_st.dX);
		}
		break;
	case VK_LEFT:
		if (m_bShift) {
			Dec(m_st.dV);
		} else if (m_bControl) {
			Dec(m_st.dU);
		} else {
			Dec(m_st.dX);
		}
		break;
	case VK_UP:
		if (m_bShift) {
			Inc(m_st.dZ);
		} else if (m_bControl) {
			Inc(m_st.dR);
		} else {
			Inc(m_st.dY);
		}
		break;
	case VK_DOWN:
		if (m_bShift) {
			Dec(m_st.dZ);
		} else if (m_bControl) {
			Dec(m_st.dR);
		} else {
			Dec(m_st.dY);
		}
		break;
	case VK_ADD:
		Dec(m_st.dZ);
		break;
	case VK_SUBTRACT:
		Inc(m_st.dZ);
		break;
	case VK_PRIOR: // page up
		Inc(m_st.dV);
		break;
	case VK_NEXT: // page down
		Dec(m_st.dV);
		break;
	case VK_HOME:
		Inc(m_st.dU);
		break;
	case VK_END:
		Dec(m_st.dU);
		break;
	case VK_INSERT:
		Inc(m_st.dR);
		break;
	case VK_DELETE:
		Dec(m_st.dR);
		break;
	case VK_NUMPAD8:
		m_st.dPov = 0;
		break;
	case VK_NUMPAD9:
		m_st.dPov = 45;
		break;
	case VK_NUMPAD6:
		m_st.dPov = 90;
		break;
	case VK_NUMPAD3:
		m_st.dPov = 135;
		break;
	case VK_NUMPAD2:
		m_st.dPov = 180;
		break;
	case VK_NUMPAD1:
		m_st.dPov = 225;
		break;
	case VK_NUMPAD4:
		m_st.dPov = 270;
		break;
	case VK_NUMPAD7:
		m_st.dPov = 315;
		break;

	default:
		break;
	}
}

// virtual
void C3dKeyInDev::OnKeyUp(UINT nChar, UINT nRepCnt, UINT nFlags)
{
	switch(nChar) {
	case VK_SHIFT:
		m_bShift = FALSE;
		break;
	case VK_CONTROL:
		m_bControl = FALSE;
		break;

	case VK_RIGHT:
	case VK_LEFT:
		if (m_bShift) {
			m_st.dV = 0;
		} else if (m_bControl) {
			m_st.dU = 0;
		} else {
			m_st.dX = 0;
		}
		break;
	case VK_UP:
	case VK_DOWN:
		if (m_bShift) {
			m_st.dZ = 0;
		} else if (m_bControl) {
			m_st.dR = 0;
		} else {
			m_st.dY = 0;
		}
		break;
	case VK_ADD:
	case VK_SUBTRACT:
		m_st.dZ = 0;
		break;
	case VK_PRIOR: // page up
	case VK_NEXT: // page down
		m_st.dV = 0;
		break;
	case VK_HOME:
	case VK_END:
		m_st.dU = 0;
		break;
	case VK_INSERT:
	case VK_DELETE:
		m_st.dR = 0;
		break;

	case VK_NUMPAD8:
	case VK_NUMPAD9:
	case VK_NUMPAD6:
	case VK_NUMPAD3:
	case VK_NUMPAD2:
	case VK_NUMPAD1:
	case VK_NUMPAD4:
	case VK_NUMPAD7:
		m_st.dPov = -1;
		break;

	default:
		break;
	}
}

//////////////////////////////////////////////////////////////////
// C3dMouseInDev

IMPLEMENT_SERIAL(C3dMouseInDev, C3dInputDevice, 0)

C3dMouseInDev::C3dMouseInDev()
: m_ptPrev(-1, -1), m_ptCur(-1, -1)
{
	memset(&m_st, 0, sizeof(m_st));
	m_st.dPov = -1;
    m_strName = "Mouse";
	m_dwFlags = 0;
	m_bCaptured = 0;
}

C3dMouseInDev::~C3dMouseInDev()
{
}

// virtual
void C3dMouseInDev::OnUserEvent(HWND hWnd, UINT uiMsg, WPARAM wParam, LPARAM lParam)
{
	switch (uiMsg) {
	case WM_LBUTTONDOWN:
		::SetCapture(hWnd);
		m_bCaptured = TRUE;
		break;

	case WM_LBUTTONUP:
		if (m_bCaptured) {
			::ReleaseCapture();
			m_bCaptured = FALSE;
		}
		break;

	case WM_MOUSEMOVE: 
		if (m_bCaptured) {
			// Note: screen coords (See C3dWnd)
			m_ptCur.x = LOWORD(lParam);
			m_ptCur.y = HIWORD(lParam);
			m_dwFlags = wParam;
		}
		break;

	default:
		break;
	}
}

// virtual
BOOL C3dMouseInDev::GetState(_3DINPUTSTATE& st)
{
	if (m_ptPrev.x < 0) {
		m_ptPrev = m_ptCur;
	}
	
	// set the initial state
	m_st.dX = 0;
	m_st.dY = 0;
	m_st.dZ = 0;
	m_st.dR = 0;
	m_st.dU = 0;
	m_st.dV = 0;

	// see where the mouse has moved to
	int dx = m_ptCur.x - m_ptPrev.x;
	int dy = m_ptCur.y - m_ptPrev.y;

	// if it's a big move, ignore it
	if ((abs(dx) > 100) || (abs(dy) > 100)) {
		dx = 0;
		dy = 0;
	}

	// provide a deadband so the object doesn't wander
	int idb = 3;
	if (dx > idb) {
		dx -= idb;
	} else if (dx < -idb) {
		dx += idb;
	} else {
		dx = 0;
	}
	if (dy > idb) {
		dy -= idb;
	} else if (dy < -idb) {
		dy += idb;
	} else {
		dy = 0;
	}

	double dScale = 0.1;
	if (dx != 0) {
		double d = dx * dScale;
		if (m_dwFlags & MK_SHIFT) {
			m_st.dV = -d;
		} else if (m_dwFlags & MK_CONTROL) {
			m_st.dU = -d;
		} else {
			m_st.dX = d;
		}
	}
	if (dy != 0) {
		double d = dy * dScale;
		if (m_dwFlags & MK_SHIFT) {
			m_st.dZ = -d;
		} else if (m_dwFlags & MK_CONTROL) {
			m_st.dR = -d;
		} else {
			m_st.dY = -d;
		}
	}
	
	m_ptPrev = m_ptCur;
	
	st = m_st;
	return TRUE;
}

