// 3dJoyDev.cpp
//
// Copyright (c) Nigel Thompson 1996
//
// Implementation for:
// C3dJoyInDev
//

#include "stdafx.h"
#include "3dplus.h"
#include <winreg.h> // registry functions

//////////////////////////////////////////////////////////////////
// C3dJoyInDev
//
// Note: only one joystick is supported
//

IMPLEMENT_SERIAL(C3dJoyInDev, C3dInputDevice, 0)

C3dJoyInDev::C3dJoyInDev()
{
	// make sure we have joystick support
	// Get the number of supported devices
	m_bDevPresent = joyGetNumDevs() > 0 ? TRUE : FALSE;

	// If we have device support, see if we have an actual device
	if (m_bDevPresent) {
		if (GetInfo() == NULL) m_bDevPresent = FALSE; // no device
	}

	if (m_bDevPresent) {

		// get the device capabilities
		MMRESULT mr;
		memset(&mr, 0, sizeof(mr));
		mr = joyGetDevCaps(0, &m_caps, sizeof(m_caps));
		m_strName = m_caps.szPname; // driver name

		// try to get the real name from the registry
		CString strKey ("System\\CurrentControlSet\\control\\MediaResources\\joystick\\");
		strKey += m_caps.szRegKey;
		strKey += "\\CurrentJoystickSettings";
		LONG l;
		HKEY hKey = NULL;
		l = ::RegOpenKeyEx(HKEY_LOCAL_MACHINE,
						   strKey,
						   0,
						   KEY_READ,
						   &hKey);
		if (l == ERROR_SUCCESS) {
			char buf[128];
			DWORD dwSize = sizeof(buf);
			l = ::RegQueryValueEx(hKey,
								  "Joystick1OEMName",
								  0,
								  NULL,
								  (BYTE*)buf,
								  &dwSize);
			if (l == ERROR_SUCCESS) {
				m_strName = buf;
			}
		}

		// initialize the configuration parameters from the registry
		CWinApp* pa = AfxGetApp();
		strKey = "\\Settings\\Joystick\\";
		strKey += m_strName;

		// See if there is an entry for this joystick
		CString strAxis(strKey);
		char buf[32];
		strcpy(buf, "\\Axis0");
		strAxis += buf;
		int i = pa->GetProfileInt(strAxis, "Input", -99);
		if (i != -99) {

			// we have an entry so load the rest of the data from it
			for (i = 0; i < 6; i++) {
				strAxis = strKey;
				sprintf(buf, "\\Axis%d", i);
				strAxis += buf; 

				m_cfg.axis[i].iInput = pa->GetProfileInt(strAxis, "Input", i);
				m_cfg.axis[i].iButton = pa->GetProfileInt(strAxis, "Button", 0);
				m_cfg.axis[i].iDead = pa->GetProfileInt(strAxis, "Dead", 10);
				CString strVal = pa->GetProfileString(strAxis, "Sens", "1.0");
				m_cfg.axis[i].dSens = atof(strVal);
			}

		} else {

			// Do a default setup
			for (i = 0; i < 6; i++) {
				m_cfg.axis[i].iInput = i;
				m_cfg.axis[i].iButton = 0;
				m_cfg.axis[i].iDead = 10;
				m_cfg.axis[i].dSens = 1.0;
			}

			// Special case any joysticks we know
			if (strcmp(m_strName, "Microsoft SideWinder 3D Pro") == 0) {
			
				// Microsoft SideWinder
				m_cfg.axis[1].dSens = -1.0;
				m_cfg.axis[2].dSens = -1.0;
				m_cfg.axis[3].iButton = 4;
				m_cfg.axis[3].iInput = 1;
				m_cfg.axis[3].dSens = -1.0;
				m_cfg.axis[4].iInput = 3;
				m_cfg.axis[5].iButton = 4;
				m_cfg.axis[5].iInput = 0;
				m_cfg.axis[5].dSens = -1.0;

			} else if (strcmp(m_strName, "Spaceball") == 0) {

				// Spaceball Avenger
				for (i = 0; i < 6; i++) {
					m_cfg.axis[i].iDead = 30; // need lots of dead band!
				}
				m_cfg.axis[1].dSens = -1.0;
				m_cfg.axis[4].dSens = -1.0;

			}
		}

	} else {
		m_strName = "Joystick (not present)";
	}

}

C3dJoyInDev::~C3dJoyInDev()
{
}

JOYINFOEX* C3dJoyInDev::GetInfo()
{
	MMRESULT mr;
	memset(&m_inf, 0, sizeof(m_inf));
	m_inf.dwSize = sizeof(m_inf);
	m_inf.dwFlags = JOY_RETURNALL;
	mr = joyGetPosEx(0, &m_inf);
	if (mr != JOYERR_NOERROR) {
		m_bDevPresent = FALSE;
		return NULL;
	}

	return &m_inf;
}

// virtual
BOOL C3dJoyInDev::GetState(_3DINPUTSTATE& st)
{
	memset(&st, 0, sizeof(_3DINPUTSTATE));

	// check we have a device
	if (!m_bDevPresent) return FALSE;

	// get the new joystick state
	if (!GetInfo()) {
		return FALSE;
	}

	// update the state info
	if (m_inf.dwFlags & JOY_RETURNPOV) {
		st.dPov = (double) m_inf.dwPOV / 100;
		if (st.dPov >= 360.0) st.dPov = -1.0; // not valid
	}
	if (m_inf.dwFlags & JOY_RETURNBUTTONS) {
		st.dwButtons = m_inf.dwButtons;
	}

	int iVal[6];
	iVal[0] = m_inf.dwXpos;
	iVal[1] = m_inf.dwYpos;
	iVal[2] = m_inf.dwZpos;
	iVal[3] = m_inf.dwRpos;
	iVal[4] = m_inf.dwUpos;
	iVal[5] = m_inf.dwVpos;
	double dOut[6];
	
	int iMin = 0;
	int iMax = 65535; // assume these for now

	// get the set of buttons used for qualifying outputs
	DWORD dwQualBtn = 0;
	for (int i = 0; i < 6; i++) {
		int iBtn = m_cfg.axis[i].iButton;
		DWORD dwBtnMask = 0;
		if (iBtn > 0) {
			dwBtnMask = 1 << (iBtn - 1);
			dwQualBtn |= dwBtnMask;
		}
	}
	
	for (i = 0; i < 6; i++) {
		// get the input channel
		int iInCh = m_cfg.axis[i].iInput;

		// get the qualifier button
		int iBtn = m_cfg.axis[i].iButton;
		DWORD dwBtnMask = 0;
		if (iBtn > 0) {
			dwBtnMask = 1 << (iBtn - 1);
		}

		// Set the deadband value
		int iDeadRange = (iMax - iMin) * m_cfg.axis[i].iDead / 100;
		int iDeadMin = (iMax - iMin - iDeadRange) / 2;
		int iDeadMax = iDeadMin + iDeadRange;

		// get the value of the input
		int iInVal = iVal[iInCh];

		int iOut = (iMin + iMax) / 2;

		// see if we require a qualifying button
		if (((dwBtnMask == 0) && !(m_inf.dwButtons & dwQualBtn))
		|| (m_inf.dwButtons & dwBtnMask)) {
				
			// compute the output value given the dead band and range
			double dscale = m_cfg.axis[i].dSens;
			if (iInVal < iDeadMin) {
				iOut -= (int)((iDeadMin - iInVal) * dscale); 
			} else if (iInVal > iDeadMax) {
				iOut += (int)((iInVal - iDeadMax) * dscale); 
			}
		}

		// save the output value
		dOut[i] = ((double)(iOut - (iMin + iMax) / 2))
				/ (double)(iMax - iMin);
	}

	st.dX = dOut[0];
	st.dY = dOut[1];
	st.dZ = dOut[2];
	st.dR = dOut[3];
	st.dU = dOut[4];
	st.dV = dOut[5];

	return TRUE;
}

// virtual 
BOOL C3dJoyInDev::ShowConfigDlg()
{
	CPropertySheet ps("Joystick Settings");
	CAxesPage ap;
	ap.m_pDev = this;
	ps.AddPage(&ap);
	if (!ps.DoModal()) {
		return FALSE;
	}

	// Save the settings in the registry
	// initialize the configuration parameters from the registry
	CWinApp* pa = AfxGetApp();
	CString	strKey = "\\Settings\\Joystick\\";
	strKey += m_strName;

	for (int i = 0; i < 6; i++) {
		CString strAxis(strKey);
		char buf[32];
		sprintf(buf, "\\Axis%d", i);
		strAxis += buf; 

		pa->WriteProfileInt(strAxis, "Input", m_cfg.axis[i].iInput);
		pa->WriteProfileInt(strAxis, "Button", m_cfg.axis[i].iButton);
		pa->WriteProfileInt(strAxis, "Dead", m_cfg.axis[i].iDead);
		sprintf(buf, "%f", m_cfg.axis[i].dSens);
		pa->WriteProfileString(strAxis, "Sens", buf);
	}

	return TRUE;
}


