// 3dShape.cpp
//
// Copyright (c) Nigel Thompson 1996
//
// Implementation for:
// C3dShape, C3dShapeList
//

#include "stdafx.h"
#include "3dplus.h"

// some math constants
static const double _pi =  3.14159265359;
static const double _twopi =  6.28318530718;

// math macros
#define SQR(x) ((x) * (x))

//////////////////////////////////////////////////////////////////
// C3dShape

IMPLEMENT_DYNAMIC(C3dShape, C3dFrame)

C3dShape::C3dShape()
{
    m_pIVisual = NULL;
    C3dFrame::Create(NULL);
    ASSERT(m_pIFrame);
	m_pIFrame->SetAppData((ULONG)this);
	m_strName = "3D Shape";
	m_pIMeshBld = NULL;
    the3dEngine.CreateMeshBuilder(&m_pIMeshBld);
    ASSERT(m_pIMeshBld);
	AttachVisual(m_pIMeshBld);
}

C3dShape::~C3dShape()
{
    if (m_pIVisual) m_pIVisual->Release();
	if (m_pIMeshBld) m_pIMeshBld->Release();
	m_ImgList.DeleteAll();
}

void C3dShape::New()
{
	// delete visual
	AttachVisual(NULL);

	// delete any child frames
	RemoveAllChildren();

	// stop rotation, reset orientation and position
	SetPosition(0.0, 0.0, 0.0);
	SetRotation(0.0, 1.0, 0.0, 0.0);
	SetDirection(0.0, 0.0, 1.0, 0.0, 1.0, 0.0);

	m_strName = "3D Shape";
}

void C3dShape::AttachVisual(IDirect3DRMVisual* pIVisual)
{
    // AddRef the new one just in case it's the same
	// as the old one
	if (pIVisual) {
		pIVisual->AddRef();
	}

	// release any existing visual
    if (m_pIVisual) {
        m_hr = m_pIFrame->DeleteVisual(m_pIVisual);
        ASSERT(SUCCEEDED(m_hr));
		m_pIVisual->Release();
        m_pIVisual = NULL;
    }

	// Make the new one current
    if (pIVisual) {
        m_pIVisual = pIVisual;
        m_hr = m_pIFrame->AddVisual(m_pIVisual);
        ASSERT(SUCCEEDED(m_hr));
    }
}

// Create from a vector set and a normal set
BOOL C3dShape::Create(D3DVECTOR* pVectors, int iVectors,
                      D3DVECTOR* pNormals, int iNormals,
                      int* pFaceData, BOOL bAutoGen)
{
    ASSERT(m_pIMeshBld);

    // Build the mesh from the vector list
    ASSERT(sizeof(ULONG) == sizeof(int));
    m_hr = m_pIMeshBld->AddFaces(iVectors, pVectors, 
                              iNormals, pNormals,
                              (ULONG*)pFaceData, NULL);
    ASSERT(SUCCEEDED(m_hr));
	if ((iNormals == 0) && bAutoGen) {
		m_pIMeshBld->GenerateNormals();
	}

	AttachVisual(m_pIMeshBld);

	// enable perspective correction
	m_pIMeshBld->SetPerspective(TRUE);

    return TRUE;
}

BOOL C3dShape::AddFaces(D3DVECTOR* pVectors, int iVectors,
				  D3DVECTOR* pNormals, int iNormals,
				  int* pFaceData)
{
    ASSERT(m_pIMeshBld);

    m_hr = m_pIMeshBld->AddFaces(iVectors, pVectors, 
                                 iNormals, pNormals,
                                 (ULONG*)pFaceData, NULL);
    return(SUCCEEDED(m_hr));
}

	
// Create a cube centered at the origin of given side size
BOOL C3dShape::CreateCuboid(double x, double y, double z)
{
    New();
	D3DVALUE x1 = D3DVAL(x / 2);
	D3DVALUE y1 = D3DVAL(y / 2);
	D3DVALUE z1 = D3DVAL(z / 2);
    D3DVECTOR vert[] = {
        {-x1, -y1, -z1},
        {-x1, -y1,  z1},
        { x1, -y1,  z1},
        { x1, -y1, -z1},
        {-x1,  y1, -z1},
        {-x1,  y1,  z1},
        { x1,  y1,  z1},
        { x1,  y1, -z1}
    };

	D3DVECTOR nlist [] = {
		{ 1,  0,  0},
		{ 0,  1,  0},
		{ 0,  0,  1},
		{-1,  0,  0},
		{ 0, -1,  0},
		{ 0,  0, -1}
	};
    
	int flist [] = {4, 0, 4, 3, 4, 2, 4, 1, 4,
                    4, 3, 0, 7, 0, 6, 0, 2,	0,
                    4, 4, 1, 5, 1, 6, 1, 7,	1,
                    4, 0, 3, 1, 3, 5, 3, 4,	3,
                    4, 0, 5, 4, 5, 7, 5, 3,	5,
                    4, 2, 2, 6, 2, 5, 2, 1,	2,
                    0
    };

    BOOL b = Create(vert, 8, nlist, 6, flist);
	m_strName = "Cuboid";

	return b;
}

BOOL C3dShape::CreateSphere(double r, int nBands)
{
    New();
	
	// make sure the number of bands is >= 3
	if (nBands < 3) {
		nBands = 3;
	}

	// compute the vertex count
	int iVertices = (nBands - 1) * nBands + 2;

	// compute the face count
	int iFaces = nBands * nBands;

    // create the list of vertices
    D3DVECTOR* Vertices = new D3DVECTOR[iVertices];
	D3DVECTOR* pv = Vertices;

    // create the face list
    int* FaceData = new int[iFaces * 5];
    int* pfd = FaceData;

	// set up the first vertex at the top
	pv->x = D3DVAL(0);
	pv->y = D3DVAL(r);
	pv->z = D3DVAL(0);
	pv++;
	int iv = 1;	// next free vertex number

	// top band
	double da = _pi / nBands;
	double a = da;
	
	// compute the y value for the bottom of the band
	double y = r * cos(a);
	double rband = r * sin(a);
	double ab = 0;

	// Write out the vertices for the bottom of this band
	int iv1 = iv; // first vertex on this band
	for (int i = 0; i < nBands; i++) {
		pv->x = D3DVAL(rband * sin(ab));
		pv->y = D3DVAL(y);
		pv->z = D3DVAL(rband * cos(ab));
		pv++;
		iv++;
		ab += da * 2;
	}

	// Write out the face values
	for (i = 0; i < nBands; i++) {
		*pfd++ = 3;
		*pfd++ = iv1 + (i % nBands);
		*pfd++ = iv1 + ((i + 1) % nBands);
		*pfd++ = 0; // top point
	}

	// now do the middle bands
	for (int iBand = 1; iBand < nBands-1; iBand++) {
		a += da;
		y = r * cos(a);
		rband = r * sin(a);
		// Write out the vertices for the bottom of this band
		ab = 0;
		int iv1 = iv; // first vertex on this band
		for (int i = 0; i < nBands; i++) {
			pv->x = D3DVAL(rband * sin(ab));
			pv->y = D3DVAL(y);
			pv->z = D3DVAL(rband * cos(ab));
			pv++;
			iv++;
			ab += da * 2;
		}

		// Write out the face values
		for (i = 0; i < nBands; i++) {
			*pfd++ = 4;
			*pfd++ = iv1 + (i % nBands);
			*pfd++ = iv1 + ((i + 1) % nBands);
			*pfd++ = iv1 - nBands + ((i + 1) % nBands);
			*pfd++ = iv1 - nBands + (i % nBands);
		}

	}

	// now do the last band
	// Write out the vertex for the bottom of this band
	iv1 = iv; // first vertex on this band
	pv->x = D3DVAL(0);
	pv->y = D3DVAL(-r);
	pv->z = D3DVAL(0);
	pv++;
	iv++;

	// Write out the face values
	for (i = 0; i < nBands; i++) {
		*pfd++ = 3;
		*pfd++ = iv1; // bottom point
		*pfd++ = iv1 - nBands + ((i + 1) % nBands);
		*pfd++ = iv1 - nBands + (i % nBands);
	}

	*pfd = 0; // end the face list

    BOOL b = Create(Vertices, iv, NULL, 0, FaceData, TRUE);
	delete [] Vertices;
	delete [] FaceData;
	m_strName = "Sphere";

	return b;
}

BOOL C3dShape::CreateSurface(double x1, double x2, double dx,
                             double z1, double z2, double dz,
                             SURFHTFN pfnHeight, void* pArg)
{
    New();
    ASSERT(dx != 0);
    ASSERT(dz != 0);
	ASSERT(pfnHeight);
    int iXSteps = (int)((x2 - x1) / dx);
    int iZSteps = (int)((z2 - z1) / dz);
    if ((iXSteps < 1) || (iZSteps < 1)) return FALSE;

    // Create the array for the vertices
    int iVertices = (iXSteps + 1) * (iZSteps + 1);
    D3DVECTOR* Vertices = new D3DVECTOR [iVertices];
    D3DVECTOR* pv = Vertices;

    // Create the array for the face data
    // each face will have 4 vertices
    int iFaces = iXSteps * iZSteps;
    int* FaceData = new int [iFaces * 5 + 1];
    int* pfd = FaceData;

    // write out the vertex set
    double x = x1;
    double z;
    for (int iRow = 0; iRow <= iXSteps; iRow++) {
        z = z1;
        for (int iCol = 0; iCol <= iZSteps; iCol++) {
            pv->x = D3DVAL(x);
            pv->z = D3DVAL(z);
            pv->y = D3DVAL(pfnHeight(x, z, pArg));
            pv++;
            z += dz;
        }
        x += dx;
    }

    // write out the face list
    int iFirst = iZSteps + 1;
    for (iRow = 0; iRow < iXSteps; iRow++) {
        for (int iCol = 0; iCol < iZSteps; iCol++) {
            *pfd++ = 4; // no of vertices per face
            *pfd++ = iFirst + iCol;
            *pfd++ = iFirst - iZSteps - 1 + iCol;
            *pfd++ = iFirst - iZSteps + iCol;
            *pfd++ = iFirst + iCol + 1;
        }
        iFirst += iZSteps + 1;
    }
    *pfd = 0; // end the list

    // Create the surface with auto-generation of the normals
    BOOL b = Create(Vertices, iVertices, NULL, 0, FaceData, TRUE);
	delete [] Vertices;
	delete [] FaceData;
	m_strName = "Surface";

	return b;
}

// Create a solid of revolution
BOOL C3dShape::CreateRSolid(double z1, double z2, double dz,
							BOOL bClosed1, BOOL bClosed2,
					        SOLIDRFN pfnRad, void* pArg,
							int nFacets)
{
    New();
    ASSERT(pfnRad);
    ASSERT(dz != 0);
    int iZSteps = (int)((z2 - z1) / dz);
    if (iZSteps < 1) return FALSE;

	int iRSteps = nFacets;
	if (iRSteps < 8) iRSteps = 8;
	double da = _twopi / iRSteps;

    // Create the array for the vertices
    int iVertices = (iZSteps + 1) * iRSteps;
    D3DVECTOR* Vertices = new D3DVECTOR [iVertices];
    D3DVECTOR* pv = Vertices;

    // Create the array for the face data
    // each face will have 4 vertices except the ends
    int iFaces = iZSteps * iRSteps;
	int iFaceEntries = iFaces * 5 + 1;
	if (bClosed1) iFaceEntries += iRSteps + 1;
	if (bClosed2) iFaceEntries += iRSteps + 1;
    int* FaceData = new int [iFaceEntries];
    int* pfd = FaceData;

    // write out the vertex set
    double z = z1;
	double r, a;
    for (int iZ = 0; iZ <= iZSteps; iZ++) {
		r = pfnRad(z, pArg);
		a = 0;
        for (int iR = 0; iR < iRSteps; iR++) {
            pv->x = D3DVAL(r * sin(a));
            pv->y = D3DVAL(r * cos(a));
            pv->z = D3DVAL(z);
            pv++;
            a += da;
        }
        z += dz;
    }

    // write out the face list
	int iFirst = iRSteps;
    for (iZ = 0; iZ < iZSteps; iZ++) {
        for (int iR = 0; iR < iRSteps; iR++) {
            *pfd++ = 4; // no of vertices per face
            *pfd++ = iFirst + iR;
            *pfd++ = iFirst + ((iR + 1) % iRSteps);
            *pfd++ = iFirst - iRSteps + ((iR + 1) % iRSteps);
            *pfd++ = iFirst - iRSteps + iR;
        }
        iFirst += iRSteps;
    }
    *pfd = 0; // end the list

    // Create the round surface with auto-generation of the normals
    BOOL b = Create(Vertices, iVertices, NULL, 0, FaceData, TRUE);

	delete [] FaceData;

	FaceData = new int [iRSteps * 2 + 2];
	D3DVECTOR nvect [] = {
		{0, 0, 1},
		{0, 0, -1}
	};

	if (bClosed1) {
		pfd = FaceData;
		*pfd++ = iRSteps;
        for (int iR = 0; iR < iRSteps; iR++) {
			*pfd++ = iR;
			*pfd++ = 1;
		}
		*pfd = 0;
		m_hr = m_pIMeshBld->AddFaces(iVertices, Vertices,  
								     2, nvect,
								     (ULONG*)FaceData, NULL);
		ASSERT(SUCCEEDED(m_hr));
	}
	if (bClosed2) {
		pfd = FaceData;
		*pfd++ = iRSteps;
		iFirst = iRSteps * iZSteps;
        for (int iR = 0; iR < iRSteps; iR++) {
			*pfd++ = iRSteps - 1 - iR + iFirst;
			*pfd++ = 0;
		}
		*pfd = 0;
		m_hr = m_pIMeshBld->AddFaces(iVertices, Vertices,  
								     2, nvect,
								     (ULONG*)FaceData, NULL);
		ASSERT(SUCCEEDED(m_hr));
	}

	delete [] Vertices;
	delete [] FaceData;
	m_strName = "Solid of revolution";

	return b;
}

// Private data for cone function
typedef struct _LINEDATA {
	double c;
	double m;
} LINEDATA;

// Cone fn
static double LineFn(double z, void* pArg)
{
	LINEDATA* pld = (LINEDATA*) pArg;
	ASSERT(pld);
	return pld->m * z + pld->c;
}

// Create a cone between points
BOOL C3dShape::CreateCone(double x1, double y1, double z1,
					double r1, BOOL bClosed1,
					double x2, double y2, double z2,
					double r2, BOOL bClosed2,
					int nFacets)
{
	if ((r1 == 0) && (r2 == 0)) return FALSE;

	// compute the length
	double l = sqrt(SQR(x2-x1)+SQR(y2-y1)+SQR(z2-z1));
	if (l <= 0) return FALSE;

	// construct the private function data
	LINEDATA ld;
	ld.c = r1;
	ld.m = (r2 - r1) / l;

	// build the cone from (0, 0, 0) to (0, 0, l)
	BOOL bResult = CreateRSolid(0, l, l,
								bClosed1, bClosed2,
								LineFn, &ld,
								nFacets);
	if (!bResult) return FALSE;

	// translate to the correct origin
	SetPosition(x1, y1, z1);

	// point it in the right direction
	SetDirection(x2 - x1, y2 - y1, z2 - z1);
	m_strName = r1 == r2 ? "Tube" : "Cone";

	return TRUE;
}

// Texture load callback function
static HRESULT C3dLoadTextureCallback(char* pszName, void* pArg, LPDIRECT3DRMTEXTURE* ppITex)
{
	C3dShape* pThis = (C3dShape*) pArg;
	ASSERT(pThis);
	ASSERT(pThis->IsKindOf(RUNTIME_CLASS(C3dShape)));
	ASSERT(pszName);

	// Load the texture map
	ASSERT(ppITex);
	C3dTexture* pTex = new C3dTexture;
	if (!pTex->Load((const char*)pszName)) {
		delete pTex;
		return E_FAIL;
	}
	*ppITex = pTex->GetInterface();

	// Add the tex. map to the shape's image list
	pThis->m_ImgList.Append(pTex);
	return NOERROR; 
}

// Load a shape from an XOF file
// If no filename is given, show a file open dialog
// returns the file name or NULL if it fails
const char* C3dShape::Load(const char* pszFileName)
{
	static CString strFile;

	if (!pszFileName || !strlen(pszFileName)) {

		// Show a file open dialog
		CFileDialog dlg(TRUE,
			            NULL,
						NULL,
						OFN_HIDEREADONLY,
						_3DOBJ_LOADFILTER,
						NULL);
		if (dlg.DoModal() != IDOK) return NULL;

		// get the file path
		strFile = dlg.m_ofn.lpstrFile;

	} else {

		strFile = pszFileName;

	}

    // remove any existing visual
	New();

    // Try to load the file
    ASSERT(m_pIMeshBld);
    m_hr = m_pIMeshBld->Load((void*)(const char*)strFile,
		                     NULL,
							 D3DRMLOAD_FROMFILE | D3DRMLOAD_FIRST,
							 C3dLoadTextureCallback,
							 this);
    if (FAILED(m_hr)) {
        return NULL;
    }

	AttachVisual(m_pIMeshBld);

	m_strName = "File object: ";
	m_strName += pszFileName;

	return strFile;
}

// Texture load callback function
static HRESULT C3dLoadTextureResCallback(char* pszName, void* pArg, LPDIRECT3DRMTEXTURE* ppITex)
{
	C3dShape* pThis = (C3dShape*) pArg;
	ASSERT(pThis);
	ASSERT(pThis->IsKindOf(RUNTIME_CLASS(C3dShape)));
	ASSERT(pszName);

	// Load the texture map
	ASSERT(ppITex);
	C3dTexture* pTex = new C3dTexture;
	if (!pTex->LoadResource((const char*)pszName)) {
		delete pTex;
		return E_FAIL;
	}
	*ppITex = pTex->GetInterface();

	// Add the tex. map to the shape's image list
	pThis->m_ImgList.Append(pTex);
	return NOERROR; 
}

// Load a shape from an XOF resource
BOOL C3dShape::Load(UINT uiResid)
{
	// remove any existing visual
	New();

	// Try to load the file
    ASSERT(m_pIMeshBld);
	D3DRMLOADRESOURCE info;
	info.hModule = AfxGetResourceHandle();
	info.lpName = MAKEINTRESOURCE(uiResid);
	info.lpType = "XOF";
    m_hr = m_pIMeshBld->Load(&info,
							NULL,
							D3DRMLOAD_FROMRESOURCE,
							C3dLoadTextureResCallback,
							this);
	ASSERT(SUCCEEDED(m_hr));
    if (FAILED(m_hr)) {
        return FALSE;
    }

	AttachVisual(m_pIMeshBld);

	m_strName = "Resource object";

	return TRUE;
}

// Get the bounding box
BOOL C3dShape::GetBox(double& x1, double& y1, double& z1,
                      double& x2, double& y2, double& z2)
{
	ASSERT(m_pIFrame);

	if (m_pIMeshBld == NULL) {
		return FALSE;
	}

	// get the bounding box
	D3DRMBOX box;
	m_hr = m_pIMeshBld->GetBox(&box);
	ASSERT(SUCCEEDED(m_hr));

	// copy the results
	x1 = box.min.x;
	y1 = box.min.y;
	z1 = box.min.z;
	x2 = box.max.x;
	y2 = box.max.y;
	z2 = box.max.z;

	return TRUE;
}

int C3dShape::GetFaceCount()
{
	ASSERT(m_pIMeshBld);
	int i = (int) m_pIMeshBld->GetFaceCount();
	return i;
}

BOOL C3dShape::SetColor(double r, double g, double b)
{
	ASSERT(m_pIMeshBld);
	m_hr = m_pIMeshBld->SetColorRGB(r, g, b);
	return SUCCEEDED(m_hr);
}

BOOL C3dShape::SetFrameColor(double r, double g, double b)
{
	ASSERT(m_pIFrame);
	m_hr = m_pIFrame->SetColorRGB(r, g, b);
	return SUCCEEDED(m_hr);
}

BOOL C3dShape::SetFaceColor(int nFace, double r, double g, double b)
{
    if (nFace >= GetFaceCount()) return FALSE;

    // Get the face list
	IDirect3DRMFaceArray* pIFaces = NULL;
	ASSERT(m_pIMeshBld);
	m_hr = m_pIMeshBld->GetFaces(&pIFaces);
	ASSERT(SUCCEEDED(m_hr));

	// Get the requested face from the list
	IDirect3DRMFace* pIFace = NULL;
	m_hr = pIFaces->GetElement(nFace, &pIFace);
	ASSERT(SUCCEEDED(m_hr));

	// Set the face color
	m_hr = pIFace->SetColorRGB(r, g, b);
	ASSERT(SUCCEEDED(m_hr));

	// Finished with the face and list interfaces
	pIFace->Release();
	pIFaces->Release();

	return TRUE;
}

BOOL C3dShape::SetMaterial(C3dMaterial* pMaterial)
{
	ASSERT(m_pIMeshBld);
	ASSERT(pMaterial);
	m_hr = m_pIMeshBld->SetMaterial(pMaterial->GetInterface());
	return SUCCEEDED(m_hr);
}

BOOL C3dShape::SetTexture(C3dTexture* pTexture)
{
	ASSERT(pTexture);
	ASSERT(m_pIMeshBld);
	m_hr = m_pIMeshBld->SetTexture(pTexture->GetInterface());
	return SUCCEEDED(m_hr);
}

BOOL C3dShape::SetFaceTexture(int nFirst, int nCount, C3dTexture* pTexture)
{
    ASSERT(pTexture);
	// Get the face list
	IDirect3DRMFaceArray* pIFaces = NULL;
	ASSERT(m_pIMeshBld);
	m_hr = m_pIMeshBld->GetFaces(&pIFaces);
	ASSERT(SUCCEEDED(m_hr));
	for (int nFace = nFirst; nFace < nFirst+nCount; nFace++) {
		IDirect3DRMFace* pIFace = NULL;
		m_hr = pIFaces->GetElement(nFace, &pIFace);
		ASSERT(SUCCEEDED(m_hr));
		m_hr = pIFace->SetTexture(pTexture->GetInterface());
		ASSERT(SUCCEEDED(m_hr));
		pIFace->Release();
	}
	pIFaces->Release();

	return TRUE;
}


// Get the normal vector for a face
C3dVector C3dShape::GetFaceNormal(int nFace)
{
    if (nFace >= GetFaceCount()) {
		return C3dVector(0, 0, 1);
	}

    // Get the face list
	IDirect3DRMFaceArray* pIFaces = NULL;
	ASSERT(m_pIMeshBld);
	m_hr = m_pIMeshBld->GetFaces(&pIFaces);
	ASSERT(SUCCEEDED(m_hr));
	IDirect3DRMFace* pIFace = NULL;
	m_hr = pIFaces->GetElement(nFace, &pIFace);
	ASSERT(SUCCEEDED(m_hr));
	D3DVECTOR rlv;
	m_hr = pIFace->GetNormal(&rlv);
	ASSERT(SUCCEEDED(m_hr));
	pIFace->Release();
	pIFaces->Release();
	return C3dVector(rlv.x, rlv.y, rlv.z);
}

// Clone a shape
// Note: any attempt to modify the new shape will fail since the
// mesh builder doesn't know about the attached visual
C3dShape* C3dShape::Clone()
{
	C3dShape* pClone = new C3dShape;
	pClone->AttachVisual(m_pIVisual);
	CString strName = GetName();
	strName += " (Clone)";
	pClone->SetName(strName);
	C3dVector p, d, u;
	GetPosition(p);
	GetDirection(d, u);
	pClone->SetPosition(p);
	pClone->SetDirection(d, u);
	return pClone;
}

#if 0
// Add the mesh from another shape
void C3dShape::Add(C3dShape* pShape)
{
	// Get the source mesh builder
	IDirect3DRMMeshBuilder* pSMB = pShape->GetMeshBuilder();
	ASSERT(pSMB);
	
	// Get the vertex list size info
	ULONG nVert = 0;
	ULONG nNorm = 0;
	DWORD dwFaceSize = 0;
	m_hr = pSMB->GetVertices(&nVert,
							 NULL,
							 &nNorm,
							 NULL,
							 &dwFaceSize,
							 NULL);
	ASSERT(SUCCEEDED(m_hr));
	
	// Allocate the buffers
	D3DVECTOR* prlvVert = new D3DVECTOR [nVert];;
	D3DVECTOR* prlvNorm = new D3DVECTOR [nNorm];
	ULONG* pFD = new ULONG [dwFaceSize];

	// Get the data
	m_hr = pSMB->GetVertices(&nVert,
							 prlvVert,
							 &nNorm,
							 prlvNorm,
							 &dwFaceSize,
							 pFD);
	ASSERT(SUCCEEDED(m_hr));
	
	// copy the vertices and faces
	ASSERT(m_pIMeshBld);
	IDirect3DRMFaceArray* piDFA = NULL;
	m_hr = m_pIMeshBld->AddFaces(nVert,
								prlvVert,
								nNorm,
								prlvNorm,
								pFD,
								&piDFA);
	ASSERT(SUCCEEDED(m_hr));

	// Delete the buffers
	delete [] prlvVert;	 
	delete [] prlvNorm;
	delete [] pFD;

	// get the face list for the source
	ASSERT(pShape);
	IDirect3DRMFaceArray* piSFA = NULL;
	m_hr = pShape->GetMeshBuilder()->GetFaces(&piSFA);
	ASSERT(SUCCEEDED(m_hr));

	// copy the texture data for each face
	ULONG nFaces = pSMB->GetFaceCount();
	for (ULONG i = 0; i < nFaces; i++) {
		IDirect3DRMFace* pSF = NULL;
		IDirect3DRMFace* pDF = NULL;
		m_hr = piSFA->GetElement(i, &pSF);
		ASSERT(SUCCEEDED(m_hr));
		m_hr = piDFA->GetElement(i, &pDF);
		ASSERT(SUCCEEDED(m_hr));

		IDirect3DRMTexture* pITex = NULL;
		m_hr = pSF->GetTexture(&pITex);
		ASSERT(SUCCEEDED(m_hr));
		m_hr = pDF->SetTexture(pITex);
		ASSERT(SUCCEEDED(m_hr));

		BOOL bu, bv;
		m_hr = pSF->GetTextureTopology(&bu, &bv);
		ASSERT(SUCCEEDED(m_hr));
		m_hr = pDF->SetTextureTopology(bu, bv);
		ASSERT(SUCCEEDED(m_hr));

	}
	piSFA->Release();
	piDFA->Release();
}
#endif

// Test if a screen pixel lies over a face
BOOL C3dShape::HitTest(CPoint pt,
					   C3dViewport* pViewport,
					   int* piFace/* = NULL*/,
					   D3DVECTOR* pvHit/* = NULL*/)
{
	ASSERT(pViewport);

	// See if we have a hit on this shape at all
	if (pViewport->HitTest(pt) != this) {
		return FALSE;
	}

	// See if we are returning a face or a vector
	if ((piFace == NULL) && (pvHit == NULL)) {
		return TRUE;
	}

	// Find which face was hit
	int nFaces = GetFaceCount();
	if (nFaces < 1) return -1;

	int iHitFace = -1;
	double dHitZ = 0;

	// Get the viewport interface
	IDirect3DRMViewport* pIViewport = pViewport->GetInterface();
	ASSERT(pIViewport);

    // Get the face list
	ASSERT(m_pIMeshBld);
	IDirect3DRMFaceArray* pIFaceList = NULL;
	m_hr = m_pIMeshBld->GetFaces(&pIFaceList);
	ASSERT(SUCCEEDED(m_hr));

	// walk the face list
	D3DVECTOR lv;
	D3DVECTOR wv;
	D3DRMVECTOR4D sv;
	for (int i = 0; i < nFaces; i++) {

		// Get the face
		IDirect3DRMFace* pIFace = NULL;
		m_hr = pIFaceList->GetElement(i, &pIFace);
		ASSERT(SUCCEEDED(m_hr));

		// Get the vertex count and allocate the screen 
		// coordinate data array
		int nVert = pIFace->GetVertexCount();
		ASSERT(nVert > 2);
		POINT* pScrnVert = new POINT [nVert];

		// Convert each vertex to screen coordinates
		double dZ = 0;
		for (int v = 0; v < nVert; v++) {

			// Get the vector in local (frame) coordinates
			m_hr = pIFace->GetVertex(v, &lv, NULL);
			ASSERT(SUCCEEDED(m_hr));

			// Convert it to world coordinates
			m_hr = m_pIFrame->Transform(&wv, &lv);
			ASSERT(SUCCEEDED(m_hr));

			// Convert world coordinates to screen coordinates
			m_hr = pIViewport->Transform(&sv,
										 &wv);
			ASSERT(SUCCEEDED(m_hr));

			// Convert the homogeneous values to absolute pixel values
			double w = sv.w;
			if (w != 0) {
				pScrnVert[v].x = (int) sv.x / w;
				pScrnVert[v].y = (int) sv.y / w;
				dZ += sv.z / w;
			} else {
				pScrnVert[v].x = 0;
				pScrnVert[v].y = 0;
			}
		}
		dZ /= nVert;

		// see if the hit point lies inside the screen polygon
		if (::_3dPointInPolygon(pScrnVert, nVert, pt)) {
			if (iHitFace < 0) {
				iHitFace = i;
				dHitZ = dZ;
			} else {
				if (dZ < dHitZ) {
					iHitFace = i;
					dHitZ = dZ;
				}
			}
		}

		// Done with this face
		delete [] pScrnVert;
		pIFace->Release();
	}

	// see if we're returning the hit vector
	if (pvHit != NULL) {

		// Compute the point on the face where the hit occured
		// Set up a vector to describe the screen point
		// we'll use the average Z value
		sv.x = pt.x;
		sv.y = pt.y;
		sv.z = dHitZ;
		sv.w = 1.0;

		// convert to world coordinates
		m_hr = pIViewport->InverseTransform(&wv, &sv);
		ASSERT(SUCCEEDED(m_hr));

		// convert to local coordinates in the frame
		m_hr = m_pIFrame->InverseTransform(&lv, &wv);
		ASSERT(SUCCEEDED(m_hr));

		// return the result
		*pvHit = lv;
	}

	// Done with the face list
	pIFaceList->Release();

	// Set the return value
	*piFace = iHitFace;


	return TRUE;
}

BOOL C3dShape::CreateShadow(C3dLight* pLight,
						    D3DVECTOR& vPt,
						    D3DVECTOR& vN)
{

	IDirect3DRMVisual* pIVisual = NULL;
	m_hr = the3dEngine.GetInterface()->CreateShadow(GetVisual(),
												    pLight->GetLight(),
												    vPt.x, vPt.y, vPt.z,
												    vN.x, vN.y, vN.z,
												    &pIVisual);
	ASSERT(SUCCEEDED(m_hr));

	// Attach the shadow to the frame
	ASSERT(m_pIFrame);
	m_hr = m_pIFrame->AddVisual(pIVisual);
	
	return SUCCEEDED(m_hr);
}

BOOL C3dShape::Save(const char* pszFilename)
{
	CString strFile;

	if (pszFilename == NULL) {
		// Show a dialog to get the name
		CFileDialog dlg   (FALSE,   // Save
						   "D3D",    // default extension
						   NULL,    // No initial file name
						   OFN_OVERWRITEPROMPT
							 | OFN_HIDEREADONLY,
						   _3DOBJ_SAVEFILTER);
		if (dlg.DoModal() != IDOK) {
			return FALSE;
		}

		strFile = dlg.GetPathName();
	} else {
		strFile = pszFilename;
	}

	ASSERT(m_pIMeshBld);
	// BUGBUG: As of build 069 only D3DRMXOF_TEXT was supported
	m_hr = m_pIMeshBld->Save(strFile,
							 D3DRMXOF_TEXT,
							 D3DRMXOFSAVE_ALL);
	return SUCCEEDED(m_hr);
}


////////////////////////////////////////////////////////////////////////////
// C3dShapeList

C3dShapeList::C3dShapeList()
{
}

C3dShapeList::~C3dShapeList()
{
	DeleteAll();
}

void C3dShapeList::Remove(C3dShape* pShape)
{
	if (!pShape) return;
	POSITION pos = CObList::Find(pShape);
	if (pos) {
		RemoveAt(pos);
	}
}

void C3dShapeList::Delete(C3dShape* pShape)
{
	if (!pShape) return;
	Remove(pShape);
	delete pShape;
}

void C3dShapeList::DeleteAll()
{
	while (!IsEmpty()) {
		C3dShape* pShape = (C3dShape*) RemoveHead();
		delete pShape;
	}
}

#if 0 // not used
// See if the list contains a shape with the same frame and
// visual attachments
C3dShape* C3dShapeList::Find(C3dShape* pShape)
{
	if (!pShape) return NULL;
	IDirect3DRMFrame* pIFrame = pShape->GetInterface();
	IDirect3DRMVisual* pIVisual = pShape->GetVisual();

	C3dShape* pListEntry = NULL;

	// walk the list
	POSITION pos = GetHeadPosition();
	while (pos) {
		pListEntry = GetNext(pos);
		if ((pListEntry->GetInterface() == pIFrame)
		&& (pListEntry->GetVisual() == pIVisual)) {
			return pListEntry;
		}
	}
	return NULL; // end of list
}
#endif

