// 3dStage.cpp
//
// Copyright (c) Nigel Thompson 1996
//
// Implementation for:
// C3dDevice, C3dViewport, C3dCamera, C3dStage
//

#include "stdafx.h"
#include "3dplus.h"

//////////////////////////////////////////////////////////////////
// C3dDevice

IMPLEMENT_DYNAMIC(C3dDevice, C3dObject)

C3dDevice::C3dDevice()
{
    m_pIDevice = NULL;
}

C3dDevice::~C3dDevice()
{
	if (m_pIDevice) {
		m_pIDevice->Release();
	}
}

BOOL C3dDevice::Create(CDirect3D* pD3D)
{
	if (m_pIDevice) {
		m_pIDevice->Release();
		m_pIDevice = NULL;
	}
	m_hr = the3dEngine.GetInterface()->CreateDeviceFromD3D(
				pD3D->GetD3DEngine(),
				pD3D->GetD3DDevice(),
				&m_pIDevice);
	if (FAILED(m_hr)) {
		return FALSE;
	}
	ASSERT(m_pIDevice);

	return TRUE;
}

int C3dDevice::GetWidth()
{
	ASSERT(m_pIDevice);
	return m_pIDevice->GetWidth();
}

int C3dDevice::GetHeight()
{
	ASSERT(m_pIDevice);
	return m_pIDevice->GetHeight();
}

void C3dDevice::Update()
{
    if (!m_pIDevice) return;

    m_hr = m_pIDevice->Update();
    ASSERT(SUCCEEDED(m_hr));
}

void C3dDevice::SetQuality(D3DRMRENDERQUALITY quality)
{
    if (!m_pIDevice) return;

    m_hr = m_pIDevice->SetQuality(quality);
    ASSERT(SUCCEEDED(m_hr));
}

//////////////////////////////////////////////////////////////////
// C3dViewport

IMPLEMENT_DYNAMIC(C3dViewport, C3dObject)

C3dViewport::C3dViewport()
{
    m_pIViewport = NULL;
}

C3dViewport::~C3dViewport()
{
	if (m_pIViewport) {
		m_pIViewport->Release();
	}
}

void C3dViewport::Clear()
{
    if (!m_pIViewport) return;

    m_hr = m_pIViewport->Clear();
    ASSERT(SUCCEEDED(m_hr));
}

BOOL C3dViewport::Create(C3dDevice* pDevice,
                         C3dCamera* pCamera,
                         int x, int y,
                         int cx, int cy)
{
    ASSERT(pDevice);
    ASSERT(pCamera);

    // See if we already have a viewport
    if (m_pIViewport == NULL) {

		// Create the viewport to fill the device area
		if (!the3dEngine.CreateViewport(pDevice,
								        pCamera,
								        0, 0, // origin
								        cx, cy, // size
								        &m_pIViewport)) {
		    return FALSE;
        }

		ASSERT(m_pIViewport);

        // set initial attributes
		m_hr = m_pIViewport->SetBack(D3DVAL(5000.0));
		ASSERT(SUCCEEDED(m_hr));

    } else {

        // save the attributes of the current viewport
        D3DVALUE rvFront = m_pIViewport->GetFront();
        D3DVALUE rvBack = m_pIViewport->GetBack();
        D3DVALUE rvField = m_pIViewport->GetField();

        // we can free the old viewport now
        m_pIViewport->Release();
        m_pIViewport = NULL;

        // Create a new viewport 
		if (!the3dEngine.CreateViewport(pDevice,
								        pCamera,
								        0, 0, // origin
								        cx, cy, // size
								        &m_pIViewport)) {
		    return FALSE;
        }

		ASSERT(m_pIViewport);

        // restore the viewport attributes
        m_pIViewport->SetBack(rvBack);
        m_pIViewport->SetFront(rvFront);
        m_pIViewport->SetField(rvField);
    }

    return TRUE;
}

BOOL C3dViewport::Render(C3dFrame* pFrame)
{
    if (!m_pIViewport) return FALSE;

    ASSERT(pFrame);
    m_hr = m_pIViewport->Render(pFrame->GetInterface());
    if (FAILED(m_hr)) return FALSE;

    return TRUE;
}

void C3dViewport::ForceUpdate(RECT* pRect)
{
	ASSERT(m_pIViewport);
	m_hr = m_pIViewport->ForceUpdate(pRect->left,
								    pRect->top,
									pRect->right = pRect->left,
									pRect->bottom - pRect->top);
}

// test for a hit on a shape
// Note: This returns NULL if there is no C++ container for the
// visible shape
C3dShape* C3dViewport::HitTest(CPoint pt)
{
	IDirect3DRMPickedArray* pIPickArray = NULL;
	ASSERT(m_pIViewport);
	m_hr = m_pIViewport->Pick(pt.x, pt.y, &pIPickArray);	
	if (FAILED(m_hr)) return NULL;

	// see if there are any items in the array
	if (pIPickArray->GetSize() == 0) {
		pIPickArray->Release();
		return NULL;
	}

	// get the first (topmost) element
	IDirect3DRMVisual* pIVisual = NULL;
	IDirect3DRMFrameArray* pIFrameList = NULL;
	m_hr = pIPickArray->GetPick(0, &pIVisual, &pIFrameList, NULL);
	ASSERT(SUCCEEDED(m_hr));
	ASSERT(pIVisual);
	ASSERT(pIFrameList);

	// get the last frame in the list
	IDirect3DRMFrame* pIFrame = NULL;
	pIFrameList->GetElement(pIFrameList->GetSize() - 1, &pIFrame);
	ASSERT(pIFrame);

	// get the 'AppData' value which should be a C++ class
	// pointer.
	C3dShape* pShape = (C3dShape*) pIFrame->GetAppData();

	if (pShape) {
		if(!pShape->IsKindOf(RUNTIME_CLASS(C3dShape))) {
			pShape = NULL;
		}
	}

	pIFrame->Release();
	pIFrameList->Release();
	pIVisual->Release();
	pIPickArray->Release();

	return pShape;
}

BOOL C3dViewport::SetFieldOfView(double f)
{
	ASSERT(m_pIViewport);
	ASSERT(f > 0);
	m_hr = m_pIViewport->SetField(f);
	return SUCCEEDED(m_hr);
}

//////////////////////////////////////////////////////////////////
// C3dCamera

IMPLEMENT_DYNAMIC(C3dCamera, C3dFrame)

//////////////////////////////////////////////////////////////////
// C3dStage

IMPLEMENT_DYNAMIC(C3dStage, C3dFrame)

C3dStage::C3dStage()
{
    m_pScene = NULL;
	m_Quality = D3DRMRENDER_GOURAUD;

	// Create frame with no parent
    C3dFrame::Create(NULL);
	ASSERT(m_pIFrame);

	// Create the camera frame as a child of the stage frame
	m_Camera.Create(this);


    // Set the initial camera position and direction
    // it's pointing in. We'll start with it looking in at the x=0, y=0 point
    m_Camera.SetPosition(0, 0, -10);
    m_Camera.SetDirection(0, 0, 1, 0, 1, 0);
}

C3dStage::~C3dStage()
{
}


// Create the stage from a Direct3D object
BOOL C3dStage::Create(CDirect3D* pD3D)
{
	// Remove any existing scene
	SetScene(NULL);
	
	// Create a new device from the Direct3D surfaces
	if (!m_Device.Create(pD3D)) return FALSE;

	// Set the current quality
	m_Device.SetQuality(m_Quality);


	// Create the viewport
    if (!m_Viewport.Create(&m_Device,
						   &m_Camera,
						   0, 0,
						   m_Device.GetWidth(), 
						   m_Device.GetHeight())) {
        return FALSE;
    }

	return TRUE;
}

void C3dStage::Render()
{
    ASSERT(m_pIFrame);

    // Clear the viewport
    m_Viewport.Clear();

    if (m_pScene) { 

        // Render the scene
        m_Viewport.Render(m_pScene);
    }

    // Update the display
    m_Device.Update();
}

void C3dStage::SetQuality(D3DRMRENDERQUALITY quality)
{
	m_Quality = quality;
	m_Device.SetQuality(m_Quality);
}

void C3dStage::SetScene(C3dScene* pScene)
{
    // remove any existing scene and background
	SetBackground(NULL);
    if (m_pScene) {
        RemoveChild(m_pScene);
		m_pScene->m_pStage = NULL;
    }
    m_pScene = pScene;
    if (m_pScene) {
        AddChild(m_pScene);
		m_pScene->m_pStage = this;

		// Apply any background from the scene
		SetBackground(m_pScene);

		// Set up the camera position etc.
		m_Camera.SetPosition(m_pScene->m_vCamPos);
		m_Camera.SetDirection(m_pScene->m_vCamDir, m_pScene->m_vCamUp);
		m_Viewport.SetFieldOfView(m_pScene->m_dCamField);
    }
}

void C3dStage::SetBackground(C3dScene* pScene)
{
	if (!m_pIFrame) return;

	if (!pScene) {

		// remove any existing background
		m_hr = m_pIFrame->SetSceneBackgroundImage(NULL);
		ASSERT(SUCCEEDED(m_hr));
		m_hr = m_pIFrame->SetSceneBackground(D3DRMCreateColorRGB(0,0,0));
		ASSERT(SUCCEEDED(m_hr));
		return;
	}

	if (pScene->m_pBkgndImg) {

		// set the background image
		IDirect3DRMTexture* pITex = NULL;

		// Create a texture from the image
		if (!the3dEngine.CreateTexture(pScene->m_pBkgndImg->GetObject(), &pITex)) {
			TRACE("Texture create failed\n");
			return;
		}
		ASSERT(pITex);
		m_hr = m_pIFrame->SetSceneBackgroundImage(pITex);
		pITex->Release();
		ASSERT(SUCCEEDED(m_hr));

	} else {

		// set the background color
		m_hr = m_pIFrame->SetSceneBackground(pScene->m_clrBkgnd);
		ASSERT(SUCCEEDED(m_hr));
	}
}