// 3dVector.cpp
//
// Copyright (c) Nigel Thompson 1996
//
// Implementation for:
// C3dVector
//

#include "stdafx.h"
#include "3dplus.h"

#define D2R  0.01745329251994

//////////////////////////////////////////////////////////////////
// C3dVector

C3dVector::C3dVector()
{
	x = 1;
	y = 1;
	z = 1;
}

C3dVector::~C3dVector()
{
}

C3dVector::C3dVector(const D3DVECTOR& r)
{
	x = r.x;
	y = r.y;
	z = r.z;
}

C3dVector::C3dVector(double _x, double _y, double _z)
{
	x = _x;
	y = _y;
	z = _z;
}

C3dVector& C3dVector::operator = (const D3DVECTOR& r)
{
	x = r.x;
	y = r.y;
	z = r.z;
	return *this;
}

C3dVector operator + (const D3DVECTOR& a, const D3DVECTOR& b)
{
    return C3dVector(a.x + b.x,
                     a.y + b.y,
                     a.z + b.z);
}

C3dVector& C3dVector::operator += (const D3DVECTOR& r)
{
    x += r.x;
    y += r.y;
    z += r.z;
    return *this;
}

C3dVector operator - (const D3DVECTOR& a, const D3DVECTOR& b)
{
    return C3dVector(a.x - b.x,
                     a.y - b.y,
                     a.z - b.z);
}

C3dVector& C3dVector::operator -= (const D3DVECTOR& r)
{
    x -= r.x;
    y -= r.y;
    z -= r.z;
    return *this;
}

C3dVector C3dVector::operator - ()
{
	return C3dVector(-x, -y, -z);
}

// vector (cross) product
C3dVector operator * (const D3DVECTOR& a, const D3DVECTOR& b)
{
    return C3dVector(a.y * b.z - a.z * b.y,
					 a.z * b.x - a.x * b.z,
					 a.x * b.y - a.y * b.x);
}

// vector (cross) product
C3dVector& C3dVector::operator *= (const D3DVECTOR& r)
{
    // deliberately create a temp to hold result
    *this = *this * r;
    return *this;
}
                      
// magnitude change
C3dVector operator * (const D3DVECTOR& a, const double s)
{
    return C3dVector(a.x * s,
					 a.y * s,
					 a.z * s);
}

// magnitude change
C3dVector& C3dVector::operator *= (const double s)
{
	x *= s;
	y *= s;
	z *= s;
    return *this;
}



// dot (scalar) product
double C3dVector::Dot(const D3DVECTOR&r)
{
	return x * r.x + y * r.y + z * r.z;
}

// magnitude (length)
double C3dVector::Mag()
{
	return sqrt(x * x + y * y + z * z);
}

// Normalize a vector (make it unit lebgth)
BOOL C3dVector::Normalize()
{
	double l = Mag();
	if (l == 0) return FALSE;
	x /= l;
	y /= l;
	z /= l;
	return TRUE;
}

// test for coincidence
BOOL C3dVector::Coincident(const D3DVECTOR& r)
{
	C3dVector a(*this);
	a.Normalize();
	C3dVector b(r);
	b.Normalize();
	if ((a.x == b.x) 
	&& (a.y == b.y)
	&& (a.z == b.z)) {
		return TRUE;
	}
	return FALSE;
}

// Compute an arbitrary up vector
C3dVector C3dVector::GenerateUp()
{
	Normalize();

	// Create an initial up vector
	C3dVector vu(0, 1, 0);

	// make sure the up and dir vectors are not the same
	if (Coincident(vu) || Coincident(-vu)) {

		// d is 'up' or 'down' so just set up to be x
		return C3dVector(1, 0, 0);

	} 

	// compute the 'right' vector from the requested up
	// vector and the direction vector
	C3dVector vr = *this * vu;

	// compute the actual up vector as from the dir and right vectors
	return vr * *this;
}

BOOL C3dVector::IsNull()
{
	if ((x == 0.0) && (y == 0) && (z ==  0)) {
		return TRUE;
	}
	return FALSE;
}
