// 3dwnd.cpp
//
// Copyright (c) Nigel Thompson 1996
//
// Implementation for:
// C3dWnd
//

#include "stdafx.h"
#include "3dplus.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

/////////////////////////////////////////////////////////////////////////////
// C3dWnd

C3dWnd::C3dWnd()
: m_rcClient(0, 0, 320, 240)
{
	m_pDD = NULL;
	m_pD3D = NULL;
	m_bAutoDelete = FALSE;
	m_pStage = NULL;
	m_pScene = NULL;
	m_bEnableUpdates = FALSE;
	m_iWidth = 320;
	m_iHeight = 240;
	m_bRepaintAll = TRUE;
	m_bAppActive = FALSE;
	m_pGrabImg = NULL;
	m_pGrabPal = NULL;
	m_pController = NULL;
	m_bEnableMouseSelection = FALSE;
	m_pSelChangeFn = NULL;
	m_pSelChangeArg = NULL;
	m_ColorModel = D3DCOLOR_MONO;
}

C3dWnd::~C3dWnd()
{
}


BEGIN_MESSAGE_MAP(C3dWnd, CWnd)
	//{{AFX_MSG_MAP(C3dWnd)
	ON_WM_CREATE()
	ON_WM_DESTROY()
	ON_WM_MOVE()
	ON_WM_SIZE()
	ON_WM_PAINT()
	ON_WM_ACTIVATEAPP()
	//ON_WM_ACTIVATE()
	ON_WM_PALETTECHANGED()
	ON_WM_ERASEBKGND()
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()

BOOL C3dWnd::Create(const char* pszCaption,
					DWORD dwStyle,
					int x, int y,
					int cx, int cy,
					CWnd* pParent/*= NULL*/)
{
	const char* pszClass = AfxRegisterWndClass(CS_VREDRAW | CS_HREDRAW,
				::LoadCursor(NULL, IDC_ARROW),
				(HBRUSH)::GetStockObject(GRAY_BRUSH));
	
	if (!pParent) {
		m_bAutoDelete = TRUE;
	}
	return CWnd::CreateEx(0,
						  pszClass,
						  pszCaption,
						  dwStyle,
						  x, y, 
						  cx, cy,
						  pParent ? pParent->GetSafeHwnd() : NULL,
						  NULL);
}

// Simpler construction when used as a child window
BOOL C3dWnd::Create(CWnd* pParent, UINT uiID, D3DCOLORMODEL cm)
{
	if (!pParent) return FALSE;

	// Save the color model for later
	m_ColorModel = cm;

	return CWnd::Create(NULL,
					    "",
						WS_CHILD | WS_VISIBLE,
						CRect(0, 0, 1, 1),
						pParent,
						uiID);
}

// Create the stage
BOOL C3dWnd::CreateStage()
{

	// Initialise the direct draw objects
	if (!m_pDD) {
		m_pDD = new CDirectDraw;
	}
	if (!m_pDD->Create()) return FALSE;

	// Set the mode for the window
	if (!m_pDD->SetWindowedMode(GetSafeHwnd(),
							   m_iWidth,
							   m_iHeight)) {
		return FALSE;
	}

	// Create the Direct3D object
	if (!m_pD3D) {
		m_pD3D = new CDirect3D;
	}
	if (!m_pD3D->Create(m_pDD)) return FALSE;

	// Set the color model we want
	if (! m_pD3D->SetMode(m_ColorModel)) return FALSE;
	
	// Create a stage
	if (!m_pStage) {
		m_pStage = new C3dStage;
	}
	if (!m_pStage->Create(m_pD3D)) return FALSE;

	// attach any current scene
	m_pStage->SetScene(m_pScene);

	return TRUE;
}

// release the DirectDraw components
void C3dWnd::ReleaseDD()
{
	if (m_pGrabImg) {
		delete m_pGrabImg;
		m_pGrabImg = NULL;
	}
	if (m_pGrabPal) {
		delete m_pGrabPal;
		m_pGrabPal = NULL;
	}
	if (m_pD3D) {
		delete m_pD3D;
		m_pD3D = NULL;
	}
	if (m_pDD) {
		delete m_pDD;
		m_pDD = NULL;
	}
}

// Release the stage and its related components
void C3dWnd::ReleaseStage()
{
	if (m_pStage) {
		m_pStage->SetScene(NULL);
		delete m_pStage;
		m_pStage = NULL;
	}
	ReleaseDD();
}

int C3dWnd::OnCreate(LPCREATESTRUCT lpCreateStruct) 
{
	if (CWnd::OnCreate(lpCreateStruct) == -1)
		return -1;
	
	// save the initial size and position
	m_iWidth = lpCreateStruct->cx;
	m_iHeight = lpCreateStruct->cy;
	m_rcClient = CRect(lpCreateStruct->x,
				    lpCreateStruct->y,
					lpCreateStruct->x + lpCreateStruct->cx,
					lpCreateStruct->y + lpCreateStruct->cy);

	// Create the stage
	if (!CreateStage()) return -1;
	
	return 0;
}

void C3dWnd::OnDestroy() 
{
	CWnd::OnDestroy();
	
	// Clean up
	ReleaseStage();
}

void C3dWnd::PostNcDestroy()
{
	if (m_bAutoDelete) {
		delete this;
	}
}

// Update the current scene, render it and draw the changes to the screen
// returns TRUE if it has something to do, FALSE if idle
BOOL C3dWnd::Update(double delta)
{
	if (!m_bAppActive || !m_bEnableUpdates) return FALSE;

	if (!m_pScene) {
		TRACE("Updates disabled\n");
		m_bEnableUpdates = FALSE;
		return FALSE;
	}

	RECT rcFrom;
	rcFrom.left = 0;
	rcFrom.top = 0;
	rcFrom.right = m_iWidth;
	rcFrom.bottom = m_iHeight;

	// if the window has been resized - force an update
	// of the entire area
	if (m_bRepaintAll) {
		m_pStage->ForceUpdate(&rcFrom);
	}

	// update the scene
	//m_pStage->Clear();
    m_pScene->Move(delta);
    m_pStage->Render();
	BOOL b;

	// Blt the back buffer to the front buffer so we
	// can see what we rendered
	b = m_pDD->GetFrontBuffer()->Blt(&m_rcClient,
								 m_pDD->GetBackBuffer(),
								 &rcFrom,
								 DDBLT_WAIT,
								 NULL);

	m_bRepaintAll = FALSE;
	return TRUE;
}


void C3dWnd::OnMove(int x, int y) 
{
	// Compute the screen coordinates of the client rectangle
	m_rcClient = CRect(x, y, x+m_iWidth, y+m_iHeight);
	if (GetParent()) {
		ClientToScreen(&m_rcClient);
	}
}

void C3dWnd::OnSize(UINT nType, int cx, int cy) 
{
	if ((cx <= 0) || (cy <= 0)) return;

	if ((cx == m_iWidth) && (cy == m_iHeight)) return;
	
	// Shut down and restart
	ReleaseStage();

	// Compute the new screen coordinates
	m_iWidth = cx;
	m_iHeight = cy;
	m_rcClient = CRect(0, 0, m_iWidth, m_iHeight);
	ClientToScreen(&m_rcClient);

	CreateStage();
	m_bRepaintAll = TRUE;
}

void C3dWnd::OnPaint() 
{
	CPaintDC dc(this); // device context for painting
	
	// Behave as though the window was resized
	m_bRepaintAll = TRUE;

	// if we are not active draw the image we grabbed
	// earlier if we have one

	if (!m_bAppActive && m_pGrabImg) {
		CPalette* pOldPal = NULL;
		if (m_pGrabPal) {
			pOldPal = dc.SelectPalette(m_pGrabPal, FALSE);
			dc.RealizePalette();
		}
		CDC dcMem;
		dcMem.CreateCompatibleDC(&dc);
		CPalette* pOldMemPal = NULL;
		if (m_pGrabPal) {
			pOldMemPal = dcMem.SelectPalette(m_pGrabPal, FALSE);
			dcMem.RealizePalette();
		}
		CBitmap* pOldBmp = dcMem.SelectObject(m_pGrabImg);
		dc.BitBlt(0, 0,
				  m_iWidth, m_iHeight,
				  &dcMem,
				  0, 0,
				  SRCCOPY);
		if (pOldMemPal) {
			dcMem.SelectPalette(pOldMemPal, FALSE);
		}
		dcMem.SelectObject(pOldBmp);
		if (pOldPal) {
			dc.SelectPalette(pOldPal, FALSE);
		}
	}
}


BOOL C3dWnd::SetScene(C3dScene* pScene)
{
	if (!m_pStage) return FALSE;
	m_pScene = pScene;
	m_pStage->SetScene(m_pScene);
	if (m_pScene) {
		// Enable the idle-time rendering
		m_bEnableUpdates = TRUE;
	} else {
		m_bEnableUpdates = FALSE;
	}
	

	return TRUE;
}

// Note: this function will only be called if the C3dWnd
// is the app's main window.  In any other case, the
// app must handle the WM_ACTIVATE message itself
void C3dWnd::OnActivateApp(BOOL bActive, HTASK hTask) 
{
	m_bAppActive = bActive;
	// Let the DirectDraw object know
	if (m_pDD) {
		m_pDD->OnActivate(bActive);
	}

	// As a refinement here when going inactive, we
	// grab the current image from the back buffer
	// and build a bitmap and palette. Then, in response to a
	// WM_PAINT message we use our palette and bitmap
	// to draw the last state of the rendered image.
	if (!bActive) {
		if (m_pGrabPal) delete m_pGrabPal;
		if (m_pGrabImg) delete m_pGrabImg;
		m_pGrabPal = m_pDD->GrabPalette();
		m_pGrabImg = m_pDD->GrabImage();
		if (m_pGrabImg) {
			InvalidateRect(NULL);
		}
	}
}

void C3dWnd::OnPaletteChanged(CWnd* pFocusWnd) 
{
	// If we are inactive, repaint using the new palette
	if (!m_bAppActive) {
		InvalidateRect(NULL, FALSE);
	}
}

BOOL C3dWnd::OnEraseBkgnd( CDC* pDC )
{
	CRect rc;
	GetClientRect(&rc);
	CBrush br(RGB(0, 0, 0));
	pDC->FillRect(&rc, &br);

	return TRUE; // say we handled it
}

BOOL C3dWnd::AttachController(C3dController* pController)
{
	m_pController = pController;
	return TRUE;
}

BOOL C3dWnd::OnWndMsg( UINT message, WPARAM wParam, LPARAM lParam, LRESULT* pResult )
{
	LPARAM lp;

	if (m_pController) {
		switch (message) {
		case WM_MOUSEMOVE:
		case WM_LBUTTONDOWN:
		case WM_LBUTTONUP:
		case WM_LBUTTONDBLCLK:
		case WM_RBUTTONDOWN:
		case WM_RBUTTONUP:
		case WM_RBUTTONDBLCLK: {

			// Convert the coordinates to screen coordinates
			CPoint pt(LOWORD(lParam), HIWORD(lParam));
			ClientToScreen(&pt);
			lp = MAKELPARAM(pt.x, pt.y);
		   }
			// fall through
		case WM_KEYDOWN:
		case WM_KEYUP:

			// send the message to the controller
			m_pController->OnUserEvent(GetSafeHwnd(), message, wParam, lp);
			break;

		default:
			break;
		}
	}

	// see if we are looking for mouse selections
	if (m_bEnableMouseSelection
	&& (message == WM_LBUTTONDOWN)) {
		CPoint pt(LOWORD(lParam), HIWORD(lParam));
		C3dShape* pShape = HitTest(pt);
		if (m_pSelChangeFn) {
			// call the notification function
			m_pSelChangeFn(pShape, pt, m_pSelChangeArg);
		}
	}

	return CWnd::OnWndMsg(message, wParam, lParam, pResult);
}

// Test for a hit on a visible object
C3dShape* C3dWnd::HitTest(CPoint pt)
{
	ASSERT(m_pStage);
	return m_pStage->GetViewport()->HitTest(pt);
}

// enable mouse hits to select an object
void C3dWnd::EnableMouseSelection(SELCHANGEFN pNotifyFn,
							      void* pArg,
							      BOOL bEnable)
{
	if (bEnable) {
		ASSERT(pNotifyFn);
		m_pSelChangeFn = pNotifyFn;
		m_pSelChangeArg = pArg;
	} else {
		m_pSelChangeFn = NULL;
		m_pSelChangeArg = NULL;
	}
	m_bEnableMouseSelection = bEnable;
}