//////////////////////////////////////////////////////////////////////////////////
// Project Name:   [ CDX Class Library - CDX.lib ]
// Author:         [ Dan Farley - 97308096@brookes.ac.uk ]
// Source File:    [ CDXSurface Implementation ]
// Revision:       [ 1.6 ]
//////////////////////////////////////////////////////////////////////////////////
#include "CDX.h"

//////////////////////////////////////////////////////////////////////////////////
// CDXSurface Constructor
//////////////////////////////////////////////////////////////////////////////////
LPDIRECTDRAWSURFACE4 CDXSurface::m_SurBack = NULL;
CDXSurface::CDXSurface(void)
{
	Initialise();
}

//////////////////////////////////////////////////////////////////////////////////
// CDXSurface Destructor
//////////////////////////////////////////////////////////////////////////////////
CDXSurface::~CDXSurface()
{
	Finalise();
}

//////////////////////////////////////////////////////////////////////////////////
// CDXSurface Initialise
//////////////////////////////////////////////////////////////////////////////////
void CDXSurface::Initialise(void)
{
	m_Surface = NULL;
	m_DC = NULL;
	m_Font = NULL;
	m_Filename = NULL;

	ZeroMemory(&m_Desc, sizeof(DDSURFACEDESC2));
	m_Desc.dwSize = sizeof(DDSURFACEDESC2);
	m_Desc.ddpfPixelFormat.dwSize = sizeof(DDPIXELFORMAT);

	m_Width = 0;
	m_Height = 0;
	SetRect(&m_SrcRect, 0, 0, 0, 0);
	SetRect(&m_DestRect, 0, 0, 0, 0);
}

//////////////////////////////////////////////////////////////////////////////////
// CDXSurface Finalise
//////////////////////////////////////////////////////////////////////////////////
void CDXSurface::Finalise(void)
{
//	DeleteDC(m_DC);
//	DeleteObject(m_Font);
	SAFE_RELEASE(m_Surface);
}

//////////////////////////////////////////////////////////////////////////////////
// CDXSurface Create
//////////////////////////////////////////////////////////////////////////////////
HRESULT CDXSurface::Create(CDXScreen* engine)
{
	HRESULT rval;

	Finalise();

	// Create the surface
	rval = engine->GetDD()->CreateSurface(&m_Desc, &m_Surface, NULL);
	if(rval != DD_OK) return rval;

	rval = m_Surface->GetSurfaceDesc(&m_Desc);
	if(rval != DD_OK) return rval;

	// Initialise some variables
	m_Width = m_Desc.dwWidth;
	m_Height = m_Desc.dwHeight;
	SetRect(&m_SrcRect, 0, 0, m_Width, m_Height);
	SetRect(&m_DestRect, 0, 0, m_Width, m_Height);

	return rval;
}

//////////////////////////////////////////////////////////////////////////////////
// CDXSurface Create
//////////////////////////////////////////////////////////////////////////////////
HRESULT CDXSurface::Create(CDXScreen *engine, int width, int height)
{
	HRESULT rval;

	Finalise();
	Initialise();

	// Initialise some variables
	m_Width = width;
	m_Height = height;
	SetRect(&m_SrcRect, 0, 0, m_Width, m_Height);
	SetRect(&m_DestRect, 0, 0, m_Width, m_Height);

	// Set the surface description
	m_Desc.dwFlags = DDSD_CAPS | DDSD_HEIGHT | DDSD_WIDTH;
	m_Desc.ddsCaps.dwCaps = DDSCAPS_OFFSCREENPLAIN;
	m_Desc.dwWidth = m_Width;
	m_Desc.dwHeight = m_Height;

	// Create the surface
	rval = engine->GetDD()->CreateSurface(&m_Desc, &m_Surface, NULL);
	if(rval != DD_OK) return rval;

	return rval;
}

//////////////////////////////////////////////////////////////////////////////////
// CDXSurface LoadBitmap
//////////////////////////////////////////////////////////////////////////////////
HRESULT CDXSurface::LoadBitmap(CDXScreen* engine, const char* filename)
{
	HRESULT rval;
	HBITMAP hbm;
	BITMAP  bm;

	Finalise();
	Initialise();

	// Try to load the bitmap from a resource
	hbm = (HBITMAP)LoadImage(GetModuleHandle(NULL), filename,
		                       IMAGE_BITMAP, 0, 0, LR_CREATEDIBSECTION);

	// Try to load the bitmap from a file
	if(hbm == NULL)
	hbm = (HBITMAP)LoadImage(NULL, filename, IMAGE_BITMAP,
	                         0, 0, LR_LOADFROMFILE | LR_CREATEDIBSECTION);

	if(hbm == NULL) return E_FAIL;

	// Get size of the bitmap
	GetObject(hbm, sizeof(bm), &bm);

	// Create a DirectDrawSurface for this bitmap
	m_Desc.dwFlags = DDSD_CAPS | DDSD_HEIGHT | DDSD_WIDTH;
	m_Desc.ddsCaps.dwCaps = DDSCAPS_OFFSCREENPLAIN;
	m_Desc.dwWidth = bm.bmWidth;
	m_Desc.dwHeight = bm.bmHeight;

	rval = engine->GetDD()->CreateSurface(&m_Desc, &m_Surface, NULL);
	if(rval != DD_OK) return rval;

	CopyBitmap(hbm, 0, 0, 0, 0);
	DeleteObject(hbm);

	// Initialise some variables
	m_Filename = filename;
	m_Width = m_Desc.dwWidth;
	m_Height = m_Desc.dwHeight;
	SetRect(&m_SrcRect, 0, 0, m_Width, m_Height);
	SetRect(&m_DestRect, 0, 0, m_Width, m_Height);

	return rval;
}

//////////////////////////////////////////////////////////////////////////////////
// CDXSurface CopyBitmap
//////////////////////////////////////////////////////////////////////////////////
HRESULT CDXSurface::CopyBitmap(HBITMAP hbm, int x, int y, int dx, int dy)
{
	HDC Image;
	BITMAP bm;
	HRESULT rval;

	if(hbm == NULL || m_Surface == NULL) return E_FAIL;

	// Make sure this surface is restored
	m_Surface->Restore();

	// Select bitmap into a MemoryDC so we can use it
	Image = CreateCompatibleDC(NULL);
	if(!Image) return E_FAIL;
	SelectObject(Image, hbm);

	// Get size of the bitmap
	GetObject(hbm, sizeof(bm), &bm);
	dx = dx == 0 ? bm.bmWidth  : dx;
	dy = dy == 0 ? bm.bmHeight : dy;

	// Get size of surface
	m_Desc.dwSize = sizeof(DDSURFACEDESC2);
	m_Desc.dwFlags = DDSD_HEIGHT | DDSD_WIDTH;
	m_Surface->GetSurfaceDesc(&m_Desc);

	rval = GetDC();
	if(rval == DD_OK)
	{
		StretchBlt(m_DC, 0, 0, m_Desc.dwWidth, m_Desc.dwHeight, Image,
		           x, y, dx, dy, SRCCOPY);
		ReleaseDC();
	}

	DeleteDC(Image);

	return rval;
}

//////////////////////////////////////////////////////////////////////////////////
// CDXSurface Draw
//////////////////////////////////////////////////////////////////////////////////
HRESULT CDXSurface::Draw(/*CDXSurface* Dest,*/ DWORD Flags)
{
	HRESULT rval;

	rval = m_SurBack->Blt(&m_DestRect, m_Surface, &m_SrcRect, Flags, NULL);
	if(rval == DDERR_SURFACELOST) Restore();

	return rval;
}

//////////////////////////////////////////////////////////////////////////////////
// CDXSurface DrawFast
//////////////////////////////////////////////////////////////////////////////////
HRESULT CDXSurface::DrawFast(int X, int Y/*, CDXSurface* Dest*/, DWORD Flags)
{
	HRESULT rval;

	rval = m_SurBack->BltFast(X, Y, m_Surface, &m_SrcRect, Flags);
	if(rval == DDERR_SURFACELOST) Restore();

	return rval;
}

//////////////////////////////////////////////////////////////////////////////////
// CDXSurface Fill
//////////////////////////////////////////////////////////////////////////////////
HRESULT CDXSurface::Fill(DWORD color)
{
	DDBLTFX bltfx;

	bltfx.dwSize = sizeof(DDBLTFX);
	bltfx.dwFillColor = color;
	return m_Surface->Blt(NULL, NULL, NULL, DDBLT_WAIT | DDBLT_COLORFILL, &bltfx);
}

/*
//////////////////////////////////////////////////////////////////////////////////
// Clip - clips a destination rectange and modifys X,Y coords appropriatly
//////////////////////////////////////////////////////////////////////////////////
static void Clip(int *DestX, int *DestY, RECT *SrcRect, RECT *DestRect)
{
	// If it's partly off the right side of the screen
	if(*DestX + (SrcRect->right - SrcRect->left) > DestRect->right)
		SrcRect->right -= *DestX + (SrcRect->right-SrcRect->left) - DestRect->right;

	// Partly off the left side of the screen
	if(*DestX < DestRect->left)
	{
		SrcRect->left += DestRect->left - *DestX;
		*DestX = DestRect->left;
	}

	// Partly off the top of the screen
	if(*DestY < DestRect->top)
	{
		SrcRect->top += DestRect->top - *DestY;
		*DestY = DestRect->top;
	}

	// If it's partly off the bottom side of the screen
	if(*DestY + (SrcRect->bottom - SrcRect->top) > DestRect->bottom)
	SrcRect->bottom -= ((SrcRect->bottom-SrcRect->top)+*DestY) - DestRect->bottom;

	return;
}

//////////////////////////////////////////////////////////////////////////////////
// CDXSurface DrawClipped
//////////////////////////////////////////////////////////////////////////////////
HRESULT CDXSurface::DrawClipped(int X, int Y, CDXSurface* Dest, LPRECT ClipRect)
{
	HRESULT rval;
	RECT ModSrc = m_SrcRect;
	Clip(&X, &Y, &ModSrc, ClipRect);

	rval = Dest->m_Surface->BltFast(X, Y, m_Surface, &ModSrc, DDBLTFAST_WAIT | DDBLTFAST_SRCCOLORKEY);
	if(rval == DDERR_SURFACELOST) Restore();

	return rval;
}
*/

//////////////////////////////////////////////////////////////////////////////////
// CDXSurface SetDest
//////////////////////////////////////////////////////////////////////////////////
void CDXSurface::SetDest(int t, int l, int b, int r)
{
	m_DestRect.top = t;
	m_DestRect.left = l;
	m_DestRect.bottom = b;
	m_DestRect.right = r;
}

//////////////////////////////////////////////////////////////////////////////////
// CDXSurface SetSrc
//////////////////////////////////////////////////////////////////////////////////
void CDXSurface::SetSrc(int t, int l, int b, int r)
{
	m_SrcRect.top = t;
	m_SrcRect.left = l;
	m_SrcRect.bottom = b;
	m_SrcRect.right = r;
}

//////////////////////////////////////////////////////////////////////////////////
// CDXSurface SetColorKey
//////////////////////////////////////////////////////////////////////////////////
HRESULT CDXSurface::SetColorKey(COLORREF col)
{
	COLORREF rgb;
	DDCOLORKEY ddck;
	DWORD dw;

	// Use GDI SetPixel to color match for us
	if(GetDC() == DD_OK)
	{
		rgb = ::GetPixel(m_DC, 0, 0);
		::SetPixel(m_DC, 0, 0, col);
		ReleaseDC();
	}

	// Now lock the surface so we can read back the converted color
	if(Lock() == DD_OK)
	{
		dw  = *(DWORD *)m_Desc.lpSurface;
		if(m_Desc.ddpfPixelFormat.dwRGBBitCount < 32)
			dw &= (1 << m_Desc.ddpfPixelFormat.dwRGBBitCount) - 1;
		UnLock();
	}

	// Now put the color that was there back
	if(GetDC() == DD_OK)
	{
		SetPixel(m_DC, 0, 0, rgb);
		ReleaseDC();
	}

	ddck.dwColorSpaceLowValue = dw;
	ddck.dwColorSpaceHighValue = dw;

	return m_Surface->SetColorKey(DDCKEY_SRCBLT, &ddck);
}

//////////////////////////////////////////////////////////////////////////////////
// CDXSurface Restore
//////////////////////////////////////////////////////////////////////////////////
HRESULT CDXSurface::Restore(void)
{
	return m_Surface->Restore();
}

//////////////////////////////////////////////////////////////////////////////////
// CDXSurface Lock
//////////////////////////////////////////////////////////////////////////////////
HRESULT CDXSurface::Lock(void)
{
	ZeroMemory(&m_Desc, sizeof(m_Desc));
	m_Desc.dwSize = sizeof(m_Desc);
	return m_Surface->Lock(NULL, &m_Desc, DDLOCK_WAIT, NULL);
}

//////////////////////////////////////////////////////////////////////////////////
// CDXSurface UnLock
//////////////////////////////////////////////////////////////////////////////////
HRESULT CDXSurface::UnLock(void)
{
	return m_Surface->Unlock(NULL);
}

//////////////////////////////////////////////////////////////////////////////////
// CDXSurface SetFont
//////////////////////////////////////////////////////////////////////////////////
HRESULT CDXSurface::SetFont(const char* FontName, int Width, int Height, int Attributes)
{
	m_Font = CreateFont(Height, Width,
	                    0, 0,
	                    Attributes,
	                    FALSE,
	                    FALSE,
	                    FALSE,
	                    ANSI_CHARSET,
	                    OUT_DEFAULT_PRECIS,
	                    CLIP_DEFAULT_PRECIS,
	                    NONANTIALIASED_QUALITY,
	                    VARIABLE_PITCH,
	                    FontName);

	if(m_Font == NULL) return E_FAIL;
	else return S_OK;
}

//////////////////////////////////////////////////////////////////////////////////
// CDXSurface TextOut
//////////////////////////////////////////////////////////////////////////////////
HRESULT CDXSurface::TextXY(int x, int y, COLORREF col, const char* pString)
{
	SelectObject(m_DC, m_Font);
	SetBkMode(m_DC, TRANSPARENT);
	SetTextColor(m_DC, col);
	if(!TextOut(m_DC, x, y, pString, strlen(pString))) return E_FAIL;
	else return DD_OK;
}
/////////////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////1.5 ׳  ..................

//////////////////////////////////////////////////////////////////////////////////
// CDXSurface DrawTrans
//////////////////////////////////////////////////////////////////////////////////
HRESULT CDXSurface::DrawTrans(int X, int Y/*, CDXSurface* lpDDest*/)
{
	HRESULT rval;

	rval = m_SurBack->BltFast(X, Y, m_Surface, &m_SrcRect, DDBLTFAST_WAIT | DDBLTFAST_SRCCOLORKEY);
	if(rval == DDERR_SURFACELOST) Restore();

	return rval;
}

//////////////////////////////////////////////////////////////////////////////////
// CDXSurface DrawClipped
//////////////////////////////////////////////////////////////////////////////////
HRESULT CDXSurface::DrawClipped(int X, int Y/*, CDXSurface* lpDDest*/, LPRECT ClipRect)
{
	HRESULT rval;
	RECT ModSrc = m_SrcRect;
	Clip(&X, &Y, &ModSrc, ClipRect);

	rval = m_SurBack->BltFast(X, Y, m_Surface, &ModSrc, DDBLTFAST_WAIT | DDBLTFAST_SRCCOLORKEY);
	if(rval == DDERR_SURFACELOST) Restore();

	return rval;
}

//////////////////////////////////////////////////////////////////////////////////
// CDXSurface DrawWindowed
//////////////////////////////////////////////////////////////////////////////////
/*HRESULT CDXSurface::DrawWindowed(CDXSurface* lpDDest)
{
	HRESULT rval;
	RECT Window;
	GetClientRect(Screen->m_hWnd, &Window);
	ClientToScreen(Screen->m_hWnd, (LPPOINT)&Window);
	ClientToScreen(Screen->m_hWnd, (LPPOINT)&Window+1);

	rval = lpDDest->m_Surface->Blt(&Window, m_Surface, NULL, DDBLT_WAIT, NULL);
	if(rval == DDERR_SURFACELOST) Restore();

	return rval;
}*/

//////////////////////////////////////////////////////////////////////////////////
// CDXSurface DrawScaled
//////////////////////////////////////////////////////////////////////////////////
HRESULT CDXSurface::DrawScaled(int X, int Y, float Factor/*, CDXSurface* lpDDS*/)
{
	HRESULT rval;

	m_DestRect.top = Y;
	m_DestRect.left = X;
	m_DestRect.bottom = Y +( m_Height * Factor);
	m_DestRect.right = X + (m_Width * Factor);

  rval = m_SurBack->Blt(&m_DestRect, m_Surface, &m_SrcRect, DDBLT_WAIT | DDBLT_KEYSRC, NULL);
	if(rval == DDERR_SURFACELOST) Restore();

	return rval;
}

#define DEG2RAD(x) (x*(float)PI/(float)180)

//////////////////////////////////////////////////////////////////////////////////
// CDXSurface DrawRotated
//////////////////////////////////////////////////////////////////////////////////
void CDXSurface::DrawRotated(int X, int Y, double Angle/*, CDXSurface* Dest*/)
{
	// NOT IMPLEMENTED
}

//////////////////////////////////////////////////////////////////////////////////
// CDXSurface DrawHFlip
//////////////////////////////////////////////////////////////////////////////////
HRESULT CDXSurface::DrawHFlip(int X, int Y/*, CDXSurface* Dest*/)
{
	DDBLTFX ddBltFx;

	ddBltFx.dwSize = sizeof(DDBLTFX);
	ddBltFx.dwDDFX = DDBLTFX_MIRRORLEFTRIGHT;
	return m_Surface->Blt(NULL, NULL, NULL, DDBLT_DDFX | DDBLT_WAIT, &ddBltFx);
}

//////////////////////////////////////////////////////////////////////////////////
// CDXSurface DrawVFlip
//////////////////////////////////////////////////////////////////////////////////
HRESULT CDXSurface::DrawVFlip(int X, int Y/*, CDXSurface* Dest*/)
{
	DDBLTFX ddBltFx;

	ddBltFx.dwSize = sizeof(DDBLTFX);
	ddBltFx.dwDDFX = DDBLTFX_MIRRORUPDOWN;
	return m_Surface->Blt(NULL, NULL, NULL, DDBLT_DDFX | DDBLT_WAIT, &ddBltFx);
}
////////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////////
// CDXSurface PutPixel
//////////////////////////////////////////////////////////////////////////////////
void CDXSurface::PutPixel(int X, int Y, int Col)
{
	BYTE *Bitmap = (BYTE*)m_Desc.lpSurface;
	Bitmap[Y * m_Desc.lPitch + X] = Col;
}

//////////////////////////////////////////////////////////////////////////////////
// CDXSurface GetPixel
//////////////////////////////////////////////////////////////////////////////////
int CDXSurface::GetPixel(int X, int Y)
{
	BYTE *Bitmap = (BYTE*)m_Desc.lpSurface;
	return Bitmap[Y * m_Desc.lPitch + X];
}
void Clip(int *DestX, int *DestY, RECT *SrcRect, RECT *DestRect)
{
	// If it's partly off the right side of the screen
	if(*DestX + (SrcRect->right - SrcRect->left) > DestRect->right)
		SrcRect->right -= *DestX + (SrcRect->right-SrcRect->left) - DestRect->right;

	// Partly off the left side of the screen
	if(*DestX < DestRect->left)
	{
		SrcRect->left += DestRect->left - *DestX;
		*DestX = DestRect->left;
	}

	// Partly off the top of the screen
	if(*DestY < DestRect->top)
	{
		SrcRect->top += DestRect->top - *DestY;
		*DestY = DestRect->top;
	}

	// If it's partly off the bottom side of the screen
	if(*DestY + (SrcRect->bottom - SrcRect->top) > DestRect->bottom)
	SrcRect->bottom -= ((SrcRect->bottom-SrcRect->top)+*DestY) - DestRect->bottom;

	return;
}




