// This program is the second version of my tilebased game demo,
// and unlike alot of my "game programming" stuff that I've put out
// I'm hoping to build some real gameplay into this project...
//
// For those who haven't seen the first version of this, I'm using
// Andre' Lamothe's code from his DirectX discussion/demo called
// dXtacy. Check out his site at http://www.xgames3d.com/ ...
// ... and of course - visit my site some time soon:
// http://members.xoom.com/JimMcCue/
//
// my name is James McCue, and I can be reached by e-mail at
// aloiterer@juno.com
//
// I'm sold on DirectX
//
// ... Included in this file are routines that will allow you to
// begin using a timing paradigm introduced to me by Dhonn Lushine
// dhonn@usa.net
//
// I do not use dhonn's routines in this source, but since I like the
// concept, I've left the routines in here...
//
// goto http://members.aol.com/dhonn to find out more about his timer
//
// LISTING 1.0 - DIRECT X 5.0 GAME CONSOLE ////////////////////////////////////

// INCLUDES ///////////////////////////////////////////////////////////////////

#define WIN32_LEAN_AND_MEAN // make sure certain headers are included correctly

#include <windows.h> // include the standard windows stuff
#include <windowsx.h> // include the 32 bit stuff
#include <mmsystem.h> // include the multi media stuff
// note you need winmm.lib also
#include <ddraw.h> // include direct draw components

#include <stdlib.h> // include all the good stuff
#include <stdarg.h>
#include <stdio.h>
#include <math.h>
#include <float.h>
#include "ddutil.h"         // DDLoadPalette, DDCopyBitmap

// DEFINES ////////////////////////////////////////////////////////////////////

#define WINDOW_CLASS_NAME "WINDOW_CLASS" // this is the name of the window class

// defines for screen parameters

#define SCREEN_WIDTH 640 // the width of the viewing surface
#define SCREEN_HEIGHT 480 // the height of the viewing surface
#define SCREEN_BPP 16 // the bits per pixel

#define NUMTEXTURES 12 // world textures
#define NUMCLOUDS   10 // floating "normal" clouds
#define NUMPLAYERFRAMES 8 // number of frames for the ship
// tile size (you can change these, but make sure the world [in tiles]
// is larger than your viewport)

#define CELL_X_SIZE 32
#define CELL_Y_SIZE 32

// number of tiles horizontally and vertically representing the 
// world map (only SOME of these are on screen at any one time)

#define TILE_X_NUM 32
#define TILE_Y_NUM 32

// though I don't do things very efficiently here, I was thinking
// of those who like to use bitwize shifting, as opposed to 
// multiplication. (Multiplication is a series of additions on
// a computer... shifting is faster)

#define TILE_X_NUM_SHIFT 5
#define TILE_Y_NUM_SHIFT 5

// The "area" of the gameworld in tiles

#define TOTAL_TILES  (TILE_X_NUM*TILE_Y_NUM)

// the maximum coordinates for the lower right hand
// corner of the screen

#define X_BOUND    ((TILE_X_NUM)*CELL_X_SIZE)
#define Y_BOUND    ((TILE_Y_NUM)*CELL_Y_SIZE)

// the characters are clouds right now,
// they're bigger than tiles here

#define CLOUD_WIDTH 64
#define CLOUD_HEIGHT 64

// Directional defines for character motion

#define MOVE_LEFT    1
#define MOVE_RIGHT   2
#define MOVE_UP      4
#define MOVE_DOWN    8

// TYPES /////////////////////////////////////////////////////////////////////

typedef unsigned char UCHAR;

typedef struct enem_typ
{
	int x;
	int y;
	int speed;
} enemy, *enemy_ptr;

// MACROS /////////////////////////////////////////////////////////////////////

// these query the keyboard in real-time, the way this is used in
// Game_Main() allows for multiple keypresses

#define KEY_DOWN(vk_code) ((GetAsyncKeyState(vk_code) & 0x8000) ? 1 : 0)
#define KEY_UP(vk_code) ((GetAsyncKeyState(vk_code) & 0x8000) ? 0 : 1)

// PROTOTYPES ////////////////////////////////////////////////////////////////

int DD_Init(HWND hwnd);
int DD_Shutdown(void);

void Game_Init(void);
void Game_Main(void);
void Game_Shutdown(void);

void Translate_Other_Objects(void);
void Draw_Other_Objects(int right, int bottom);
void Render_Scene(int right, int bottom);

// some very short routines that use MCI for music

void Open_Music();
void Start_Or_Restart_Track();
void Stop_Music();
void Close_Music();

// PROTOTYPES for dhonn lushine's timer functions ///////////////////////////

BOOL initftime(void); 
double ftime(void);

// DIRECTDRAW GLOBALS ////////////////////////////////////////////////////////

LPDIRECTDRAW lpdd = NULL; // dd object
LPDIRECTDRAWSURFACE lpddsprimary = NULL; // dd primary surface
LPDIRECTDRAWSURFACE lpddsback    = NULL; // dd back buffer
LPDIRECTDRAWSURFACE lpddstextures[12] = {NULL,NULL,NULL,NULL,NULL,NULL,
                                         NULL,NULL,NULL,NULL,NULL,NULL};
LPDIRECTDRAWSURFACE lpddsplayer[8] = {NULL,NULL,NULL,NULL,NULL,NULL,
                                      NULL,NULL};
LPDIRECTDRAWSURFACE lpddsenemy = NULL;
LPDIRECTDRAWPALETTE lpddpal = NULL; // a pointer to the created dd palette
PALETTEENTRY color_palette[256]; // holds the shadow palette entries
DDSURFACEDESC ddsd; // a direct draw surface description struct
DDSCAPS ddscaps; // a direct draw surface capabilities struct
HRESULT ddrval; // result back from dd calls
HWND main_window_handle = NULL; // used to store the window handle
UCHAR *video_buffer = NULL; // pointer to video ram
UCHAR *double_buffer = NULL; // pointer to double buffer
int lpitch; // the linear pitch
enemy cloud[NUMCLOUDS];

BOOL music_enabled;

// High resolution timer stuff


__int64 startclock;  // we want a start time
double rate_inv;


// GAME GLOBALS GO HERE ///////////////////////////////////////////////////// 

// Names of bmps that will be loaded from disk

char szBitmap1[] = "TILE1.bmp";
char szBitmap2[] = "TILE2.bmp";
char szBitmap3[] = "TILE3.bmp";
char szBitmap4[] = "TILE4.bmp";
char szBitmap5[] = "TILE5.bmp";
char szBitmap6[] = "TILE6.bmp";
char szBitmap7[] = "TILE7.bmp";
char szBitmap8[] = "TILE8.bmp";
char szBitmap9[] = "TILE9.bmp";
char szBitmap10[] = "TILE10.bmp";
char szBitmap11[] = "TILE11.bmp";
char szBitmap12[] = "TILE12.bmp";

char szEnemy[] = "ENEM1.bmp";
char szPlayer1[] = "PLAY1.bmp";
char szPlayer2[] = "PLAY2.bmp";
char szPlayer3[] = "PLAY3.bmp";
char szPlayer4[] = "PLAY4.bmp";
char szPlayer5[] = "PLAY5.bmp";
char szPlayer6[] = "PLAY6.bmp";
char szPlayer7[] = "PLAY7.bmp";
char szPlayer8[] = "PLAY8.bmp";

// think of this array as something you would construct a model
// train set on...

char terrain_texture[] =

     { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,10,11,0,10,11,0,0,0,0,0,0,0,0,0,0,0,
       0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,8,9,0,8,9,0,0,0,0,0,0,0,0,0,0,0,
       0,0,1,1,1,1,1,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
       0,0,1,0,0,0,0,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
       0,0,1,0,0,0,0,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,1,1,1,1,0,
       0,0,1,0,0,0,0,1,1,1,1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,0,1,0,0,0,1,0,
       0,0,1,0,0,0,0,1,0,0,0,0,0,0,0,0,0,1,0,0,0,0,0,0,0,0,1,0,0,0,1,0,
       0,0,1,0,0,0,0,1,0,0,0,0,0,0,0,0,0,1,0,0,0,0,0,0,0,0,1,0,0,0,1,0,
       0,0,1,0,0,0,0,1,0,0,0,0,0,0,0,0,0,1,1,1,1,1,1,1,1,1,1,0,0,0,1,0,
       0,0,1,0,0,0,0,1,0,0,0,0,0,0,0,0,0,1,0,0,0,0,0,0,0,1,0,0,0,0,1,0,
       0,0,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,1,0,0,0,0,1,0,
       0,0,0,0,0,0,0,1,0,0,0,0,0,0,0,0,0,1,0,0,0,0,0,0,0,1,0,0,0,0,1,0,
       0,0,0,0,0,0,0,1,0,0,0,0,0,0,0,0,0,1,0,0,0,0,0,0,0,1,0,0,0,0,1,0,
       0,0,0,0,0,0,0,1,0,0,0,0,0,0,0,0,0,1,0,0,0,0,0,0,0,1,0,0,0,0,1,0,
       0,0,0,0,0,0,0,1,0,0,0,0,0,10,11,0,0,1,0,0,0,0,0,0,1,1,0,0,0,0,1,0,
       0,0,0,0,0,0,0,1,0,0,0,0,0,8,9,0,0,1,0,0,0,0,0,1,1,0,0,0,0,0,1,0,
       0,0,0,0,0,0,0,1,1,1,1,1,1,1,1,1,1,1,0,0,0,0,0,1,0,0,0,0,0,0,1,0,
       0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,0,0,0,1,0,
       0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,0,0,0,1,0,
       0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,0,0,0,1,0,
       0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,0,0,0,0,0,0,1,0,0,0,0,0,0,1,0,
       0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,0,0,0,0,0,1,0,0,0,0,0,0,1,0,
       0,0,0,0,0,0,0,0,0,0,5,2,6,0,0,0,0,0,7,0,0,0,0,1,1,1,1,1,1,1,1,0,
       0,0,0,0,0,0,0,0,0,0,2,2,2,0,0,0,0,0,0,7,0,0,0,0,0,0,0,0,0,0,0,0,
       0,0,0,1,1,1,1,1,0,0,2,2,2,0,0,0,0,0,0,0,7,0,0,0,0,0,0,0,0,0,0,0,
       0,0,0,1,0,0,0,1,0,0,2,2,2,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,0,0,0,
       0,0,0,1,0,0,0,1,0,0,2,2,2,0,0,0,0,0,0,0,0,0,0,0,1,0,1,0,0,0,0,0,
       0,0,0,1,0,0,0,1,0,0,3,2,4,0,0,0,0,0,0,0,0,0,0,1,0,0,0,1,0,0,0,0,
       0,0,0,1,1,1,1,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,0,0,0,0,0,0,
       0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,0,0,0,0,0,0,0,
       0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,
       0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,0,1,0,0,0,1,0,0,0,
      
     };

// SHORT MUSIC ROUTINES, THAT USE MCI //////////////////////////////////

void Open_Music(void)
{
	char strCommandString[80];
	MCIERROR ret;

	sprintf(strCommandString,"open PASSPORT.MID type sequencer alias jazz");
   
    ret = mciSendString(strCommandString, 0, 0, 0);

	if (ret == 0)
		music_enabled = TRUE;
} // end Open_Music

void Start_Or_Restart_Track(void)
{
    MCIERROR ret;

	if (music_enabled)
	    ret = mciSendString("play jazz from 0", 0, 0, 0);

	if (ret)
		music_enabled=FALSE;

} // end Start_Or_Restart_Track(void)

void Stop_Music(void)
{
	MCIERROR ret;

	if (music_enabled)
        ret = mciSendString("stop jazz", 0, 0, 0);
} // end Stop_Music

void Close_Music(void)
{
	MCIERROR ret;

	if (music_enabled)
	{
		ret = mciSendString("stop jazz", 0, 0, 0);
		ret = mciSendString("close jazz", 0, 0, 0);
	} // end if music enabled
} // end close music

// HIREZ TIMER FUNCTIONS - by dhonn lushine /////////////////////////////

BOOL initftime(void) {
  __int64 rate;

  // we need the accuracy
  if(!QueryPerformanceFrequency((LARGE_INTEGER*)&rate)) {
    return FALSE; // win errors
  }

  // usually the rate will be 1193180
  if(!rate) {
    return FALSE;
  }

  rate_inv=1.0/(double)rate;

  if(!QueryPerformanceCounter((LARGE_INTEGER*)&startclock)) {
    return FALSE; // win errors
  }

  return TRUE; // there is a clock
}
// you would have to start up with initftime() at the beginning
// of the game. And check for errors
 
double ftime(void) {
  // by dividing by its rate you get accurate seconds

  __int64 endclock;

  QueryPerformanceCounter((LARGE_INTEGER*)&endclock);

  return (double)(endclock-startclock)*rate_inv;

  // note: I recommend that you multiply but the inverse of a constant.
  // (speed reasons)
}

// DIRECT X FUNCTIONS /////////////////////////////////////////////////////////

int DD_Init(HWND hwnd)
{
	// this function is responsible for initializing direct draw, it creates a
	// primary surface 
    
	HDC hdc;
	int index; // looping index

	// now that the windows portion is complete, start up direct draw
	if (DirectDrawCreate(NULL,&lpdd,NULL)!=DD_OK)
	{
		// shutdown any other dd objects and kill window
		DD_Shutdown();
		return(0);
	} // end if

	// now set the coop level to exclusive and set for full screen and mode x
	if (lpdd->SetCooperativeLevel(hwnd, DDSCL_ALLOWREBOOT | DDSCL_EXCLUSIVE |
	DDSCL_FULLSCREEN | DDSCL_ALLOWMODEX)!=DD_OK)
	{
		// shutdown any other dd objects and kill window
		DD_Shutdown();
		return(0);
	} // end if

	// now set the display mode
	if (lpdd->SetDisplayMode(SCREEN_WIDTH,SCREEN_HEIGHT,SCREEN_BPP)!=DD_OK)
	{
		// shutdown any other dd objects and kill window
		DD_Shutdown();
		return(0);
	} // end if
    
     // Create the primary surface with 1 back buffer
    ddsd.dwSize = sizeof( ddsd );
    ddsd.dwFlags = DDSD_CAPS | DDSD_BACKBUFFERCOUNT;
    ddsd.ddsCaps.dwCaps = DDSCAPS_PRIMARYSURFACE |
			  DDSCAPS_FLIP |
			  DDSCAPS_COMPLEX;
    ddsd.dwBackBufferCount = 1;
    ddrval = lpdd->CreateSurface( &ddsd, &lpddsprimary, NULL );
    if( ddrval != DD_OK )
    {
		DD_Shutdown();
		return(0);
   }

    ddscaps.dwCaps = DDSCAPS_BACKBUFFER;
    ddrval = lpddsprimary->GetAttachedSurface(&ddscaps, &lpddsback);
    if( ddrval != DD_OK )
    {
		DD_Shutdown();
		return(0);
    }

  
	ddsd.dwFlags = DDSD_CAPS | DDSD_HEIGHT | DDSD_WIDTH;
	ddsd.ddsCaps.dwCaps = DDSCAPS_OFFSCREENPLAIN;
	ddsd.dwHeight = CELL_X_SIZE;
	ddsd.dwWidth = CELL_Y_SIZE;

	for (index=0; index<NUMTEXTURES; index++)
		if (lpdd->CreateSurface(&ddsd,&lpddstextures[index],NULL)!=DD_OK)
		{
			// shutdown any other dd objects and kill window
			DD_Shutdown();
			return(0);
		} // end if

	ddsd.dwFlags = DDSD_CAPS | DDSD_HEIGHT | DDSD_WIDTH;
	ddsd.ddsCaps.dwCaps = DDSCAPS_OFFSCREENPLAIN;
	ddsd.dwHeight = CLOUD_HEIGHT;
	ddsd.dwWidth = CLOUD_WIDTH;

	if (lpdd->CreateSurface(&ddsd,&lpddsenemy,NULL)!=DD_OK)
	{
		// shutdown any other dd objects and kill window
		DD_Shutdown();
		return(0);
	} // end if

	for (index=0; index<NUMPLAYERFRAMES; index++)
		if (lpdd->CreateSurface(&ddsd,&lpddsplayer[index],NULL)!=DD_OK)
		{
			// shutdown any other dd objects and kill window
			DD_Shutdown();
			return(0);
		} // end if


    // print some stuff

	if (lpddsprimary->GetDC(&hdc) == DD_OK)
	{
		SetBkColor( hdc, RGB( 0, 0, 255 ) );
		SetTextColor( hdc, RGB( 255, 255, 0 ) );
		TextOut( hdc, 0, 0, "Welcome Aboard!", 15 );
		lpddsprimary->ReleaseDC(hdc);
	}   

	// return success if we got this far
	return(1);

} // end DD_Init

///////////////////////////////////////////////////////////////////////////////

int DD_Shutdown(void)
{
// this function tests for dd components that have been created and releases
// them back to the operating system
int index;

	// test if the dd object exists
	if (lpdd)
	{
		// test if there is a primary surface
		if(lpddsprimary)
		{
			// release the memory and set pointer to NULL
			lpddsprimary->Release();
			lpddsprimary = NULL;
		} // end if

        for (index = 0; index<NUMTEXTURES; index++)
			if(lpddstextures[index])
			{
				// release the memory and set pointer to NULL
				lpddstextures[index]->Release();
				lpddstextures[index] = NULL;
			} // end if

		for (index = 0; index<NUMPLAYERFRAMES; index++)
		{
			if(lpddsplayer[index])
			{
				// release the memory and set pointer to NULL
				lpddsplayer[index]->Release();
				lpddsplayer[index] = NULL;
			} // end if
        }

		if(lpddsenemy)
		{
			// release the memory and set pointer to NULL
			lpddsenemy->Release();
			lpddsenemy = NULL;
		} // end if

		// now release the dd object itself
		lpdd->Release();
		lpdd = NULL;

		// return success
		return(1);

	} // end if
	else
		return(0);

} // end DD_Shutdown


// WINDOWS CALLBACK FUNCTION ////////////////////////////////////////////////// 

LRESULT CALLBACK WindowProc(HWND hwnd, UINT msg, WPARAM wparam, LPARAM lparam)
{
// this is the main message handler of the system

HDC hdc; // handle to graphics context
PAINTSTRUCT ps; // used to hold the paint info

	// what is the message?

	switch(msg)
	{ 
		case WM_CREATE:
		{
			// do windows inits here
			return(0);
		} break;

		// having made this program on a 486, I've notice it
		// can be a bit jittery, I wonder if this paint event
		// could be part of the reason. Anybody?

		case WM_PAINT:
		{
			// this message occurs when your window needs repainting
			hdc = BeginPaint(hwnd,&ps); 
			EndPaint(hdc,&ps);

			return(0);
		} break;

		case WM_DESTROY:
		{
			// this message is sent when your window is destroyed
			PostQuitMessage(0);
			return(0);
		} break;

		default:break;

	} // end switch

	// let windows process any messages that we didn't take care of 
	return (DefWindowProc(hwnd, msg, wparam, lparam));

} // end WinProc

// WINMAIN ////////////////////////////////////////////////////////////////////

int WINAPI WinMain( HINSTANCE hinstance,
HINSTANCE hprevinstance,
LPSTR lpcmdline,
int ncmdshow)
{
WNDCLASSEX winclass; // this holds the windows class info
HWND hwnd; // this holds the handle of our new window
MSG msg; // this holds a generic message
int do_or_die = 1; // give the user the option of avoiding running
                   // the program

	// first fill in the window class stucture

	winclass.cbSize = sizeof(WNDCLASSEX);
	winclass.style = CS_DBLCLKS | CS_OWNDC | CS_HREDRAW | CS_VREDRAW;
	winclass.lpfnWndProc = WindowProc;
	winclass.cbClsExtra = 0;
	winclass.cbWndExtra = 0;
	winclass.hInstance = hinstance;
	winclass.hIcon = LoadIcon(NULL, IDI_APPLICATION);
	winclass.hIconSm = LoadIcon(NULL, IDI_APPLICATION);
	winclass.hCursor = LoadCursor(NULL, IDC_ARROW);
	winclass.hbrBackground = GetStockObject(BLACK_BRUSH);
	winclass.lpszMenuName = NULL;
	winclass.lpszClassName = WINDOW_CLASS_NAME;

	// register the window class
	if (!RegisterClassEx(&winclass))
		return(0);

	// create the window
	if (!(hwnd = CreateWindowEx(WS_EX_TOPMOST,
								WINDOW_CLASS_NAME, // class
								"Jim's directx tasklist marque", // title
								WS_VISIBLE | WS_POPUP,
								0,0, // x,y
								GetSystemMetrics(SM_CXSCREEN),
								GetSystemMetrics(SM_CYSCREEN), 
								NULL, // parent
								NULL, // menu
								hinstance, // instance
								NULL))) // creation parms
		return(0);

	// give the user the option of bailing out now...

	do_or_die = MessageBox(hwnd, 
		                   "This program requires that you have DirectX 5 installed. Would you like to continue?",
						   "Jim's 2D tileworld demo",
						   MB_YESNO);

	if (do_or_die!=IDYES)
	{
		DestroyWindow(hwnd);
		return(0);
	} // end if abort program

	// hide the mouse cursor
	ShowCursor(0);

	// save the window handle
	main_window_handle = hwnd;

	// initialize direct draw
	if (!DD_Init(hwnd))
	{
		DestroyWindow(hwnd);
		return(0);
	} // end if

	// initialize game
	Game_Init();

	// enter main event loop
	while(1)
	{
		if (PeekMessage(&msg,NULL,0,0,PM_REMOVE))
		{ 
			// test if this is a quit
			if (msg.message == WM_QUIT)
			break;

			// translate any accelerator keys
			TranslateMessage(&msg);

			// send the message to the window proc
			DispatchMessage(&msg);
		} // end if
		else
		{
		// do asynchronous processing here

		// call main logic module
		Game_Main();


		} // end else

	} // end while

    // shut down direct draw
	DD_Shutdown();

	// shutdown game
	Game_Shutdown();

	// return to Windows
	return(msg.wParam);

} // end WinMain

// HERE ARE OUR GAME CONSOLE FUNCTIONS ///////////////////////////////////////////////////////

// Game_Init opens the music file, loads the bmps into direct draw 
// surfaces (offscreen plain)
// and also sets the source color keys for 
// bitmaps I want to Bltfast transparently...

void Game_Init(void)
{
// do any initialization here

    HBITMAP hbm;
	HRESULT smelly; // return value I only bother to check once...
    int index;

    Open_Music();

   // Load our bitmap resource.GetModuleHandle(NULL)LR_CREATEDIBSECTION

    hbm = (HBITMAP)LoadImage(NULL, szBitmap1, IMAGE_BITMAP, CELL_X_SIZE, CELL_Y_SIZE, LR_LOADFROMFILE);
    smelly = DDCopyBitmap(lpddstextures[0], hbm, 0, 0,   CELL_X_SIZE, CELL_Y_SIZE);
    hbm = (HBITMAP)LoadImage(NULL, szBitmap2, IMAGE_BITMAP, CELL_X_SIZE, CELL_Y_SIZE, LR_LOADFROMFILE);
    smelly = DDCopyBitmap(lpddstextures[1], hbm, 0, 0,   CELL_X_SIZE, CELL_Y_SIZE);
    hbm = (HBITMAP)LoadImage(NULL, szBitmap3, IMAGE_BITMAP, CELL_X_SIZE, CELL_Y_SIZE, LR_LOADFROMFILE);
    smelly = DDCopyBitmap(lpddstextures[2], hbm, 0, 0,   CELL_X_SIZE, CELL_Y_SIZE);
    hbm = (HBITMAP)LoadImage(NULL, szBitmap4, IMAGE_BITMAP, CELL_X_SIZE, CELL_Y_SIZE, LR_LOADFROMFILE);
    smelly = DDCopyBitmap(lpddstextures[3], hbm, 0, 0,   CELL_X_SIZE, CELL_Y_SIZE);
    hbm = (HBITMAP)LoadImage(NULL, szBitmap5, IMAGE_BITMAP, CELL_X_SIZE, CELL_Y_SIZE, LR_LOADFROMFILE);
    smelly = DDCopyBitmap(lpddstextures[4], hbm, 0, 0,   CELL_X_SIZE, CELL_Y_SIZE);
    hbm = (HBITMAP)LoadImage(NULL, szBitmap6, IMAGE_BITMAP, CELL_X_SIZE, CELL_Y_SIZE, LR_LOADFROMFILE);
    smelly = DDCopyBitmap(lpddstextures[5], hbm, 0, 0,   CELL_X_SIZE, CELL_Y_SIZE);
    hbm = (HBITMAP)LoadImage(NULL, szBitmap7, IMAGE_BITMAP, CELL_X_SIZE, CELL_Y_SIZE, LR_LOADFROMFILE);
    smelly = DDCopyBitmap(lpddstextures[6], hbm, 0, 0,   CELL_X_SIZE, CELL_Y_SIZE);
    hbm = (HBITMAP)LoadImage(NULL, szBitmap8, IMAGE_BITMAP, CELL_X_SIZE, CELL_Y_SIZE, LR_LOADFROMFILE);
    smelly = DDCopyBitmap(lpddstextures[7], hbm, 0, 0,   CELL_X_SIZE, CELL_Y_SIZE);
    hbm = (HBITMAP)LoadImage(NULL, szBitmap9, IMAGE_BITMAP, CELL_X_SIZE, CELL_Y_SIZE, LR_LOADFROMFILE);
    smelly = DDCopyBitmap(lpddstextures[8], hbm, 0, 0,   CELL_X_SIZE, CELL_Y_SIZE);
    hbm = (HBITMAP)LoadImage(NULL, szBitmap10, IMAGE_BITMAP, CELL_X_SIZE, CELL_Y_SIZE, LR_LOADFROMFILE);
    smelly = DDCopyBitmap(lpddstextures[9], hbm, 0, 0,   CELL_X_SIZE, CELL_Y_SIZE);
    hbm = (HBITMAP)LoadImage(NULL, szBitmap11, IMAGE_BITMAP, CELL_X_SIZE, CELL_Y_SIZE, LR_LOADFROMFILE);
    smelly = DDCopyBitmap(lpddstextures[10], hbm, 0, 0,   CELL_X_SIZE, CELL_Y_SIZE);
    hbm = (HBITMAP)LoadImage(NULL, szBitmap12, IMAGE_BITMAP, CELL_X_SIZE, CELL_Y_SIZE, LR_LOADFROMFILE);
    smelly = DDCopyBitmap(lpddstextures[11], hbm, 0, 0,   CELL_X_SIZE, CELL_Y_SIZE);
 
    hbm = (HBITMAP)LoadImage(NULL, szEnemy, IMAGE_BITMAP, CLOUD_WIDTH, CLOUD_HEIGHT, LR_LOADFROMFILE);
    smelly = DDCopyBitmap(lpddsenemy, hbm, 0, 0,   CLOUD_WIDTH, CLOUD_HEIGHT);

	// ,load the frames of player directions

	hbm = (HBITMAP)LoadImage(NULL, szPlayer1, IMAGE_BITMAP, CLOUD_WIDTH, CLOUD_HEIGHT, LR_LOADFROMFILE);
    smelly = DDCopyBitmap(lpddsplayer[0], hbm, 0, 0,   CLOUD_WIDTH, CLOUD_HEIGHT);
	DDSetColorKey(lpddsplayer[0], RGB(0,0,0));

	hbm = (HBITMAP)LoadImage(NULL, szPlayer2, IMAGE_BITMAP, CLOUD_WIDTH, CLOUD_HEIGHT, LR_LOADFROMFILE);
    smelly = DDCopyBitmap(lpddsplayer[1], hbm, 0, 0,   CLOUD_WIDTH, CLOUD_HEIGHT);
	DDSetColorKey(lpddsplayer[1], RGB(0,0,0));

	hbm = (HBITMAP)LoadImage(NULL, szPlayer3, IMAGE_BITMAP, CLOUD_WIDTH, CLOUD_HEIGHT, LR_LOADFROMFILE);
    smelly = DDCopyBitmap(lpddsplayer[2], hbm, 0, 0,   CLOUD_WIDTH, CLOUD_HEIGHT);
	DDSetColorKey(lpddsplayer[2], RGB(0,0,0));

	hbm = (HBITMAP)LoadImage(NULL, szPlayer4, IMAGE_BITMAP, CLOUD_WIDTH, CLOUD_HEIGHT, LR_LOADFROMFILE);
    smelly = DDCopyBitmap(lpddsplayer[3], hbm, 0, 0,   CLOUD_WIDTH, CLOUD_HEIGHT);
	DDSetColorKey(lpddsplayer[3], RGB(0,0,0));

	hbm = (HBITMAP)LoadImage(NULL, szPlayer5, IMAGE_BITMAP, CLOUD_WIDTH, CLOUD_HEIGHT, LR_LOADFROMFILE);
    smelly = DDCopyBitmap(lpddsplayer[4], hbm, 0, 0,   CLOUD_WIDTH, CLOUD_HEIGHT);
	DDSetColorKey(lpddsplayer[4], RGB(0,0,0));

	hbm = (HBITMAP)LoadImage(NULL, szPlayer6, IMAGE_BITMAP, CLOUD_WIDTH, CLOUD_HEIGHT, LR_LOADFROMFILE);
    smelly = DDCopyBitmap(lpddsplayer[5], hbm, 0, 0,   CLOUD_WIDTH, CLOUD_HEIGHT);
	DDSetColorKey(lpddsplayer[5], RGB(0,0,0));

	hbm = (HBITMAP)LoadImage(NULL, szPlayer7, IMAGE_BITMAP, CLOUD_WIDTH, CLOUD_HEIGHT, LR_LOADFROMFILE);
    smelly = DDCopyBitmap(lpddsplayer[6], hbm, 0, 0,   CLOUD_WIDTH, CLOUD_HEIGHT);
	DDSetColorKey(lpddsplayer[6], RGB(0,0,0));

	hbm = (HBITMAP)LoadImage(NULL, szPlayer8, IMAGE_BITMAP, CLOUD_WIDTH, CLOUD_HEIGHT, LR_LOADFROMFILE);
    smelly = DDCopyBitmap(lpddsplayer[7], hbm, 0, 0,   CLOUD_WIDTH, CLOUD_HEIGHT);
	DDSetColorKey(lpddsplayer[7], RGB(0,0,0));



    DeleteObject(hbm);

    if (smelly!=DD_OK)
	{
         PostQuitMessage(0);
    }
 
	DDSetColorKey(lpddsenemy, RGB(0,0,0));

	// init regular cloud

	for (index = 0; index<NUMCLOUDS; index++)
	{
         cloud[index].x = (rand()%(X_BOUND-130)) + CLOUD_WIDTH;
		 cloud[index].y = (rand()%(Y_BOUND-130)) + CLOUD_HEIGHT;
		 cloud[index].speed = (rand()%5) + 1;
    } // end for index

} // end Game_Init

/////////////////////////////////////////////////////////////////////////////////////////////

void Game_Shutdown(void)
{
// cleanup and release all resources here
    Stop_Music();
    Close_Music();

} // end Game_Shutdown

/////////////////////////////////////////////////////////////////////////////////////////////

// this looks pretty straight forward from the top, but it
// gets a little bit murky when I start trying to do as follows:
//
//  - I wanted to do one of those deals where the "player" is 
// centered on the screen until the viewport reaches the edges of
// the screen, then I wanted him or her ; ) to roam freely...
// when the player moves back toward center, in a way that should
// move the viewport, well... this does that... just not that elegantly.

void Game_Main(void)
{
// process your game logic and draw next frame

// right now, it's just a cheesy clipping tile engine!
 
    HDC hdc;
	LPDIRECTDRAWSURFACE pdds; // an alias we'll use to the
	                          // tile to be used in BltFast!
    RECT rcRect;              /// used for clipping tiles
	HRESULT             ddrval;
	static int px = SCREEN_WIDTH, // upperleft corner of screen
	           py = SCREEN_HEIGHT; 
	static int player_left = (SCREEN_WIDTH/2) - 32;
	static int player_top = (SCREEN_HEIGHT/2) - 32;
	int sprite_left, sprite_top;
    int centering_factor_x,
		centering_factor_y;
    static int player_direction_frame = 0;
    static int movementspeed = 4;

	// get input, note how keyboard is accessed with constants "VK_"
	// these are defined in "winuser.h" are basically the scan codes
	// for letters just use capital ASCII codes

	if (KEY_DOWN(VK_ESCAPE))
	{
		PostMessage(main_window_handle,WM_QUIT,0,0); // this is how you exit you game
    }

	// want to change the speed of the ship?

	if (KEY_DOWN('D'))
		if ((--movementspeed)<1) movementspeed = 1;

	if (KEY_DOWN('A'))
		if ((++movementspeed)>12) movementspeed = 12;

    // which way is player moving
	if (KEY_DOWN(VK_RIGHT))
	{
        if ((player_left+=movementspeed)>(X_BOUND-CLOUD_WIDTH)) player_left = (X_BOUND-CLOUD_WIDTH);
 
	    player_direction_frame = 0;
	}

	if (KEY_DOWN(VK_LEFT))
	{
 		if ((player_left-=movementspeed)<0) player_left = 0;

		player_direction_frame = 4;
	}

	if (KEY_DOWN(VK_UP))
	{
        if ((player_top-=movementspeed)<0) player_top = 0;
 
		player_direction_frame = 2;
	}

	if (KEY_DOWN(VK_DOWN))
	{
        if ((player_top+=movementspeed)>(Y_BOUND-CLOUD_HEIGHT)) player_top = (Y_BOUND-CLOUD_HEIGHT);
 
        player_direction_frame = 6;
	}

	if (KEY_DOWN(VK_RIGHT) && KEY_DOWN(VK_UP))
		player_direction_frame = 1;

	if (KEY_DOWN(VK_RIGHT) && KEY_DOWN(VK_DOWN))
		player_direction_frame = 7;

	if (KEY_DOWN(VK_LEFT) && KEY_DOWN(VK_UP))
		player_direction_frame = 3;

	if (KEY_DOWN(VK_LEFT) && KEY_DOWN(VK_DOWN))
		player_direction_frame = 5;

    // would you like some tunage?

	if (KEY_DOWN('M'))
        Start_Or_Restart_Track();

    if (KEY_DOWN('N'))
        Stop_Music();

	// this is where I check to see if the player is roaming where the
	// "viewport" cannot -> at least not with the player in the center of
	// it...
    
	centering_factor_x = (SCREEN_WIDTH>>1)+(CLOUD_WIDTH>>1);
	centering_factor_y = (SCREEN_HEIGHT>>1)+(CLOUD_HEIGHT>>1);

    px = player_left + centering_factor_x;
  
	if (px > X_BOUND) px = X_BOUND;
	if (px < SCREEN_WIDTH) px = SCREEN_WIDTH;

    py = player_top + centering_factor_y;

    if (py > Y_BOUND) py = Y_BOUND;
	if (py < SCREEN_HEIGHT) py = SCREEN_HEIGHT;
	// render the ground..

	Render_Scene(px,py);

	// moves the other "normal" clouds around...
    Translate_Other_Objects();

	// draws them...

	Draw_Other_Objects(px,py);

	// This next section takes the players world coordinates, and
	// converts them into screen coordinates...

	sprite_left = player_left-(px-SCREEN_WIDTH);
	sprite_top = player_top-(py-SCREEN_HEIGHT);

	// we need to clip the player...

	rcRect.top = 0;
	rcRect.left = 0;
	rcRect.right = CLOUD_WIDTH;
	rcRect.bottom = CLOUD_HEIGHT;

	pdds = lpddsplayer[player_direction_frame];
     
	ddrval = lpddsback->BltFast( sprite_left, sprite_top, pdds, &rcRect, DDBLTFAST_WAIT | DDBLTFAST_SRCCOLORKEY);
 
	// print a message on the backbuffer

	if (lpddsback->GetDC(&hdc) == DD_OK)
	{
		SetBkColor( hdc, RGB( 0, 0, 255 ) );
		SetTextColor( hdc, RGB( 255, 255, 0 ) );
		TextOut( hdc, 0, 0, "Keys: LEFT, RIGHT, UP, DOWN, .. and ESC... OH - and M Music (on), N (off), A/D Speed", 84 );
		lpddsback->ReleaseDC(hdc);
	}   

	// flip back buffer to primary buffer
    lpddsprimary->Flip(NULL,DDFLIP_WAIT);

// return and let windows have some time - NO WAY, since I'm not checking
// for "lost" surfaces, I don't let you switch to other applications while
// this is running, we'll cros that bridge when we come to it...

} // end Game_Main

// this just moves the "normal" clouds..

void Translate_Other_Objects(void)
{
	int index;

	static wind = 0;
	static wind_timer = 100;

	for (index = 0; index<NUMCLOUDS; index++)
	{
		cloud[index].x += cloud[index].speed;

		if (cloud[index].x > (X_BOUND))
			cloud[index].x = -CLOUD_WIDTH;
		if(cloud[index].x < -CLOUD_WIDTH) 
            cloud[index].x = X_BOUND;

		if (wind)
		{
			cloud[index].y += wind;

			if (cloud[index].y > (Y_BOUND))
				cloud[index].y = -CLOUD_HEIGHT;
			if(cloud[index].y < -CLOUD_HEIGHT) 
				cloud[index].y = Y_BOUND;
        }

	} // end for cloud

	if ((--wind_timer) == 0)
	{
		wind = (rand()%5) - 2;
		wind_timer = ((rand()%5)*10) + 60;
	}
} // end for Translate_Other_Objects

// this just draws the "normal" clouds

// this routine does clipping, in that it resets the values
// in rcRect, which are used in the call to BltFast...

// Again, not the neatest way

void Draw_Other_Objects(int right, int bottom)
{
	LPDIRECTDRAWSURFACE pdds; // an alias we'll use to the
	                          // tile to be used in BltFast!
    RECT rcRect;        // used by Bltfast, to determine what
	                    // part of our "tiles/sprites" are drawn
	HRESULT ddrval;
	int actual_screen_x, // these are the actual screen coordinates
		actual_screen_y; // for the upper left hand corner of each sprite
	int left, top;       // the left and top virtual coordinates of the 
	                     // viewport
    int index;

	for (index = 0; index<NUMCLOUDS; index++)
	{
        // start of assuming we won't need to clip

		rcRect.left = 0;
		rcRect.top = 0;
		rcRect.right = CLOUD_WIDTH;
		rcRect.bottom = CLOUD_HEIGHT;

		// the right side and bottom of the viewport are
		// sent, use the screenwidth and height to determine
		// the left and top sides..

		left = right - SCREEN_WIDTH;
		top = bottom - SCREEN_HEIGHT;

		// if at least a sliver of the cloud will be in view... then...

		if ((cloud[index].x >= left-CLOUD_WIDTH) &&
			(cloud[index].x < right)  &&
			(cloud[index].y >= top-CLOUD_HEIGHT) &&
			(cloud[index].y < bottom))
		{
            // do we need to clip it to the left side of the viewport?

			if (cloud[index].x < left)
			{
				actual_screen_x = 0;
				rcRect.left = left-cloud[index].x;
			}
			else // apparently not...
			{
				actual_screen_x = cloud[index].x - left;

				// how about the right side?

				if (cloud[index].x > (right - CLOUD_WIDTH))
				{
                    rcRect.right = right - cloud[index].x;
				}
			}

			// do we need to clip it to the top of the viewport?

			if (cloud[index].y < top)
			{
				actual_screen_y = 0;
				rcRect.top = top-cloud[index].y;
			}
			else // nope...
			{
				actual_screen_y = cloud[index].y - top;

				// how about the bottom?

				if (cloud[index].y > (bottom - CLOUD_HEIGHT))
				{
					rcRect.bottom = bottom-cloud[index].y;
				}
			} // end else

			pdds = lpddsenemy;
 
			ddrval = lpddsback->BltFast( actual_screen_x, 
										 actual_screen_y, 
										 pdds, 
										 &rcRect, 
										 DDBLTFAST_WAIT | DDBLTFAST_SRCCOLORKEY);

        } // end if not culled
	} // end for cloud
} // end Draw_Other_Objects

// this routine lays down all the tiles...

void Render_Scene(int right, int bottom)
{
	LPDIRECTDRAWSURFACE pdds; // an alias we'll use to the
	                          // tile to be used in BltFast!
    RECT rcRect;              /// used for clipping tiles
	HRESULT             ddrval;
	int start_left, start_top;
	int start_right, start_bottom;
    int start_x_cell, start_y_cell;
	int render_it_x,render_it_y;
	int clip_right, clip_bottom;
    int which_cell,cell_x_save;
	int num_x_tiles, num_y_tiles;
	int x_offset, y_offset;
    int orig_left_clip,orig_top_clip;

	// my muddleheaded setup for the tile engine

	start_left = right-SCREEN_WIDTH;
	start_top = bottom-SCREEN_HEIGHT;

	// find the cell of the "map" to inspect (the map is very close
	// to the top of this source, you could have loaded it from a disk
	// if you wished, later versions of this tile engine will do that

	start_x_cell = start_left / CELL_X_SIZE;
	start_y_cell = start_top / CELL_Y_SIZE;

	// since we're tiling left to right, top to bottom, we only
	// *really* need to keep the orig_left_clip, which is the clipping
	// info for the left side of all the rows of tiles we'll be rendering

	orig_left_clip = rcRect.left = start_left%CELL_X_SIZE;
	orig_top_clip = rcRect.top = start_top%CELL_Y_SIZE;

	// you know something, this might be a completely extraneous step..

    start_right = (start_left+CELL_X_SIZE)%CELL_X_SIZE;
	start_bottom = (start_top+CELL_Y_SIZE)%CELL_Y_SIZE;

	if (start_right<=start_left)
		rcRect.right = CELL_X_SIZE;
	else
		rcRect.right = start_right;

	if (start_bottom<=start_top)
		rcRect.bottom = CELL_Y_SIZE;
	else
		rcRect.bottom = start_bottom;

	// get starting texture to draw with

	cell_x_save = start_x_cell + (start_y_cell<<TILE_Y_NUM_SHIFT);

	// we'll save it for calculating the starting tile for each row,
	// because sometimes we'll be drawing more tiles accross than at other
	// times, because the numbers I have chosen can draw 10 tiles to
	// cover the screen, but if you 
	// move horizontally, you'll need 11

	which_cell = cell_x_save;

	x_offset = 0;
	y_offset = 0;

	num_x_tiles = ((SCREEN_WIDTH+start_left)/ CELL_X_SIZE) + 1;
	num_y_tiles = ((SCREEN_HEIGHT+start_top)/ CELL_Y_SIZE) + 1;

// loop thru entire scene left to right, top to bottom

	for (render_it_y = 0; render_it_y<num_y_tiles; render_it_y++)
	{
		x_offset = 0;

		// get original start left clip for each x loop

        rcRect.left = orig_left_clip;

		// assume, for now, that we've got the whole tile
		// to work with horizontally

        rcRect.right = CELL_X_SIZE;

		// draw tiles left to right

		for (render_it_x = 0; render_it_x<num_x_tiles; render_it_x++)
		{
			pdds = lpddstextures[terrain_texture[which_cell]];
             
			ddrval = lpddsback->BltFast( x_offset, y_offset, pdds, &rcRect, DDBLTFAST_WAIT );
 
			// the following will only evaluate to a value other
			// than CELL_X_SIZE once for every row of tiles

			x_offset+=CELL_X_SIZE-rcRect.left;

			++which_cell;

	        rcRect.left = 0;
	        rcRect.right = CELL_X_SIZE;

			// determine where to kick out of the loop
			// or just clip the right hand side of the rect

			clip_right = SCREEN_WIDTH - x_offset;

			if ((clip_right)<0)
				break;

			if ((clip_right)<CELL_X_SIZE)
				if ((clip_right)==0)
			    {
				    ++which_cell;
				    break;
			    }
				else
					rcRect.right = clip_right;
			// sync time (optional)
		} // end for render_it_x

		// this will only evaluate to a value other than CELL_Y_SIZE
		// once...

		y_offset+=CELL_Y_SIZE-rcRect.top;

		cell_x_save += TILE_X_NUM;
        which_cell = cell_x_save;

		rcRect.top = 0;
	    rcRect.bottom = CELL_Y_SIZE;

		// clip tiles w/ respect to Y screen boundary

		clip_bottom = SCREEN_HEIGHT - y_offset;

		if ((clip_bottom)<CELL_Y_SIZE)
			if ((clip_bottom)<=0)
			    break;
		    else
				rcRect.bottom = SCREEN_HEIGHT-y_offset;

			
    } // end for render_it_y

} // Render_Scene