// This version of the 2D tile demo actually starts getting good
// because I'm using my texturemapping routines from my DJGPP 3d polygon
// tutorials in a cool way - to give my little ship more than 8 different
// angles of attack
//
// This source contains some of the best 16-bit pixel plotting code I've 
// seen (dhonn helped me with it) I almost wanted to make another 16-bit
// pixel plotting demo... but just dig it out of the source....
//  
// Get ready for some rotation my friend!
//
// This demo and source code are copyright 1998 - by James McCue.
//
// ... and James McCue is me! - so let's go...
//
// ... Included in this file are routines that will allow you to
// begin using a timing paradigm introduced to me by Dhonn Lushine
// dhonn@usa.net
//
// I USE this timing code this time around... and I discuss it a bit
// in the HTML file that probably accompanied this source...
//
// goto http://members.aol.com/dhonn to find out more about his timer
//
// LISTING 1.0 - DIRECT X 5.0 GAME CONSOLE ////////////////////////////////////

// INCLUDES ///////////////////////////////////////////////////////////////////

#define WIN32_LEAN_AND_MEAN // make sure certain headers are included correctly

#include <windows.h> // include the standard windows stuff
#include <windowsx.h> // include the 32 bit stuff
#include <winuser.h>
#include <mmsystem.h> // include the multi media stuff
// note you need winmm.lib also
#include <ddraw.h> // include direct draw components

#include <stdlib.h> // include all the good stuff
#include <stdarg.h>
#include <stdio.h>
#include <math.h>
#include <float.h>
#include "ddutil.h"         // DDLoadPalette, DDCopyBitmap
#include "resource.h"
#include "gamedefs.h" // some global definitions
#include "typedefs.h" // program structures
#include "shrdvars.h" // some shared variables
#include "polydraw.h" // my pixel plotting, rotation, and polygon drawing
                      // code

// DEFINES ////////////////////////////////////////////////////////////////////

#define WINDOW_CLASS_NAME "WINDOW_CLASS" // this is the name of the window class

#define STRICT

// TYPES /////////////////////////////////////////////////////////////////////
// I put these in their own module typedefs.h

// MACROS /////////////////////////////////////////////////////////////////////

// these query the keyboard in real-time, the way this is used in
// Game_Main() allows for multiple keypresses

#define KEY_DOWN(vk_code) ((GetAsyncKeyState(vk_code) & 0x8000) ? 1 : 0)
#define KEY_UP(vk_code) ((GetAsyncKeyState(vk_code) & 0x8000) ? 0 : 1)

// PROTOTYPES ////////////////////////////////////////////////////////////////

int WINAPI WinMain(HINSTANCE hinstance, HINSTANCE hprevinstance, PSTR szcmdline, int icmdshow);

int DD_Init(void);
int DD_Shutdown(void);

void Game_Init(void);
void Game_Main(void);
void Game_Shutdown(void);

void Wind_Effects_All(void);
void Translate_Other_Objects(void);
void Draw_Other_Objects(int right, int bottom);
void Render_Tiled_Background(int right, int bottom);

// some very short routines that use MCI for music

void Open_Music();
void Start_Or_Restart_Track();
void Continue_Track();
void Stop_Music();
void Close_Music();

// self explanatory (almost)

void Load_Bitmaps_Into_DDSurfaces(void);
void Reload_Bitmaps(void);
void Restore_Surfaces(void);

// PROTOTYPES for dhonn lushine's timer functions ///////////////////////////

BOOL initftime(void); 
double ftime(void);

BOOL CALLBACK DialogProc(HWND hDlg, UINT iMsg, WPARAM wParam, LPARAM lParam);

// DIRECTDRAW GLOBALS ////////////////////////////////////////////////////////

LPDIRECTDRAW lpdd = NULL; // dd object
LPDIRECTDRAWSURFACE lpddsprimary = NULL; // dd primary surface
LPDIRECTDRAWSURFACE lpddsback    = NULL; // dd back buffer
LPDIRECTDRAWSURFACE lpddstextures[12] = {NULL,NULL,NULL,NULL,NULL,NULL,
                                         NULL,NULL,NULL,NULL,NULL,NULL};
LPDIRECTDRAWSURFACE lpddsplayer = NULL;
LPDIRECTDRAWSURFACE lpddsenemy = NULL;
DDSURFACEDESC ddsd; // a direct draw surface description struct
DDSCAPS ddscaps; // a direct draw surface capabilities struct
HRESULT ddrval; // result back from dd calls
HWND main_window_handle = NULL; // used to store the window handle
enemy cloud[NUMCLOUDS];
BOOL music_enabled;
polygon ship;

int iScreenWidth =640; // the width of the viewing surface
int iScreenHeight =480; // the height of the viewing surface

int iHalfWidth =320;
int iHalfHeight =240;

// lookup tables for rotation and movement

unsigned char *double_buffer = NULL; // pointer to double buffer
int lpitch; // the linear pitch
// High resolution timer stuff

BOOL paused = FALSE;

__int64 startclock;  // we want a start time
double rate_inv;

double start_time, end_time;
double frametime;

BOOL game_is_running = TRUE;
BOOL music_playing = FALSE;
WORD SpriteRGB16[SHIP_WIDTH*SHIP_HEIGHT];

// GAME GLOBALS GO HERE ///////////////////////////////////////////////////// 

// Names of bmps that will be loaded from disk

char szBitmap[NUMTEXTURES][11] = {"TILE1.bmp",
                                "TILE2.bmp",
								"TILE3.bmp",
								"TILE4.bmp",
								"TILE5.bmp",
								"TILE6.bmp",
								"TILE7.bmp",
								"TILE8.bmp",
								"TILE9.bmp",
								"TILE10.bmp",
								"TILE11.bmp",
								"TILE12.bmp"};

char szEnemy[] = "ENEM1.bmp";
char szPlayer1[] = "PLAY1.bmp";

// think of this array as something you would construct a model
// train set on...

char terrain_texture[] =

     { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,10,11,0,10,11,0,0,0,0,0,0,0,0,0,0,0,
       0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,8,9,0,8,9,0,0,0,0,0,0,0,0,0,0,0,
       0,0,1,1,1,1,1,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
       0,0,1,0,0,0,0,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
       0,0,1,0,0,0,0,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,1,1,1,1,0,
       0,0,1,0,0,0,0,1,1,1,1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,0,1,0,0,0,1,0,
       0,0,1,0,0,0,0,1,0,0,0,0,0,0,0,0,0,1,0,0,0,0,0,0,0,0,1,0,0,0,1,0,
       0,0,1,0,0,0,0,1,0,0,0,0,0,0,0,0,0,1,0,0,0,0,0,0,0,0,1,0,0,0,1,0,
       0,0,1,0,0,0,0,1,0,0,0,0,0,0,0,0,0,1,1,1,1,1,1,1,1,1,1,0,0,0,1,0,
       0,0,1,0,0,0,0,1,0,0,0,0,0,0,0,0,0,1,0,0,0,0,0,0,0,1,0,0,0,0,1,0,
       0,0,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,1,0,0,0,0,1,0,
       0,0,0,0,0,0,0,1,0,0,0,0,0,0,0,0,0,1,0,0,0,0,0,0,0,1,0,0,0,0,1,0,
       0,0,0,0,0,0,0,1,0,0,0,0,0,0,0,0,0,1,0,0,0,0,0,0,0,1,0,0,0,0,1,0,
       0,0,0,0,0,0,0,1,0,0,0,0,0,0,0,0,0,1,0,0,0,0,0,0,0,1,0,0,0,0,1,0,
       0,0,0,0,0,0,0,1,0,0,0,0,0,10,11,0,0,1,0,0,0,0,0,0,1,1,0,0,0,0,1,0,
       0,0,0,0,0,0,0,1,0,0,0,0,0,8,9,0,0,1,0,0,0,0,0,1,1,0,0,0,0,0,1,0,
       0,0,0,0,0,0,0,1,1,1,1,1,1,1,1,1,1,1,0,0,0,0,0,1,0,0,0,0,0,0,1,0,
       0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,0,0,0,1,0,
       0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,0,0,0,1,0,
       0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,0,0,0,1,0,
       0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,0,0,0,0,0,0,1,0,0,0,0,0,0,1,0,
       0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,0,0,0,0,0,1,0,0,0,0,0,0,1,0,
       0,0,0,0,0,0,0,0,0,0,5,2,6,0,0,0,0,0,7,0,0,0,0,1,1,1,1,1,1,1,1,0,
       0,0,0,0,0,0,0,0,0,0,2,2,2,0,0,0,0,0,0,7,0,0,0,0,0,0,0,0,0,0,0,0,
       0,0,0,1,1,1,1,1,0,0,2,2,2,0,0,0,0,0,0,0,7,0,0,0,0,0,0,0,0,0,0,0,
       0,0,0,1,0,0,0,1,0,0,2,2,2,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,0,0,0,
       0,0,0,1,0,0,0,1,0,0,2,2,2,0,0,0,0,0,0,0,0,0,0,0,1,0,1,0,0,0,0,0,
       0,0,0,1,0,0,0,1,0,0,3,2,4,0,0,0,0,0,0,0,0,0,0,1,0,0,0,1,0,0,0,0,
       0,0,0,1,1,1,1,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,0,0,0,0,0,0,
       0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,0,0,0,0,0,0,0,
       0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,
       0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,0,1,0,0,0,1,0,0,0,
      
     };

// SHORT MUSIC ROUTINES, THAT USE MCI //////////////////////////////////

void Open_Music(void)
{
	char strCommandString[80];
	MCIERROR ret;

	sprintf(strCommandString,"open PASSPORT.MID type sequencer alias jazz");
   
    ret = mciSendString(strCommandString, 0, 0, 0);

	if (ret)
	{
		MessageBox(main_window_handle,"Failed to load PASSPORT.MID, check the directory tiled3.exe is in, if it's not there, redownload tiled3.zip, if it is, let aloiterer@juno.com know about this error","MCI music load error",MB_OK);
        music_enabled = FALSE;
	}
	else
		music_enabled = TRUE;

} // end Open_Music

void Start_Or_Restart_Track(void)
{
    MCIERROR ret;

	if (music_enabled)
	    ret = mciSendString("play jazz from 0", 0, 0, 0);

	if (ret)
		music_enabled=FALSE;

	music_playing=TRUE;
} // end Start_Or_Restart_Track(void)

void Continue_Track(void)
{
    MCIERROR ret;

	if (music_enabled)
	    ret = mciSendString("play jazz", 0, 0, 0);

	if (ret)
		music_enabled=FALSE;

	music_playing=TRUE;
} // end Continue_Track(void)

void Stop_Music(void)
{
	MCIERROR ret;

	if (music_enabled)
        ret = mciSendString("stop jazz", 0, 0, 0);

//	music_playing=FALSE;
} // end Stop_Music

void Close_Music(void)
{
	MCIERROR ret;

	if (music_enabled)
	{
		ret = mciSendString("stop jazz", 0, 0, 0);
		ret = mciSendString("close jazz", 0, 0, 0);
	} // end if music enabled

	music_playing=FALSE;
} // end close music


// HIREZ TIMER FUNCTIONS - by dhonn lushine /////////////////////////////

BOOL initftime(void) 
{
    __int64 rate;

    // we need the accuracy
    if(!QueryPerformanceFrequency((LARGE_INTEGER*)&rate)) 
	{
        return FALSE; // win errors
    }

    // usually the rate will be 1193180
    if(!rate) 
	{
        return FALSE;
    }

    rate_inv=1.0/(double)rate;

    if(!QueryPerformanceCounter((LARGE_INTEGER*)&startclock)) 
	{
        return FALSE; // win errors
    }

    return TRUE; // there is a clock
}
// you would have to start up with initftime() at the beginning
// of the game. And check for errors
 
double ftime(void) 
{
    // by dividing by its rate you get accurate seconds

    __int64 endclock;

    QueryPerformanceCounter((LARGE_INTEGER*)&endclock);

    return (double)(endclock-startclock)*rate_inv;

    // note: I recommend that you multiply but the inverse of a constant.
    // (speed reasons)
}

// DIRECT X FUNCTIONS /////////////////////////////////////////////////////////

int DD_Init(void)
{
	// this function is responsible for initializing direct draw, it creates a
	// primary surface 
    char buffer[80]; 
	HDC hdc;
	int index; // looping index

	// now that the windows portion is complete, start up direct draw
	if (DirectDrawCreate(NULL,&lpdd,NULL)!=DD_OK)
	{
		MessageBox(main_window_handle,"Sorry, can't initialize directdraw, if you would, please let aloiterer@juno.com know if you recieve this message. Thanks for trying to check this out!","DD Initialization error",MB_OK);

		// shutdown any other dd objects and kill window
		DD_Shutdown();
		return(0);
	} // end if

	// now set the coop level to exclusive and set for full screen and mode x
	if (lpdd->SetCooperativeLevel(main_window_handle, DDSCL_ALLOWREBOOT | DDSCL_EXCLUSIVE |
	DDSCL_FULLSCREEN | DDSCL_ALLOWMODEX)!=DD_OK)
	{
		MessageBox(main_window_handle,"Couldn't set 'cooperative level', if you would, please let aloiterer@juno.com know if you recieve this message. Thanks for trying to check this out!","DD Initialization error",MB_OK);
		// shutdown any other dd objects and kill window
		DD_Shutdown();
		return(0);
	} // end if

	// now set the display mode
	if (lpdd->SetDisplayMode(iScreenWidth,iScreenHeight,SCREEN_BPP)!=DD_OK)
	{
		MessageBox(main_window_handle,"Couldn't set video mode, if you would, please let aloiterer@juno.com know if you recieve this message. Thanks for trying to check this out!","DD Initialization error",MB_OK);

		// shutdown any other dd objects and kill window
		DD_Shutdown();
		return(0);
	} // end if
   
	memset(&ddsd,0,sizeof(ddsd));

     // Create the primary surface with 1 back buffer
    ddsd.dwSize = sizeof( ddsd );
    ddsd.dwFlags = DDSD_CAPS | DDSD_BACKBUFFERCOUNT;
    ddsd.ddsCaps.dwCaps = DDSCAPS_PRIMARYSURFACE |
			  DDSCAPS_FLIP |
			  DDSCAPS_COMPLEX;
    ddsd.dwBackBufferCount = 1;
    ddrval = lpdd->CreateSurface( &ddsd, &lpddsprimary, NULL );
    if( ddrval != DD_OK )
    {
		MessageBox(main_window_handle,"Failed to create the PRIMARY surface, if you would, please let aloiterer@juno.com know if you recieve this message. Thanks for trying to check this out!","DD Initialization error",MB_OK);

		DD_Shutdown();
		return(0);
    }

    ddscaps.dwCaps = DDSCAPS_BACKBUFFER;
    ddrval = lpddsprimary->GetAttachedSurface(&ddscaps, &lpddsback);
    if( ddrval != DD_OK )
    {
		MessageBox(main_window_handle,"Failed to create the backbuffer, if you would, please let aloiterer@juno.com know if you recieve this message. Thanks for trying to check this out!","DD Initialization error",MB_OK);

		DD_Shutdown();
		return(0);
    }

	// create surfaces for world tile textures

	ddsd.dwFlags = DDSD_CAPS | DDSD_HEIGHT | DDSD_WIDTH;
	ddsd.ddsCaps.dwCaps = DDSCAPS_OFFSCREENPLAIN;
	ddsd.dwHeight = CELL_X_SIZE;
	ddsd.dwWidth = CELL_Y_SIZE;

	for (index=0; index<NUMTEXTURES; index++)
		if (lpdd->CreateSurface(&ddsd,&lpddstextures[index],NULL)!=DD_OK)
		{
		    sprintf(buffer,"Failed to create surface for texture #%d, please let aloiterer@juno.com know if you recieve this error. Thanks for trying to run this program",index+1);

    		MessageBox(main_window_handle,buffer,"DD Initialization error",MB_OK);

			// shutdown any other dd objects and kill window
			DD_Shutdown();
			return(0);
		} // end if

	// create surfaces for cloud bitmaps

	ddsd.dwFlags = DDSD_CAPS | DDSD_HEIGHT | DDSD_WIDTH;
	ddsd.ddsCaps.dwCaps = DDSCAPS_OFFSCREENPLAIN;
	ddsd.dwHeight = CLOUD_HEIGHT;
	ddsd.dwWidth = CLOUD_WIDTH;

	if (lpdd->CreateSurface(&ddsd,&lpddsenemy,NULL)!=DD_OK)
	{
		MessageBox(main_window_handle,"Couldn't create surface to hold cloud image, if you would, please let aloiterer@juno.com know if you recieve this message. Thanks for trying to check this out!","DD Initialization error",MB_OK);

		// shutdown any other dd objects and kill window
		DD_Shutdown();
		return(0);
	} // end if

	// create surfaces for the player

	ddsd.dwFlags = DDSD_CAPS | DDSD_HEIGHT | DDSD_WIDTH;
	ddsd.ddsCaps.dwCaps = DDSCAPS_OFFSCREENPLAIN;
	ddsd.dwHeight = SHIP_HEIGHT;
	ddsd.dwWidth = SHIP_WIDTH;

	if (lpdd->CreateSurface(&ddsd,&lpddsplayer,NULL)!=DD_OK)
	{
		MessageBox(main_window_handle,"Couldn't create surface to hold the space ship, if you would, please let aloiterer@juno.com know if you recieve this message. Thanks for trying to check this out!","DD Initialization error",MB_OK);

		// shutdown any other dd objects and kill window
		DD_Shutdown();
		return(0);
	} // end if

    // print some stuff

	if (lpddsprimary->GetDC(&hdc) == DD_OK)
	{
		SetBkColor( hdc, RGB( 0, 0, 255 ) );
		SetTextColor( hdc, RGB( 255, 255, 0 ) );
		TextOut( hdc, 0, 0, "Welcome Aboard!", 15 );
		lpddsprimary->ReleaseDC(hdc);
	}   
    else
	    MessageBox(main_window_handle,"Couldn't get the display context for the primary surface, if you would, please let aloiterer@juno.com know if you recieve this message. Thanks for trying to check this out!","Getting HDC of surface error",MB_OK);


	// return success if we got this far
	return(1);

} // end DD_Init

///////////////////////////////////////////////////////////////////////////////

int DD_Shutdown(void)
{
// this function tests for dd components that have been created and releases
// them back to the operating system
int index;

	// test if the dd object exists
	if (lpdd)
	{
		// test if there is a primary surface
		if(lpddsprimary)
		{
			// release the memory and set pointer to NULL
			lpddsprimary->Release();
			lpddsprimary = NULL;
		} // end if

        for (index = 0; index<NUMTEXTURES; index++)
			if(lpddstextures[index])
			{
				// release the memory and set pointer to NULL
				lpddstextures[index]->Release();
				lpddstextures[index] = NULL;
			} // end if

		if(lpddsplayer)
		{
			// release the memory and set pointer to NULL
			lpddsplayer->Release();
			lpddsplayer = NULL;
		} // end if

		if(lpddsenemy)
		{
			// release the memory and set pointer to NULL
			lpddsenemy->Release();
			lpddsenemy = NULL;
		} // end if

		// now release the dd object itself
		lpdd->Release();
		lpdd = NULL;

		// return success
		return(1);

	} // end if
	else
		return(0);

} // end DD_Shutdown

// This routine will restore surfaces that have been lost...
// It should only be called after you have switched to another task,
// and returned to it...but I put if(game_is_running) just in case

void Restore_Surfaces(void)
{
// this function tests if surfaces have been lost and restores the memory
// - THOUGH not the contents of them...

int index;


    if (game_is_running)
		return;

    if ((lpddsprimary->IsLost())==DDERR_SURFACELOST)
		lpddsprimary->Restore();
 
    for (index = 0; index<NUMTEXTURES; index++)
        if ((lpddstextures[index]->IsLost())==DDERR_SURFACELOST)
		    lpddstextures[index]->Restore();

    if ((lpddsenemy->IsLost())==DDERR_SURFACELOST)
    	lpddsenemy->Restore();

	if (music_playing)
		Continue_Track();
} // end Restore_Surfaces

////////////////////////////////////////////////////////////////

void Load_Bitmaps_Into_DDSurfaces(void)
{
	HBITMAP hbm;
    HRESULT smelly;
    int index;
    char buffer[80];

   // Load our bitmap resource.GetModuleHandle(NULL)LR_CREATEDIBSECTION

	for (index = 0; index<NUMTEXTURES; index++)
	{
		hbm = (HBITMAP)LoadImage(NULL, szBitmap[index], IMAGE_BITMAP, CELL_X_SIZE, CELL_Y_SIZE, LR_LOADFROMFILE);

		smelly = DDCopyBitmap(lpddstextures[index], hbm, 0, 0,   CELL_X_SIZE, CELL_Y_SIZE);
        
		if (smelly!=DD_OK)
		{
			sprintf(buffer,"couldn't load bitmap TILE%d.BMP, please check the directory tiled3.exe is in, if the file isn't there, download tiled3.zip again, if it is, contact aloiterer@juno.com, thank you!",index+1);

		    MessageBox(main_window_handle,buffer,"Bitmap loading error",MB_OK);
            PostQuitMessage(0);

        }
	} // end for index

    hbm = (HBITMAP)LoadImage(NULL, szEnemy, IMAGE_BITMAP, CLOUD_WIDTH, CLOUD_HEIGHT, LR_LOADFROMFILE);
    smelly = DDCopyBitmap(lpddsenemy, hbm, 0, 0,   CLOUD_WIDTH, CLOUD_HEIGHT);

	if (smelly!=DD_OK)
	{
		MessageBox(main_window_handle,"Couldn't load ENEM1.BMP, if its not in the directory w/ this executable, redownload tiled3.zip, if it is contact me - aloiterer@juno.com, thanks...","Bitmap loading error",MB_OK);
        PostQuitMessage(0);

    }

	DDSetColorKey(lpddsenemy, RGB(0,0,0));
} // end Load_Bitmaps_Into_DDSurfaces

////////////////////////////////////////////////////////////////

void Reload_Bitmaps(void)
{

	if (game_is_running)
		return;

	Load_Bitmaps_Into_DDSurfaces();
} // end Reload_Bitmaps

// HERE ARE OUR GAME CONSOLE FUNCTIONS ///////////////////////////////////////////////////////

// Game_Init opens the music file, loads the bmps into direct draw 
// surfaces (offscreen plain)
// and also sets the source color keys for 
// bitmaps I want to Bltfast transparently...

void Game_Init(void)
{
// do any initialization here

    int index,x_index,y_index;
	HRESULT smelly;
	HBITMAP sprite_data;
 
    Open_Music(); // calls function which in turn calls makes an
	              // mciSendString call to load PASSPORT.MID from disk

	// initializes sine and cosine lookup tables

	Init_Rotation_and_Movement_Tables();

	// NOTE: since I like to use 2 different sets of local vertices
	// for my polygons, and render using vertices_local_rotated values
	// to get the final position... I have to initialize both
	// vertices_local_original and vertices_local_rotated here

	memset(&ship,0,sizeof(polygon));

    ship.vertices_local_original[0].x =
		ship.vertices_local_rotated[0].x = (float)-(SHIP_WIDTH/2);
    ship.vertices_local_original[0].y =	
		ship.vertices_local_rotated[0].y = (float)-(SHIP_HEIGHT/2);
    ship.vertices_local_original[1].x =	
		ship.vertices_local_rotated[1].x = (float)-(SHIP_WIDTH/2);
    ship.vertices_local_original[1].y =	
		ship.vertices_local_rotated[1].y = (float)(SHIP_HEIGHT/2);
    ship.vertices_local_original[2].x =	
		ship.vertices_local_rotated[2].x = (float)(SHIP_WIDTH/2);
    ship.vertices_local_original[2].y =	
		ship.vertices_local_rotated[2].y = (float)(SHIP_HEIGHT/2);
    ship.vertices_local_original[3].x =	
		ship.vertices_local_rotated[3].x = (float)(SHIP_WIDTH/2);
    ship.vertices_local_original[3].y =	
		ship.vertices_local_rotated[3].y = (float)-(SHIP_HEIGHT/2);

	// I position screen polygons based on their centers - not their
	// upper left hand corners like I do w/ plain old 2d sprites

	ship.x = (float)320;
	ship.y = (float)240;

	// load bitmaps that will go into surfaces

	Load_Bitmaps_Into_DDSurfaces();

	// init regular cloud positions and x-wise movement magnitude

	for (index = 0; index<NUMCLOUDS; index++)
	{
         cloud[index].x = (float)(rand()%(X_BOUND-130)) + CLOUD_WIDTH;
		 cloud[index].y = (float)(rand()%(Y_BOUND-130)) + CLOUD_HEIGHT;
		 cloud[index].magnitude_per_second = (rand()%5) * 20 + 20;
    } // end for index

	// ,load the frames of player 

	// EXCELLENT PART, extracting RGB16s from images for fast 16-bit
	// pixel plotting

	sprite_data = (HBITMAP)LoadImage(NULL, szPlayer1, IMAGE_BITMAP, SHIP_WIDTH, SHIP_HEIGHT, LR_LOADFROMFILE);
    smelly = DDCopyBitmap(lpddsplayer, sprite_data, 0, 0,   SHIP_WIDTH, SHIP_HEIGHT);

	if (smelly!=DD_OK)
	{
		MessageBox(main_window_handle,"couldn't load bitmap PLAY1.BMP, please check the directory tiled3.exe is in, if the file isn't there, download tiled3.zip again, if it is, contact aloiterer@juno.com, thank you!","Bitmap loading error",MB_OK);
        PostQuitMessage(0);

    }

    // lock the lpddsplayer surface, and use double_buffer as a word
	// pointer, for ease of access

	memset(&ddsd,0,sizeof(ddsd));
	ddsd.dwSize = sizeof(ddsd);
	lpddsplayer->Lock(NULL,&ddsd,DDLOCK_SURFACEMEMORYPTR,NULL);
	double_buffer = (unsigned char *)ddsd.lpSurface;

	// here's the pitch acquisition, I think I redundantly get this value
	// in PlotPixel

    lpitch = ddsd.lPitch;
    
	// loop across surface, storing values I'll use to draw with
	// in a WORD buffer called SpriteRGB16

	for (y_index=0; y_index<SHIP_WIDTH; y_index++)
	{
	    for (x_index=0; x_index<SHIP_WIDTH; x_index++)
        {
            SpriteRGB16[x_index + (y_index*SHIP_WIDTH)] = 	*((WORD *)(double_buffer + y_index*lpitch + (x_index<<1)));
		} // end for x_index
	} // end for y_index

	// Unlock the buffer!!

	lpddsplayer->Unlock(NULL);

	// Don't need this hbitmap anymore

    DeleteObject(sprite_data);

} // end Game_Init

/////////////////////////////////////////////////////////////////////////////////////////////

void Game_Shutdown(void)
{
// cleanup and release all resources here
    Stop_Music();
    Close_Music();

} // end Game_Shutdown

/////////////////////////////////////////////////////////////////////////////////////////////

// this looks pretty straight forward from the top, but it
// gets a little bit murky when I start trying to do as follows:
//
//  - I wanted to do one of those deals where the "player" is 
// centered on the screen until the viewport reaches the edges of
// the screen, then I wanted him or her ; ) to roam freely...
// when the player moves back toward center, in a way that should
// move the viewport, well... this does that... just not that elegantly.

void Game_Main(void)
{
// process your game logic and draw next frame

// right now, it's just a cheesy clipping tile engine!
 
    HDC hdc;
	static int px = iScreenWidth, // upperleft corner of screen
	           py = iScreenHeight; 
	int sprite_mid_x, sprite_mid_y;
    int centering_factor_x,
		centering_factor_y;
    static float magnitude_per_second = 50.0f;
	float angles_per_second = 200.0f-(float)(magnitude_per_second/4.0f);
	float anglespeed = 0.0f;
	float movementspeed = 0.0f;
    static float angle = 0.0f;
    char buffer[80];
 
////////////////////////////////////////////////////////////////////////////
/// dhonn lushine's magical high resolution timer deal... (dhonn@usa.net)
/// 
/// usually, we move x amount of pixels per frame when we make stuff...
/// and so the movement per second is determined by the framerate
///
/// WITH the "timing paradigm" used here however, we dictate how far per
/// second we'd like to travel, how many degrees we'd like to turn, etc...
///
/// and then the frame-rate determines the RESOLUTION w/ which we see this
/// time pass...
///
/// get's choppy on slow machines, but this demo goes around 15 fps on my 
/// 486 - just keep in mind I'm tracking positions w/ floats, so I don't
/// lose any fractional additions to my ship's x and y positions.
///
/// this timing paradigm is used in X-wing, id software creations,
/// and many other fine games
    do 
	{
        // get the time till its not equal to start frame time, it should,
        // never loop at the rate of this timer, but if it does then
        // it will cap to the frame rate to the rate of this timer which
        // is 1193180 hz
	    start_time = ftime();

    } while(start_time==end_time);

	// the total time it took to put together the last frame and
	// get back here...

	frametime = start_time-end_time;
    end_time = start_time;

	// slow-poke rules are in effect... if your machine is running my
	// demo at less than 5 frames per second - aside from telling you
	// that I stink, this will keep my program from becoming unbelievably
	// choppy. ACTUALLY - I should set this value to 0.1 (representing
	// 10 fps)

	if (frametime>(double)0.2)
	    frametime = 0.2;
///////////////////////////////////////////////////////////////////////////

// we've got what we need from dhonn's timer - which will be used in
// Wind_Effects_All() and Translate_Other_Objects(), as well as this 
// routine
///////////////////////////////////////////////////////////////////////////

	// store the framerate for the last frame in the char buffer, which
	// I'll print below

	sprintf(buffer,"last fps rate: %4d ",(int)((double)1/frametime));

	// first things first - are we pausing?

	if (KEY_DOWN('P'))
	{
		while (KEY_DOWN('P'));
	    paused = !paused;

		if (!paused)
		{
			end_time = ftime();
		}
    }

	// not paused, do some stuff

	if (paused)
	{
		while ((lpddsprimary->GetDC(&hdc)) != DD_OK);

		SetBkColor( hdc, RGB( 0, 0, 255 ) );
		SetTextColor( hdc, RGB( 255, 255, 0 ) );
		TextOut( hdc, 230, 230, "Game Paused, press P to continue", 32 );
 		
		while ((lpddsprimary->ReleaseDC(hdc)) != DD_OK);

	}
	else
	{
		if (KEY_DOWN(VK_ESCAPE))
		{
			PostMessage(main_window_handle,WM_QUIT,0,0); // this is how you exit you game
		}

		// want to change the speed of the ship?

		if (KEY_DOWN(VK_DOWN))
		{
			if ((magnitude_per_second-=5.0f)<10.0f) 
			{
			    magnitude_per_second = 10.0f;
			}
		}
        else
		if (KEY_DOWN(VK_UP))
		{
			if ((magnitude_per_second+=5.0f)>300.0f) 
			{
			    magnitude_per_second = 300.0f;
			}
		}
        else
			if ((magnitude_per_second-=2.0f)<10.0f)
			    magnitude_per_second = 10.0f;

////////////////////////////////////////////////////////////////////////////

        // here is where we use frametime to get the amount of
		// of translation/rotation we can move this frame

        movementspeed = ((float)magnitude_per_second*frametime);
        anglespeed = ((float)angles_per_second*frametime);

		// which way is player moving and/or rotating

		if (KEY_DOWN(VK_RIGHT))
		{
			angle = angle - anglespeed;
			if (angle < 0.0f) 
				angle=angle+360.0f;

			Rotate_Ship((polygon_ptr)&ship, (int)angle);
		}

		if (KEY_DOWN(VK_LEFT))
		{

			angle = angle + anglespeed;
			if (angle >= 360.0f) 
				angle -= 360.0f;

			Rotate_Ship((polygon_ptr)&ship, (int)angle);
		}

		Translate_Other_Objects();

		Wind_Effects_All();

		// also move ship by our own power

		ship.x += (float)cos(angle*3.14159/180)*movementspeed;
		ship.y += (float)sin(angle*3.14159/180)*movementspeed;

		// test boundaries

		if (ship.x < 0.0f) ship.x = (float)0.0;
		if (ship.x > (float)(X_BOUND)) ship.x = (float)(X_BOUND);

		if (ship.y < (float)0) ship.y = (float)0.0;
		if (ship.y > (float)(Y_BOUND)) ship.y = (float)Y_BOUND;

		// would you like some tunage?

		if (KEY_DOWN('M'))
			Start_Or_Restart_Track();

		if (KEY_DOWN('N'))
			Stop_Music();

	////////////////////////////////////////////////////////////////////////

	/// I like those kind of scrollers that center the player in the viewport
	/// This part of Game_Main() takes care of that, though there are 
	/// probably neater ways

		// this is where I check to see if the player is roaming where the
		// "viewport" cannot -> at least not with the player in the center of
		// it...
    
		centering_factor_x = iHalfWidth;
		centering_factor_y = iHalfHeight;

		px = (int)ship.x + centering_factor_x;
  
		if (px >= X_BOUND) px = X_BOUND-1;
		if (px < iScreenWidth) px = iScreenWidth;

		py = (int)ship.y + centering_factor_y;

		if (py >= Y_BOUND) py = Y_BOUND-1;
		if (py < iScreenHeight) py = iScreenHeight;

		// render the ground..

		Render_Tiled_Background(px,py);

		// draws THEM...

		Draw_Other_Objects(px,py);

	///////////////////////////////////////////////////////////////////////

		// This next section takes the players world coordinates, and
		// converts them into screen coordinates...

		sprite_mid_x = (int)ship.x-(px-iScreenWidth);
		sprite_mid_y = (int)ship.y-(py-iScreenHeight);

// FINALLY - draw the polygon ship!
		// LOCK the backbuffer surface so we can 
		// draw directly to it

		memset(&ddsd,0,sizeof(ddsd));
		ddsd.dwSize = sizeof(ddsd);
		
		while((lpddsback->Lock(NULL,&ddsd,DDLOCK_SURFACEMEMORYPTR,NULL)) != DD_OK);

		double_buffer = (unsigned char *)ddsd.lpSurface;

		// here's the pitch acquisition, I think I redundantly get this value
		// in PlotPixel

		lpitch = ddsd.lPitch;
    

		Draw_Ship((polygon_ptr)&ship,sprite_mid_x,sprite_mid_y);

		// don't forget to unlock the surface!

	    while((lpddsback->Unlock(NULL))!=DD_OK);

		// print a message on the backbuffer

		while ((lpddsback->GetDC(&hdc)) != DD_OK);

		SetBkColor( hdc, RGB( 0, 0, 255 ) );
		SetTextColor( hdc, RGB( 255, 255, 0 ) );
		TextOut( hdc, 0, 0, "Keys: LEFT, RIGHT, UP, DOWN, .. and ESC... OH - and M Music (on), N (off).", 74 );
        TextOut( hdc, 0, iScreenHeight-25, buffer, 20 ); 
		
		while ((lpddsback->ReleaseDC(hdc)) != DD_OK);

		// flip back buffer to primary buffer
		lpddsprimary->Flip(NULL,DDFLIP_WAIT);

	// return and let windows have some time - NO WAY, since I'm not checking
	// for "lost" surfaces, I don't let you switch to other applications while
	// this is running, we'll cros that bridge when we come to it...
    } // end if NOT paused

} // end Game_Main

void Wind_Effects_All(void)
{
	int index;
    float movement_this_frame;
	static float wind_per_sec = 0.0f;
	static int wind_timer = 5;
    float fps;

    movement_this_frame = (float)((float)wind_per_sec*frametime);

	for (index = 0; index<NUMCLOUDS; index++)
	{
		if (wind_per_sec)
		{
			cloud[index].y += movement_this_frame;

			if (cloud[index].y > (float)(Y_BOUND))
				cloud[index].y = (float)-CLOUD_HEIGHT;
			if(cloud[index].y < (float)-CLOUD_HEIGHT) 
				cloud[index].y = (float)Y_BOUND;
        }

	} // end for cloud

	// I test the ship against the bounds when I get back
	// to Game_Main(), so I don't need to check it here

	if ((int)wind_per_sec)
	    ship.y+=(float)((float)wind_per_sec*frametime);

	if ((--wind_timer) == 0)
	{
		fps = (float)((double)1.0/frametime);
		wind_per_sec = (float)((rand()%5) - 2)*fps;
		wind_timer = (int)((rand()%5)*fps) + (int)(fps);
	}
} // end for Wind_Effects_All

		
// this just moves the "normal" clouds..

void Translate_Other_Objects(void)
{
	int index;
    float movement_this_frame;

	for (index = 0; index<NUMCLOUDS; index++)
	{
	    movement_this_frame = ((float)cloud[index].magnitude_per_second*frametime);	

		cloud[index].x += movement_this_frame;

		if ((int)cloud[index].x >= (X_BOUND))
			cloud[index].x = (float)-CLOUD_WIDTH+1;
		if((int)cloud[index].x <= -CLOUD_WIDTH) 
            cloud[index].x = (float)X_BOUND-1;

	} // end for cloud

} // end for Translate_Other_Objects

// this just draws the "normal" clouds

// this routine does clipping, in that it resets the values
// in rcRect, which are used in the call to BltFast...

// Again, not the neatest way

void Draw_Other_Objects(int right, int bottom)
{
	LPDIRECTDRAWSURFACE pdds; // an alias we'll use to the
	                          // tile to be used in BltFast!
    RECT rcRect;        // used by Bltfast, to determine what
	                    // part of our "tiles/sprites" are drawn
	HRESULT ddrval;
	int actual_screen_x, // these are the actual screen coordinates
		actual_screen_y; // for the upper left hand corner of each sprite
	int left, top;       // the left and top virtual coordinates of the 
	                     // viewport
    int index;

	for (index = 0; index<NUMCLOUDS; index++)
	{
        // start of assuming we won't need to clip

		rcRect.left = 0;
		rcRect.top = 0;
		rcRect.right = CLOUD_WIDTH;
		rcRect.bottom = CLOUD_HEIGHT;

		// the right side and bottom of the viewport are
		// sent, use the screenwidth and height to determine
		// the left and top sides..

		left = right - iScreenWidth;
		top = bottom - iScreenHeight;

		// if at least a sliver of the cloud will be in view... then...

		if (((int)cloud[index].x >= left-CLOUD_WIDTH) &&
			((int)cloud[index].x < right)  &&
			((int)cloud[index].y >= top-CLOUD_HEIGHT) &&
			((int)cloud[index].y < bottom))
		{
            // do we need to clip it to the left side of the viewport?

			if ((int)cloud[index].x < left)
			{
				actual_screen_x = 0;
				rcRect.left = (long)(left - cloud[index].x);
			}
			else // apparently not...
			{
				actual_screen_x = (int)(cloud[index].x - left);

				// how about the right side?

				if ((int)cloud[index].x > (right - CLOUD_WIDTH))
				{
                    rcRect.right = (long)(right - cloud[index].x);
				}
			}

			// do we need to clip it to the top of the viewport?

			if ((int)cloud[index].y < top)
			{
				actual_screen_y = 0;
				rcRect.top = (long)(top-cloud[index].y);
			}
			else // nope...
			{
				actual_screen_y = (int)(cloud[index].y - top);

				// how about the bottom?

				if ((int)cloud[index].y > (bottom - CLOUD_HEIGHT))
				{
					rcRect.bottom = (long)(bottom-cloud[index].y);
				}
			} // end else

			pdds = lpddsenemy;
 
			ddrval = lpddsback->BltFast( actual_screen_x, 
										 actual_screen_y, 
										 pdds, 
										 &rcRect, 
										 DDBLTFAST_WAIT | DDBLTFAST_SRCCOLORKEY);

        } // end if not culled
	} // end for cloud
} // end Draw_Other_Objects

// this routine lays down all the tiles...

void Render_Tiled_Background(int right, int bottom)
{
	LPDIRECTDRAWSURFACE pdds; // an alias we'll use to the
	                          // tile to be used in BltFast!
    RECT rcRect;              /// used for clipping tiles
	HRESULT             ddrval;
	int start_left, start_top;
	int start_right, start_bottom;
    int start_x_cell, start_y_cell;
	int render_it_x,render_it_y;
	int clip_right, clip_bottom;
    int which_cell,cell_x_save;
	int num_x_tiles, num_y_tiles;
	int x_offset, y_offset;
    int orig_left_clip,orig_top_clip;

	// my muddleheaded setup for the tile engine

	start_left = right-iScreenWidth;
	start_top = bottom-iScreenHeight;

	// find the cell of the "map" to inspect (the map is very close
	// to the top of this source, you could have loaded it from a disk
	// if you wished, later versions of this tile engine will do that

	start_x_cell = start_left / CELL_X_SIZE;
	start_y_cell = start_top / CELL_Y_SIZE;

	// since we're tiling left to right, top to bottom, we only
	// *really* need to keep the orig_left_clip, which is the clipping
	// info for the left side of all the rows of tiles we'll be rendering

	orig_left_clip = rcRect.left = start_left%CELL_X_SIZE;
	orig_top_clip = rcRect.top = start_top%CELL_Y_SIZE;

	// you know something, this might be a completely extraneous step..

    start_right = (start_left+CELL_X_SIZE)%CELL_X_SIZE;
	start_bottom = (start_top+CELL_Y_SIZE)%CELL_Y_SIZE;

	start_right = start_left + CELL_X_SIZE;

	if (start_right>=CELL_X_SIZE)
		rcRect.right = CELL_X_SIZE;

	start_bottom = start_top + CELL_Y_SIZE;

	if (start_bottom>=CELL_Y_SIZE)
		rcRect.bottom = CELL_Y_SIZE;

	// get starting texture to draw with

	cell_x_save = start_x_cell + (start_y_cell<<TILE_Y_NUM_SHIFT);

	// we'll save it for calculating the starting tile for each row,
	// because sometimes we'll be drawing more tiles accross than at other
	// times, because the numbers I have chosen can draw 10 tiles to
	// cover the screen, but if you 
	// move horizontally, you'll need 11

	which_cell = cell_x_save;

	x_offset = 0;
	y_offset = 0;

	num_x_tiles = ((iScreenWidth+start_left)/ CELL_X_SIZE) + 1;
	num_y_tiles = ((iScreenHeight+start_top)/ CELL_Y_SIZE) + 1;

// loop thru entire scene left to right, top to bottom

	for (render_it_y = 0; render_it_y<num_y_tiles; render_it_y++)
	{
		x_offset = 0;

		// get original start left clip for each x loop

        rcRect.left = orig_left_clip;

		// assume, for now, that we've got the whole tile
		// to work with horizontally

        rcRect.right = CELL_X_SIZE;

		// draw tiles left to right

		for (render_it_x = 0; render_it_x<num_x_tiles; render_it_x++)
		{
			pdds = lpddstextures[terrain_texture[which_cell]];
             
			ddrval = lpddsback->BltFast( x_offset, y_offset, pdds, &rcRect, DDBLTFAST_WAIT );
 
			// the following will only evaluate to a value other
			// than CELL_X_SIZE once for every row of tiles

			x_offset+=CELL_X_SIZE-rcRect.left;

			++which_cell;

	        rcRect.left = 0;
	        rcRect.right = CELL_X_SIZE;

			// determine where to kick out of the loop
			// or just clip the right hand side of the rect

			clip_right = iScreenWidth - x_offset;

			if ((clip_right)<0)
				break;

			if ((clip_right)<CELL_X_SIZE)
				if ((clip_right)==0)
			    {
				    ++which_cell;
				    break;
			    }
				else
					rcRect.right = clip_right;
			// sync time (optional)
		} // end for render_it_x

		// this will only evaluate to a value other than CELL_Y_SIZE
		// once...

		y_offset+=CELL_Y_SIZE-rcRect.top;

		cell_x_save += TILE_X_NUM;
        which_cell = cell_x_save;

		rcRect.top = 0;
	    rcRect.bottom = CELL_Y_SIZE;

		// clip tiles w/ respect to Y screen boundary

		clip_bottom = iScreenHeight - y_offset;

		if ((clip_bottom)<CELL_Y_SIZE)
			if ((clip_bottom)<=0)
			    break;
		    else
				rcRect.bottom = iScreenHeight-y_offset;

			
    } // end for render_it_y

} // Render_Tiled_Background

// WINDOWS CALLBACK FUNCTION ////////////////////////////////////////////////// 

LRESULT CALLBACK WindowProc(HWND hwnd, UINT msg, WPARAM wparam, LPARAM lparam)
{
// this is the main message handler of the system

HDC hdc; // handle to graphics context
PAINTSTRUCT ps; // used to hold the paint info
WORD fActive;
static BOOL lastwparam = TRUE;

	// what is the message?

	switch(msg)
	{ 
		case WM_CREATE:
		{
			// do windows inits here
			return(0);
		} break;

		// THIS is great, it allows you to alt+tab out of this program
		// without crashing... it rules...

		// thanks to Inedag@aol.com, dhonn@usa.net, and others who
		// all took time out to help a moron like me w/ this

		case WM_ACTIVATEAPP:
		{
			fActive = (BOOL)wparam;           // activation flag 

			if (fActive)
			{
				if (fActive!=lastwparam)
				{
					Restore_Surfaces();
					Reload_Bitmaps();
					game_is_running = TRUE;
				}
			}
			else
			{
				Stop_Music();
				game_is_running = FALSE;
				paused=TRUE;
            }
			lastwparam = fActive;
		}
		break;

		case WM_PAINT:
		{
			// this message occurs when your window needs repainting
			hdc = BeginPaint(hwnd,&ps); 
			EndPaint(hdc,&ps);

			return(0);
		} break;

		case WM_DESTROY:
		{

			// shut down direct draw
			DD_Shutdown();

			// shutdown game
			Game_Shutdown();

			// this message is sent when your window is destroyed
			PostQuitMessage(0);
			return(0);
		} break;

		default:break;

	} // end switch

	// let windows process any messages that we didn't take care of 
	return (DefWindowProc(hwnd, msg, wparam, lparam));

} // end WinProc

////////////////////////////////////////////////////////////////

BOOL CALLBACK DialogProc(HWND hDlg, UINT iMsg, WPARAM wParam, LPARAM lParam)
{
	switch (iMsg)
	{
		case WM_INITDIALOG:             
		    return TRUE;

		/////////////////////////

		case WM_COMMAND:
			switch (LOWORD(wParam))
			{
				case ID640:
					iScreenWidth = 640;
					iScreenHeight = 480;
					iHalfWidth = 320;
					iHalfHeight = 240;
					EndDialog(hDlg, TRUE);
					return TRUE;

				case ID800:
					iScreenWidth = 800;
					iScreenHeight = 600;
					iHalfWidth = 400;
					iHalfHeight = 300;
					EndDialog(hDlg, TRUE);
					return TRUE;

				case IDCANCEL:
					EndDialog(hDlg, FALSE);
					return FALSE;
			} 

		    break;
	}
	return FALSE;
}

// WINMAIN ////////////////////////////////////////////////////////////////////

int WINAPI WinMain( HINSTANCE hinstance,
HINSTANCE hprevinstance,
LPSTR lpcmdline,
int ncmdshow)
{
WNDCLASSEX winclass; // this holds the windows class info
HWND hwnd; // this holds the handle of our new window
MSG msg; // this holds a generic message
BOOL timed;
int do_or_die = 1;
	// first fill in the window class stucture


	// give the user the option of bailing out now...

	winclass.cbSize = sizeof(WNDCLASSEX);
	winclass.style = CS_DBLCLKS | CS_OWNDC | CS_HREDRAW | CS_VREDRAW;
	winclass.lpfnWndProc = WindowProc;
	winclass.cbClsExtra = 0;
	winclass.cbWndExtra = 0;
	winclass.hInstance = hinstance;
	winclass.hIcon = LoadIcon(NULL, IDI_APPLICATION);
	winclass.hIconSm = LoadIcon(NULL, IDI_APPLICATION);
	winclass.hCursor = LoadCursor(NULL, IDC_ARROW);
	winclass.hbrBackground = GetStockObject(BLACK_BRUSH);
	winclass.lpszMenuName = NULL;
	winclass.lpszClassName = WINDOW_CLASS_NAME;

	// register the window class
	if (!RegisterClassEx(&winclass))
		return(0);

	// create the window
	if (!(hwnd = CreateWindowEx(WS_EX_TOPMOST,
								WINDOW_CLASS_NAME, // class
								"Jim's directx tasklist marque", // title
								WS_VISIBLE | WS_POPUP,
								0,0, // x,y
								GetSystemMetrics(SM_CXSCREEN),
								GetSystemMetrics(SM_CYSCREEN), 
								NULL, // parent
								NULL, // menu
								hinstance, // instance
								NULL))) // creation parms
		return(0);

	MessageBeep(0);

	do_or_die =     DialogBox(hinstance, MAKEINTRESOURCE(IDD_OPENINGDLG), hwnd, (DLGPROC)DialogProc);

	if (!do_or_die)
	{
		DestroyWindow(hwnd);
		return(0);
	} // end if abort program

	// hide the mouse cursor
	ShowCursor(0);

	// save the window handle
	main_window_handle = hwnd;

	// initialize direct draw
	if (!DD_Init())
	{
		DestroyWindow(hwnd);
		return(0);
	} // end if

	// initialize game
	Game_Init();

	timed = initftime();
    if (!timed)
	{
		MessageBox(main_window_handle,"Please notify dhonn@usa.net that Jim the loiterer did a poor job trying to install his timer!","Timer installation error!",MB_OK);

		return(0);
	}

	end_time = ftime();

	// enter main event loop
	while(1)
	{
		if (PeekMessage(&msg,NULL,0,0,PM_REMOVE))
		{ 
			// test if this is a quit
			if (msg.message == WM_QUIT)
			break;

			// translate any accelerator keys
			TranslateMessage(&msg);

			// send the message to the window proc
			DispatchMessage(&msg);
		} // end if
		else
		{
			// do asynchronous processing here

			// call main logic module
			if (game_is_running)
				Game_Main();


		} // end else

	} // end while

    // shut down direct draw
	DD_Shutdown();

	// shutdown game
	Game_Shutdown();

	// return to Windows
	return(msg.wParam);

} // end WinMain

