#include "Craft.hpp"
#include <math.h>

const int OURPLANE_MAX_ENERGY = 40;
const int OURPLANE_DEFAULT_LIFE = 3;
const int OURPLANE_DEFAULT_BOMB_COUNT = 2;

SCOurPlane::SCOurPlane()
{
	SetID(SI_OURPLANE);

	SetSurface(&Game->GetSFPlayer()[1]);
	SetShadowSurface(&Game->GetSFPlayerShadow()[1]);
	SetExplodeSurface(&(Game->GetSFPlayerExplode()[0]));
	SetContactSurface(&Game->GetSFPlayerContact()[1]);

	itsEnergy = OURPLANE_MAX_ENERGY;

	itsbInvincible = itsbShoot = itsbDropBomb = FALSE;

	itsBomb = NULL;
	itsBombCount = OURPLANE_DEFAULT_BOMB_COUNT;

	itsFrame = 1;
	itsLife = OURPLANE_DEFAULT_LIFE;
	itsPlayerNumber = 0;
	itsScore = 0;

	itsNormalBulletLevel = 1;
	itsGuidedBulletLevel = 0;
	itsSideBulletLevel = 0;
	itsCurrentBullet = SI_OURNORMALBULLET;
};

const int OURPLANE_INITIAL_X[2] = { 310, 110 };
const int OURPLANE_INITIAL_Y = 400;

void SCOurPlane::Create(int pPlayerNumber)
{
	SetX(OURPLANE_INITIAL_X[pPlayerNumber]);
	SetY(OURPLANE_INITIAL_Y);

	SetSurface(&Game->GetSFPlayer()[pPlayerNumber * 3 + 1]);

	itsPlayerNumber = pPlayerNumber;
};

BOOL SCOurPlane::CheckBound()
{
	if (itsState == PS_DEAD) return FALSE;

	if (GetX() < 0) SetX(0);
	if (GetX() > STATUS_X - GetWidth() - 15) SetX(STATUS_X - GetWidth() - 15);

	if (GetY() < CLIP_RECT.top) SetY(CLIP_RECT.top);
	if (GetY() > CLIP_RECT.bottom - GetHeight()) SetY(CLIP_RECT.bottom - GetHeight());

	return TRUE;
};

void SCOurPlane::Damage(int pDamage)
{
	if (itsbInvincible == TRUE || itsState == PS_BOMB || itsState == PS_DEAD) return;

	itsEnergy -= pDamage;
	itsState = PS_CONTACT;

	Game->OurHitBuffer()->Play();

	if (itsEnergy < 0) 
	{
		itsState = PS_BOMB;
		itsBombTime = 0;
		SetSurface(GetExplodeSurface());
		Game->SoundBuffer()[SB_BOMBEXPLODE].Play();
	};
};

void SCOurPlane::Draw()
{
	if (itsState == PS_DEAD) return;
	if (itsbInvincible == TRUE)
		if (Game->GetStage()->itsFrameCount % 3 == 0) return;

	switch (itsState)
	{
	case PS_NORMAL :
		GetSurface()->DrawTransFast(GetX(), GetY(), GetBack());
		break;
	case PS_CONTACT :
		GetContactSurface()->DrawTransFast(GetX(), GetY(), GetBack());
		break;
	case PS_BOMB :
		GetSurface()->Draw(GetX() - GetWidth(), GetY() - GetHeight(), GetBack());
		GetSurface()->DrawHFlip(GetX(), GetY() - GetHeight(), GetBack());
		GetSurface()->DrawVFlip(GetX() - GetWidth(), GetY(), GetBack());
		GetSurface()->DrawHVFlip(GetX(), GetY(), GetBack());
		break;
	};
};

void SCOurPlane::DrawShadow()
{
	if (itsState == PS_DEAD) return;
	if (itsbInvincible == TRUE)
		if (Game->GetStage()->itsFrameCount % 3 == 0) return;

	switch (itsState)
	{
	case PS_NORMAL :
	case PS_CONTACT :
		SCSprite::DrawShadow();
		break;
	};
};

const int energyColor[2][10] = {
	{ 84, 191, 190, 189, 188, 187, 186, 185, 184, 183 },
	{ 60, 59, 199, 198, 197, 196, 195, 194, 193, 192}
};

const int OURPLANE_MAX_BOMB = 5;

const int ENERGY_X = 25;
const int ENERGY_WIDTH = 20;
const int ENERGY_Y[2] = { 20, 320 };
const int ENERGY_HEIGHT = 140;

const int BOMB_X = 60;
const int BOMB_Y[2] = { 20, 320 };

const int LEVEL_X = 95;
const int LEVEL_WIDTH = 8;
const int LEVEL_Y[6] = { 35, 85, 140, 340, 392, 442 };
const int LEVEL_HEIGHT = 20;
const int LEVEL_BOX_X = 90;
const int LEVEL_BOX_WIDTH = 38;
const int LEVEL_BOX_Y[6] = { 30, 80, 135, 335, 387, 437 };
const int LEVEL_BOX_HEIGHT = 30;

const int SCORE_X = 35;
const int SCORE_Y[2] = { 195, 260 };

void SCOurPlane::DrawStatus()
{
	if (itsState == PS_DEAD) return;

	int i;

	for (i = 0; i < itsEnergy; i++)
		GetBack()->FillRect(
			STATUS_X + ENERGY_X,
			ENERGY_Y[itsPlayerNumber] + ENERGY_HEIGHT - (i + 1) * ENERGY_HEIGHT / OURPLANE_MAX_ENERGY,
			STATUS_X + ENERGY_X + ENERGY_WIDTH,
			ENERGY_Y[itsPlayerNumber] + ENERGY_HEIGHT - i * ENERGY_HEIGHT / OURPLANE_MAX_ENERGY,
			energyColor[itsPlayerNumber][i / 4]
		);

	for (i = 0; i < itsBombCount; i++)
		Game->GetSFBomb()->DrawFast(
			STATUS_X + BOMB_X,
			BOMB_Y[itsPlayerNumber] + i * ENERGY_HEIGHT / OURPLANE_MAX_BOMB,
			GetBack()
		);

	for (i = 0; i < itsNormalBulletLevel; i++)
		GetBack()->FillRect(
			STATUS_X + LEVEL_X + i * (LEVEL_WIDTH + 2),
			LEVEL_Y[itsPlayerNumber * 3 + 0],
			STATUS_X + LEVEL_X + i * (LEVEL_WIDTH + 2) + LEVEL_WIDTH,
			LEVEL_Y[itsPlayerNumber * 3 + 0] + LEVEL_HEIGHT,
			RED_COLOR
		);

	for (i = 0; i < itsGuidedBulletLevel; i++)
		GetBack()->FillRect(
			STATUS_X + LEVEL_X + i * (LEVEL_WIDTH + 2),
			LEVEL_Y[itsPlayerNumber * 3 + 1],
			STATUS_X + LEVEL_X + i * (LEVEL_WIDTH + 2) + LEVEL_WIDTH,
			LEVEL_Y[itsPlayerNumber * 3 + 1] + LEVEL_HEIGHT,
			RED_COLOR
		);

	for (i = 0; i < itsSideBulletLevel; i++)
		GetBack()->FillRect(
			STATUS_X + LEVEL_X + i * (LEVEL_WIDTH + 2),
			LEVEL_Y[itsPlayerNumber * 3 + 2],
			STATUS_X + LEVEL_X + i * (LEVEL_WIDTH + 2) + LEVEL_WIDTH,
			LEVEL_Y[itsPlayerNumber * 3 + 2] + LEVEL_HEIGHT,
			RED_COLOR
		);

	int k = 0;

	switch (itsCurrentBullet)
	{
	case SI_OURNORMALBULLET : k = 0; break;
	case SI_OURGUIDEDBULLET : k = 1; break;
	};

	GetBack()->Lock();

	GetBack()->Rect(
		STATUS_X + LEVEL_BOX_X,
		LEVEL_BOX_Y[itsPlayerNumber * 3 + k],
		STATUS_X + LEVEL_BOX_X + LEVEL_BOX_WIDTH,
		LEVEL_BOX_Y[itsPlayerNumber * 3 + k] + LEVEL_BOX_HEIGHT,
		WHITE_COLOR
	);

	GetBack()->UnLock();

	GetBack()->GetDC();
	GetBack()->SetFont();

	char scoreStr[6];
	wsprintf(scoreStr, "%6d", itsScore);

	GetBack()->TextXY(
		STATUS_X + SCORE_X,
		SCORE_Y[itsPlayerNumber],
		RGB(0, 255, 0),
		scoreStr
	);

	GetBack()->ReleaseDC();
};

void SCOurPlane::DropBomb()
{
	if (itsState == PS_DEAD) return;

	itsbDropBomb = FALSE;

	if (itsBomb != NULL || itsBombCount <= 0) return;

	itsBomb = new SCBomb(this);
	itsBombCount--;
};

const int COIN_SCORE = 10;

void SCOurPlane::GainCoin()
{
	if (itsState == PS_DEAD) return;

	Game->ItemBuffer()->Play();

	itsScore += COIN_SCORE;
};

const int ITEM_SCORE = 5;
const int OURPLANE_ENERGY_ITEM = 10;
const int OURPLANE_MAX_NORMAL_BULLET_LEVEL = 3;
const int OURPLANE_MAX_GUIDED_BULLET_LEVEL = 2;
const int OURPLANE_MAX_SIDE_BULLET_LEVEL = 1;

void SCOurPlane::GainItem(ITEMID pItemID)
{
	if (itsState == PS_DEAD || itsState == PS_BOMB) return;

	Game->ItemBuffer()->Play();
	itsScore += ITEM_SCORE;

	switch (pItemID)
	{
	case II_ENERGY :
		itsEnergy += OURPLANE_ENERGY_ITEM;
		if (itsEnergy > OURPLANE_MAX_ENERGY) itsEnergy = OURPLANE_MAX_ENERGY;
		break;
	case II_BOMB :
		increase(&itsBombCount, OURPLANE_MAX_BOMB, FALSE);
		break;
	case II_SPEED :
		break;
	case II_SIDEBULLET :
		increase(&itsSideBulletLevel, OURPLANE_MAX_SIDE_BULLET_LEVEL, FALSE);
		break;
	case II_NORMALBULLET :
		increase(&itsNormalBulletLevel, OURPLANE_MAX_NORMAL_BULLET_LEVEL, FALSE);
		break;
	case II_GUIDEDBULLET :
		increase(&itsGuidedBulletLevel, OURPLANE_MAX_GUIDED_BULLET_LEVEL, FALSE);
		break;
	};
};

const int OURPLANE_COLLISION_DAMAGE = 5;

BOOL SCOurPlane::InterAct()
{
	if (itsState == PS_DEAD) return FALSE;
	if (itsState == PS_BOMB) return TRUE;

	SCObject* Node = NULL;
	SCObject* Save = NULL;

	for (Node = Game->GetEnemyList()->GetFirst(); Node != Game->GetEnemyList()->GetLink(); Node = Save)
	{
		Save = Node->GetNext();

		if (CheckHit((SCBitmap *)Node) == TRUE)
		{
			if (((SCPlane *)Node)->GetState() == PS_BOMB) continue;
			((SCEnemyPlane *)Node)->Damage(this, OURPLANE_COLLISION_DAMAGE);
			Damage(OURPLANE_COLLISION_DAMAGE);
		};
	};

	for (Node = Game->GetEnemyBulletList()->GetFirst(); Node != Game->GetEnemyBulletList()->GetLink(); Node = Save)
	{
		Save = Node->GetNext();

		if (CheckHit((SCBitmap *)Node) == TRUE)
		{
			Damage(((SCBullet *)Node)->GetDamage());
			Game->GetEnemyBulletList()->Delete(Node);
		};
	};

	return TRUE;
};

void SCOurPlane::Move()
{
	if (itsState == PS_DEAD) return;

	SCSprite::Move();

	CheckBound();
	SetVelX(0), SetVelY(0);
};

BOOL SCOurPlane::NewLife()
{
	return FALSE;
};

void SCOurPlane::Shoot()
{
	if (itsState == PS_DEAD) return;

	itsbShoot = FALSE;
	
	switch (itsCurrentBullet)
	{
	case SI_OURNORMALBULLET :
		ShootNormalBullet();
		break;
	case SI_OURGUIDEDBULLET :
		ShootGuidedBullet();
		break;
	};

	ShootSideBullet();
};

const int OURNORMALBULLET_SPEED = 8;
const int OURNORMALBULLET_SURFACE_NUMBER = 0;

void SCOurPlane::ShootNormalBullet()
{
	Game->OurNormalShootBuffer()->Play();

	switch (itsNormalBulletLevel)
	{
		case 0 :
			return;
		case 1 :
			Game->GetOurBulletList()->Add(new SCOurNormalBullet(
				this,
				GetX() + GetWidth() / 2,
				GetY() + GetHeight() / 2,
				0,
				-OURNORMALBULLET_SPEED));
			break;
		case 2 :
			Game->GetOurBulletList()->Add(new SCOurNormalBullet(
				this,
				GetX() + GetWidth() / 2 - Game->GetSFOurBullet()[OURNORMALBULLET_SURFACE_NUMBER].m_PixelWidth * 0.5 - 2,
				GetY() + GetHeight() / 2,
				0,
				-OURNORMALBULLET_SPEED));
			Game->GetOurBulletList()->Add(new SCOurNormalBullet(
				this,
				GetX() + GetWidth() / 2 + Game->GetSFOurBullet()[OURNORMALBULLET_SURFACE_NUMBER].m_PixelWidth * 0.5 + 2,
				GetY() + GetHeight() / 2,
				0,
				-OURNORMALBULLET_SPEED));
			break;
		case 3 :
			Game->GetOurBulletList()->Add(new SCOurNormalBullet(
				this,
				GetX() + GetWidth() / 2 - Game->GetSFOurBullet()[OURNORMALBULLET_SURFACE_NUMBER].m_PixelWidth * 0.5 - 2,
				GetY() + GetHeight() / 2,
				0,
				-OURNORMALBULLET_SPEED));
			Game->GetOurBulletList()->Add(new SCOurNormalBullet(
				this,
				GetX() + GetWidth() / 2 + Game->GetSFOurBullet()[OURNORMALBULLET_SURFACE_NUMBER].m_PixelWidth * 0.5 + 2,
				GetY() + GetHeight() / 2,
				0,
				-OURNORMALBULLET_SPEED));
			Game->GetOurBulletList()->Add(new SCOurNormalBullet(
				this,
				GetX() + GetWidth() / 2 - Game->GetSFOurBullet()[OURNORMALBULLET_SURFACE_NUMBER].m_PixelWidth * 1.5,
				GetY() + GetHeight() / 2,
				-OURNORMALBULLET_SPEED / 2,
				-OURNORMALBULLET_SPEED * sqrt(3) / 2));
			Game->GetOurBulletList()->Add(new SCOurNormalBullet(
				this,
				GetX() + GetWidth() / 2 + Game->GetSFOurBullet()[OURNORMALBULLET_SURFACE_NUMBER].m_PixelWidth * 1.5,
				GetY() + GetHeight() / 2,
				OURNORMALBULLET_SPEED / 2,
				-OURNORMALBULLET_SPEED * sqrt(3) / 2));
			break;
	};
};

void SCOurPlane::ShootGuidedBullet()
{
	Game->OurGuidedShootBuffer()->Play();

	switch (itsGuidedBulletLevel)
	{
	case 0 :
		break;
	case 1 :
		Game->GetOurBulletList()->Add(new SCOurGuidedBullet(this,
			GetX() + GetWidth() / 2,
			GetY() + GetHeight() / 2
			));
		break;
	case 2 :
		Game->GetOurBulletList()->Add(new SCOurGuidedBullet(this,
			GetX() + GetWidth() / 2 - Game->GetSFOurBullet()[OURNORMALBULLET_SURFACE_NUMBER + 1].m_PixelWidth * 0.5 - 2,
			GetY() + GetHeight() / 2
			));
		Game->GetOurBulletList()->Add(new SCOurGuidedBullet(this,
			GetX() + GetWidth() / 2 - Game->GetSFOurBullet()[OURNORMALBULLET_SURFACE_NUMBER + 1].m_PixelWidth * 0.5 + 2,
			GetY() + GetHeight() / 2
			));
		break;
	};
};

void SCOurPlane::ShootSideBullet()
{
	switch (itsSideBulletLevel)
	{
	case 0 :
		return;
	case 1 :
		Game->GetOurBulletList()->Add(new SCOurSideBullet(this, D_LEFT));
		Game->GetOurBulletList()->Add(new SCOurSideBullet(this, D_RIGHT));
		return;
	};
};

const int OURPLANE_BOMB_TIME = 5;

BOOL SCOurPlane::Update()
{
	if (itsState == PS_DEAD) return FALSE;

	if (itsBomb != NULL && itsBomb->Update() == FALSE)
	{
		delete itsBomb;
		itsBomb = NULL;
	};

	switch (itsState)
	{
	case PS_CONTACT :
		itsState = PS_NORMAL;
	case PS_NORMAL :
		SetSurface(&(Game->GetSFPlayer()[itsPlayerNumber * 3 + itsFrame]));
		SetShadowSurface(&(Game->GetSFPlayerShadow()[itsPlayerNumber * 3 + itsFrame]));
		SetContactSurface(&(Game->GetSFPlayerContact()[itsPlayerNumber * 3 + itsFrame]));
		Move();
		if (itsbShoot == TRUE) Shoot();
		if (itsbDropBomb == TRUE) DropBomb();
		break;
	case PS_BOMB :
		itsBombTime++;
		SetSurface(GetExplodeSurface() + itsBombTime);
		if (itsBombTime >= OURPLANE_BOMB_TIME)
		{
			if (NewLife() == FALSE) 
			{
				itsState = PS_DEAD;
				return FALSE;
			}
			else return TRUE;
		};
		break;
	};

	return TRUE;
};