unit SubUnit1;

interface

uses
   Windows;

const
   SOURCE_X_WIDE        = 320;
   SOURCE_Y_WIDE        = 200;
   DESTINATION_X_WIDE   = 320*2;
   DESTINATION_Y_WIDE   = 200*2;
   PCX_SIZE             = SOURCE_X_WIDE * SOURCE_Y_WIDE;
   MAX_PALETTE          = 256;
   MAX_BUMP_OBJECT      = 2;

type
   TBumpParameter = record
      xlight, ylight, wide : integer;
   end;

   EBumpObject = (boExplose, boPulse);

   TBumpObject = class

      constructor Create(x, y, wide : byte; min, max, add, angle_inc : integer; _type : EBumpObject);
      destructor  Destroy; override;

   private
      m_x, m_y, m_wide     : byte;
      m_angle, m_angle_inc : integer;
      m_min, m_max, m_add  : integer;

   public
      m_type               : EBumpObject;
      procedure   IncAngle;
      function    DoAction(var BumpParameter : TBumpParameter) : boolean;
   end;

type
   TColor  = array[1..3] of byte;

   TLogPalette = record
      palVersion    : word;
      palNumEntries : word;
      palPalEntry   : array[0..pred(MAX_PALETTE)] of TPaletteEntry;
   end;

   PScreen = ^TScreen;
   TScreen = array[0..pred(SOURCE_Y_WIDE),0..pred(SOURCE_X_WIDE)] of byte;

   TBump   = class

      constructor Create(file_name : string);
      destructor  Destroy; override;
      procedure   DoAction;

   private
      FLogPalette : TLogPalette;
      FPalette    : hPalette;
      FSourcePage : TScreen;
      FScreenPage : TScreen;
      FBumpObject : array[1..MAX_BUMP_OBJECT] of TBumpObject;

   public
      procedure MakePalette(r1, g1, b1, r2, g2, b2, _begin, _end : byte);
      procedure UpdatePalette(_set : integer);
      procedure ExcuteBumping(BumpParameter : TBumpParameter);
      procedure FlipPage;

   end;

const
   MAX_SECTION = 7;
   color_set : array[1..MAX_SECTION,1..2] of TColor = (
      (( 0, 0, 0),(200,160,180)),
      ((32, 8,16),(212, 60,128)),
      (( 0, 0, 0),(100,212,  0)),
      (( 8,16,32),( 40,128,212)),
      (( 0, 0, 0),(212,176, 96)),
      (( 0, 0, 0),( 96,212,176)),
      (( 0, 0, 0),(176, 96,212))
   );


implementation

uses
   MainUnit;

var
   X_table : array[0..360] of integer;
   Y_table : array[0..360] of integer;


(* Begin Of TBumpObject class *)

constructor TBumpObject.Create;
begin
   m_x         := x;
   m_y         := y;
   m_wide      := wide;
   m_add       := add;
   m_angle     := 0;
   m_angle_inc := angle_inc;
   m_min       := min;
   m_max       := max;
   m_type      := _type;
end;

procedure TBumpObject.IncAngle;
begin
   m_angle := (m_angle + m_angle_inc) mod 360;
end;

function TBumpObject.DoAction(var BumpParameter : TBumpParameter) : boolean;
begin
   doAction := TRUE;
   case m_type of
      boExplose :
      begin
         BumpParameter.xlight := m_x;
         BumpParameter.ylight := m_y;
         BumpParameter.wide   := m_wide;
      end;
      boPulse   :
      begin
         BumpParameter.xlight := 160 + X_table[m_angle];
         BumpParameter.ylight := 100 - Y_table[m_angle];
         BumpParameter.wide   := m_wide;
         inc(m_x,3);
         inc(m_y,2);
         IncAngle;
      end;
   end;
   inc(m_wide,m_add);
   if m_wide > m_max then m_add := -m_add;
   if m_wide < m_min then begin
      if m_type = boExplose then begin
         doAction := FALSE;
      end;
      m_add := -m_add;
   end;
end;

destructor  TBumpObject.Destroy; begin end;

(* End Of TBumpObject class *)

(* Begin Of TBump class *)

constructor TBump.Create(file_name : string);
var
   f : file;
   i : integer;
begin
   AssignFile(f,file_name);
   Reset(f,1);
   for i := pred(SOURCE_Y_WIDE) downto 0 do begin
      BlockRead(f,FSourcePage[i],SOURCE_X_WIDE);
   end;
   CloseFile(f);

   FLogPalette.palVersion    := $300;
   FLogPalette.palNumEntries := MAX_PALETTE;

   UpdatePalette(MAX_SECTION);

   FBumpObject[1] := TBumpObject.Create(0,0,100,50,150,0,5,boPulse);
   FBumpObject[2] := TBumpObject.Create(random(320),random(200),10,10,92,10,3,boExplose);
{
   FBumpObject[3] := TBumpObject.Create(0,0,40,20,40,1,5,boVerticalBand);
   FBumpObject[4] := TBumpObject.Create(0,0,40,20,40,1,5,boHorizontalBand);
}
end;

destructor  TBump.Destroy;
begin
   FBumpObject[1].Free;
   FBumpObject[2].Free;
{
   FBumpObject[3].Free;
   FBumpObject[4].Free;
}
   DeleteObject(FPalette);
end;

procedure   TBump.MakePalette(r1, g1, b1, r2, g2, b2, _begin, _end : byte);
const
   DIVISION = 256;
var
   i                                     : integer;
   red, green, blue, r_inc, g_inc, b_inc : longint;
begin
   r_inc := longint(succ(r2-r1)) * DIVISION div succ(_end-_begin);
   g_inc := longint(succ(g2-g1)) * DIVISION div succ(_end-_begin);
   b_inc := longint(succ(b2-b1)) * DIVISION div succ(_end-_begin);
   red   := longint(r1) * DIVISION;
   green := longint(g1) * DIVISION;
   blue  := longint(b1) * DIVISION;

   with FLogPalette do
   for i := _begin to _end do begin
      palPalEntry[i].peRed   := byte(red   shr 8);
      palPalEntry[i].peGreen := byte(green shr 8);
      palPalEntry[i].peBlue  := byte(blue  shr 8);
      palPalEntry[i].peFlags := 0;
      inc(red  ,r_inc);
      inc(green,g_inc);
      inc(blue ,b_inc);
   end;
end;

procedure   TBump.UpdatePalette(_set : integer);
begin
   if _set in [1..MAX_SECTION] then begin
      makePalette(color_set[_set,1,1],color_set[_set,1,2],color_set[_set,1,3],
                  color_set[_set,2,1],color_set[_set,2,2],color_set[_set,2,3],0,127);
      makePalette(color_set[_set,2,1],color_set[_set,2,2],color_set[_set,2,3],
                  255-color_set[_set,1,1],255-color_set[_set,1,2],255-color_set[_set,1,3],128,255);
   end;
   FPalette := CreatePalette(Windows.TLogPalette((@FLogPalette)^));
end;

procedure   TBump.ExcuteBumping(BumpParameter : TBumpParameter);
var
   x, y, temp3          : Dword;
   xp, yp, temp1, temp2 : integer;
   source, destination  : PScreen;
begin
   source      := PScreen(@FSourcePage);
   destination := PScreen(@FScreenPage);
   x := 1; y := 1;
   for yp := BumpParameter.ylight downto BumpParameter.ylight-(SOURCE_Y_WIDE-3) do begin
      for xp := BumpParameter.xlight downto BumpParameter.xlight-(SOURCE_X_WIDE-3) do begin
         temp1 := BumpParameter.wide - abs(source^[y][x+1] - source^[y][x-1] + xp);
         temp2 := BumpParameter.wide - abs(source^[y+1][x] - source^[y-1][x] + yp);
         if (temp1 > 0) and (temp2 > 0) then begin
            temp3 := word(temp1 * temp2 div (BumpParameter.wide div 2));
            temp3 := destination^[y][x] + temp3;
            if temp3 > $FF then temp3 := $FF;
            destination^[y][x] := byte(temp3);
         end;
         inc(x);
      end;
      inc(y);
      x := 1;
   end;
end;

procedure   TBump.FlipPage;
type
   TBitMapInfo = record
      bmiHeader : TBitmapInfoHeader;
      bmiColors : array[0..pred(MAX_PALETTE)] of TRGBQuad;
   end;
var
   i            : integer;
   temp_palette : hPalette;
   BitmapInfo   : TBitMapInfo;
   DC           : hDC;
begin

   FillChar(BitmapInfo,sizeof(BitmapInfo),0);
   with BitmapInfo.bmiHeader do begin
      biSize        := sizeof(BitmapInfo.bmiHeader);
      biWidth       := SOURCE_X_WIDE;
      biHeight      := SOURCE_Y_WIDE;
      biPlanes      := 1;
      biBitCount    := 8;
      biCompression := BI_RGB;
      biSizeImage   := SOURCE_X_WIDE*SOURCE_Y_WIDE;
   end;

   with BitmapInfo, FLogPalette do
   for i := 0 to 255 do begin
      bmiColors[i].rgbRed   := palPalEntry[i].peRed;
      bmiColors[i].rgbGreen := palPalEntry[i].peGreen;
      bmiColors[i].rgbBlue  := palPalEntry[i].peBlue;
   end;

   DC := Basic.Canvas.Handle;

   temp_palette := SelectPalette(DC,FPalette,FALSE);
   RealizePalette(DC);

   SetDIBitsToDevice(DC,0,0,SOURCE_X_WIDE,SOURCE_Y_WIDE,0,0,0,SOURCE_Y_WIDE,@FScreenPage,Windows.TBitmapInfo((@BitmapInfo)^),DIB_RGB_COLORS);
{
   StretchDIBits(DC,0,0,DESTINATION_X_WIDE,DESTINATION_Y_WIDE,
                    0,0,SOURCE_X_WIDE,SOURCE_Y_WIDE,
                    @FScreenPage,Windows.TBitmapInfo((@BitmapInfo)^),DIB_RGB_COLORS,SRCCOPY);
{
   StretchDIBits(DC,0,0,800,600,
                    0,0,SOURCE_X_WIDE,SOURCE_Y_WIDE,
                    @FScreenPage,Windows.TBitmapInfo((@BitmapInfo)^),DIB_RGB_COLORS,SRCCOPY);
}
   SelectPalette(DC,temp_palette,FALSE);

end;

procedure   TBump.DoAction;
var
   i             : integer;
   BumpParameter : TBumpParameter;
begin
   FillChar(FScreenPage,sizeof(FScreenPage),0);

   for i := 1 to MAX_BUMP_OBJECT do begin
      if FBumpObject[i].doAction(BumpParameter) then begin
         ExcuteBumping(BumpParameter);
      end else begin
         FBumpObject[i].Free;
         FBumpObject[i] := nil;
         FBumpObject[i] := TBumpObject.Create(random(320),random(200),10,10,92,10,3,boExplose);
      end;
   end;

   FlipPage;
end;

(* End Of TBump class *)

var
   i : integer;

Initialization

   for i := 0 to 359 do Y_table[i] := round(cos(i*PI/180)*64);
   for i := 0 to 359 do X_table[i] := round(sin(i*PI/180)*128);

end.
