/*
	8bit & 24bit PCX file viewer for Windows 95
	
	Coder: Kim Seong Wan

	created : 1998. 10. 21
	revised : 1998. 11. 13
*/

#include <windows.h>
#include <stdio.h>
#include <stdlib.h>

#include "resource.h"

struct PCX_HEADER_TYPE {
	char manufacturer;
	char version;
	char encoding;
	char bits_per_pixel;
	short  sx, sy, ex, ey;
	short  hres, vres;
	char pallete[48];
	char reserved;
	char color_planes;
	short  bytes_per_line;
	short  pallete_type;
	char filler[58];
} PCX_HEADER;

struct PCXpalette {
	char r, g, b;
} palette[256];

struct myBMINFO {
	BITMAPINFOHEADER bmiHeader;
	RGBQUAD	bmiColors [1];
} BMInfo;

HWND hWnd;
HCURSOR hCursor;
HBITMAP hBM;
BYTE *Bits;
BYTE *buffer;

unsigned int height, width;
unsigned int sheight, swidth;

int count = 0;
char bIsActive = 0;
BOOL bValideFileLoaded = FALSE;
BOOL bIs24Bit = TRUE;

BOOL InitApplication(HINSTANCE hInstance);
BOOL InitInstance(HINSTANCE hInstance,int nCmdShow);
LRESULT CALLBACK WindowProc(HWND hWnd,UINT message,
                            WPARAM wParam,LPARAM lParam);

void UpdateFrame(void);
void DecodePCX(unsigned char *buffer, int bpl, FILE *fp);
BOOL LoadPCX(char * fname);
void ShowPCX24(struct myBMINFO * lpBM);
void ShowPCX8(struct myBMINFO * lpBM);
BOOL FileOpenDlg(HWND hwnd);
void MakeDIBSection(int width, int height);

int WINAPI WinMain(HINSTANCE hInstance, // handle to current instance 
					HINSTANCE hPrevInstance, // handle to previous instance 
					LPSTR lpCmdLine, // pointer to command line 
					int nCmdShow) // show state of window 
{
    MSG msg;

	if (!InitApplication(hInstance))
		return FALSE;

    if (!InitInstance(hInstance,nCmdShow))
        return FALSE;

/*    
	while(GetMessage(&msg,NULL,0,0))
    {
        TranslateMessage(&msg);
        DispatchMessage(&msg);
    }
*/

	// message loop for game app
    while( 1 )
    {
        if( PeekMessage( &msg, NULL, 0, 0, PM_NOREMOVE ) )
        {
            if( !GetMessage( &msg, NULL, 0, 0 ) )
            {
               	free(buffer);
				return msg.wParam;
            }
            else
			{
                TranslateMessage(&msg);
                DispatchMessage(&msg);
            }
        }
        else if ( bIsActive )
        {
            UpdateFrame();
        }
        else
        {
            WaitMessage();
        }
    }
    
}

BOOL InitApplication(HINSTANCE hInstance)
{
    WNDCLASS wc;

    wc.style            = CS_HREDRAW|CS_VREDRAW;
    wc.lpfnWndProc      = WindowProc;
    wc.cbClsExtra       = 0;
    wc.cbWndExtra       = 0;
    wc.hInstance        = hInstance;
    wc.hIcon            = LoadIcon(hInstance, (char *)IDI_ICON1);
    wc.hCursor          = LoadCursor(NULL, IDC_ARROW);
    wc.hbrBackground    = GetStockObject(WHITE_BRUSH);
    wc.lpszMenuName     = (char *)IDR_MENU1;
    wc.lpszClassName    = "PCX View Class";

    return RegisterClass(&wc);
}

BOOL InitInstance(HINSTANCE hInstance,int nCmdShow)
{
    hWnd = CreateWindow(
        "PCX View Class",          // pointer to registered class name
        "Simple PCX Viewer",          // pointer to window name
        WS_OVERLAPPEDWINDOW,    // window style
        CW_USEDEFAULT,          // horizontal position of window
        CW_USEDEFAULT,          // vertical position of window
        640,//CW_USEDEFAULT,          // window width
        480,//CW_USEDEFAULT,          // window height
        NULL,                   // handle to parent or owner window
        NULL,               // handle to menu or child-window identifier
        hInstance,          // handle to application instance
        NULL);              // pointer to window-creation data

    if (!hWnd) return FALSE;

	hCursor = LoadCursor(hInstance, (char *)IDC_CURSOR1);
	
    ShowWindow(hWnd,nCmdShow);
    UpdateWindow(hWnd);

    return TRUE;
}

LRESULT CALLBACK WindowProc(HWND hWnd,UINT message,
                            WPARAM wParam,LPARAM lParam)
{
    HDC hDC;
    PAINTSTRUCT ps;
	//RECT rect;

    switch(message)
    {
		case WM_CREATE:
			return 0;
		
		case WM_PAINT:
			hDC = BeginPaint(hWnd, &ps);
			bIsActive = 1;
			UpdateFrame();
			EndPaint(hWnd, &ps);
			return 0;
		
		case WM_DESTROY:
			PostQuitMessage(0);
			return 0;
		
		case WM_COMMAND:
		{
			switch (LOWORD(wParam))
			{
				case ID_OPEN:
					FileOpenDlg(hWnd);
					bIsActive = 0;
					return 0;

				case ID_EXIT:
					PostQuitMessage(0);
					return 0;
				case ID_ABOUT:
					MessageBox(hWnd, // handle of owner window 
						"Simple PCX Viewer \n\n (c)1998 Kim Seong Wan", // address of text in message box 
						"About", // address of title of message box 
						MB_OK); // style of message box 
	

					return 0;
					
			}
		}
    }
    
	return DefWindowProc(hWnd,message,wParam,lParam);
}

void PutPixel(int x, int y, char color)
{
	unsigned offset;

	offset = y * swidth * 3 + x * 3;
	*(Bits + offset + 0) = color;
	*(Bits + offset + 1) = color;
	*(Bits + offset + 2) = color;
}

void DrawBox(int x1, int y1, int x2, int y2, char color)
{
	int i, j;

	for ( i = x1; i <= x2; i++ )
		PutPixel(i, y1, color);
	for ( i = x1; i <= x2; i++ )
		PutPixel(i, y2, color);
	for ( j = y1; j <= y2; j++ )
		PutPixel(x1, j, color);
	for ( j = y1; j <= y2; j++ )
		PutPixel(x2, j, color);
}

void UpdateFrame(void)
{
	HDC ScreenDC, Context;
	RECT Dest;
	HBITMAP DefaultBitmap;

	if (bIs24Bit)
		ShowPCX24(&BMInfo);
	else
		ShowPCX8(&BMInfo);

	//PutPixel(320, 240, (char)0xff);
	//DrawBox(300, 200, 340, 280, (char)0xff);

	ScreenDC = GetDC(hWnd);
	GetClientRect(hWnd, &Dest);

	Context = CreateCompatibleDC(0);
	DefaultBitmap = SelectObject(Context, hBM);

	BitBlt(ScreenDC, 0, 0, Dest.right - Dest.left, Dest.bottom - Dest.top, Context, 0, 0, SRCCOPY);

	SelectObject(Context, DefaultBitmap);
	DeleteDC(Context);
	DeleteObject(DefaultBitmap);
	ReleaseDC(hWnd, ScreenDC);
	
	bIsActive = 0;
}

void DecodePCX(unsigned char *buffer, int bpl, FILE *fp)
{
    int i, data, num;

    for(i = 0; i< bpl;) {
        data = fgetc(fp);
        if (data >= 0xc0) {
             num = data & 0x3f;
             data = fgetc(fp);
             while(num--)
                buffer[i++] = data;
        }
        else buffer[i++] = data;
    }
}

BOOL LoadPCX(char * fname)
{
	FILE *fp;
	char AppTitle[80];

	fp = fopen(fname,"rb");
    if( fp == NULL )
    {
		MessageBox(hWnd, // handle of owner window 
			"Error: file Not found!!", // address of text in message box 
			"ERROR MESSAGE", // address of title of message box 
			 MB_OK|MB_ICONERROR); // style of message box 
		
		return FALSE;
	}
        
	fread(&PCX_HEADER, sizeof(PCX_HEADER), 1, fp);

	width = PCX_HEADER.ex - PCX_HEADER.sx + 1;
	height = PCX_HEADER.ey - PCX_HEADER.sy + 1;

/*
        printf("PCX file : %s\n", argv[1]);
        printf("- HEADER INFO -\n");
        printf("  manufacturer: %d\n", PCX_HEADER.manufacturer);
        printf("       version: %d\n", PCX_HEADER.version);
        printf("      encoding: %d\n", PCX_HEADER.encoding);
        printf("bits_per_pixel: %d\n", PCX_HEADER.bits_per_pixel);
        printf("sx, sy, ex, ey: %d, %d, %d, %d\n", PCX_HEADER.sx, PCX_HEADER.sy, PCX_HEADER.ex, PCX_HEADER.ey);
        printf("    hres, vres: %d, %d\n", PCX_HEADER.hres, PCX_HEADER.vres);
        printf("   pallete[48]:\n");
        printf("      reserved:\n");
        printf("  color_planes: %d\n", PCX_HEADER.color_planes);
        printf("bytes_per_line: %d\n", PCX_HEADER.bytes_per_line);
        printf("  pallete_type: %d\n", PCX_HEADER.pallete_type);
        printf("    filler[58]:\n\n");
*/
	if ( (PCX_HEADER.manufacturer != 10) || (PCX_HEADER.version != 5) )
	{
		MessageBox(hWnd, // handle of owner window 
			"Error: Not a PCX file!", // address of text in message box 
			"ERROR MESSAGE", // address of title of message box 
			 MB_OK|MB_ICONERROR); // style of message box 
		
		fclose(fp);
		bValideFileLoaded = FALSE;
		return FALSE;
	}

	MakeDIBSection(width, height);

	if (buffer != NULL) free(buffer);
	buffer = malloc(height * width * 3);
	if (buffer == NULL)
	{
		MessageBox(hWnd, // handle of owner window 
			"Error: Not enough memory!", // address of text in message box 
			"ERROR MESSAGE", // address of title of message box 
			 MB_OK|MB_ICONERROR); // style of message box 
		
		fclose(fp);
	
		return FALSE;
	}

	if ( PCX_HEADER.color_planes == 3 )
	{
		DecodePCX(buffer, width * height * 3, fp);
		bIs24Bit = TRUE;	// 24bit color
	}
	else // color_planes == 1 
	{
		DecodePCX(buffer, width * height, fp);
		fgetc(fp);
		fread(palette, 256 * 3, 1, fp);
		bIs24Bit = FALSE;   // 8bit color
	}
	
	bIsActive = 1;
	bValideFileLoaded = TRUE;
	
	sprintf(AppTitle, "Simple PCX Viewer  [%s (%d x %d %dbit color)]", fname, width, height, PCX_HEADER.color_planes * 8); 

	SetWindowText(hWnd, AppTitle); 
	
	fclose(fp);

	return TRUE;
}

void ShowPCX24(struct myBMINFO * lpBM)
{
	unsigned int i, j;
	unsigned int dest, src;
	unsigned int DIB_bytes_per_line;
	unsigned int lcount;

	if(bValideFileLoaded)
	{
		swidth	= lpBM->bmiHeader.biWidth;
		sheight = -(lpBM->bmiHeader.biHeight);

		lcount = swidth;
		DIB_bytes_per_line = (swidth * 3 + 3)/4 * 4;
		if ( width < swidth ) lcount = width;
		if ( height < sheight ) sheight = height;

		for(j = 0; j < sheight; j++)
		{
			src = j * PCX_HEADER.bytes_per_line * 3;
			dest = j * DIB_bytes_per_line;
			for( i = 0; i < lcount * 3; i += 3)
			{
				*(Bits + dest + 2) = buffer[src];								 //R
				*(Bits + dest + 1) = buffer[src + PCX_HEADER.bytes_per_line];    //G
				*(Bits + dest + 0) = buffer[src + PCX_HEADER.bytes_per_line * 2];//B
				dest += 3;
				src++;
			}
		}
	}
}

void ShowPCX8(struct myBMINFO * lpBM)
{
	unsigned int i, j;
	unsigned int dest, src;
	unsigned int lcount;

	if(bValideFileLoaded)
	{
		swidth	= lpBM->bmiHeader.biWidth;
		sheight = -(lpBM->bmiHeader.biHeight);

		lcount = swidth;
		if ( width < swidth ) lcount = width;
		if ( height < sheight ) sheight = height;

		for(j = 0; j < sheight; j++)
		{
			src = j * PCX_HEADER.bytes_per_line;
			dest = j * swidth * 3;
			for( i = 0; i < lcount * 3; i += 3)
			{
				*(Bits + dest + 2) = palette[ buffer[src] ].r; //R
				*(Bits + dest + 1) = palette[ buffer[src] ].g; //G
				*(Bits + dest + 0) = palette[ buffer[src] ].b; //B
				dest += 3;
				src++;
			}
		}
	}
}

BOOL FileOpenDlg(HWND hwnd)
{ 
	OPENFILENAME ofn;
	RECT Dest;
	char str[] = "PCX Files (*.PCX)\0*.pcx\0";
	char lpDir[1024] = "";
	char lpFile[1024] = "*.pcx";
	char lpTitle[1024] = "";

	ofn.lStructSize			= sizeof(OPENFILENAME);
	ofn.hwndOwner			= hwnd;
	ofn.hInstance			= NULL;
	ofn.lpstrFilter			= str;
	ofn.lpstrCustomFilter	= NULL;
	ofn.nMaxCustFilter		= 40;
	ofn.nFilterIndex		= 0;
	ofn.lpstrFile			= lpFile;//"*.pcx";
	ofn.nMaxFile			= 256;
	ofn.lpstrFileTitle		= lpTitle;
	ofn.nMaxFileTitle		= 1024;
	ofn.lpstrInitialDir		= lpDir;//NULL;
	ofn.lpstrTitle			= "Open PCX";
	ofn.Flags				= 0;
	ofn.nFileOffset			= 0;
	ofn.nFileExtension		= 0;
	ofn.lpstrDefExt			= "pcx";
	ofn.lCustData			= 0;
	ofn.lpfnHook			= NULL;
	ofn.lpTemplateName		= NULL;

	if( !GetOpenFileName(&ofn) )
	{
		UpdateFrame();
		MessageBox(hWnd, // handle of owner window 
			"Error: No File to open!", // address of text in message box 
			"ERROR MESSAGE", // address of title of message box 
			 MB_OK|MB_ICONERROR); // style of message box 
		
		GetClientRect(hWnd, &Dest);
		InvalidateRect(hWnd, &Dest, TRUE);
			
		return FALSE;
	}

	UpdateFrame();
	GetClientRect(hWnd, &Dest);
	InvalidateRect(hWnd, &Dest, TRUE);
	
	return ( LoadPCX(ofn.lpstrFileTitle) );
}

void MakeDIBSection(int width, int height)
{
	HWND ActiveWindow;
	HDC ScreenDC;

	ActiveWindow = GetActiveWindow();
	ScreenDC = GetDC(ActiveWindow);

	BMInfo.bmiHeader.biSize				= sizeof (BITMAPINFOHEADER);
	BMInfo.bmiHeader.biWidth			= width;
	BMInfo.bmiHeader.biHeight			= -height;
	BMInfo.bmiHeader.biPlanes			= 1;
	BMInfo.bmiHeader.biBitCount			= 24;
	BMInfo.bmiHeader.biCompression		= BI_RGB;
	BMInfo.bmiHeader.biSizeImage		= 0;
	BMInfo.bmiHeader.biXPelsPerMeter	= 0;
	BMInfo.bmiHeader.biYPelsPerMeter	= 0;
	BMInfo.bmiHeader.biClrUsed			= 0;
	BMInfo.bmiHeader.biClrImportant		= 0;

	hBM = CreateDIBSection (ScreenDC, (BITMAPINFO *)&BMInfo, DIB_RGB_COLORS, (VOID *)&Bits, NULL, 0);

	ReleaseDC(ActiveWindow, ScreenDC);
}