/////////////////////////////////////////////////////////////////////////////
//
// COpenGLView.cpp : implementation of the COpenGLView class
//
/////////////////////////////////////////////////////////////////////////////

#include "stdafx.h"

//#include "resource.h"       // main symbols

#include "COpenGLView.h"

#include <mmsystem.h> // for MM timers (you'll need WINMM.LIB)

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif



// add support for OpenGL 1.1 if we're using an old header
// These are new PIXELFORMATDESCRIPTOR flags for OpenGL 1.1

#ifndef PFD_GENERIC_ACCELERATED
#define PFD_GENERIC_ACCELERATED		0x00001000
#endif

#ifndef PFD_DEPTH_DONTCARE
#define PFD_DEPTH_DONTCARE			0x20000000
#endif

#define INSTALLABLE_DRIVER_TYPE_MASK  (PFD_GENERIC_ACCELERATED|PFD_GENERIC_FORMAT)


/////////////////////////////////////////////////////////////////////////////

const char* const COpenGLView::_ErrorStrings[]= {
				{"No Error"},					// 0
				{"Unable to get a DC"},			// 1
				{"ChoosePixelFormat failed"},	// 2
				{"SelectPixelFormat failed"},	// 3
				{"wglCreateContext failed"},	// 4
				{"wglMakeCurrent failed"},		// 5
				{"wglDeleteContext failed"},	// 6
				{"SwapBuffers failed"},			// 7

		};

/////////////////////////////////////////////////////////////////////////////
// COpenGLView

IMPLEMENT_DYNCREATE(COpenGLView, CView)

BEGIN_MESSAGE_MAP(COpenGLView, CView)
	//{{AFX_MSG_MAP(COpenGLView)
	ON_WM_CREATE()
	ON_WM_DESTROY()
	ON_WM_ERASEBKGND()
	ON_WM_SIZE()
	// If you don't have an ID_ANIMATE, you can either add one
	// to your menu (Add an Animate command) or comment out the
	// references
#if defined ID_ANIMATE
	ON_COMMAND(ID_ANIMATE, OnAnimate)
	ON_UPDATE_COMMAND_UI(ID_ANIMATE, OnUpdateAnimate)
#else
	#pragma message( "No Animation Accelerator Interface Defined in COpenGLView" )
#endif
	ON_WM_KEYDOWN()
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// COpenGLView construction/destruction

COpenGLView::COpenGLView() 
	: m_StockSceneListIndex(0), m_DefaultTextID(0),
	m_DefaultFlatTextID(0),
	m_SelectedStockScene( eStockSceneCheckerboard ),
	m_bAnimationRunning(FALSE),
	m_PixelFormat(0),m_hRC(0), m_pDC(0),
	m_ErrorString(_ErrorStrings[0]),
	m_deltaX(0), m_deltaY(0), m_deltaZ(0),
	m_accelX(0), m_accelY(0), m_accelZ(0), 
	m_rotdeltaX(0), m_rotdeltaY(0), m_rotdeltaZ(0), 
	m_rotaccelX(0), m_rotaccelY(0), m_rotaccelZ(0),
	m_rotX(0), m_rotY(0), m_rotZ(0),
	m_X(0), m_Y(0), m_Z(0),
	m_MaxVelocity(5.0),	m_MaxRotation(90.0), // in units/sec
	m_ElapsedTime( ~(DWORD)0 ), // initialize to a big number
	m_eCurrentDriver(eGeneric)	// driver for current pixel format

{
	// TODO: add construction code here

}

/////////////////////////////////////////////////////////////////////////////
COpenGLView::~COpenGLView()
{
	// clean up the extensions array
	if ( m_ExtensionPtr )
		{
		m_ExtensionArray.RemoveAll();
		delete[] m_ExtensionPtr;
		}
}

/////////////////////////////////////////////////////////////////////////////
BOOL COpenGLView::PreCreateWindow(CREATESTRUCT& cs) 
{
	// TODO: Add your specialized code here and/or call the base class

	// An OpenGL window must be created with the following flags and must not
    // include CS_PARENTDC for the class style. Refer to SetPixelFormat
    // documentation in the "Comments" section for further information.
    cs.style |= WS_CLIPSIBLINGS | WS_CLIPCHILDREN;

	return CView::PreCreateWindow(cs);
}


/////////////////////////////////////////////////////////////////////////////
// COpenGLView drawing

void COpenGLView::OnDraw(CDC* pDC)
{

	CDocument* pDoc = GetDocument();
	ASSERT_VALID(pDoc);

	if ( m_bAnimationRunning )
		{
		// Get the system time, in milliseconds.
		// The system time is the time elapsed since Windows was started.
		// NOTE: This value wraps about every 50 days, not that I worry
		// about it.

		// The accuracy depends upon the OS that you're running
		// Win95 has accuracy to 1 ms, while WinNT will average about
		// 5 ms or more (fine for our purposes). If you're running NT you can
		// set the accuracy to a finer value. See the documentation.
		// (However, this runs fine under NT as it is...)
		m_ElapsedTime = ::timeGetTime(); // get current time

		// Now invalidate the rect and post the message to repaint.
		// Note that posting merely places the message in the queue.
		// We HAVE to do this before we short circut the animation code
		// since we'd never get called again if we didn't.
		InvalidateRect( 0, FALSE );
		GetParent()->PostMessage(WM_PAINT);

		// this statement will limit the refresh rate of the
		// window to 1000/30 frames/sec (about 33).
		// Comment it out or modify it if you really want
		// faster rates. See if you can tell the difference between a
		// limited animation and an unlimited one. Resize the window
		// and watch the rates change. With a tiny window you can easily get
		// it over 150 frames/sec (with a simple scene).
		if ( ElapsedTimeinMSSinceLastRender() < 30 )
			return;

		// Now do the time-based dynamic calculations for things like
		// accelerations, velocity, rotations, etc...
		CalculateDynamics();

		}

	// Clear out the color & depth buffers
	::glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );

	PreRenderScene();

	::glPushMatrix();
	RenderStockScene();
 	::glPopMatrix();

	::glPushMatrix();
	RenderScene();
	::glPopMatrix();

	// Tell OpenGL to flush its pipeline
	::glFinish();

	// Now Swap the buffers
	if ( FALSE == ::SwapBuffers( m_pDC->GetSafeHdc() ) )
		{
		SetError(7);
		}

	PostRenderScene();

	// the very last thing we do is to save
	// the elapsed time, this is used with the
	// next elapsed time to calculate the
	// elapsed time since a render and the frame rate
	m_previousElapsedTime = m_ElapsedTime;

}

//////////////////////////////////////////////////////////////////////////////
// PostRenderScene
// perform post display processing
//
// The default PostRenderScene places the framerate in the
// view's title. Replace this with your own title if you like.
void COpenGLView::PostRenderScene( void )
{
	// Only update the title every 15 redraws (this is about
	// every 1/2 second)
	static int updateFrame = 15;

	if ( !m_bAnimationRunning || 16 > ++updateFrame )
		return;

	updateFrame = 0;

	char string[256];
	_snprintf( string, 200, "%s ( %d Frames/sec )",
		(const char*)m_WindowTitle, FramesPerSecond() );

	GetParentFrame()->SetWindowText( string );

}

//////////////////////////////////////////////////////////////////////////////
// FramesPerSecond
// fetch frame rate calculations
int COpenGLView::FramesPerSecond( void )
{
	double eTime = ElapsedTimeinMSSinceLastRender();

	if ( 0 == (int)eTime )
		return 0;

	return (int)(1000/(int)eTime);
}



/////////////////////////////////////////////////////////////////////////////
// COpenGLView diagnostics

#ifdef _DEBUG
/////////////////////////////////////////////////////////////////////////////
void COpenGLView::AssertValid() const
{
	CView::AssertValid();
}

/////////////////////////////////////////////////////////////////////////////
void COpenGLView::Dump(CDumpContext& dc) const
{
	CView::Dump(dc);
}

/////////////////////////////////////////////////////////////////////////////
CDocument* COpenGLView::GetDocument() // non-debug version is inline
{
	ASSERT(m_pDocument->IsKindOf(RUNTIME_CLASS(CDocument)));
	return (CDocument*)m_pDocument;
}
#endif //_DEBUG

/////////////////////////////////////////////////////////////////////////////

int COpenGLView::OnCreate(LPCREATESTRUCT lpCreateStruct) 
{
	if (CView::OnCreate(lpCreateStruct) == -1)
		return -1;
	
	// TODO: Add your specialized creation code here
	GetParentFrame()->GetWindowText( m_WindowTitle );

	// get rid of that " - Untitled" stuff
	GetParentFrame()->ModifyStyle(FWS_ADDTOTITLE,0);

	InitializeOpenGL();

	// Now we initialize the animation code

	m_StartTime = ::timeGetTime(); // get time in ms

	// need a previous time if we start off animated
	if ( m_bAnimationRunning )
		{
		m_previousElapsedTime = m_StartTime;
		}

	m_HomeX = m_X;	// initialize dynamic starting values
	m_HomeY = m_Y;
	m_HomeZ = m_Z;
	m_HomerotX = m_rotX;
	m_HomerotY = m_rotY;
	m_HomerotZ = m_rotZ;

	return 0;
}


/////////////////////////////////////////////////////////////////////////////
void COpenGLView::OnDestroy() 
{
	CView::OnDestroy();
	
	// TODO: Add your message handler code here

//	NOTE:wglDeleteContext makes the RC non-current, so this step is unnecessary
//	(but you can do it if it makes you feel secure)
//    if ( FALSE ==  ::wglMakeCurrent( 0, 0 ) )
//		{
//		SetError(2);
//        return FALSE;
//		}


    if ( FALSE == ::wglDeleteContext( m_hRC ) )
		{
		SetError(6);
 		}

//	For Color-Index mode, you should reset the palette to the original here

    if ( m_pDC )
		{
        delete m_pDC;
		}
}


/////////////////////////////////////////////////////////////////////////////
// OnEraseBkgnd
// OpenGL has its own routine to erase the background. Here we tell MFC
// not to do it, that we'll take care of it. If we didn't the scene might
// flash.	
BOOL COpenGLView::OnEraseBkgnd(CDC* pDC) 
{
	// TODO: Add your message handler code here and/or call default
	
	//	return CView::OnEraseBkgnd(pDC);
	return TRUE; // tell Windows not to erase the background
}

/////////////////////////////////////////////////////////////////////////////
// OnSize
// We need to set up the viewport when the size changes, and this is the best
// place for it, as long as you don't need to render more than one scene, in which
// case you'd have to do it just before each scene gets rendered.
// We also set up the viewing volumn here since we're using perspective mode. For
// Orthographic you could do it anywhere since you don't need the aspect ratio.
// Finally we also set up the default veiwing transform. For an animated scene you'd
// have to do it just before the scene was rendered.

void COpenGLView::OnSize(UINT nType, int cx, int cy) 
{
	CView::OnSize(nType, cx, cy);
	
	// TODO: Add your message handler code here
	GLdouble aspect_ratio; // width/height ratio
	
	if ( 0 >= cx || 0 >= cy )
		{
		return;
		}

	SetupViewport( cx, cy );

	// compute the aspect ratio
	// this will keep all dimension scales equal
	aspect_ratio = (GLdouble)cx/(GLdouble)cy;

	// select the projection matrix and clear it
    ::glMatrixMode(GL_PROJECTION);
    ::glLoadIdentity();

	// select the viewing volumn
	SetupViewingFrustum( aspect_ratio );
	
	// switch back to the modelview matrix and clear it
    ::glMatrixMode(GL_MODELVIEW);
    ::glLoadIdentity();
  
	// now perform any viewing transformations
	SetupViewingTransform();
}


//////////////////////////////////////////////////////////////////////////////
// SetError-error string manipulation

/////////////////////////////////////////////////////////////////////////////
void COpenGLView::SetError( int e )
{
	// if there was no previous error,
	// then save this one
	if ( _ErrorStrings[0] == m_ErrorString ) 
		{
		m_ErrorString = _ErrorStrings[e];
		}
}


//////////////////////////////////////////////////////////////////////////////
// InitializeOpenGL
// - just that. This is set up for RGB mode, though I've indicated
// where you would add code for color-index mode.
BOOL COpenGLView::InitializeOpenGL()
{
	// Can we put this in the constructor?
    m_pDC = new CClientDC(this);

    if ( NULL == m_pDC ) // failure to get DC
		{
		SetError(1);
		return FALSE;
		}

	if (!SetupPixelFormat())
		{
        return FALSE;
		}

//	For Color-Index mode, you'd probably create your palette here, right
//	after you select the pixel format

    if ( 0 == (m_hRC = ::wglCreateContext( m_pDC->GetSafeHdc() ) ) )
		{
		SetError(4);
		return FALSE;
		}

    if ( FALSE == ::wglMakeCurrent( m_pDC->GetSafeHdc(), m_hRC ) )
		{
		SetError(5);
		return FALSE;
		}	

	// get the extended information about the currently running
	// OpenGL implementation
    FetchExtendedInformation();

	// select our default display fonts
	// (this may take a while....)
	SetDefaultFonts();

	// specify black as clear color
    ::glClearColor(0.0f, 0.0f, 0.0f, 0.0f);
	// specify the back of the buffer as clear depth
    ::glClearDepth(1.0f);
	// enable depth testing
    ::glEnable(GL_DEPTH_TEST);

	// Save the initial positions and rotations
	m_HomeX = m_X;
	m_HomeY = m_Y;
	m_HomeZ = m_Y;
	m_HomerotX = m_rotX;
	m_HomerotY = m_rotY;
	m_HomerotZ = m_rotY;

	return TRUE;
}


/////////////////////////////////////////////////////////////////////////////
BOOL COpenGLView::SetupPixelFormat()
{
  static PIXELFORMATDESCRIPTOR pfd = 
	{
        sizeof(PIXELFORMATDESCRIPTOR),  // size of this pfd
        1,                              // version number
        PFD_DRAW_TO_WINDOW |            // support window
          PFD_SUPPORT_OPENGL |          // support OpenGL
          PFD_DOUBLEBUFFER,             // double buffered
        PFD_TYPE_RGBA,                  // RGBA type
        24,                             // 24-bit color depth
        0, 0, 0, 0, 0, 0,               // color bits ignored
        0,                              // no alpha buffer
        0,                              // shift bit ignored
        0,                              // no accumulation buffer
        0, 0, 0, 0,                     // accum bits ignored
        16,                             // 16-bit z-buffer
        0,                              // no stencil buffer
        0,                              // no auxiliary buffer
        PFD_MAIN_PLANE,                 // main layer
        0,                              // reserved
        0, 0, 0                         // layer masks ignored
    };
    
    if ( 0 == (m_PixelFormat = ::ChoosePixelFormat(m_pDC->GetSafeHdc(), &pfd)) )
	    {
		SetError(2);
        return FALSE;
		}

    if ( FALSE == ::SetPixelFormat(m_pDC->GetSafeHdc(), m_PixelFormat, &pfd) )
	    {
       	SetError(3);
        return FALSE;
		}

    return TRUE;
}

/////////////////////////////////////////////////////////////////////////////
// This is a helper function that's used to parse the strings

static const char* GetString( GLenum theEnum  );
// This is a helper function
static const char* GetString( GLenum theEnum  )
{
    char* string;

    // returns a pointer to a STATIC string!
    string = (char*)::glGetString( theEnum );
    if ( GL_NO_ERROR != ::glGetError() ) // failed!
        {
        static char* error = "no-information-available" "";
        string = error;
        }

    return string;

}


/////////////////////////////////////////////////////////////////////////////
void COpenGLView::FetchExtendedInformation()
{


// Get information about the particular pixel format that's running. From this we
// can tell if the OpenGL implementation is strictly software, partially in hardware with
// some of the routines being processed by the generic software driver (the Mini-Client driver)
// or if the entire implementation is in hardware (the Installable Client driver)

	PIXELFORMATDESCRIPTOR pfd;

	// Get information about the DC's current pixel format 
	::DescribePixelFormat( m_pDC->GetSafeHdc(), m_PixelFormat,
		sizeof(PIXELFORMATDESCRIPTOR), &pfd ); 
 
	// Now that we've set up a pixel format, get information about the current
	// driver for this format. You can use this info to determine which pixel format
	// is the fastest. Note that the Mini-Client driver is only available in OpenGL 1.1
	// or later.

	if ( 0 == ( INSTALLABLE_DRIVER_TYPE_MASK & pfd.dwFlags ) )
		m_eCurrentDriver = eInstallableClient; // fully in hardware (fastest)
	else if ( INSTALLABLE_DRIVER_TYPE_MASK == ( INSTALLABLE_DRIVER_TYPE_MASK & pfd.dwFlags ) )
		m_eCurrentDriver = eMiniClient;	// partially in hardware (pretty fast, maybe..)
	else  // plain old generic
		m_eCurrentDriver = eGeneric;	// software

// end of the current driver specific area

// Get general information about the OpenGL driver

//  GL_VENDOR
//  Returns the company responsible for this GL implementation.
//  This name does not change from release to release. 
    m_pVender = GetString( GL_VENDOR );

//  GL_RENDERER
//  Returns the name of the renderer. This name is typically specific
//  to a particular configuration of a hardware platform.
//  It does not change from release to release. 
    m_pRenderer = GetString( GL_RENDERER );

//  GL_VENDOR and GL_RENDERER together uniquely specify a platform,
//  and will not change from release to release. They should be used
//  to identify the platform.

//  GL_VERSION
//  Returns a version or release number. 
    m_pVersion = GetString( GL_VERSION );

//  GL_EXTENSIONS
//  Returns a space-separated list of supported extensions to GL.
//  Because GL does not include queries for the performance
//  characteristics of an implementation, it is expected that some
//  applications will be written to recognize known platforms and will
//  modify their GL usage based on known performance characteristics
//  of these platforms.

//  The format and contents of the string that glGetString() returns depend
//  on the implementation, except that extension names will not include
//  space characters and will be separated by space characters
//  in the GL_EXTENSIONS string, and that all strings are null-terminated.

    char *extension, *c;

	const char* e = GetString( GL_EXTENSIONS );

	// This string is deleted in the destructor
	m_ExtensionPtr = c = new char[ strlen(e)+1 ];

	if ( 0 == m_ExtensionPtr )
		{
		TRACE("Couldn't allocate memory for the extension list.\n");
		return;
		}

	// make a local copy to mess with
	::strcpy( m_ExtensionPtr, e );

	BOOL bDone = FALSE;

	bDone = TRUE;

	while ( !bDone )
		{
		extension = c;
		while ( (0 != *c) && (' ' != *c)  )
			c++;
		if ( 0 == *c )
			bDone = TRUE;
		*c = 0; // terminate the string
		// Now create a new CString object and add it to the array
        m_ExtensionArray.Add( extension );

        // skip to next string
		extension = ++c;
		}

// End of the extended driver information

}	

/////////////////////////////////////////////////////////////////////////////
// SetupViewport
BOOL COpenGLView::SetupViewport( int cx, int cy )
{
	// select the full client area
    ::glViewport(0, 0, cx, cy);

	return TRUE;
}

/////////////////////////////////////////////////////////////////////////////
// SetupViewingFrustum
BOOL COpenGLView::SetupViewingFrustum( GLdouble aspect_ratio )
{
	// select a default viewing volumn
    ::gluPerspective(40.0f, aspect_ratio, .1f, 20.0f);
	// here's an ortho view
//	glOrtho( -2.0f, 2.0f, -2.0f, 2.0f, -.10f, 20.0f );

	return TRUE;
}


/////////////////////////////////////////////////////////////////////////////
// SetupViewingTransform
// This is a static, default viewing transform
BOOL COpenGLView::SetupViewingTransform()
{
	// select a default viewing transformation
	// of a 20 degree rotation about the X axis
	// then a -5 unit transformation along Z
	// (It's as good as any I guess...)
	::glTranslatef( 0.0f, 0.0f, -5.0f );
	::glRotatef( 20.0f, 1.0f, 0.0f, 0.0f );
    return TRUE;
}


/////////////////////////////////////////////////////////////////////////////
// RenderScene
// This is the default scene for the COpenGLView class.
// draw a red wire sphere inside a light blue cube
BOOL COpenGLView::RenderScene()
{
	// rotate the wire sphere so it's vertically
	// oriented
	::glRotatef( 90.0f, 1.0f, 0.0f, 0.0f );
	::glColor3f( 1.0f, 0.0f, 0.0f );
	::auxWireSphere( .5 );
	::glColor3f( 0.5f, 0.5f, 1.0f );
	::auxWireCube( 1.0 );
    return TRUE;
}	


/////////////////////////////////////////////////////////////////////////////
// GetNewDisplayListIDs
// This is just a nice wrapper for the glGenLists() function.
GLuint COpenGLView::GetNewDisplayListIDs( GLsizei range )
{
	ASSERT( 0 < range );

	return ::glGenLists( range );
}

 
/////////////////////////////////////////////////////////////////////////////
// RenderStockScene
// This routine performs a lazy-rendering of the stock scene; i.e. if the
// stock scene isn't yet a disply list, it generates the display list. In any
// event it then renders the disply list.
void COpenGLView::RenderStockScene()
{
	// if the display list for the stock
	// scene hasn't been generated, then do so
	// the selected varaible is only used to change
	// the stock scene
	if ( m_SelectedStockScene )
		{
		GenerateStockScene( m_SelectedStockScene );
		m_SelectedStockScene = eStockSceneSet; // clear it
		}

	if ( 0 != m_StockSceneListIndex )
		{
		::glCallList( m_StockSceneListIndex );
		}
}

 
/////////////////////////////////////////////////////////////////////////////
// GenerateStockScene
// This routine simply generates the display list for the stock scene(s)
BOOL COpenGLView::GenerateStockScene( eStockSceneID id )
{
	// if the display list for the stock
	// scene hasn't been generated, then do so
	// we'll reuse this ID for all stock scenes
	if ( 0 == m_StockSceneListIndex )
		{
		// get a unique id
		m_StockSceneListIndex = GetNewDisplayListID( );

		if ( 0 == m_StockSceneListIndex ) // failed
			{
			return FALSE;
			}
		}

	// we have an ID, so set (or reset) it
	// and create the new stock scene

	::glNewList( m_StockSceneListIndex, GL_COMPILE );
	GenerateThisStockScene( id );
	::glEndList();

	return TRUE;
}


/////////////////////////////////////////////////////////////////////////////
// GenerateThisStockScene
// This routine generates the stock scenes. The functions
// called are simply expected to render a scene
void COpenGLView::GenerateThisStockScene( eStockSceneID id )
{

	switch( id )
		{
		case eStockSceneSet:
		case eStockSceneNone:
			; // an empty list
			break;

		case eStockSceneUserDefined:
			StockSceneUserDefined();
			break;

		case eStockSceneTriangles:
			StockSceneTriangles();
			break;

		case eStockSceneCheckerboard:
			StockSceneCheckerboard();
			break;

		case eStockSceneFlatSurface:
			StockSceneFlatSurface();
			break;

		default:
			break;
		}
}


/////////////////////////////////////////////////////////////////////////////
// StockSceneCheckerboard
// Draw a black and white checkerboard
// NOTE: You can use a quad strip and get better performance
// as long as you don't need smooth shading.
void COpenGLView::StockSceneCheckerboard( )
{
	// define all vertices   X     Y     Z
	GLfloat v0[3], v1[3], v2[3], v3[3], delta;
	int color = 0;

	delta = 0.5f;

	// define the two colors
	GLfloat color1[3] = { 0.9f, 0.9f, 0.9f };
 	GLfloat color2[3] = { 0.05f, 0.05f, 0.05f };

	v0[1] = v1[1] = v2[1] = v3[1] = 0.0f;

	::glBegin( GL_QUADS );

	for ( int x = -5 ; x <= 5 ; x++ )
		{
		for ( int z = -5 ; z <= 5 ; z++ )
			{
			::glColor3fv( (color++)%2 ? color1 : color2 );
		
			v0[0] = 0.0f+delta*z;
			v0[2] = 0.0f+delta*x;

			v1[0] = v0[0]+delta;
			v1[2] = v0[2];

			v2[0] = v0[0]+delta;
			v2[2] = v0[2]+delta;

			v3[0] = v0[0];
			v3[2] = v0[2]+delta;

			::glVertex3fv( v0 );
			::glVertex3fv( v1 );
			::glVertex3fv( v2 );
			::glVertex3fv( v3 );
			}
		}
	::glEnd();	
	
}

/////////////////////////////////////////////////////////////////////////////
// StockSceneTriangles
// Draw a square surface of red and blue triangles
// all touching the origin.
void COpenGLView::StockSceneTriangles( )
{
	// define all vertices   X     Y     Z
	GLfloat surface0[3] = { 0.0f, 0.0f, 0.0f };
	GLfloat surface1[3] = {+5.0f, 0.0f, 0.0f };
	GLfloat surface2[3] = {+5.0f, 0.0f,-5.0f };
	GLfloat surface3[3] = { 0.0f, 0.0f,-5.0f };
	GLfloat surface4[3] = {-5.0f, 0.0f,-5.0f };
	GLfloat surface5[3] = {-5.0f, 0.0f, 0.0f };
	GLfloat surface6[3] = {-5.0f, 0.0f,+5.0f };
	GLfloat surface7[3] = { 0.0f, 0.0f,+5.0f };
	GLfloat surface8[3] = {+5.0f, 0.0f,+5.0f };
	GLfloat surface9[3] = {+5.0f, 0.0f, 0.0f };

	// define the two colors
	GLfloat color1[3] = { 0.5f, 0.0f, 0.0f };
 	GLfloat color2[3] = { 0.0f, 0.0f, 0.5f };

	::glBegin( GL_TRIANGLES );
		::glColor3fv( color1 );
		::glVertex3fv( surface0 );
		::glVertex3fv( surface1 );
		::glVertex3fv( surface2 );
		::glColor3fv( color2 );
		::glVertex3fv( surface0 );
		::glVertex3fv( surface2 );
		::glVertex3fv( surface3 );
		::glColor3fv( color1 );
		::glVertex3fv( surface0 );
		::glVertex3fv( surface3 );
		::glVertex3fv( surface4 );
		::glColor3fv( color2 );
		::glVertex3fv( surface0 );
		::glVertex3fv( surface4 );
		::glVertex3fv( surface5 );
		::glColor3fv( color1 );
		::glVertex3fv( surface0 );
		::glVertex3fv( surface5 );
		::glVertex3fv( surface6 );
		::glColor3fv( color2 );
		::glVertex3fv( surface0 );
		::glVertex3fv( surface6 );
		::glVertex3fv( surface7 );
		::glColor3fv( color1 );
		::glVertex3fv( surface0 );
		::glVertex3fv( surface7 );
		::glVertex3fv( surface8 );
		::glColor3fv( color2 );
		::glVertex3fv( surface0 );
		::glVertex3fv( surface8 );
		::glVertex3fv( surface9 );
	::glEnd();	
	
}



/////////////////////////////////////////////////////////////////////////////
// StockSceneFlatSurface
// Draw a dark-grey square surface
void COpenGLView::StockSceneFlatSurface( void )
{
	// define vertices 
	GLfloat v0[2], v1[2];
	
	// define the color (not too black!)
	GLfloat color0[3] = { 0.2f, 0.2f, 0.2f };
 	
	v0[0] = v1[1] = -5.0f;
	v0[1] = v1[0] =  5.0f;

	::glColor3fv( color0 );
	
	// glRect draws on the z=0 plane and we want it on the
	// y=0 plane, so rotate about the x axis

	::glRotatef( -90.0f, 1.0f, 0.0f, 0.0f );

	::glRectfv( v0, v1 );
	
}

/////////////////////////////////////////////////////////////////////////////
// Draw3DAxes
// Draws lines along the current 3 axes from "start" units to "finish", with 
// "ticks" tickmarks spaced out along it.
void COpenGLView::Draw3DAxes( float start, float finish, int ticks )
{
	// make sure that start < finish
	if ( start > finish )
		{
		float temp = start;
		start = finish;
		finish = start;
		}

	// if ticks < 0 and delta is larger than 1, place the ticks
	// on each scales unit length
	if ( 0 > ticks )
		{
		float delta = finish-start;
		ticks = delta > 1.0 ? (int)delta : 0;
		}

	// draw the tickmarked axes
	Draw3DAxesLine( start, finish, 0, ticks );
	Draw3DAxesLine( start, finish, 1, ticks );
	Draw3DAxesLine( start, finish, 2, ticks );
}

/////////////////////////////////////////////////////////////////////////////
// Draw3DAxesLine
// This routine draws a colored line along a specified axis.
// axis_id = 0 for the x, 1 for the y, and anything else for the z
// start and finish are the starting and ending location, start < finish.
// ticks is the number of ticks to place along the axis.
// If you are using lighting/materials, you might want to wrapper this routine
// so that it's called with lighting disabled, or else the axis lines will be effected
// by lighting claculations - which generally means hard to see.
void COpenGLView::Draw3DAxesLine( float start, float finish, int axis_id, int ticks )
{
	float *px, *py, *pz, zero = 0.0f;
	float tickx, ticky, tickz;
	float *pdx, *pdy, *pdz, tinytick, delta = (finish-start)/(ticks<1?1:ticks);
	GLfloat negativeColor[3] = { 1.0f, 0.0f, 0.0f };
	GLfloat positiveColor[3] = { 0.0f, 1.0f, 0.0f };

	pdx = pdy = pdz = px = py = pz = &zero;
	tickx = ticky = tickz = 0.0f;
	tinytick = 0.05f;

	// select which of the 3 axes is going to vary
	if ( 0 == axis_id ) // X axis
		{
		pdx = &delta;	  
		ticky = tinytick;	  
		px = &start;	  
		}
	else if ( 1 == axis_id ) // Y axis
		{
		pdy = &delta;	  
		tickx = tinytick;	  
		py = &start;	  
		}
	else 	// default Z axis
		{
		pdz = &delta;	  
		ticky = tinytick;	  
		pz = &start;	  
		}

	// turn off the lighting effects
	// since we don't want the axes lines effected by the
	// lighting. You might need to call ::glDisable(GL_LIGHTING)
	// before this routine

	::glBegin(GL_LINES);

	// now draw the two lines that make up the axis
	::glColor3fv( negativeColor ); // negative color
	::glVertex3f( *px, *py, *pz );
	::glVertex3f( 0.0f, 0.0f, 0.0f );

	::glColor3fv( positiveColor ); // positive color
	::glVertex3f( 0.0f, 0.0f, 0.0f );
	::glVertex3f( *px+*pdx*ticks, *py+*pdy*ticks, *pz+*pdz*ticks );

	// now draw the tick marks
	for (int i = 0; i < ticks  ; i++ )
		{
		if ( i < ticks/2 )
			{
			::glColor3fv( negativeColor );
			}
		else
			{
			::glColor3fv( positiveColor );
			}

		::glVertex3f( *px-tickx, *py-ticky, *pz-tickz );
		::glVertex3f( *px+tickx, *py+ticky, *pz+tickz );

		*px += *pdx;
		*py += *pdy;
		*pz += *pdz;
		}

	::glEnd();

	// don't forget to turn lighting effects back on
	// via glEnable(GL_LIGHTING)

}


/////////////////////////////////////////////////////////////////////////////
// COpenGLView Text-Specific routines follow
/////////////////////////////////////////////////////////////////////////////


/////////////////////////////////////////////////////////////////////////////
// GenerateDefaultFonts
// Generate the default 3D and flat text disp;ay lists
void COpenGLView::GenerateDefaultFonts( char* font3D, char* fontFlat )
{
   m_DefaultTextID		= GenerateDisplayListForFont( font3D );
   m_DefaultFlatTextID	= GenerateBitmapListForFont( fontFlat );
}

/////////////////////////////////////////////////////////////////////////////
// GLTextOut
// Output text (either Bitmap or Outline)
void COpenGLView::GLTextOut( GLuint id, const char * const textstring )
{
	if ( 0 == id || 0 == textstring )
		{
		return;
		}

	GLsizei size = strlen( textstring );

	::glListBase( id );
	::glCallLists( size, GL_UNSIGNED_BYTE, (const GLvoid*)textstring ); 

} 

/////////////////////////////////////////////////////////////////////////////
// GenerateDisplayListForFont
// The routines used for generating 3D text.
GLuint COpenGLView::GenerateDisplayListForFont( char* fontname, double xt )
{
	GLuint id;

	if ( 0 == m_pDC ||
		 0 == fontname || 
		 (GLuint)0 == (id = GetNewDisplayListIDs(256)) )
		{
        return 0;
		}

	LOGFONT     logfont;
	GLYPHMETRICSFLOAT gmf[256];

	// lfHeight can't be used to change the font size
	logfont.lfHeight		= -12; // use glScale to change size
	logfont.lfWidth			= 0;
	logfont.lfEscapement	= 0;
	logfont.lfOrientation	= logfont.lfEscapement;
	logfont.lfWeight		= FW_NORMAL;
	logfont.lfItalic		= FALSE;
	logfont.lfUnderline		= FALSE;
	logfont.lfStrikeOut		= FALSE;
	logfont.lfCharSet		= ANSI_CHARSET;
	logfont.lfOutPrecision	= OUT_TT_ONLY_PRECIS;
	logfont.lfClipPrecision	= CLIP_DEFAULT_PRECIS;
	logfont.lfQuality		= DEFAULT_QUALITY;
	logfont.lfPitchAndFamily = FF_DONTCARE|DEFAULT_PITCH;
	lstrcpy ( logfont.lfFaceName, fontname );

	CFont newfont;

	// returns 0 if it fails
	BOOL success = newfont.CreateFontIndirect( &logfont );
	CFont* oldfont =	m_pDC->SelectObject( &newfont );	
	ASSERT( 0 != oldfont );

    // Create a set of display lists based on the glyphs of the TrueType font 
	// notice that we really waste the first 32 spaces....
	// if there's a problem delete the display lists
	// Note that this single call takes MOST of the initialization time
	// for the COpenGLView class (probably a couple of seconds!) so if you don't
	// want/need 3D text, you might comment this functionality out.
	if (	0 == success ||
		FALSE == ::wglUseFontOutlines( m_pDC->m_hDC, 0, 256, id, 0.0f,
			(float)xt, WGL_FONT_POLYGONS,  gmf) )
		{
		::glDeleteLists( id, 256 );
		id  = 0;
		}
	else
		{
		m_pDC->SelectObject( oldfont );
		}

	return id;
}

/////////////////////////////////////////////////////////////////////////////
// GenerateBitmapListForFont
// The routines used for generating flat text
GLuint COpenGLView::GenerateBitmapListForFont( char* fontname )
{
	GLuint id;

	if ( 0 == m_pDC ||
		 (GLuint)0 == (id = GetNewDisplayListIDs(256)) )
		{
        return 0;
		}

	CFont newfont;
	CFont* oldfont;
	BOOL success;

	if ( NULL != fontname )
		{

		LOGFONT     logfont;
	
		logfont.lfHeight		= -12;
		logfont.lfWidth			= 0;
		logfont.lfEscapement	= 0;
		logfont.lfOrientation	= logfont.lfEscapement;
		logfont.lfWeight		= FW_NORMAL;
		logfont.lfItalic		= FALSE;
		logfont.lfUnderline		= FALSE;
		logfont.lfStrikeOut		= FALSE;
		logfont.lfCharSet		= ANSI_CHARSET;
		logfont.lfOutPrecision	= OUT_DEFAULT_PRECIS;
		logfont.lfClipPrecision	= CLIP_DEFAULT_PRECIS;
		logfont.lfQuality		= DEFAULT_QUALITY;
		logfont.lfPitchAndFamily = FF_DONTCARE|DEFAULT_PITCH;
		lstrcpy ( logfont.lfFaceName, fontname );

		// returns 0 if it fails
		success = newfont.CreateFontIndirect( &logfont );
		oldfont =	m_pDC->SelectObject( &newfont );
		ASSERT( 0 != oldfont );
		}
	else
		{
		// make the system font the device context's selected font
		oldfont =	(CFont*)m_pDC->SelectStockObject( SYSTEM_FONT );
		ASSERT( 0 != oldfont );
		}
	
    // Create a set of display lists based on the glyphs of the font 
	// notice that we really waste the first 32 spaces....
	// if there's a problem delete the display lists
   if (	0 == success ||
		FALSE == ::wglUseFontBitmaps( m_pDC->m_hDC, 0, 256, id ) )
		{
		::glDeleteLists( id, 256 );
		id  = 0;
		}
   else
		{
		m_pDC->SelectObject( oldfont );
		}

   return id;
}


/////////////////////////////////////////////////////////////////////////////
// Animation routines
/////////////////////////////////////////////////////////////////////////////


/////////////////////////////////////////////////////////////////////////////
void COpenGLView::OnAnimate() 
{
	// TODO: Add your command handler code here
	ToggleAnimationFlag();

	if ( m_bAnimationRunning )
		{
		InvalidateRect( 0, FALSE );
		UpdateWindow();
		}
	else
		{
		// turn off animation message
		GetParentFrame()->SetWindowText( (const char*)m_WindowTitle );
		}
}

/////////////////////////////////////////////////////////////////////////////
void COpenGLView::OnUpdateAnimate(CCmdUI* pCmdUI) 
{
	// TODO: Add your command update UI handler code here
	pCmdUI->SetCheck( m_bAnimationRunning );
}

/////////////////////////////////////////////////////////////////////////////
// CalculateDynamics
// This routine uses simple velocity-based values to change the viewpoint.
// Thus there is no acceleration component, just a velocity component.
// Most people can't get used to an acceleration-based control as it's
// too difficult to get used to. This setup is much more predictable.
//
// The values used for velocity, rotation, and acceleration are
// all in units per second. This allows you to pick reasonable,
// intuitive values.
void COpenGLView::CalculateDynamics()
{
	ASSERT( AnimationRunning() );

   	double eTime = (double)ElapsedTimeinMSSinceLastRender()/1000.0;

	// Limit translational acceleration values
	if ( m_deltaZ > m_MaxVelocity )
		m_deltaZ = m_MaxVelocity;
 	if ( m_deltaX > m_MaxVelocity )
		m_deltaX = m_MaxVelocity;
	if ( m_deltaY > m_MaxVelocity )
		m_deltaY = m_MaxVelocity;
   	if ( -m_deltaZ > m_MaxVelocity )
		m_deltaZ = -m_MaxVelocity;
 	if ( -m_deltaX > m_MaxVelocity )
		m_deltaX = -m_MaxVelocity;
	if ( -m_deltaY > m_MaxVelocity )
		m_deltaY = -m_MaxVelocity;

	// Limit rotational acceleration values
	if ( m_rotdeltaZ > m_MaxRotation )
		m_rotdeltaZ = m_MaxRotation;
 	if ( m_rotdeltaX > m_MaxRotation )
		m_rotdeltaX = m_MaxRotation;
	if ( m_rotdeltaY > m_MaxRotation )
		m_rotdeltaY = m_MaxRotation;
   	if ( -m_rotdeltaZ > m_MaxRotation )
		m_rotdeltaZ = -m_MaxRotation;
 	if ( -m_rotdeltaX > m_MaxRotation )
		m_rotdeltaX = -m_MaxRotation;
	if ( -m_rotdeltaY > m_MaxRotation )
		m_rotdeltaY = -m_MaxRotation;


	// This is where the actual values for rotation and
	// acceleration change. Note that I'm really ignoring the acceleration
	// values. Most people simply find changing velocity by
	// tweaking the acceleration values too hard to understand, so I
	// take the easy way out and hook up the user controls to the velocity
	// instead. If you're building a space simulation, then you might want to
	// redo these equations.

	// (Actually, when I said I'm ingoring the acceleration values,
	// I really meant that I was using the acceleration variables in place
	// of the velocity variables in these equations. I do this so that if you
	// wanted to correctly use acceleration to tweak velocity (via the UI)
	// and then velocity to tweak position, only these equations would change.
	m_X += m_accelX * eTime;
  	m_Y += m_accelY * eTime;
	m_Z += m_accelZ * eTime;

	m_rotX += m_rotaccelX * eTime;
  	m_rotY += m_rotaccelY * eTime;
	m_rotZ += m_rotaccelZ * eTime;

}


/////////////////////////////////////////////////////////////////////////////
// ViewpointOriginRotation
// This viewpoint spins about the origin and accepts
// changes in rotation rates and distance from the origin.
// The distance to the origin can be varied
BOOL COpenGLView::ViewpointOriginRotation()
{

    ::glLoadIdentity();

	// Move away from the origin
	::glTranslatef( 0.0f, 0.0f, (GLfloat)m_Z );

	// Rotate about the origin
	::glRotatef( (GLfloat)m_rotX, 1.0f, 0.0f, 0.0f );
	::glRotatef( (GLfloat)m_rotY, 0.0f, 1.0f, 0.0f );
	::glRotatef( (GLfloat)m_rotZ, 0.0f, 0.0f, 1.0f );

    return TRUE;
}


/////////////////////////////////////////////////////////////////////////////
// OnKeyDown
// This routine runs the keyboard interface.
// Most of the keys accepted are from the numeric keypad

// HOME: Resets viewpoint to initial values (animation keeps going)
// SHIFT-HOME: reset viewpoint & clear movement vectors
// 5: Clear movement vectors
// Up/Down Arrows: change velocity in Z (if shifted, change rotation)
// Left/Right Arrows: change velocity in X (if shifted, change rotation)
// Plus/Minus: change velocity in Y (if shifted, change rotation)
// ESC: Exit animation

void COpenGLView::OnKeyDown(UINT nChar, UINT nRepCnt, UINT nFlags) 
{
	// TODO: Add your message handler code here and/or call default
	BOOL ctrl = 0x8000 & GetKeyState( VK_CONTROL );
	BOOL shift= 0x8000 & GetKeyState( VK_SHIFT );

	if ( AnimationRunning() )
		{
		switch ( nChar )
			{
			case VK_HOME: // reset position to viewpoint's origin
				m_X = m_HomeX;
				m_Y = m_HomeY;
				m_Z = m_HomeZ;
				m_rotX = m_HomerotX;
				m_rotY = m_HomerotY;
				m_rotZ = m_HomerotZ;

			 	if ( !shift )  // drop through if shifted
					return;
				// i.e shift home resets viewpoint AND movement

			case VK_CLEAR: // numeric 5 keypad key
				// kill velocity & acceleration & rotation
			  	m_deltaX = m_deltaY = m_deltaZ = 0.0f;
			  	m_accelX = m_accelY = m_accelZ = 0.0f;
			  	m_rotdeltaX = m_rotdeltaY = m_rotdeltaZ = 0.0f;
			  	m_rotaccelX = m_rotaccelY = m_rotaccelZ = 0.0f;
				return;
   			case VK_UP: // up arrow
				shift ?
					IncreaseRotation( m_rotaccelZ ):
					IncreaseVelocity(m_accelZ);
					return;
			case VK_DOWN: // down arrow
				shift ?
					DecreaseRotation( m_rotaccelZ ):
					DecreaseVelocity(m_accelZ);
					return;
 			case VK_LEFT: // left arrow
				shift ?
					IncreaseRotation( m_rotaccelX ):
					IncreaseVelocity(m_accelX);
					return;
			case VK_RIGHT: // right arrow
				shift ?
					DecreaseRotation( m_rotaccelX ):
					DecreaseVelocity(m_accelX);
				return;
 			case VK_SUBTRACT: // minus 
				shift ?
					IncreaseRotation( m_rotaccelY ):
					IncreaseVelocity( m_accelY );
				return;
			case VK_ADD: // plus
				shift ?
					DecreaseRotation( m_rotaccelY ):
					DecreaseVelocity( m_accelY );
				return;
		}
	// end if animated
	}

	switch ( nChar )
	   {
	   case VK_ESCAPE:	// close down the app
			GetParent()->PostMessage(WM_CLOSE);
			return;
	   default:
			return;
	   }

	// Probably don't ever need this
	//CView::OnKeyDown(nChar, nRepCnt, nFlags);
}
