/**************************************************
   solar.c

   Program to demonstrate how to use a local
   coordinate method to position parts of a 
   model in relation to other model parts.
   
   Pressing the "a" key toggles the animation
   Pressing the up and down arrow keys will
   increase/decrease the animation rate

**************************************************/

#include "glos.h" // MS specifc stuff

#include <GL/gl.h> // system OpenGL includes
#include <GL/glu.h>
#include <GL/glaux.h>

static GLenum spinMode = GL_TRUE;

void OpenGLInit(void);

static void CALLBACK Animate(void );
static void CALLBACK Key_a(void );
static void CALLBACK Key_up(void );
static void CALLBACK Key_down(void );
static void CALLBACK ResizeWindow(GLsizei w, GLsizei h);

static int HourOfDay = 0, DayOfYear = 0;
static int AnimateIncrement = 24; // in hours


static void CALLBACK Key_a(void)
{
    spinMode = !spinMode;
}

static void CALLBACK Key_up(void)
{
    AnimateIncrement *= 2;
	if ( 0 == AnimateIncrement )
		AnimateIncrement = 1;
}

static void CALLBACK Key_down(void)
{
    AnimateIncrement /= 2;
	
}

static void CALLBACK Animate(void)
{
	// clear the redering window
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

    if (spinMode)
		{
		// calc animation parameters
        HourOfDay += AnimateIncrement;
        DayOfYear += HourOfDay/24;

        HourOfDay = HourOfDay%24;
        DayOfYear = DayOfYear%365;
		}
	
	// clear current matrix (Modelview)
    glLoadIdentity();
	// back off six units 
    glTranslatef ( 0.0f, 0.0f, -5.0f );
	// rotate the plane of the elliptic
	// (rotate the model's plane about the
	// x axis by five degrees)
	glRotatef( 5.0f, 1.0f, 0.0f, 0.0f );

    // draw the sun
    glColor3f( 1.0, 1.0, 1.0 );
    auxWireSphere( 1.0 );

    // draw the Earth
    glRotatef( (GLfloat)(360.0*DayOfYear/365.0),
		0.0, 1.0, 0.0 );
    glTranslatef( 4.0, 0.0, 0.0 );
    glPushMatrix(); // save matrix state
	glRotatef( (GLfloat)(360.0*HourOfDay/24.0) 
		  , 0.0, 1.0, 0.0 );
    glColor3f( 0.2, 0.2, 1.0 );
    auxWireSphere( 0.2 );
    glPopMatrix(); // restore matrix state

   	glRotatef( (GLfloat)(360.0*12.5*DayOfYear/365.0),
		0.0, 1.0, 0.0 );
    glTranslatef( 0.5, 0.0, 0.0 );
    glColor3f( 0.3, 0.3, 0.3 );
    auxWireSphere( 0.05 );

	// flush the pipeline, swap the buffers
    glFlush();
    auxSwapBuffers();

}

// initialize OpenGL
void OpenGLInit(void)
{
    glShadeModel( GL_FLAT );
    glClearColor( 0.0f, 0.0f, 0.0f, 0.0f );
    glClearDepth( 1.0f );
	glDepthFunc( GL_LEQUAL );
    glEnable( GL_DEPTH_TEST );
}

// called when the window is resized
static void CALLBACK ResizeWindow(GLsizei w, GLsizei h)
{
    h = (h == 0) ? 1 : h;
	w = (w == 0) ? 1 : w;
	glViewport( 0, 0, w, h );
    glMatrixMode( GL_PROJECTION );
    glLoadIdentity();
    gluPerspective( 45.0, (GLfloat)w/(GLfloat)h, 1.0f, 20.0f );
	// select the Modelview matrix
    glMatrixMode( GL_MODELVIEW );
}


// main routine
// set up OpenGL, hook up callbacks
// and start the main loop
int main( int argc, char** argv )
{
	// we're going to animate it, so double buffer 
    auxInitDisplayMode(AUX_DOUBLE | AUX_RGB );
    auxInitPosition( 0, 0, 620, 160 );
    auxInitWindow( "Solar System Example" );

	// Initialize OpenGL as we like it..
    OpenGLInit();

	// set up callback functions
    auxKeyFunc( AUX_UP, Key_up ); // faster
    auxKeyFunc( AUX_DOWN, Key_down ); // slower
	auxKeyFunc( AUX_a, Key_a ); //animate
    auxReshapeFunc( ResizeWindow );

	// call this when idle
    auxIdleFunc( Animate );
	// call this in main loop
	auxMainLoop( Animate );

    return(0);
}
