/////////////////////////////////////////////////////////////////////////////////////////////
// CDX demo - Fullscreen bump mapping demo & keyboard/mouse input
// You must link to winmm.lib, ddraw.lib, dinput.lib dxguid.lib and cdx.lib
/////////////////////////////////////////////////////////////////////////////////////////////

// Olivier Couvreur : ocouvreur@ubisoft.fr

// This tiny demo was inpired by a program found on ftp:\\x2ftp.oulu.fi : Sorry, I don't remind the author !
// CDX lib is very cool because it simplifies greatly use of DirectX.
// I would like to try it, so it's my little contribution.

// Moreover it's working (except DirectInput => use bInputOk flag ) with Windows NT 4.0 with service pack 3 (emulation only => slower)

// FPS is displayed by pressing SPACE key
// You can move the light with mouse or arrow keys

// Not too bad : There is still a big problem : Why the screen becomes black
// after pressing ALT+TAB (task switching => application minimized) and maximizing 
// the application ? 
// Could someone help me ? What is missing ?
// May be the surface containing HEIGHT.BMP is lost ?

// Allow warning level 4 checking without those ugly windows warnings !
#pragma warning (disable : 4244 4201 4514 )

#include <windows.h>
#include <windowsx.h>
#include <stdio.h>
#include <math.h>

#include "CDX.h"

CDXScreen* Screen;  // The screen object, every program must have one
CDXSurface* Surface;// The surface object holds height map for bump mapping
CDXInput Input;     // The input object

HWND hWndApp;

// Is the program running?
BOOL bActive = TRUE;  

// Is DirectInput Ok
BOOL bInputOk=FALSE;

////////////////////////////////////////////////////////////////////////////////////////////
#define WSCR 320
#define HSCR 200
#define HALF_WSCR (WSCR>>1)
#define HALF_HSCR (HSCR>>1)

// Precomputed Lite Map in InitBump
#define WLITE 256
#define HLITE 256
#define HALF_WLITE (WLITE>>1)
#define HALF_HLITE (HLITE>>1)
char LiteMap[WLITE*HLITE];

// Mouse coordinates
#define DX 2
#define DY 2
int MouseX=HALF_WSCR,MouseY=HALF_HSCR;
int MouseDX=0,MouseDY=0;

// Frame rate counters
BOOL FpsFlag=0;
DWORD LastTime = 0;
DWORD CurTime = 0;
DWORD FpsTime = 0;
DWORD DeltaTime = 0;
DWORD FramesRendered = 0;
DWORD Fps = 0;

// Precompute Lite Map : A kind of phong light model : I could store only a quarter but i'm too lazy
void InitBump()
{
  float nX,nY,nZ;
  int x,y;

  // sum must be 255
#define A 191
#define B 64
  for (y=0;y<HLITE;y++)
  {
    for (x=0;x<WLITE;x++)
    {
      nX=(float) (x-HALF_WLITE)/(float) (HALF_WLITE);
      nY=(float) (y-HALF_HLITE)/(float) (HALF_HLITE);
      nZ=1.0-sqrt(nX*nX+nY*nY);
      if (nZ<0.0) nZ=0.0;
      nZ=nZ*(A+nZ*nZ*nZ*nZ*nZ*nZ*nZ*nZ*B);
      LiteMap[x+y*WLITE]=(char)(nZ<=255.0?nZ:255);
    }
  }
#undef B
#undef A
}

// Do the bump : this is not an optimized version !
void Bump(CDXSurface *HeightSurface,CDXSurface *BackBuffer,int center_x,int center_y)
{
  int x,y;
  int nx=0,ny=0;
  int i,j;
  int lHeightMapPitch,lScreenBufferPitch;
  unsigned char *HeightMapPtr,*ScreenBufferPtr;
  unsigned char *LiteMapPtr=(unsigned char*) &LiteMap[0];
  
  // Very important to accces surfaces content
  HeightSurface->Lock();
  BackBuffer->Lock();

  HeightMapPtr=(unsigned char*) HeightSurface->m_DDSD.lpSurface;
  ScreenBufferPtr=(unsigned char*) BackBuffer->m_DDSD.lpSurface;

  lHeightMapPitch=HeightSurface->m_DDSD.lPitch;
  lScreenBufferPitch=BackBuffer->m_DDSD.lPitch;

  i=lHeightMapPitch+1;
  j=lScreenBufferPitch+1;
  for(y=1;y<HSCR-1;y++)
  {
    for(x=1;x<WSCR-1;x++,i++,j++)
    {
	  nx=(HeightMapPtr[i+1]-HeightMapPtr[i-1])-(x-center_x)+HALF_WLITE;
      if (nx>=WLITE || nx<0)
      {
		ScreenBufferPtr[j]=0;
		continue;
      }

      ny=(HeightMapPtr[i+lHeightMapPitch]-HeightMapPtr[i-lHeightMapPitch])-(y-center_y)+HALF_HLITE;
      if (ny>=HLITE || ny<0)
      {
		ScreenBufferPtr[j]=0;
		continue;
      }

      ScreenBufferPtr[j]=LiteMapPtr[nx+ny*WLITE];
    }
    i+=2+(lHeightMapPitch-WSCR);
	j+=2+(lScreenBufferPitch-WSCR);
  }

  // Very important too : never forget these lines
  BackBuffer->UnLock();
  HeightSurface->UnLock();
}

/////////////////////////////////////////////////////////////////////////////////////////////
// WinProc
/////////////////////////////////////////////////////////////////////////////////////////////
long PASCAL WinProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	switch(message)
	{
		case WM_ACTIVATEAPP:
            bActive = wParam;
			break;
		
		case WM_SETCURSOR:
			if (bActive)
			{
				SetCursor(NULL);
				return 1;
			}
			break;

		case WM_KEYDOWN:
			switch(wParam)
			{
				case VK_ESCAPE:
					PostMessage(hWnd, WM_CLOSE, 0, 0);
					break;

				case VK_SPACE:
					if (!bInputOk)
					{
					  FpsFlag ^=1;
					}
					break;

				case VK_LEFT:
					if (!bInputOk)
					{
                      MouseDX=-DX;
					}
					break;

				case VK_RIGHT:
					if (!bInputOk)
					{
                      MouseDX=+DX;
					}
					break;

				case VK_UP:
					if (!bInputOk)
					{
                      MouseDY=-DY;
					}
					break;

				case VK_DOWN:
					if (!bInputOk)
					{
                      MouseDY=+DY;
					}
					break;
			}
			break;

		case WM_MOUSEMOVE:
			if (!bInputOk)
			{
              MouseDX=0;
			  MouseDY=0;
			  MouseX=LOWORD(lParam);
			  MouseY=HIWORD(lParam);  
			}
			break;

		case WM_DESTROY:
			PostQuitMessage(0);
			break;
	}

	return DefWindowProc(hWnd, message, wParam, lParam);
}

/////////////////////////////////////////////////////////////////////////////////////////////
// InitApp - Create the window and the CDX objects
/////////////////////////////////////////////////////////////////////////////////////////////
char szAppName[]="Bump Mapping Demo";
BOOL InitApp(HINSTANCE hInst, int nCmdShow)
{
	WNDCLASS WndClass;

	if ((hWndApp=FindWindow(szAppName,0))==NULL)
	{
		WndClass.style = CS_DBLCLKS | CS_HREDRAW | CS_VREDRAW;
		WndClass.lpfnWndProc = WinProc;
		WndClass.cbClsExtra = 0;
		WndClass.cbWndExtra = 0;
		WndClass.hInstance = hInst;
		WndClass.hIcon = LoadIcon(hInst, "AppIcon");
		WndClass.hCursor = LoadCursor(0, IDC_ARROW);
		WndClass.hbrBackground = GetStockObject(BLACK_BRUSH);
		WndClass.lpszMenuName = 0;
		WndClass.lpszClassName = szAppName;
		if (!RegisterClass(&WndClass))
		{
		  return(FALSE);
		}
    }
	else
	{
      if (IsIconic(hWndApp))
	  {
        ShowWindow(hWndApp,SW_RESTORE);
		SetForegroundWindow(hWndApp);
	  }
	  return(TRUE);
	}

	hWndApp = CreateWindowEx(
			WS_EX_TOPMOST,
			szAppName,
			szAppName,
			WS_POPUP | WS_CAPTION | WS_SYSMENU,
			WSCR,//0,
			HSCR,//0,
			WSCR,//GetSystemMetrics(SM_CXSCREEN),
			HSCR,//GetSystemMetrics(SM_CYSCREEN),
			NULL,
			NULL,
			hInst,
			NULL);

	if(!hWndApp) return FALSE;

	ShowWindow(hWndApp, nCmdShow);
	UpdateWindow(hWndApp);

	// Create the CDXSreen object
	Screen = new CDXScreen();
	Screen->CreateFullScreen(hWndApp, WSCR, HSCR, 8);

	// Load the palette from the bitmap
	Screen->LoadPalette("HEIGHT.BMP");

	// Load the height map
	Surface = new CDXSurface(Screen,"HEIGHT.BMP");

	// choose font
	Screen->GetBack()->ChangeFont("Times New Roman", 0, 18, FW_NORMAL);

	// Create input devices
	bInputOk=Input.Create(hInst, hWndApp);
    if (!bInputOk) SetCursorPos(MouseX,MouseY);
	
	// Init Bump Mapping
	InitBump();

	return TRUE;
}

/////////////////////////////////////////////////////////////////////////////////////////////
// WinMain
/////////////////////////////////////////////////////////////////////////////////////////////
int PASCAL WinMain(HINSTANCE hInst, HINSTANCE hPrevInst, LPSTR lpCmdLine, int nCmdShow)
{
	MSG msg;

	hPrevInst=hPrevInst;
	lpCmdLine=lpCmdLine;
	if(!InitApp(hInst, nCmdShow)) return FALSE;

	for(;;)
	{
		if(PeekMessage(&msg, NULL, 0, 0, PM_NOREMOVE))
		{
			if(!GetMessage(&msg, NULL, 0, 0 )) 
			{

				return msg.wParam;
			}
			TranslateMessage(&msg); 
			DispatchMessage(&msg);
		}
		else if (bActive)
		{
			// Refresh the input devices
			if (bInputOk)
			{
				Input.Update();  

				// Read Input X
				if(Input.Keys[DIK_LEFT])  MouseDX=-DX;
				else if(Input.Keys[DIK_RIGHT]) MouseDX=+DX;
				else MouseDX=Input.Mouse.x;

				// Read Input Y
				if(Input.Keys[DIK_UP])    MouseDY=-DY;
				else if(Input.Keys[DIK_DOWN])  MouseDY=+DY;
				else  MouseDY=Input.Mouse.y;          

				// FPS Flags
				if (Input.Keys[DIK_SPACE]) FpsFlag ^=1;
			}

			// Clip X move
			MouseX+=MouseDX;
			if (MouseX<0) MouseX=0;
			if (MouseX>=WSCR) MouseX=WSCR-1;

			// Clip Y move
			MouseY+=MouseDY;
			if (MouseY<0) MouseY=0;
			if (MouseY>=HSCR) MouseY=HSCR-1;

			// Draw the picture into Back surface
  		    Bump(Surface,Screen->GetBack(),MouseX,MouseY);

			// Frame rate calculations
			CurTime   = timeGetTime();
			DeltaTime = CurTime - LastTime;
			LastTime  = CurTime;
			FpsTime  += DeltaTime;

			FramesRendered++;

			if (FpsTime > 1000)
			{
				Fps = FramesRendered;
				FramesRendered  = 0;
				FpsTime = 0;
			}

			// Print the frame rate to the screen
			if (FpsFlag)
			{
	 			char str[12];
				sprintf(str, "FPS: %d", Fps);

				Screen->GetBack()->GetDC();
				Screen->GetBack()->SetFont();
				Screen->GetBack()->TextXY(5, 5, RGB(255, 255, 255), str);
				Screen->GetBack()->ReleaseDC();      
			}

			// Then display it
			Screen->Flip();  
		}
		else WaitMessage();
	}
}
